﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Log.h>
#include <nn/nfc.h>

#define NFC_PASSTHROUGHDEMO_ENABLE_LOG

#if defined( NFC_PASSTHROUGHDEMO_ENABLE_LOG )
#define NFC_PASSTHROUGHDEMO_LOG(...)     NN_LOG("NFC_PASSTHROUGHDEMO: " __VA_ARGS__)
#else
#define NFC_PASSTHROUGHDEMO_LOG(...)
#endif // defined( NFC_PASSTHROUGHDEMO_ENABLE_LOG )

namespace nns { namespace nfc { namespace pt {

    /*!
        @brief      NFC デバイスの最大数です。
     */
    const int DeviceCountMax = nn::nfc::DeviceCountMax;

    /*!
        @brief      amiibo のアプリケーション専用領域にアクセスするために必要な ID です。
        @details    ID が異なるアプリケーション専用領域はオープンできません。
     */
    const nn::Bit32 AccessId = 0x0FF41E00;

    /*!
        @brief      アプリケーションの状態を表す定数です。
     */
    enum State
    {
        State_None,                     //!< NFC ライブラリを初期化していない状態です。
        State_Initializing,             //!< NFC ライブラリを初期中です。
        State_Initialized,              //!< NFC ライブラリが利用可能な状態です。
        State_DeviceListing,            //!< NFC デバイスリストを取得中です。
        State_DeviceInitialize,         //!< NFC デバイスを初期化中です。
        State_DeviceSelect,             //!< NFC デバイスを選択中です。
        State_StartDetection,           //!< タグの検知を開始しています。
        State_StopDetection,            //!< タグの検知を停止しています。
        State_TagRelease,               //!< タグの喪失待機中です。
        State_TagSearch,                //!< タグを検索中です。
        State_DeviceCheck,              //!< NFC デバイスの状態を確認中です。
        State_TagSendReadCommand,       //!< タグに読み込みコマンド送信中です。
        State_TagCheck,                 //!< タグを確認中です。
        State_TagSendWriteCommand,      //!< タグに書き込みコマンドを送信中です。
        State_TagEnabled,               //!< 利用可能なタグの情報を取得できた状態です。
        State_Finalize,                 //!< NFC ライブラリを終了中です。
        State_Error,                    //!< エラー状態です。
        State_Exit,                     //!< 終了状態です。

        State_CountMax                  //!< アプリケーションの状態の総数です。
    };

    /*!
        @brief      アプリケーションで扱うデータです。
     */
    struct ApplicationData
    {
        State                    state;          //!< アプリケーションの状態です。
        nn::Result               lastResult;     //!< 最後に実行した NFC 関連の処理の結果です。
        uint32_t                 frame;          //!< フレーム番号です。
        uint32_t                 transitFrame;   //!< 最後の状態遷移したときのフレーム番号です。
        nn::hid::NpadIdType      npadIds[ DeviceCountMax ]; //!< NFC デバイスの NpadId です。
        int                      deviceCount;    //!< NFC デバイスの数です。
        int                      deviceIndex;    //!< 操作対象の NFC デバイスです。
        nn::nfc::TagInfo         tagInfo;        //!< タグの情報です。
        bool                     isReceivedReadNak;//!< READ コマンド時に NAK を受け取ったか否か。
        bool                     isReceivedWriteNak;//!< WRITE コマンド時に NAK を受け取ったか否か。
        uint8_t                  counter;        //!< NTAG21x ページ 4 の 1 バイト目から読み込んだカウンタです。
    };

}}} // end of namespace nns::nfc::pt
