﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{NetdiagNatTypeDetection.cpp,PageSampleNetdiagNatTypeDetection}
 *
 * @brief
 *  NAT タイプ判定のサンプルプログラム
 */

/**
 * @page PageSampleNetdiagNatTypeDetection NetdiagNatTypeDetection
 * @tableofcontents
 *
 * @brief
 *  NAT タイプの判定を行うサンプルの解説です。
 *
 * @section PageSampleNetdiagNatTypeDetection_SectionBrief 概要
 *  ネットワーク接続後、ソケット通信を行って NAT タイプ判定サーバーへ接続し、その結果から NAT タイプを判定します。
 *
 * @section PageSampleNetdiagNatTypeDetection_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/NetdiagNatTypeDetection Samples/Sources/Applications/NetdiagNatTypeDetection @endlink 以下にあります。
 *
 * @section PageSampleNetdiagNatTypeDetection_SectionNecessaryEnvironment 必要な環境
 *  事前に設定マネージャを使用してネットワーク接続設定をインポートする必要があります。
 *  詳細は @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} をご覧ください。
 *
 * @section PageSampleNetdiagNatTypeDetection_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleNetdiagNatTypeDetection_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleNetdiagNatTypeDetection_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleNetdiagNatTypeDetection_SectionDetail 解説
 *
 * @subsection PageSampleNetdiagNatTypeDetection_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  NetdiagNatTypeDetection.cpp
 *  @includelineno NetdiagNatTypeDetection.cpp
 *
 * @subsection PageSampleNetdiagNatTypeDetection_SectionSampleDetail サンプルプログラムの解説
 *  サンプルプログラムの全体像は以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリを初期化します。
 * - nn::netdiag::DetectNatType で NAT タイプを取得します。
 *   - 内部では、NAT タイプ判定サーバーと UDP ソケット通信にてパケットの送受信を行い、その結果で NAT タイプを決定します。
 *   - nn::netdiag::DetectNatType が返す値は nn::netdiag::NatType_A / B / C / D / Z のいずれかです。
 * - NAT タイプを NATTYPE A / B / C / D / F で表示します。
 * - nn::netdiag::NatType_Z に対応する表示は "NATTYPE F" となります。
 * - 詳細は nn::netdiag::NatType のリファレンス、「機能/NETDIAGライブラリ/マニュアル」を参照してください。
 *
 */

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>

#include <nn/socket.h>

#include <nn/netdiag/netdiag_Result.h>
#include <nn/netdiag/netdiag_NatApi.h>

const char* NatTypeString[] = {
    "NATTYPE A",
    "NATTYPE B",
    "NATTYPE C",
    "NATTYPE D",
    "NATTYPE F"
};

//----------------------------------------------------------------
// NIFM 初期化
nn::Result InitializeNifm()
{
    auto result = nn::nifm::Initialize();
    if ( result.IsFailure() )
    {
        NN_LOG("nifm::Initialize Failed.\n");
        return result;
    }

    nn::nifm::SubmitNetworkRequest();
    while( nn::nifm::IsNetworkRequestOnHold() )
    {
        NN_LOG("Wainitg for network interface availability...\n");
        nn::os::SleepThread( nn::TimeSpan::FromSeconds(1) );
    }

    if ( ! nn::nifm::IsNetworkAvailable() )
    {
        NN_LOG("\nError: network is not available.\n");
        return nn::netdiag::ResultNotInitialized();
    }

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
// ソケット初期化
namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}
nn::Result InitializeSocket()
{
    auto result = nn::socket::Initialize(g_SocketConfigWithMemory);

    if( result.IsFailure() )
    {
        NN_LOG("\nError: nn::socket::Initialize() failed. (error 0x%x)\n\n", result.GetDescription());
        return nn::netdiag::ResultNotInitialized();
    }

    NN_RESULT_SUCCESS;
}

//----------------------------------------------------------------
//
//  メイン関数です。
//
extern "C" void nnMain()
{
    nn::Result result;

    // Nifm 初期化
    result = InitializeNifm();
    if ( result.IsFailure() )
    {
        goto EXIT_NN_MAIN;
    }

    // Socket 初期化
    result = InitializeSocket();
    if ( result.IsFailure() )
    {
        goto EXIT_NN_MAIN;
    }

    nn::netdiag::NatType natType;
    result = nn::netdiag::DetectNatType( &natType );

    if ( result.IsSuccess() )
    {
        NN_LOG("NATTYPE = [%s]\n", NatTypeString[natType] );
    }
    else
    {
        NN_LOG("NATTYPE = [???] (error 0x%x)\n", result.GetDescription() );
    }

EXIT_NN_MAIN:
    nn::nifm::CancelNetworkRequest();

    NN_LOG("\nEnd Sample\n");
}
