﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <vector>

#include <nn/os.h>
#include <nn/fs.h>
#include <nn/nn_Result.h>

#include <nns/mm.h>

/**
 * @brief
 * MovieVideoInputHandler class for handling video decoder input data.
 *
 */
void MovieVideoInputHandlerThreadFunction(void *arg);
class MovieVideoInputHandler
{
public:
   /**
    * @brief
    * MovieVideoInputHandler constructor.
    *
    * @param[in] extractor                      Movie extractor for video track.
    * @param[in] videoTrackIndex                Video track index.
    * @param[in] videoDecoder                   Movie video decoder for video track.
    *
    */
    MovieVideoInputHandler(movie::Extractor* extractor, int32_t videoTrackIndex, movie::Decoder* videoDecoder);

   /**
    * @brief
    * Initialize MovieVideoInputHandler instance.
    *
    * @param[in]  coreMask                      Usable CPU cores in core mask.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * Usable cores in core mask. If Nth bit is set to 1, the thread can run on CPU core number N. The core mask need
    * to be set before creation of thread. @class MovieVideoInputHandler thread is created. A message queue is created
    * and the thread will wait on this message queue.
    *
    */
    movie::Status Initialize(uint64_t coreMask);

   /**
    * @brief
    * ReadInputDataFromExtractorSendTodecoder. API to read data from extractor and send the same to decoder.
    *
    * @param[in] index                         	Video decoder input buffer index.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_EndOfStream
    * @retval ::Status_UnknownError
    *
    * @details
    * This API can be used to acquire an input buffer from decoder and read input data from extractor. If data is
    * available, input buffer is filled with data from extractor and send to decoder. Video time stamp for the input
    * video frame is retrieved from extractor and passed to decoder.
    *
    */
    movie::Status ReadInputDataFromExtractorSendTodecoder(int index);

   /**
    * @brief
    * CheckForInputBuffersAndReadVideoInputData. API to check for decoder input buffers and fill with video data to
    * decode.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    * @retval ::Status_EndOfStream
    * @retval ::Status_UnknownError
    *
    * @details
    * This API can be used to check for available input buffer. If an input buffer is available and extractor has data
    * ReadInputDataFromExtractorSendTodecoder() API is called.
    *
    */
    movie::Status CheckForInputBuffersAndReadVideoInputData();

   /**
    * @brief
    * GetVideoIndexListSize. API to get current size of input buffer list.
    *
    * @param[out] indexListSize                	Video decoder input buffer list size.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to get current size of @class MovieVideoInputHandler input buffer list. This info can be
    * used whether to proceed with reading data from extractor. If there are no input buffer indices the client need
    * to retry at later time.
    *
    */
    movie::Status GetVideoIndexListSize(int32_t* indexListSize);

   /**
    * @brief
    * AddBufferToVideoIndexList. API to add an input buffer index to @class MovieVideoInputHandler input list
    *
    * @param[in]  bufferIndex                         Video decoder input buffer index.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to add decoder input buffer index to  @class MovieVideoInputHandler input list.
    * @class MovieDecoderEventHandler will acquire buffer index from decoder, when it receives decoder event. Buffer
    * index is passed to @class MovieVideoInputHandler for further processing.
    */
    movie::Status AddBufferToVideoIndexList(int32_t bufferIndex);

   /**
    * @brief
    * RemoveBufferFromVideoIndexList. API to get an input buffer index.
    *
    * @param[out]  bufferIndex                         Video decoder input buffer index
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API can be used to remove an input buffer index from the list. An input buffer can be requested from decoder
    * for the given buffer index. Buffer indices are acquired from movie decoder by @class MovieDecoderEventHandler. If
    * no indices are available, -1 value will be set to bufferIndex.
    *
    */
    movie::Status RemoveBufferFromVideoIndexList(int32_t* bufferIndex);

   /**
    * @brief
    * VideoInputBufferAvailableEvent. API to indicate input buffer availability
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to create an input buffer available message. The message is sent to @class MovieVideoInputHandler
    * message queue using SignalVideoInputBufferAvailable() API.
    *
    */
    movie::Status VideoInputBufferAvailableEvent();

   /**
    * @brief
    * SignalVideoInputBufferAvailable. API to send input buffer available message.
    *
    * @return ::movie::Status
    * @retval ::Status_Success
    *
    * @details
    * This API is used to send input buffer available message to @class MovieVideoInputHandler message queue.
    *
    */
    movie::Status SignalVideoInputBufferAvailable();

   /**
    * @brief
    * Thread function for MovieVideoInputHandler thread.
    *
    * @param[in] arg                            Arguments for thread.
    *
    * @return None
    *
    * @details
    * The thread loop function. The thread will wait on a message queue.
    *
    */
    friend void MovieVideoInputHandlerThreadFunction(void *arg);

   /**
    * @brief
    * SignalFlush. API to send flush message to @class MovieVideoInputHandler
    *
    * @return None
    *
    * @details
    * This API is used to send a flush message to @class MovieVideoInputHandler message queue. The message handler will
    * call Flush() API.
    *
    */
    void SignalFlush();

   /**
    * @brief
    * Flush. API to flush input buffer.
    *
    * @return None
    *
    * @details
    * This API is used to flush @class MovieVideoInputHandler input buffer index list. API will simply clear the list.
    *
    */
    void Flush();

   /**
    * @brief
    * MovieVideoInputHandler destructor.
    *
    * @return None
    */
    ~MovieVideoInputHandler();

private:
    movie::Extractor* m_Extractor;
    movie::Decoder* m_VideoDecoder;
    int32_t m_VideoTrackIndex;
    size_t m_ThreadStackSize;
    void* m_ThreadStack;
    nn::os::ThreadType m_ThreadType;
    std::vector<int> m_VideoBufferIndexList;
    nn::os::MutexType m_VideoListMutex;

    size_t m_MessageQueueBufferSize;
    uintptr_t *m_MessageQueueBuffer;
    nn::os::MessageQueueType m_MessageQueue;
    bool m_FlushInput;

    enum MoviePlayerMessage
    {
        MoviePlayerMessage_VideoInputAvailable  = 1,
        MoviePlayerMessage_VideoInputThreadExit = 2,
        MoviePlayerMessage_VideoInputInputFlush = 3,
    };
};
