﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "MovieVideoInputHandler.h"

/**
 * @brief
 * MovieVideoInputHandler class for handling video decoder input events and feed input data to the decoder
 *
 * @details
 *
 * Purpose:
 *     The purpose of @class MovieVideoInputHandler is to handle audio decoder input events. @class MovieDecoderEventHandler
 * registers events with decoder. When any input buffer related events are signaled from the decoder, @class MovieDecoderEventHandler
 * will sent a message to @class MovieVideoInputHandler message queue. When the @class MovieVideoInputHandler receives
 * input buffer available message, it will check whether the extractor has data for video track. If extractor has data
 * for video track it will get input buffer from decoder and reads input video data from the extractor. The filled input buffer is
 * sent for decoding. It also handles flush message by clearing all input buffers from it queue.
 *
 * Setup:
 *     @class MoviePlayer class will create an instance of @class MovieVideoInputHandler. An extractor and video decoder is passed
 * in the constructor. During initialization a thread is created. The thread will wait for messages in this message queue. Other
 * resources like input buffer list are initialized.
 *
 * Main Processing:
 *     When @class MovieDecoderEventHandler receives input buffer available event from video decoder, it sends a message to
 *  @class MovieVideoInputHandler message queue. When the message is serviced, extractor is checked for video data availability.
 * If video data is available, video decoder input buffer is requested from decoder. This will be filled with compressed
 * video data. Filled input buffer is sent to decoder for decoding. This process continues as long as input data is available
 * from extractor. When end of stream is received from extractor, an input buffer with EOS flag is sent to the decoder.
 *
 *
 * Teardown:
 *    When @class MovieDecoderEventHandler receives EOS buffer from video decoder, media playback will be stopped. At the end player will
 * delete @class MovieVideoInputHandler instance. This will result in sending thread exit message to @class MovieVideoInputHandler thread.
 * The thread will exit and resources are freed.
 *
 */
MovieVideoInputHandler::MovieVideoInputHandler(movie::Extractor* extractor,
    int32_t videoTrackIndex,
    movie::Decoder* videoDecoder)
    : m_Extractor(extractor),
      m_VideoDecoder(videoDecoder),
      m_VideoTrackIndex(videoTrackIndex),
      m_ThreadStackSize(1024 * 256),
      m_ThreadStack(nullptr),
      m_MessageQueueBufferSize(256),
      m_MessageQueueBuffer(nullptr),
      m_FlushInput(false)
{
}

MovieVideoInputHandler::~MovieVideoInputHandler()
{
    nn::os::SendMessageQueue(&m_MessageQueue, MovieVideoInputHandler::MoviePlayerMessage_VideoInputThreadExit);
    nn::os::WaitThread(&m_ThreadType);
    nn::os::DestroyThread(&m_ThreadType);

    // Finalize MessageQueue used by movie decoder player
    nn::os::FinalizeMessageQueue(&m_MessageQueue);
    if( m_MessageQueueBuffer != nullptr )
    {
        delete[] m_MessageQueueBuffer;
    }
    // Delete thread stack memory.
    if( m_ThreadStack != nullptr )
    {
        free(m_ThreadStack);
    }
}
movie::Status MovieVideoInputHandler::Initialize(uint64_t coreMask)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();
    m_VideoBufferIndexList.reserve(64);
    //set only valid core mask
    if( coreMask )
    {
        movie::SetCoreMask(coreMask);
    }

    m_MessageQueueBuffer = new uintptr_t[ m_MessageQueueBufferSize ];
    if( m_MessageQueueBuffer == nullptr )
    {
        return movie::Status_OutOfMemory;
    }

    // Initialize MovieVideoInputHandler MessageQueue.
    nn::os::InitializeMessageQueue(&m_MessageQueue, m_MessageQueueBuffer, m_MessageQueueBufferSize);
    nn::os::InitializeMutex(&m_VideoListMutex, false, 0);

    // Create video InputReader thread.
    m_ThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_ThreadStack == NULL )
    {
        movieStatus = movie::Status_OutOfMemory;
    }
    else
    {
        nnResult = nn::os::CreateThread(&m_ThreadType,
            &MovieVideoInputHandlerThreadFunction,
            ( void* )this,
            m_ThreadStack,
            m_ThreadStackSize,
            nn::os::DefaultThreadPriority);
        if( nnResult.IsSuccess() )
        {
            nn::os::SetThreadName(&m_ThreadType, "MovieVideoInputReadThread");
            nn::os::StartThread(&m_ThreadType);
        }
        else
        {
            movieStatus = movie::Status_FailedToCreateThread;
        }
    }

    return movieStatus;
}

movie::Status MovieVideoInputHandler::GetVideoIndexListSize(int32_t* indexListSize)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_VideoListMutex);
    *indexListSize = m_VideoBufferIndexList.size();
    nn::os::UnlockMutex(&m_VideoListMutex);
    return movieStatus;
}

movie::Status MovieVideoInputHandler::AddBufferToVideoIndexList(int32_t bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_VideoListMutex );
    m_VideoBufferIndexList.push_back(bufferIndex);
    nn::os::UnlockMutex( &m_VideoListMutex );
    return movieStatus;
}

movie::Status MovieVideoInputHandler::VideoInputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    SignalVideoInputBufferAvailable();
    return movieStatus;
}

movie::Status MovieVideoInputHandler::SignalVideoInputBufferAvailable()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue(&m_MessageQueue, MovieVideoInputHandler::MoviePlayerMessage_VideoInputAvailable);
    return movieStatus;
}

movie::Status MovieVideoInputHandler::RemoveBufferFromVideoIndexList(int32_t* bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_VideoListMutex );
    int32_t index = -1;
    if( m_VideoBufferIndexList.size() > 0 )
    {
        index = m_VideoBufferIndexList.front();
        m_VideoBufferIndexList.erase(m_VideoBufferIndexList.begin());
    }
    *bufferIndex = index;
    nn::os::UnlockMutex( &m_VideoListMutex );
    return movieStatus;
}

movie::Status MovieVideoInputHandler::ReadInputDataFromExtractorSendTodecoder(int index)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;
    movieStatus = m_VideoDecoder->GetInputBuffer(index, &buffer);
    if( movieStatus == movie::Status_Success )
    {
        buffer.SetRange(0, 0);
        movieStatus = m_Extractor->Read(&buffer);
        if( ( movieStatus != movie::Status_Success ) ||
            ( buffer.Size() <= 0 ) ||
            ( movieStatus == movie::Status_EndOfStream ) )
        {
            flags = movie::BufferFlags_EndOfStream;
        }
        m_Extractor->GetSampleTime(&presentationTimeUs);
        m_VideoDecoder->SendInputBufferForDecode(index, buffer.Offset(), buffer.Size(), presentationTimeUs, flags);
         m_Extractor->Advance();
    }
    return movieStatus;
}

movie::Status MovieVideoInputHandler::CheckForInputBuffersAndReadVideoInputData()
{
    movie::Status movieStatus = movie::Status_NotFound;
    size_t currentTrackIndex = -1;

    if( m_Extractor != NULL )
    {
        movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
        if( movieStatus == movie::Status_EndOfStream )
        {
            currentTrackIndex = m_VideoTrackIndex;
        }
        else if( movieStatus != movie::Status_Success )
        {
            return movieStatus;
        }
    }
    movieStatus = movie::Status_NotFound;
    if( currentTrackIndex == m_VideoTrackIndex )
    {
        int index = -1;
        RemoveBufferFromVideoIndexList(&index);
        if( index != -1 )
        {
            movieStatus = ReadInputDataFromExtractorSendTodecoder(index);
        }
    }
    return movieStatus;
}

void MovieVideoInputHandler::Flush()
{
    nn::os::LockMutex(&m_VideoListMutex);
    m_VideoBufferIndexList.clear();
    nn::os::UnlockMutex(&m_VideoListMutex);
    m_FlushInput = false;
}

void MovieVideoInputHandler::SignalFlush()
{
    m_FlushInput = true;
    nn::os::SendMessageQueue(&m_MessageQueue, MovieVideoInputHandler::MoviePlayerMessage_VideoInputInputFlush);
}

void MovieVideoInputHandlerThreadFunction(void *arg)
{
    MovieVideoInputHandler *movieVideoInputHandler = ( MovieVideoInputHandler*) arg;
    if( movieVideoInputHandler != nullptr )
    {
        for( ;; )
        {
            uintptr_t data = 0;
            nn::os::ReceiveMessageQueue(&data, &movieVideoInputHandler->m_MessageQueue);
            if( data == MovieVideoInputHandler::MoviePlayerMessage_VideoInputAvailable )
            {
                movie::Status movieStatus = movieVideoInputHandler->CheckForInputBuffersAndReadVideoInputData();
                if( movieStatus != movie::Status_Success )
                {
                    if( movieVideoInputHandler->m_FlushInput == false )
                    {
                        int64_t delayNs = 10000; // Retry after 10000Ns. This sleep can be tweaked based on application need.
                                                 // Short sleep is needed when heavy(system calls at nanosecond intervals)
                                                 // system calls are done in another thread.
                        nn::os::SleepThread(nn::TimeSpan::FromNanoSeconds(delayNs));
                        movieVideoInputHandler->SignalVideoInputBufferAvailable();
                    }
                }
            }
            else if( data == MovieVideoInputHandler::MoviePlayerMessage_VideoInputThreadExit )
            {
                break;
            }
            else if( data == MovieVideoInputHandler::MoviePlayerMessage_VideoInputInputFlush )
            {
                movieVideoInputHandler->Flush();
            }
            else
            {
                movieVideoInputHandler->CheckForInputBuffersAndReadVideoInputData();
            }
        }
    }
}
