﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{MovieDecoderPlayer.cpp,PageSampleMovieDecoderPlayer}
 *
 * @brief
 *  movie API による動画再生サンプル
 */
 /**
 * @page PageSampleMovieDecoderPlayer MovieDecoderPlayer
 * @tableofcontents
 *
 * @brief
 * movie API による動画再生サンプルプログラムです。
 *
 * @section PageSampleMovieDecoderPlayer_SectionBrief 概要
 * NX 本体上で mp4 ファイルを入力し、再生を行うサンプルです。
 *
 * @section PageSampleMovieDecoderPlayer_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/MovieDecoderPlayer
 * Samples/Sources/Applications/MovieDecoderPlayer @endlink 以下にあります。
 *
 * @section PageSampleMovieDecoderPlayer_SectionNecessaryEnvironment 必要な環境
 * 追加で必要となるものはありません。
 *
 * @section PageSampleMovieDecoderPlayer_SectionHowToOperate 操作方法
 * サンプルプログラムを実行すると、自動的に指定した動画の再生が始まります。
 * 再生終了後は、自動的にプログラムは終了します。
 *
 * @section PageSampleMovieDecoderPlayer_SectionPrecaution 注意事項
 *  特にありません。
 *
 * @section PageSampleMovieDecoderPlayer_SectionHowToExecute 実行手順
 *  @<MovieDecoderPlayer executable@> @<media path@> -me @<media extension@> @<optional parameters@>
 *
 * @<media extension@> : mp4, m4a, webm, mkv, ogg, ivf, h264
 *
 * @<optional parameters@>
 *
 * - -l 0: ループ再生をしない。
 * - -l 1: ループ再生をする。
 *
 * - -aging 0: エージングテスト無効（エージングテストとは、Decoder や Extractor および他リソースを毎回破棄して再生を繰り返すテスト）
 * - -aging 1: エージングテスト有効
 *
 * - -core 1: CPUコアマスクは 0x1=0001、つまり Core 0 を使用する。（CPUコアマスクは CPU プロファイラのログを使って確認可能）
 * - -core 2: CPUコアマスクは 0x2=0010、つまり Core 1 を使用する。
 * - -core 4: CPUコアマスクは 0x4=0100、つまり Core 2 を使用する。
 *
 * - -t 0: メモリアロケーションのトラッキングを無効化
 * - -t 1: メモリアロケーションのトラッキングを有効化（注意：有効化により再生時のパフォーマンスは劣化するかもしれません。）
 *
 * - -dm 1: movie::DecoderMode_Cpu モードを有効化する。
 * - -dm 2: movie::DecoderMode_NativeTexture モードを有効化する。
 *         アプリケーションは video デコーダーに NVN_FORMAT_RGBA8 フォーマットの NVN テクスチャを渡す。（デフォルト）
 *
 * 例（ホストPCから動画データを取得）
 * MovieDecoderPlayer.nsp c:@\H264_AAC_1280x720_30sec_01.mp4 -me mp4
 *
 * @section PageSampleMovieDecoderPlayer_SectionDetail 解説
 * 指定した動画ファイルを NX 本体上で再生します。
 */

#include "MovieDecoderPlayer.h"
#include <movie/Utils.h>
#include <movie/Common.h>
#include <movie/Status.h>
#include <nn/nn_SdkLog.h>
#include <nn/init.h>
#include <nn/lmem/lmem_ExpHeap.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <string>
#include <cinttypes>
#include <stddef.h>
#include <memory>

#include <nn/mem/mem_StandardAllocator.h>
#include <nv/nv_MemoryManagement.h>

#include "ByteStreamReader.h"
#include <nns/mm.h>

#include "ExampleMediaDataVisitors.h"

const int FsHeapSize = 512 * 1024;
const int mmHeapSize = 512 << 20;

uint8_t                     g_FsHeapBuffer[FsHeapSize];
nn::lmem::HeapHandle        g_FsHeap;
char                        g_mmHeapBuffer[mmHeapSize];

const size_t  g_GraphicsSystemMemorySize = 8 * 1024 * 1024;
uint8_t g_GraphicsHeap[ g_GraphicsSystemMemorySize ] __attribute__(( aligned(4096) ));

bool g_TrackMemoryAllocation = false;

void FsInitHeap()
{
    g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, FsHeapSize, nn::lmem::CreationOption_DebugFill);
}

void* FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, p);
}

static MemoryTracker &MovieMemoryTracker()
{
    static MemoryTracker g_MovieMemoryTracker;
    return g_MovieMemoryTracker;
}

static MemoryTracker &CoreMemoryTracker()
{
    static MemoryTracker g_CoreMemoryTracker;
    return g_CoreMemoryTracker;
}

static MemoryTracker &MallocMemoryTracker()
{
    static MemoryTracker g_MallocMemoryTracker;
    return g_MallocMemoryTracker;
}

static MemoryTracker &NewMemoryTracker()
{
    static MemoryTracker g_NewMemoryTracker;
    return g_NewMemoryTracker;
}

static nn::mem::StandardAllocator &GlobalAllocator()
{
    static nn::mem::StandardAllocator g_MultimediaAllocator(g_mmHeapBuffer, sizeof(g_mmHeapBuffer));
    return g_MultimediaAllocator;
}

void* MovieAllocate(size_t size, size_t alignment, void *userPtr)
{
    void *address = GlobalAllocator().Allocate(size, alignment);
    static size_t totalAllocatedSize = 0;
    totalAllocatedSize += size;
    if( g_TrackMemoryAllocation )
    {
        const char* threadName = ::nn::os::GetThreadNamePointer(::nn::os::GetCurrentThread());
        static_cast<MemoryTracker*>( userPtr )->Track(address, size, threadName);
    }
    return address;
}

void MovieDeallocate(void *address, void *userPtr)
{
    GlobalAllocator().Free(address);
    if( g_TrackMemoryAllocation )
    {
        static_cast<MemoryTracker*>( userPtr )->Untrack(address);
    }
}

void *MovieReallocate(void* address, size_t newSize, void *userPtr)
{
    if( g_TrackMemoryAllocation )
    {
        if( address != nullptr)
        {
            static_cast< MemoryTracker* >( userPtr )->Untrack(address);
        }
    }
    void *memory = GlobalAllocator().Reallocate(address, newSize);
    if( g_TrackMemoryAllocation )
    {
        const char* threadName = ::nn::os::GetThreadNamePointer(::nn::os::GetCurrentThread());
        static_cast<MemoryTracker*>( userPtr )->Track(memory, newSize, threadName);
    }
    return memory;
}

extern "C" void* malloc(size_t size)
{
    return MovieAllocate(size, 8, &MallocMemoryTracker());
}

extern "C" void free(void* address)
{
    if( address != nullptr )
        MovieDeallocate(address, &MallocMemoryTracker());
}

extern "C" void* calloc(size_t num, size_t size)
{
    const size_t sum = num * size;
    void* address = malloc(sum);
    if( address != nullptr )
        ::std::memset(address, 0, sum);
    return address;
}

extern "C" void* realloc(void* address, size_t newSize)
{
    return MovieReallocate(address, newSize, &MallocMemoryTracker());
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    return MovieAllocate(size, alignment, &MallocMemoryTracker());
}

void* operator new( ::std::size_t size, const ::std::nothrow_t& ) throw( )
{
    return MovieAllocate(static_cast<int>( size ), 8, &NewMemoryTracker());
}

void* operator new[](::std::size_t size, const ::std::nothrow_t&) throw( )
{
    return operator new( size, ::std::nothrow_t() );
}

void* operator new( ::std::size_t size ) throw( ::std::bad_alloc )
{
    return operator new( size, ::std::nothrow_t() );
}

void* operator new[](::std::size_t size) throw( ::std::bad_alloc )
{
    return operator new( size, ::std::nothrow_t() );
}

void operator delete( void* address ) throw( )
{
    if( address != nullptr )
        MovieDeallocate(address, &NewMemoryTracker());
}

void operator delete[](void* address) throw( )
{
    operator delete( address );
}

/* Process startup setup */
extern "C" void nninitStartup()
{
    FsInitHeap();
    nn::fs::SetAllocator(FsAllocate, FsDeallocate);
}

/* MovieDecoderPlayer Player Usage */
static void Usage()
{
    NN_SDK_LOG( "\n MovieDecoderPlayer ::Invalid options, refer Readme.MovieDecoderPlayer.txt \n");
}
static void PrintMemoryUsageStats()
{
    if( g_TrackMemoryAllocation )
    {
        NN_LOG("\n [Movie] memory allocator statistics \n");
        MovieMemoryTracker().OutputUsage();
        NN_LOG("\n [Core] memory allocator statistics \n");
        CoreMemoryTracker().OutputUsage();
        NN_LOG("\n [Malloc] memory allocator statistics \n");
        MallocMemoryTracker().OutputUsage();
        NN_LOG("\n [New] memory allocator statistics \n");
        NewMemoryTracker().OutputUsage();
    }
}

/* Process main entry */
extern "C" void nnMain()
{
    {
        int argc = nn::os::GetHostArgc();
        char** argv = nn::os::GetHostArgv();

        /* Set allocator callback functions */
        movie::SetAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, &MovieMemoryTracker());
        // Donate memory for graphics driver to work in
        nv::InitializeGraphics(g_GraphicsHeap, g_GraphicsSystemMemorySize);
        nv::SetGraphicsAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, &CoreMemoryTracker());
        nv::SetGraphicsDevtoolsAllocator(MovieAllocate, MovieDeallocate, MovieReallocate, &CoreMemoryTracker());

        if( argc <= 1 )
        {
            Usage();
            return;
        }
        int32_t playbackAgingRepeatCount = 0;
        bool runPlaybackAgingTest = false;
        uint64_t coreMask = 0;
        int32_t resetCount = 0;
        do
        {
            bool invalidOptions = false;
            bool loopPlayback = false;
            StreamFormat streamFormat = StreamFormat::StreamFormat_Unknown;
            movie::ContainerType container = movie::ContainerType_Unknown;
            movie::DecoderMode videoDecoderMode = movie::DecoderMode_NativeTexture;
            movie::DecoderMode audioDecoderMode = movie::DecoderMode_Cpu;
            movie::DecoderOutputFormat videoDecoderOutputFormat = movie::DecoderOutputFormat_VideoColorAbgr;
            movie::DecoderOutputFormat audioDecoderOutputFormat = movie::DecoderOutputFormat_AudioPcm16;
            for( int options = 0; options < argc; options++ )
            {
                if( !strcmp(argv[ options ], "-me") )
                {
                    options++;
                    invalidOptions = true;
                    if( options < argc )
                    {
                        if( ( !strcmp(argv[ options ], "mp4") ) || ( !strcmp(argv[ options ], "m4a") ) )
                        {
                            container = movie::ContainerType_Mpeg4;
                            invalidOptions = false;
                        }
                        else if( ( !strcmp(argv[ options ], "webm") ) )
                        {
                            container = movie::ContainerType_WebM;
                            invalidOptions = false;
                        }
                        else if( ( !strcmp(argv[ options ], "mkv") ) )
                        {
                            container = movie::ContainerType_Matroska;
                            invalidOptions = false;
                        }
                        else if( ( !strcmp(argv[ options ], "ogg") ) )
                        {
                            container = movie::ContainerType_Ogg;
                            invalidOptions = false;
                        }
                        else if( ( !strcmp(argv[ options ], "ivf") ) )
                        {
                            streamFormat = StreamFormat_ivf;
                            invalidOptions = false;
                        }
                        else if( ( !strcmp(argv[ options ], "h264") ) )
                        {
                            streamFormat = StreamFormat_h264;
                            invalidOptions = false;
                        }
                        else
                            break;
                    }
                }
                else if( !strcmp(argv[ options ], "-l") )
                {
                    options++;
                    invalidOptions = true;
                    if( options < argc )
                    {
                        if( ( !strcmp(argv[ options ], "0") ) || ( !strcmp(argv[ options ], "1") ) )
                        {
                            loopPlayback = atoi(argv[ options ]);
                            invalidOptions = false;
                        }
                        else
                            break;
                    }
                }
                else if( !strcmp(argv[ options ], "-aging") )
                {
                    options++;
                    invalidOptions = true;
                    if( options < argc )
                    {
                        if( ( !strcmp(argv[ options ], "0") ) || ( !strcmp(argv[ options ], "1") ) )
                        {
                            runPlaybackAgingTest = atoi(argv[ options ]);
                            invalidOptions = false;
                        }
                        else
                            break;
                    }
                }
                else if( !strcmp(argv[ options ], "-core") )
                {
                    options++;
                    invalidOptions = true;
                    if( options < argc )
                    {
                        if( ( !strcmp(argv[ options ], "1") ) || ( !strcmp(argv[ options ], "2") ) || ( !strcmp(argv[ options ], "4") ) )
                        {
                            coreMask = atoi(argv[ options ]);
                            invalidOptions = false;
                        }
                        else
                            break;
                    }
                }
                else if( !strcmp(argv[ options ], "-reset") )
                {
                    options++;
                    invalidOptions = true;
                    if( options < argc )
                    {
                        resetCount = atoi(argv[ options ]);
                        invalidOptions = false;
                    }
                }
                else if( !strcmp(argv[ options ], "-t") )
                {
                    options++;
                    invalidOptions = true;
                    if( ( !strcmp(argv[ options ], "0") ) || ( !strcmp(argv[ options ], "1") ) )
                    {
                        g_TrackMemoryAllocation = atoi(argv[ options ]);
                        invalidOptions = false;
                    }
                    else
                        break;
                }
                else if( !strcmp(argv[ options ], "-dm") )
                {
                    options++;
                    invalidOptions = true;
                    if( ( !strcmp(argv[ options ], "1") ) || ( !strcmp(argv[ options ], "2") ) )
                    {
                        if( 1 == atoi(argv[ options ]) )
                        {
                            videoDecoderMode = movie::DecoderMode_Cpu;
                            videoDecoderOutputFormat = movie::DecoderOutputFormat_VideoColorNv12;
                        }
                        invalidOptions = false;
                    }
                    else
                        break;
                }
            }
            if( ( invalidOptions == true )
                || ( ( streamFormat == StreamFormat::StreamFormat_Unknown ) && ( container == movie::ContainerType::ContainerType_Unknown ) ) )
            {
                Usage();
                return;
            }

            if( loopPlayback == true )
                runPlaybackAgingTest = false;
            bool doesMediaFileExists = false;
            bool sdcardMounted = false;
            std::string path = argv[ 1 ];
            if( false == path.empty() )
            {
                std::string delimiter = ":";
                std::string token = path.substr(0, path.find(delimiter));
                if( token == "sdcard" )
                {
                    nn::Result resultSdcardMount = nn::fs::MountSdCardForDebug("sdcard");
                    if( resultSdcardMount.IsFailure() )
                    {
                        NN_SDK_LOG("\n nn::fs::SD card mount failure. Module:%d, Description:%d\n",
                            resultSdcardMount.GetModule(),
                            resultSdcardMount.GetDescription());
                        return;
                    }
                    sdcardMounted = true;
                }
                else
                {
                    nn::Result resultHostMount = nn::fs::MountHostRoot();
                    if( resultHostMount.IsFailure() )
                    {
                        NN_SDK_LOG("\n nn::fs::Host root mount failure. Module:%d, Description:%d\n",
                            resultHostMount.GetModule(),
                            resultHostMount.GetDescription());
                        return;
                    }
                }
                nn::fs::FileHandle fileHandle;
                fileHandle.handle = NULL;
                nn::Result result = nn::fs::OpenFile(&fileHandle, path.c_str(), nn::fs::OpenMode_Read);
                if( result.IsSuccess() )
                {
                    doesMediaFileExists = true;
                    nn::fs::CloseFile(fileHandle);
                }
            }
            if( doesMediaFileExists == false )
            {
                NN_SDK_LOG("\n MovieDecoderPlayer:: Cannot open input [%s] file Terminating Application !\n", path.c_str());
                if( sdcardMounted == true )
                    nn::fs::Unmount("sdcard");
                else
                    nn::fs::UnmountHostRoot();
                return;
            }
            NN_SDK_LOG("\n MovieDecoderPlayer:: Input File Name : %s\n", path.c_str());
            if( videoDecoderMode == movie::DecoderMode_NativeTexture )
                NN_LOG("\n Using [movie::DecoderMode_NativeTexture] mode for video decoder \n");
            else
                NN_LOG("\n Using [movie::DecoderMode_Cpu] mode for video decoder \n");
            movie::Status movieStatus = movie::Status_Success;
            std::unique_ptr<MovieDecoderPlayer> movieDecoderPlayer { new MovieDecoderPlayer{} };
            if( movieDecoderPlayer == nullptr )
            {
                NN_SDK_LOG("\n Failed to create  MovieDecoderPlayer, Error : Status_OutOfMemory - Terminating Application ! \n");
                if( sdcardMounted == true )
                    nn::fs::Unmount("sdcard");
                else
                    nn::fs::UnmountHostRoot();
                nv::FinalizeGraphics();
                return;
            }
            movieDecoderPlayer->m_SdcardMounted = sdcardMounted;
            movieStatus = movieDecoderPlayer->Initialize(coreMask);
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG("\n Failed to initialize  MovieDecoderPlayer, Error : 0x%x - Terminating Application ! \n", movieStatus);
                movieDecoderPlayer->HandleErrorExit();
                movieDecoderPlayer.reset();
                return;
            }
            movieDecoderPlayer->m_LoopPlayback = loopPlayback;
            // Check for raw streams
            if( ( streamFormat == StreamFormat_ivf ) || ( streamFormat == StreamFormat_h264 ) )
            {
                movieDecoderPlayer->m_ByteStreamReader = new ByteStreamReader();
                if( movieDecoderPlayer->m_ByteStreamReader == NULL )
                {
                    NN_SDK_LOG("\n Failed to create IVF stream reader - Terminating Application ! \n");
                    movieDecoderPlayer->HandleErrorExit();
                    movieDecoderPlayer.reset();
                    return;
                }
                else
                {
                    movieStatus = movieDecoderPlayer->m_ByteStreamReader->Open(path.c_str(), streamFormat);
                    if( movieStatus == movie::Status_Success )
                    {
                        movieStatus = movieDecoderPlayer->m_ByteStreamReader->Prepare();
                        if( movieStatus == movie::Status_Success )
                        {
                            movieStatus = movieDecoderPlayer->m_ByteStreamReader->GetStreamProperties(&movieDecoderPlayer->m_VideoDecoderType,
                                &movieDecoderPlayer->m_Width,
                                &movieDecoderPlayer->m_Height,
                                &movieDecoderPlayer->m_VideoFrameRate);
                            if( movieStatus == movie::Status_Success )
                            {
                                movieStatus = movieDecoderPlayer->CreateDecoder(&movieDecoderPlayer->m_VideoDecoder,
                                    movieDecoderPlayer->m_VideoDecoderType,
                                    videoDecoderMode,
                                    videoDecoderOutputFormat);
                                if( movieStatus == movie::Status_Success )
                                {
                                    movie::MediaData config;
                                    movieDecoderPlayer->m_VideoDecoderCreated = true;
                                    if( movieDecoderPlayer->m_VideoDecoderType == movie::DecoderType_VideoVp8 )
                                    {
                                        CreateVp8DecoderConfigurationData(movieDecoderPlayer->m_Width, movieDecoderPlayer->m_Height,
                                            movieDecoderPlayer->m_VideoFrameRate,
                                            &config);
                                    }
                                    else if( movieDecoderPlayer->m_VideoDecoderType == movie::DecoderType_VideoVp9 )
                                    {
                                        CreateVp9DecoderConfigurationData(movieDecoderPlayer->m_Width,
                                            movieDecoderPlayer->m_Height,
                                            movieDecoderPlayer->m_VideoFrameRate
                                            , &config);
                                    }
                                    else if( movieDecoderPlayer->m_VideoDecoderType == movie::DecoderType_VideoAvc )
                                    {
                                        CreateH264DecoderConfigurationData(movieDecoderPlayer->m_ByteStreamReader->m_H264Sps,
                                            movieDecoderPlayer->m_ByteStreamReader->m_H264SpsSize,
                                            movieDecoderPlayer->m_ByteStreamReader->m_H264Pps,
                                            movieDecoderPlayer->m_ByteStreamReader->m_H264PpsSize,
                                            &config);
                                    }
                                    else
                                        movieStatus = movie::Status_UnknownError;
                                    if( movieStatus == movie::Status_Success )
                                    {
                                        movieStatus = movieDecoderPlayer->ConfigureDecoder(movieDecoderPlayer->m_VideoDecoder,
                                            &movieDecoderPlayer->m_VideoDecoderEvents,
                                            &config);
                                    }
                                    movieDecoderPlayer->ResetVideoDecodeComplete();
                                    movieDecoderPlayer->m_VideoTrackIndex = 0;
                                    movieDecoderPlayer->m_TrackFormat.push_back(config);
                                }
                            }
                        }
                    }
                }
            }
            else if( container != movie::ContainerType_Unknown )
            {
                // Create extractor to parse media data
                movieStatus = movieDecoderPlayer->CreateExtractor(path.c_str(),
                    container,
                    &movieDecoderPlayer->m_Extractor,
                    &movieDecoderPlayer->m_ClientReader);
                if( movieStatus == movie::Status_Success )
                {
                    // Create audio and video decoder for decoding media data
                    movieStatus = movieDecoderPlayer->CreateDecoders(movieDecoderPlayer->m_Extractor,
                        audioDecoderMode,
                        audioDecoderOutputFormat,
                        videoDecoderMode,
                        videoDecoderOutputFormat);
                }
            }
            else
                movieStatus = movie::Status_UnknownError;
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG("\n Failed to createDecoders and Extractor, Error : 0x%x - Terminating Application ! \n", movieStatus);
                movieDecoderPlayer->HandleErrorExit();
                movieDecoderPlayer.reset();
                return;
            }
            // Create audio and video renderer for rendering decoded audio and video data
            movieStatus = movieDecoderPlayer->CreateAudioVideoRenderer();
            if( movieStatus == movie::Status_Success )
            {
                // Setup AudioOut audio renderer
                if( movieDecoderPlayer->IsAudioDecoderCreated() )
                {
                    NN_SDK_LOG("\n MovieDecoderPlayer:: Create AudioOut renderer \n");
                    movieStatus = movieDecoderPlayer->m_MovieAudioOutputHandler->Initialize(movieDecoderPlayer->m_MediaClock);
                    if( movieStatus == movie::Status_Success )
                    {
                        if( ( container == movie::ContainerType_Matroska ) || ( container == movie::ContainerType_WebM ) ||
                            ( container == movie::ContainerType_Ogg ) )
                        {
                            movieDecoderPlayer->m_VariableSizeAudioFrames = true;
                        }
                    }
                }
                if( movieStatus == movie::Status_Success )
                {
                    if( movieDecoderPlayer->IsVideoDecoderCreated() )
                    {
                        NN_SDK_LOG("\n MovieDecoderPlayer:: Create Video renderer \n");
                        movieStatus = movieDecoderPlayer->m_MovieVideoOutputHandler->Initialize(movieDecoderPlayer->m_MediaClock, nullptr);
                        if( movieStatus == movie::Status_Success )
                        {
                            if( movieDecoderPlayer->IsAudioDecoderCreated() )
                                movieDecoderPlayer->m_VideoFrameRate = -1;
                        }
                    }
                }
            }
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG("\n Failed to create AudioVideoRenderer, Error : 0x%x - Terminating Application ! \n", movieStatus);
                movieDecoderPlayer->HandleErrorExit();
                movieDecoderPlayer.reset();
                return;
            }
            NN_SDK_LOG("\n MovieDecoderPlayer:: Start Input Reader Thread \n");
            // Start audio input reader thread
            movieDecoderPlayer->StartAudioInputReaderThread();
            // Start video input reader thread
            movieDecoderPlayer->StartVideoInputReaderThread();
            // Start worker thread to receive decoder events
            movieDecoderPlayer->StartAudioVideoWorkerThread();
            NN_SDK_LOG("\n MovieDecoderPlayer:: start Codecs.\n");
            // Start decoders
            if( movieDecoderPlayer->m_AudioDecoder != NULL )
                movieStatus = movieDecoderPlayer->StartDecoder(movieDecoderPlayer->m_AudioDecoder);
            if( movieDecoderPlayer->m_VideoDecoder != NULL )
                movieStatus = movieDecoderPlayer->StartDecoder(movieDecoderPlayer->m_VideoDecoder);
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG("\n Failed to  StartDecoders , Error : 0x%x - Terminating Application ! \n", movieStatus);
                movieDecoderPlayer->HandleErrorExit();
                movieDecoderPlayer.reset();
                return;
            }
            if( movieDecoderPlayer->m_Extractor != NULL )
            {
                NN_SDK_LOG("\n MovieDecoderPlayer:: Wait for cache fill \n");
                movieStatus = movieDecoderPlayer->m_Extractor->SeekTo(0ll);
                if( movieStatus == movie::Status_Success )
                {
                    int64_t startTimeUs = 0ll;
                    int64_t endTimeUs = 0ll;
                    // Cache 5 seconds worth of data
                    int64_t cacheTimeUs = 5000000ll;
                    // Wait maximum of 5 Seconds
                    int sleepTimeMs = 10;
                    int waitCount = 500;
                    while( waitCount > 0 )
                    {
                        movieStatus = movieDecoderPlayer->m_Extractor->GetBufferedRange(&startTimeUs, &endTimeUs);
                        if( ( movieStatus != movie::Status_Success ) || ( endTimeUs >= cacheTimeUs ) )
                            break;
                        waitCount--;
                        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(sleepTimeMs));
                    }
                }
            }
            if( movieDecoderPlayer->m_VideoDecoder == nullptr )
                movieDecoderPlayer->m_VideoFormatChangeEventReceived = true;
            if( movieDecoderPlayer->m_AudioDecoder == nullptr )
                movieDecoderPlayer->m_AudioFormatChangeEventReceived = true;
            // Wait for audio and video FormatChange events.
            movieDecoderPlayer->WaitForFormatChange();
            if( movieDecoderPlayer->IsAudioDecoderCreated() )
            {
                movieStatus = movieDecoderPlayer->m_MovieAudioOutputHandler->Open(movieDecoderPlayer->m_AudioDecoder,
                    movieDecoderPlayer->m_SampleRate,
                    movieDecoderPlayer->m_NumChannels,
                    movieDecoderPlayer->m_VariableSizeAudioFrames);
            }
            if( movieDecoderPlayer->IsVideoDecoderCreated() )
            {
                movieStatus = movieDecoderPlayer->m_MovieVideoOutputHandler->Open(movieDecoderPlayer->m_VideoDecoder,
                    movieDecoderPlayer->m_Width,
                    movieDecoderPlayer->m_Height);
            }
            if( movieStatus != movie::Status_Success )
            {
                NN_SDK_LOG("\n Failed to open audio and video renderer, Error : 0x%x - Terminating Application ! \n", movieStatus);
                movieDecoderPlayer->HandleErrorExit();
                movieDecoderPlayer.reset();
                return;
            }
            NN_LOG("\n MovieDecoderPlayer:: Playback is Started \n");
            // Wait till we have sufficient audio and video output buffers to start playback
            // Wait maximum of one second
            int startPlaybackCount = 0;
            while( 1 )
            {
                if( true == movieDecoderPlayer->StartPlayback() || ( startPlaybackCount >= 200 ) )
                {
                    if( movieDecoderPlayer->m_MovieAudioOutputHandler != NULL )
                    {
                        movieStatus = movieDecoderPlayer->m_MovieAudioOutputHandler->Start();
                        if( movieStatus != movie::Status_Success )
                        {
                            movieDecoderPlayer->SetVideoDecodeComplete();
                            movieDecoderPlayer->SetAudioDecodeComplete();
                            break;
                        }
                    }
                    if( movieDecoderPlayer->m_MovieVideoOutputHandler != NULL )
                    {
                        movieStatus = movieDecoderPlayer->m_MovieVideoOutputHandler->Start(movieDecoderPlayer->m_Width,
                            movieDecoderPlayer->m_Height);
                        if( movieStatus != movie::Status_Success )
                        {
                            movieDecoderPlayer->SetVideoDecodeComplete();
                            movieDecoderPlayer->SetAudioDecodeComplete();
                            break;
                        }
                    }
                    break;
                }
                startPlaybackCount++;
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(5));
            }
            movieDecoderPlayer->m_PlaybackTime = Clock::GetNowUs();
            // Wait till playback is complete
            int logCount = 0;
            NN_LOG("\n ******* Waiting for Playback to complete ******* \n");
            while( 1 )
            {
                bool isPlaybackComplete = movieDecoderPlayer->IsPlaybackComplete();
                if( isPlaybackComplete == true )
                {
                    if( loopPlayback == true )
                    {
                        if( movie::Status_Success != movieDecoderPlayer->HandleLoopBack() )
                            break;
                    }
                    else
                        break;
                }
                logCount++;
                // Log once in 5 seconds
                if( logCount == 50 )
                {
                    NN_SDK_LOG("\n ******* Waiting for Playback to complete ******* \n");
                    logCount = 0;
                }
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                if( resetCount > 0 )
                {
                    nn::os::SleepThread(nn::TimeSpan::FromSeconds(10));
                    movieDecoderPlayer->Reset();
                    --resetCount;
                }
            }
            movieDecoderPlayer->m_PlaybackTime = ( Clock::GetNowUs() - movieDecoderPlayer->m_PlaybackTime ) / 1000;
            NN_LOG(" \n ******* Total Playback Time = : %lld msec \n", movieDecoderPlayer->m_PlaybackTime);
            // Finalize resources
            movieDecoderPlayer->Finalize();
            if( ( runPlaybackAgingTest == true ) && ( playbackAgingRepeatCount > 0 ) )
                NN_SDK_LOG("\n MovieDecoderPlayer:: Aging test PlayBack repeat count : %d \n", ++playbackAgingRepeatCount);
        } while( runPlaybackAgingTest == true );
        nv::FinalizeGraphics();
    }
    PrintMemoryUsageStats();
}//NOLINT(impl/function_size)

MovieDecoderPlayer::MovieDecoderPlayer()
{
    m_Extractor = NULL;
    m_VideoDecoder = NULL;
    m_AudioDecoder = NULL;
    m_MovieAudioOutputHandler = NULL;
    m_MovieVideoOutputHandler = NULL;
    m_ByteStreamReader = NULL;
    m_AudioDecoderType = movie::DecoderType::DecoderType_Unknown;
    m_VideoDecoderType = movie::DecoderType::DecoderType_Unknown;
    m_ThreadStackSize = 1024 * 256;
    m_Tracks = 0;
    m_Width = -1;
    m_Height = -1;
    m_NumChannels = -1;
    m_SampleRate = -1;
    m_AudioTrackIndex = -1;
    m_VideoTrackIndex = -2;
    m_VideoFrameRate = -1.0;
    m_TrackDurationUs = -1;
    SetAudioDecodeComplete();
    SetVideoDecodeComplete();
    m_AudioDecoderCreated = false;
    m_VideoDecoderCreated = false;
    m_AudioOutputBufferReceived = 0;
    m_VideoOutputBufferReceived = 0;
    m_ClientReader = NULL;
    m_MediaClock = nullptr;
    m_PlaybackComplete = false;
    m_PlaybackTime = 0;
    m_VideoPresentationTimeUs = 0;
    m_VariableSizeAudioFrames = false;
    nn::os::InitializeMutex( &m_VideoListMutex, false, 0 );
    nn::os::InitializeMutex( &m_AudioListMutex, false, 0 );
}

movie::Status MovieDecoderPlayer::Initialize(uint64_t coreMask)
{
    movie::Status movieStatus = movie::Status_Success;

    // Maximum number of messages in Audio and Video input reader of message queue.
    m_BufferSize = 128;
    m_LoopCount = 0;
    m_PlaybackComplete = false;
    m_PlaybackTime = 0;
    m_VideoPresentationTimeUs = 0;
    m_AudioFormatChangeEventReceived = false;
    m_VideoFormatChangeEventReceived = false;
    if( coreMask )//set only valid core mask
        movie::SetCoreMask(coreMask);
    // Initialize Event for audio video worker thread
    nn::os::InitializeEvent( &m_AudioVideoWorkerThreadExitEvent, false, nn::os::EventClearMode_ManualClear );
    // Initialize MultiWaitHolder for AudioVideoWorkerThread
    nn::os::InitializeMultiWaitHolder(&m_AudioVideoWorkerThreadExitEventHolder, &m_AudioVideoWorkerThreadExitEvent);
    // Initialize MultiWait for AudioVideoWorkerThread
    nn::os::InitializeMultiWait(&m_AudioVideoWorkerMultiWait);
    // Initialize Events to register with audio decoder
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.errorEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.formatChangedEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.inputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_AudioDecoderEventTypes.outputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);
    // Fill movie::DecoderEvents structure with audio events (for audio decoder)
    m_AudioDecoderEvents.errorEvent = &m_AudioDecoderEventTypes.errorEvent;
    m_AudioDecoderEvents.formatChangedEvent = &m_AudioDecoderEventTypes.formatChangedEvent;
    m_AudioDecoderEvents.inputBufferAvailableEvent = &m_AudioDecoderEventTypes.inputBufferAvailableEvent;
    m_AudioDecoderEvents.outputBufferAvailableEvent = &m_AudioDecoderEventTypes.outputBufferAvailableEvent;
    // Initialize AudioDecoderMultiWaitHolder with audio events (Used by movie decoder player thread to listen for audio events)
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.errorEventHolder, &m_AudioDecoderEventTypes.errorEvent);
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder, &m_AudioDecoderEventTypes.formatChangedEvent);
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder, &m_AudioDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::InitializeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder, &m_AudioDecoderEventTypes.outputBufferAvailableEvent);
    // Initialize Events to register with video decoder
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.errorEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.formatChangedEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.inputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);
    nn::os::InitializeEvent(&m_VideoDecoderEventTypes.outputBufferAvailableEvent, false, nn::os::EventClearMode_ManualClear);
    // Fill movie::DecoderEvents structure with video events (for video decoder)
    m_VideoDecoderEvents.errorEvent = &m_VideoDecoderEventTypes.errorEvent;
    m_VideoDecoderEvents.formatChangedEvent = &m_VideoDecoderEventTypes.formatChangedEvent;
    m_VideoDecoderEvents.inputBufferAvailableEvent = &m_VideoDecoderEventTypes.inputBufferAvailableEvent;
    m_VideoDecoderEvents.outputBufferAvailableEvent = &m_VideoDecoderEventTypes.outputBufferAvailableEvent;
    // Initialize VideoDecoderMultiWaitHolderTypes with video events (Used by movie decoder player thread to listen for video events)
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.errorEventHolder, &m_VideoDecoderEventTypes.errorEvent);
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder, &m_VideoDecoderEventTypes.formatChangedEvent);
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder, &m_VideoDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::InitializeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder, &m_VideoDecoderEventTypes.outputBufferAvailableEvent);
    // Initialize Events for Player to wait for format changes.
    nn::os::InitializeEvent(&m_FormatChangedPlayerEvent, false, nn::os::EventClearMode_ManualClear);
    m_VideoInputMessageBuffer = new uintptr_t[m_BufferSize];
    if( m_VideoInputMessageBuffer == NULL )
        return movie::Status_OutOfMemory;
    m_AudioInputMessageBuffer = new uintptr_t[ m_BufferSize ];
    if( m_AudioInputMessageBuffer == NULL )
        return movie::Status_OutOfMemory;
    // Initialize MessageQueue used by movie decoder player
    nn::os::InitializeMessageQueue( &m_VideoInputMessageQueue, m_VideoInputMessageBuffer, m_BufferSize );
    // Initialize MessageQueue used by movie decoder player
    nn::os::InitializeMessageQueue(&m_AudioInputMessageQueue, m_AudioInputMessageBuffer, m_BufferSize);
    // Create Audio and Video InputReader threads
    movieStatus = CreateThreadForVideoInputReader();
    if( movieStatus == movie::Status_Success )
    {
        // Create AudioInputReader thread
        movieStatus = CreateThreadForAudioInputReader();
        if( movieStatus == movie::Status_Success )
        {
            // Create AudioVideoWorker thread
            movieStatus = CreateThreadForAudioVideoWorker();
        }
    }
    m_MediaClock = new MediaClock();
    if( m_MediaClock == nullptr )
        movieStatus = movie::Status_OutOfMemory;

    m_AudioInputReadThreadStarted = false;
    m_VideoInputReadThreadStarted = false;
    m_AudioVideoWorkerThreadStarted = false;
    return movieStatus;
}

void MovieDecoderPlayer::Finalize()
{
    // Stop decoder and finalize resources
    if( m_AudioDecoder != NULL )
        StopDecoder(m_AudioDecoder);
    if( m_VideoDecoder != NULL )
        StopDecoder(m_VideoDecoder);
    StopAndDestroyVideoInputReaderThread();
    StopAndDestroyAudioInputReaderThread();
    StopAudioVideoRenderer();
    DestroypAudioVideoRenderer();
    if( m_AudioDecoder != NULL )
    {
        DestroyDecoder(m_AudioDecoder);
        m_AudioDecoder = NULL;
    }
    if( m_VideoDecoder != NULL )
    {
        DestroyDecoder(m_VideoDecoder);
        m_VideoDecoder = NULL;
    }
    m_TrackFormat.clear();
    if( m_Extractor != NULL )
    {
        DestroyExtractor(m_Extractor);
        m_Extractor = NULL;
    }
    if( m_ClientReader != NULL )
    {
        delete m_ClientReader;
        m_ClientReader = NULL;
    }
    if( m_ByteStreamReader != NULL )
    {
        m_ByteStreamReader->Close();
        delete m_ByteStreamReader;
        m_ByteStreamReader = NULL;
    }
    StopAndDestroyAudioVideoWorkerThread();
    if( m_AudioVideoWorkerThreadStack != NULL )
    {
        free(m_AudioVideoWorkerThreadStack);
        m_AudioVideoWorkerThreadStack = NULL;
    }
    if( m_VideoInputReadThreadStack != NULL )
    {
        free(m_VideoInputReadThreadStack);
        m_VideoInputReadThreadStack = NULL;
    }
    if( m_AudioInputReadThreadStack != NULL )
    {
        free(m_AudioInputReadThreadStack);
        m_AudioInputReadThreadStack = NULL;
    }
    // Finalize AudioVideoWorkerThreadExit event
    nn::os::FinalizeEvent( &m_AudioVideoWorkerThreadExitEvent );
    // Finalize VideoDecoder events
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.errorEvent);
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.formatChangedEvent);
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::FinalizeEvent(&m_VideoDecoderEventTypes.outputBufferAvailableEvent);
    // Finalize VideoDecoder MultiWaitHolder
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);
    // Finalize AudioDecoder events
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.errorEvent);
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.formatChangedEvent);
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.inputBufferAvailableEvent);
    nn::os::FinalizeEvent(&m_AudioDecoderEventTypes.outputBufferAvailableEvent);
    // Finalize AudioDecoder MultiWaitHolder
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::FinalizeMultiWaitHolder(&m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);
    // Finalize AudioVideoWorkerThread MultiWait
    nn::os::FinalizeMultiWait(&m_AudioVideoWorkerMultiWait);
    // Finalize Mutex used by Audio and video list
    nn::os::FinalizeMutex( &m_VideoListMutex );
    nn::os::FinalizeMutex( &m_AudioListMutex );
    nn::os::FinalizeEvent(&m_FormatChangedPlayerEvent);
    // Finalize MessageQueue used by movie decoder player
    nn::os::FinalizeMessageQueue( &m_VideoInputMessageQueue );
    if( m_VideoInputMessageBuffer != NULL )
        delete [] m_VideoInputMessageBuffer;
    nn::os::FinalizeMessageQueue(&m_AudioInputMessageQueue);
    if( m_AudioInputMessageBuffer != NULL )
        delete[] m_AudioInputMessageBuffer;
    if( m_MediaClock != nullptr )
    {
        delete m_MediaClock;
        m_MediaClock = nullptr;
    }
    if( m_SdcardMounted == true )
        nn::fs::Unmount("sdcard");
    else
        nn::fs::UnmountHostRoot();
}

void MovieDecoderPlayer::HandleErrorExit()
{
    Finalize();
    nv::FinalizeGraphics();
}

void MovieDecoderPlayer::ResetPlaybackComplete()
{
    ResetAudioDecodeComplete();
    ResetVideoDecodeComplete();
    m_PlaybackComplete = false;
}

bool MovieDecoderPlayer::IsPlaybackComplete()
{
    bool audioPlaybackComplete = true;
    if( m_MovieAudioOutputHandler != NULL )
        audioPlaybackComplete = m_MovieAudioOutputHandler->AllFramesRendered();
    bool videoPlaybackComplete = true;
    if( m_MovieVideoOutputHandler != NULL )
        videoPlaybackComplete = m_MovieVideoOutputHandler->AllFramesRendered();
    if( ( videoPlaybackComplete == true ) && ( audioPlaybackComplete == true ) )
        m_PlaybackComplete = true;
    if( m_PlaybackComplete == true )
        return true;
    else
        return false;
}

bool MovieDecoderPlayer::StartPlayback()
{
    bool hasSufficientAudioBuffers = true;
    if( IsAudioDecoderCreated() )
    {
        if( m_AudioOutputBufferReceived < m_AudioOutputBufferCountToStartPlayback )
            hasSufficientAudioBuffers = false;
    }
    bool hasSufficientVideoBuffers = true;
    if( IsVideoDecoderCreated() )
    {
        if( m_VideoOutputBufferReceived < m_VideoOutputBufferCountToStartPlayback )
            hasSufficientVideoBuffers = false;
    }
    if( ( hasSufficientAudioBuffers == true ) && ( hasSufficientVideoBuffers == true ) )
        return true;
    else
        return false;
}

movie::Status MovieDecoderPlayer::CreateThreadForVideoInputReader()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();
    m_VideoInputReadThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_VideoInputReadThreadStack == NULL )
        movieStatus = movie::Status_OutOfMemory;
    else
    {
        m_VideoInputReadThreadDone = false;
        nnResult = nn::os::CreateThread(&m_VideoInputReadThreadType,
                &VideoInputReadThreadFunction,
                (void*)this,
                m_VideoInputReadThreadStack,
                m_ThreadStackSize,
                nn::os::DefaultThreadPriority );
        if( nnResult.IsSuccess() )
        {
            nn::os::SetThreadName( &m_VideoInputReadThreadType, "VideoInputReadThread" );
            m_VideoInputReadThreadCreated = true;
        }
        else
            movieStatus = movie::Status_FailedToCreateThread;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CreateThreadForAudioInputReader()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();
    m_AudioInputReadThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_AudioInputReadThreadStack == NULL )
        movieStatus = movie::Status_OutOfMemory;
    else
    {
        nnResult = nn::os::CreateThread(&m_AudioInputReadThreadType,
            &AudioInputReadThreadFunction,
            ( void* )this,
            m_AudioInputReadThreadStack,
            m_ThreadStackSize,
            nn::os::DefaultThreadPriority);
        if( nnResult.IsSuccess() )
        {
            nn::os::SetThreadName(&m_AudioInputReadThreadType, "AudioInputReadThread");
            m_AudioInputReadThreadCreated = true;
        }
        else
            movieStatus = movie::Status_FailedToCreateThread;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CreateThreadForAudioVideoWorker()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();
    m_AudioVideoWorkerThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_AudioVideoWorkerThreadStack == NULL )
        movieStatus = movie::Status_OutOfMemory;
    else
    {
        nnResult = nn::os::CreateThread(&m_AudioVideoWorkerThreadType,
                &AudioVideoWorkerThreadFunction,
                (void*)this,
            m_AudioVideoWorkerThreadStack,
                m_ThreadStackSize,
                nn::os::DefaultThreadPriority );
        if( nnResult.IsSuccess() )
        {
            nn::os::SetThreadName( &m_AudioVideoWorkerThreadType, "AudioVideoWorkerThread" );
            m_AudioVideoWorkerThreadCreated = true;
        }
        else
            movieStatus = movie::Status_FailedToCreateThread;
    }
    return movieStatus;
}

void MovieDecoderPlayer:: StartVideoInputReaderThread()
{
    nn::os::StartThread( &m_VideoInputReadThreadType );
    m_VideoInputReadThreadStarted = true;
}

void MovieDecoderPlayer::StartAudioInputReaderThread()
{
    nn::os::StartThread(&m_AudioInputReadThreadType);
    m_AudioInputReadThreadStarted = true;
}

void MovieDecoderPlayer:: StartAudioVideoWorkerThread()
{
    nn::os::StartThread( &m_AudioVideoWorkerThreadType );
    m_AudioVideoWorkerThreadStarted = true;
}

void MovieDecoderPlayer:: StopAndDestroyVideoInputReaderThread()
{
    if( m_VideoInputReadThreadCreated == true )
    {
        m_VideoInputReadThreadDone = true;
        if( m_VideoInputReadThreadStarted == true )
        {
            SignalVideoInputReadThreadExit();
            nn::os::WaitThread(&m_VideoInputReadThreadType);
        }
        nn::os::DestroyThread( &m_VideoInputReadThreadType );
        m_VideoInputReadThreadCreated = false;
    }
}

void MovieDecoderPlayer::StopAndDestroyAudioInputReaderThread()
{
    if( m_AudioInputReadThreadCreated == true )
    {
        if( m_AudioInputReadThreadStarted == true )
        {
            SignalAudioInputReadThreadExit();
            nn::os::WaitThread(&m_AudioInputReadThreadType);
        }
        nn::os::DestroyThread(&m_AudioInputReadThreadType);
        m_AudioInputReadThreadCreated = false;
    }
}

void MovieDecoderPlayer:: StopAndDestroyAudioVideoWorkerThread()
{
    if( m_AudioVideoWorkerThreadCreated == true )
    {
        if( m_AudioVideoWorkerThreadStarted == true )
        {
            SignalAudioVideoWorkerThreadExit();
            nn::os::WaitThread(&m_AudioVideoWorkerThreadType);
        }
        nn::os::DestroyThread( &m_AudioVideoWorkerThreadType );
        m_AudioVideoWorkerThreadCreated = false;
    }
}

movie::Status MovieDecoderPlayer::CreateAudioVideoRenderer()
{
    movie::Status movieStatus = movie::Status_Success;
    if( IsAudioDecoderCreated() )
    {
        m_MovieAudioOutputHandler = new MovieAudioOutputHandler();
        if( m_MovieAudioOutputHandler == NULL )
        {
            movieStatus = movie::Status_OutOfMemory;
            return movieStatus;
        }
    }
    if( IsVideoDecoderCreated() )
    {
        m_MovieVideoOutputHandler = new MovieVideoOutputHandler(nullptr, m_VideoDecoderMode, m_VideoDecoderOutputFormat);
        if( m_MovieVideoOutputHandler == NULL )
            movieStatus = movie::Status_OutOfMemory;
    }
    return movieStatus;
}

void MovieDecoderPlayer::StopAudioVideoRenderer()
{
    if( m_MovieAudioOutputHandler != NULL )
    {
        m_MovieAudioOutputHandler->Stop();
        m_MovieAudioOutputHandler->Close();
        m_MovieAudioOutputHandler->Finalize();
    }
    if( m_MovieVideoOutputHandler != NULL )
    {
        m_MovieVideoOutputHandler->Stop();
        m_MovieVideoOutputHandler->Close();
        m_MovieVideoOutputHandler->Finalize();
    }
}

void MovieDecoderPlayer::DestroypAudioVideoRenderer()
{
    if( m_MovieAudioOutputHandler != NULL )
        delete m_MovieAudioOutputHandler;
    if( m_MovieVideoOutputHandler != NULL )
        delete m_MovieVideoOutputHandler;
}

movie::Status MovieDecoderPlayer::CreateExtractor(const char* uri, movie::ContainerType container, movie::Extractor** extractor, movie::ClientStreamReader **clientReader)
{
    movie::Status movieStatus = movie::Status_Success;
    // Creates Extractor with an internal data cache of default size (movie::CacheSize_20MB)
    // Data cache size can be set by creating extractor using a constructor which takes cache size as input parameter
    // Example: Create an extractor for mp4 container type with 10MB of cache
    // movie::Extractor* movieExtractor = new movie::Extractor(movie::ContainerType_Mpeg4, movie::CacheSize_10MB);
    std::unique_ptr<movie::Extractor> movieExtractor{ new movie::Extractor(container) };
    if( movieExtractor == NULL )
    {
        *extractor = NULL;
        movieStatus = movie::Status_OutOfMemory;
    }
    else
    {
        nn::fs::FileHandle fileHandle;
        fileHandle.handle = NULL;
        nn::Result result = nn::fs::OpenFile(&fileHandle, uri, nn::fs::OpenMode_Read);
        if( result.IsFailure() )
        {
            NN_SDK_LOG("\n Failed to open %s \n \n Terminating PlayerSample MediaDecoderPlayer\n", uri);
            DestroyExtractor(movieExtractor.release());
            movieStatus = movie::Status_NotFound;
            return movieStatus;
        }
        else
            nn::fs::CloseFile(fileHandle);
        movie::ClientStreamReader* movieClientReader = new StreamReader();
        if( movieClientReader == NULL )
        {
            movieStatus = movie::Status_OutOfMemory;
            *clientReader = NULL;
            return movieStatus;
        }
        movieStatus = movieExtractor->SetDataSource(movieClientReader, uri);
        if( movieStatus == movie::Status_Success )
        {
            m_Tracks = 0;
            movieStatus = movieExtractor->GetTrackCount(&m_Tracks);
            if( movieStatus == movie::Status_Success )
            {
                if( m_Tracks <= 0 )
                    movieStatus = movie::Status_TrackNotFound;
            }
        }
        *extractor = movieExtractor.release();
        *clientReader = movieClientReader;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CreateDecoder(movie::Decoder** decoder, movie::DecoderType decoderType, movie::DecoderMode decoderMode, movie::DecoderOutputFormat decoderOutputFormat)
{
    if( decoder == NULL )
        return movie::Status_ErrorBadValue;
    else
    {
        if( true == IsAudioDecoder(decoderType) )
        {
            m_AudioDecoderMode = decoderMode;
            m_AudioDecoderOutputFormat = decoderOutputFormat;
        }
        if( true == IsVideoDecoder(decoderType) )
        {
            m_VideoDecoderMode = decoderMode;
            m_VideoDecoderOutputFormat = decoderOutputFormat;
        }
        movie::Decoder* movieDecoder = new movie::Decoder(decoderType, decoderOutputFormat, decoderMode);
        if( movieDecoder == NULL )
        {
            *decoder = NULL;
            return movie::Status_OutOfMemory;
        }
        else
        {
            *decoder = movieDecoder;
            return movie::Status_Success;
        }
    }
}

movie::Status MovieDecoderPlayer::ConfigureDecoder(movie::Decoder* decoder, movie::DecoderEvents* decoderEvents, movie::MediaData* configuration)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( decoder == NULL )
        return movie::Status_ErrorBadValue;
    movieStatus = decoder->RegisterClientEvents(decoderEvents);
    if( movieStatus == movie::Status_Success )
        movieStatus = decoder->Configure(configuration);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::StartDecoder(movie::Decoder* decoder)
{
    if( decoder == NULL )
        return movie::Status_NullDecoder;
    else
        return decoder->Start();
}

movie::Status MovieDecoderPlayer::StopDecoder(movie::Decoder* decoder)
{
    if( decoder == NULL )
        return movie::Status_NullDecoder;
    else
        return decoder->Stop();
}

movie::Status MovieDecoderPlayer::FlushDecoder(movie::Decoder* decoder)
{
    if( decoder == NULL )
        return movie::Status_NullDecoder;
    else
        return decoder->Flush();
}

void MovieDecoderPlayer::DestroyDecoder(movie::Decoder* decoder)
{
    if( decoder != NULL )
        delete decoder;
}

movie::Status MovieDecoderPlayer::CreateDecoders(movie::Extractor*  extractor,
    movie::DecoderMode audioDecoderMode,
    movie::DecoderOutputFormat audioDecoderOutputFormat,
    movie::DecoderMode videoDecoderMode,
    movie::DecoderOutputFormat videoDecoderOutputFormat)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::MediaData trackFormat;
    if( extractor == NULL )
        movieStatus = movie::Status_NullExtractor;
    else
    {
        for( int i = 0; i < m_Tracks; i ++ )
        {
            movie::DecoderType decoderType = movie::DecoderType::DecoderType_Unknown;
            extractor->GetTrackConfiguration(i, &trackFormat);

            movie::sample::SamplePrintVisitor printer_;    // demonstrates the movie::MediaData's Visit/Visitor functionality
            trackFormat.Visit(printer_);                    // prints all of the entries in our trackFormat object

            const char *mime;
            if( trackFormat.FindString("mime", &mime) )
            {
                if( ( !strncasecmp("video/", mime, 6) ) && ( m_VideoDecoder == NULL) )
                {
                    if( !strncasecmp("video/avc", mime, 9) )
                        decoderType = movie::DecoderType_VideoAvc;
                    else if( !strncasecmp("video/x-vnd.on2.vp8", mime, 19) )
                        decoderType = movie::DecoderType_VideoVp8;
                    else if( !strncasecmp("video/x-vnd.on2.vp9", mime, 19) )
                        decoderType = movie::DecoderType_VideoVp9;
                    movieStatus = CreateDecoder(&m_VideoDecoder, decoderType, videoDecoderMode, videoDecoderOutputFormat);
                    if( movieStatus == movie::Status_Success )
                    {
                        movieStatus = ConfigureDecoder(m_VideoDecoder, &m_VideoDecoderEvents, &trackFormat);
                        if( movieStatus == movie::Status_Success )
                        {
                            m_VideoTrackIndex = i;
                            int32_t width = 0;
                            int32_t height = 0;
                            trackFormat.FindInt32("width", &width);
                            m_Width = width;
                            trackFormat.FindInt32("height", &height);
                            m_Height = height;
                            int32_t outSize = 0;
                            int32_t outalign = 0;
                            int32_t videoFrameRate = 0;
                            trackFormat.FindInt32("frame-rate", &videoFrameRate);
                            if( videoFrameRate > 0 )
                                m_VideoFrameRate = videoFrameRate;
                            else
                                m_VideoFrameRate = 60;
                            m_VideoDecoder->GetOutputBufferSize(&outSize, &outalign);
                            movieStatus = extractor->SelectTrack(i);
                            m_VideoDecoderCreated = true;
                            m_VideoDecoderType = decoderType;
                            ResetVideoDecodeComplete();
                            NN_SDK_LOG( "\n Video decoder created \n" );
                        }
                        else
                        {
                            delete m_VideoDecoder;
                            m_VideoDecoder = NULL;
                        }
                    }
                }
                if( ( !strncasecmp("audio/", mime, 6) ) && ( m_AudioDecoder == NULL) )
                {
                    if( !strncasecmp("audio/mp4a-latm", mime, 15) )
                        decoderType = movie::DecoderType_AudioAac;
                    else if( !strncasecmp("audio/vorbis", mime, 12) )
                        decoderType = movie::DecoderType_AudioVorbis;
                    int32_t sampleRate = 0;
                    trackFormat.FindInt32("sample-rate", &sampleRate);
                    m_SampleRate = sampleRate;
                    int32_t numChannels = 0;
                    trackFormat.FindInt32("channel-count", &numChannels);
                    m_NumChannels = numChannels;
                    if( ( sampleRate >= 16000 ) && ( numChannels <= 6) )
                    {
                        movieStatus = CreateDecoder(&m_AudioDecoder, decoderType, audioDecoderMode, audioDecoderOutputFormat);
                        if( movieStatus == movie::Status_Success )
                        {
                            movieStatus = ConfigureDecoder(m_AudioDecoder, &m_AudioDecoderEvents, &trackFormat);
                            if( movieStatus == movie::Status_Success )
                            {
                                m_AudioTrackIndex = i;
                                int32_t outSize = 0;
                                int32_t outalign = 0;
                                m_AudioDecoder->GetOutputBufferSize(&outSize, &outalign);
                                movieStatus = extractor->SelectTrack(i);
                                m_AudioDecoderCreated = true;
                                m_AudioDecoderType = decoderType;
                                ResetAudioDecodeComplete();
                                NN_SDK_LOG("\n Audio decoder created \n");
                            }
                        }
                    }
                    else
                    {
                        if( numChannels > 6 )
                        {
                            NN_SDK_LOG("\n This media file has audio track with %d channels, ", numChannels);
                            NN_SDK_LOG(" MovieDecodePlayer Sample supports up to 5.1 (6) channels for audio playback \n");
                        }
                        if( sampleRate < 16000 )
                        {
                            NN_SDK_LOG("\n This media file has audio track with %d Hz Sample rate, ", sampleRate);
                            NN_SDK_LOG(" MovieDecodePlayer Sample does not supports sample rate lower than 16 KHz for audio playback \n");
                        }
                        movieStatus = movie::Status_UnknownError;
                    }
                    if( movieStatus != movie::Status_Success )
                    {
                        m_AudioDecoderType = movie::DecoderType::DecoderType_Unknown;
                        m_AudioDecoderCreated = false;
                        delete m_AudioDecoder;
                        m_AudioDecoder = NULL;
                    }
                }
            }
            m_TrackFormat.push_back(trackFormat);
            trackFormat.Clear();
        }
    }
    if( ( m_AudioDecoder == NULL ) && ( m_VideoDecoder == NULL ) )
        movieStatus = movie::Status_UnknownError;
    return movieStatus;
}//NOLINT(impl/function_size)

/**
 * Destroy extractor and release resources
 */
void MovieDecoderPlayer::DestroyExtractor(movie::Extractor* extractor )
{
    if( extractor != NULL )
        delete extractor;
}

/**
 * Empty buffers (input) to be filled with compressed audio data
 */
movie::Status MovieDecoderPlayer::AddBufferToAudioIndexList(int32_t bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_AudioListMutex );
    m_AudioBufferIndexList.push_back(bufferIndex);
    nn::os::UnlockMutex( &m_AudioListMutex );
    return movieStatus;
}

movie::Status MovieDecoderPlayer::RemoveBufferFromAudioIndexList(int32_t* bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_AudioListMutex );
    int32_t index = -1;
    if( m_AudioBufferIndexList.size() > 0 )
    {
        index = m_AudioBufferIndexList.front();
        m_AudioBufferIndexList.erase(m_AudioBufferIndexList.begin());
    }
    *bufferIndex = index;
    nn::os::UnlockMutex( &m_AudioListMutex );
    return movieStatus;
}

movie::Status MovieDecoderPlayer::GetAudioIndexListSize(int32_t* indexListSize)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_AudioListMutex);
    *indexListSize = m_AudioBufferIndexList.size();
    nn::os::UnlockMutex(&m_AudioListMutex);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::GetVideoIndexListSize(int32_t* indexListSize)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex(&m_VideoListMutex);
    *indexListSize = m_VideoBufferIndexList.size();
    nn::os::UnlockMutex(&m_VideoListMutex);
    return movieStatus;
}

/**
 * Empty buffers (input) to be filled with compressed video data
 */
movie::Status MovieDecoderPlayer::AddBufferToVideoIndexList(int32_t bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_VideoListMutex );
    m_VideoBufferIndexList.push_back(bufferIndex);
    nn::os::UnlockMutex( &m_VideoListMutex );
    return movieStatus;
}

movie::Status MovieDecoderPlayer::RemoveBufferFromVideoIndexList(int32_t* bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_VideoListMutex );
    int32_t index = -1;
    if( m_VideoBufferIndexList.size() > 0 )
    {
        index = m_VideoBufferIndexList.front();
        m_VideoBufferIndexList.erase(m_VideoBufferIndexList.begin());
    }
    *bufferIndex = index;
    nn::os::UnlockMutex( &m_VideoListMutex );
    return movieStatus;
}

movie::Status MovieDecoderPlayer::ReadInputDataFromExtractorSendTodecoder(ByteStreamReader* byteStreamReader, int index, movie::Decoder* decoder)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;
    decoder->GetInputBuffer(index, &buffer);
    int32_t frameDataSize = 0;

    movieStatus = byteStreamReader->ReadNextFrame((char*)buffer.Base(), buffer.Capacity(), &frameDataSize, &presentationTimeUs);
    if( movieStatus == movie::Status_EndOfStream )
    {
        flags = movie::BufferFlags_EndOfStream;
        // End of stream reached
        movieStatus = movie::Status_EndOfStream;
    }
    else if( movieStatus == movie::Status_NotEnoughData )
    {
        int32_t retryReadCount = 16;
        while( retryReadCount -- )
        {
            movieStatus = byteStreamReader->ReadNextFrame(( char* ) buffer.Base(), buffer.Capacity(), &frameDataSize, &presentationTimeUs);
            if( movieStatus == movie::Status_Success )
                break;
            else if( movieStatus == movie::Status_NotEnoughData )
                continue;
            else
            {
                flags = movie::BufferFlags_EndOfStream;
                movieStatus = movie::Status_EndOfStream;
                break;
            }
        }
    }
    decoder->SendInputBufferForDecode(index, 0, frameDataSize, presentationTimeUs, flags);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::ReadInputDataFromExtractorSendTodecoder(int index, movie::Decoder* decoder)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;
    movieStatus = decoder->GetInputBuffer(index, &buffer);
    if( movieStatus == movie::Status_Success )
    {
        buffer.SetRange(0, 0);
        movieStatus = m_Extractor->Read(&buffer);
        if( ( movieStatus != movie::Status_Success ) || ( buffer.Size() <= 0 ) || ( movieStatus == movie::Status_EndOfStream ) )
            flags = movie::BufferFlags_EndOfStream;
        m_Extractor->GetSampleTime(&presentationTimeUs);
        decoder->SendInputBufferForDecode(index, buffer.Offset(), buffer.Size(), presentationTimeUs, flags);
        m_Extractor->Advance();
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SendEosToDecoder()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    movie::Buffer buffer;
    bool audioEosSent = true;
    bool videoEosSent = true;
    uint32_t flags = movie::BufferFlags_EndOfStream;
    if( true == IsAudioDecoderCreated() )
    {
        nn::os::LockMutex(&m_AudioListMutex);
        int audioInputListSize = m_AudioBufferIndexList.size();
        nn::os::UnlockMutex(&m_AudioListMutex);
        audioEosSent = false;
        for( int i = 0; i < audioInputListSize; i++ )
        {
            int index = -1;
            RemoveBufferFromAudioIndexList(&index);
            if( index != -1 )
            {
                m_AudioDecoder->GetInputBuffer(index, &buffer);
                m_AudioDecoder->SendInputBufferForDecode(index, 0, 0, 0, flags);
                audioEosSent = true;
            }
        }
    }
    if( true == IsVideoDecoderCreated() )
    {
        videoEosSent = false;
        nn::os::LockMutex(&m_VideoListMutex);
        int videoInputListSize = m_VideoBufferIndexList.size();
        nn::os::UnlockMutex(&m_VideoListMutex);
        for( int i = 0; i < videoInputListSize; i++ )
        {
            int index = -1;
            RemoveBufferFromVideoIndexList(&index);
            if( index != -1 )
            {
                m_VideoDecoder->GetInputBuffer(index, &buffer);
                m_VideoDecoder->SendInputBufferForDecode(index, 0, 0, 0, flags);
                videoEosSent = true;
            }
        }
    }
    if( ( audioEosSent == true ) && ( videoEosSent == true ) )
        movieStatus = movie::Status_Success;
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CheckForInputBuffersAndReadVideoInputData()
{
    movie::Status movieStatus = movie::Status_Success;
    size_t currentTrackIndex = -1;
    int index = -1;
    // For byte stream there is only one track [only raw video streams (h264, vp8, vp9) are supported]
    if( m_ByteStreamReader == NULL )
    {
        if( m_Extractor != NULL )
        {
            movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
            if( ( movieStatus != movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
            {
                SendEosToDecoder();
                return movieStatus;
            }
        }
    }
    else
    {
        if( ( m_VideoDecoderType    == movie::DecoderType_VideoVp8 )
            || ( m_VideoDecoderType == movie::DecoderType_VideoVp9 )
            || ( m_VideoDecoderType == movie::DecoderType_VideoAvc ) )
        {
            currentTrackIndex = m_VideoTrackIndex;
        }
    }
    int videoIndexListSize = 0;
    if( currentTrackIndex == m_VideoTrackIndex )
    {
        RemoveBufferFromVideoIndexList(&index);
        if( index != -1 )
        {
            if( m_ByteStreamReader != NULL )
                ReadInputDataFromExtractorSendTodecoder(m_ByteStreamReader, index, m_VideoDecoder);
            else
                ReadInputDataFromExtractorSendTodecoder(index, m_VideoDecoder);
        }
        currentTrackIndex = -1;
        if( m_Extractor != NULL )
        {
            movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
            if( movieStatus == movie::Status_Success )
            {
                if( currentTrackIndex == m_VideoTrackIndex )
                {
                    GetVideoIndexListSize(&videoIndexListSize);
                    if( videoIndexListSize > 0 )
                        SignalVideoInputBufferAvailable();
                }
            }
        }
    }
    else
    {
        // Try to read audio data from Extractor
        int audioIndexListSize = 0;
        GetAudioIndexListSize(&audioIndexListSize);
        if( audioIndexListSize > 0 )
            SignalAudioInputBufferAvailable();
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::CheckForInputBuffersAndReadAudioInputData()
{
    movie::Status movieStatus = movie::Status_Success;
    size_t currentTrackIndex = -1;
    int index = -1;
    int audioIndexListSize = 0;
    int videoIndexListSize = 0;
    // For byte stream there is only one track [only raw video streams (h264, vp8, vp9) are supported]
    if( m_ByteStreamReader == NULL )
    {
        if( m_Extractor != NULL )
        {
            currentTrackIndex = -1;
            movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
            if( ( movieStatus != movie::Status_Success ) || ( movieStatus == movie::Status_EndOfStream ) )
            {
                SendEosToDecoder();
                return movieStatus;
            }
        }
        else
        {
            if( ( m_VideoDecoderType == movie::DecoderType_VideoVp8 )
                || ( m_VideoDecoderType == movie::DecoderType_VideoVp9 )
                || ( m_VideoDecoderType == movie::DecoderType_VideoAvc ) )
            {
                currentTrackIndex = m_VideoTrackIndex;
            }
        }
        if( currentTrackIndex == m_AudioTrackIndex )
        {
            GetAudioIndexListSize(&audioIndexListSize);
            if( audioIndexListSize > 0 )
            {
                RemoveBufferFromAudioIndexList(&index);
                if( index != -1 )
                {
                    if( m_ByteStreamReader != NULL )
                        ReadInputDataFromExtractorSendTodecoder(m_ByteStreamReader, index, m_AudioDecoder);
                    else
                        ReadInputDataFromExtractorSendTodecoder(index, m_AudioDecoder);
                }
            }
            if( m_Extractor != NULL )
            {
                currentTrackIndex = -1;
                movieStatus = m_Extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
                if( movieStatus == movie::Status_Success )
                {
                    if( currentTrackIndex == m_AudioTrackIndex )
                    {
                        audioIndexListSize = 0;
                        GetAudioIndexListSize(&audioIndexListSize);
                        if( audioIndexListSize > 0 )
                            SignalAudioInputBufferAvailable();
                    }
                }
            }
        }
        else
        {
            videoIndexListSize = 0;
            GetVideoIndexListSize(&videoIndexListSize);
            if( videoIndexListSize > 0 )
                SignalVideoInputBufferAvailable();
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SignalVideoInputBufferAvailable()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue( &m_VideoInputMessageQueue, MovieDecoderPlayer::MoviePlayerMessage_VideoInputAvailable);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SignalAudioInputBufferAvailable()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue(&m_AudioInputMessageQueue, MovieDecoderPlayer::MoviePlayerMessage_AudioInputAvailable);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SignalAudioInputReadThreadExit()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue(&m_AudioInputMessageQueue, MovieDecoderPlayer::MoviePlayerMessage_AudioInputThreadExit);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SignalVideoInputReadThreadExit()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SendMessageQueue( &m_VideoInputMessageQueue, MovieDecoderPlayer::MoviePlayerMessage_VideoInputThreadExit);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SignalAudioVideoWorkerThreadExit()
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::SignalEvent(&m_AudioVideoWorkerThreadExitEvent );
    return movieStatus;
}

void VideoInputReadThreadFunction(void *arg)
{
    MovieDecoderPlayer *movieDecoderPlayer = (MovieDecoderPlayer*) arg;
    for (;;)
    {
        uintptr_t data;
        nn::os::ReceiveMessageQueue( &data, &movieDecoderPlayer->m_VideoInputMessageQueue );
        if( data == MovieDecoderPlayer::MoviePlayerMessage_VideoInputAvailable )
            movieDecoderPlayer->CheckForInputBuffersAndReadVideoInputData();
        else if( data == MovieDecoderPlayer::MoviePlayerMessage_VideoInputThreadExit )
        {
            movieDecoderPlayer->m_VideoInputReadThreadDone = true;
            break;
        }
    }
}

void AudioInputReadThreadFunction(void *arg)
{
    MovieDecoderPlayer *movieDecoderPlayer = ( MovieDecoderPlayer* ) arg;
    for( ;;)
    {
        uintptr_t data;
        nn::os::ReceiveMessageQueue(&data, &movieDecoderPlayer->m_AudioInputMessageQueue);
        if( data == MovieDecoderPlayer::MoviePlayerMessage_AudioInputAvailable )
            movieDecoderPlayer->CheckForInputBuffersAndReadAudioInputData();
        else if( data == MovieDecoderPlayer::MoviePlayerMessage_AudioInputThreadExit )
            break;
    }
}

movie::Status MovieDecoderPlayer::VideoInputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    movie::Buffer buffer;
    int32_t numberOfBuffersAvailable = 0;;
    movieStatus = m_VideoDecoder->AcquireInputBufferIndex(&index, &numberOfBuffersAvailable);
    if( movieStatus == movie::Status_Success )
    {
        AddBufferToVideoIndexList(index);
        SignalVideoInputBufferAvailable();
        for( int32_t i = 0; i < numberOfBuffersAvailable; i ++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_VideoDecoder->AcquireInputBufferIndex(&index, &remainingBuffers);
            if( movieStatus == movie::Status_Success )
            {
                 AddBufferToVideoIndexList(index);
                 SignalVideoInputBufferAvailable();
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::VideoOutputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    int64_t presentationTimeUs = 0ll;
    uint32_t flags = 0;
    int32_t numberOfBuffersAvailable = 0;

    movie::Buffer buffer;

    movieStatus = m_VideoDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &numberOfBuffersAvailable);
    if( movieStatus == movie::Status_Success )
    {
        if( m_ByteStreamReader != NULL )
            presentationTimeUs = m_VideoPresentationTimeUs;
        m_MovieVideoOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
        m_VideoOutputBufferReceived ++;
        if( m_ByteStreamReader != NULL )
            m_VideoPresentationTimeUs += m_ByteStreamReader->GetPresentationTimeIncrementInUs();
        if( flags == movie::BufferFlags_EndOfStream )
            SetVideoDecodeComplete();
        for( int32_t i = 0; i < numberOfBuffersAvailable; i ++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_VideoDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &remainingBuffers);
            if( movieStatus == movie::Status_Success )
            {
                if( flags == movie::BufferFlags_EndOfStream )
                     SetVideoDecodeComplete();
                if( m_ByteStreamReader != NULL )
                    presentationTimeUs = m_VideoPresentationTimeUs;
                m_MovieVideoOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
                m_VideoOutputBufferReceived ++;
                if( m_ByteStreamReader != NULL )
                    m_VideoPresentationTimeUs += m_ByteStreamReader->GetPresentationTimeIncrementInUs();
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::VideoFormatChangedEvent()
{
    NN_SDK_LOG( "\n VideoFormatChangedEvent \n" );
    movie::Status movieStatus = movie::Status_Success;
    movie::MediaData format;
    movieStatus = m_VideoDecoder->GetOutputFormat(&format);
    if( movieStatus == movie::Status_Success )
    {
        int32_t width = 0;
        int32_t height = 0;
        format.FindInt32("width", &width);
        m_Width = width;
        format.FindInt32("height", &height);
        m_Height = height;
        NN_SDK_LOG("VideoFormatChangedEvent Width %d, Height %d\n", width, height);
    }
    format.Clear();
    m_VideoFormatChangeEventReceived = true;
    CheckAndSignalFormatChangeWaiter();
    return movieStatus;
}

movie::Status MovieDecoderPlayer::AudioInputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    movie::Buffer buffer;
    int32_t numberOfBuffersAvailable = 0;;
    movieStatus = m_AudioDecoder->AcquireInputBufferIndex(&index, &numberOfBuffersAvailable);
    if( movieStatus == movie::Status_Success )
    {
        AddBufferToAudioIndexList(index);
        SignalAudioInputBufferAvailable();
        for( int32_t i = 0; i < numberOfBuffersAvailable; i ++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_AudioDecoder->AcquireInputBufferIndex(&index, &remainingBuffers);
            if( movieStatus == movie::Status_Success )
            {
                 AddBufferToAudioIndexList(index);
                 SignalAudioInputBufferAvailable();
            }
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::AudioOutputBufferAvailableEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    int32_t index = -1;
    int64_t presentationTimeUs= 0ll;
    uint32_t flags = 0;
    int32_t numberOfBuffersAvailable = 0;
    movieStatus = m_AudioDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &numberOfBuffersAvailable);
    if( movieStatus == movie::Status_Success )
    {
        m_MovieAudioOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
        m_AudioOutputBufferReceived ++;
        if( flags == movie::BufferFlags_EndOfStream )
            SetAudioDecodeComplete();
        for( int32_t i = 0; i < numberOfBuffersAvailable; i ++)
        {
            int32_t remainingBuffers = 0;
            movieStatus = m_AudioDecoder->AcquireOutputBufferIndex(&index, &presentationTimeUs, &flags, &remainingBuffers);
            if( movieStatus == movie::Status_Success )
            {
                m_AudioOutputBufferReceived ++;
                m_MovieAudioOutputHandler->OnOutputAvailable(index, presentationTimeUs, flags);
            }
            if( flags == movie::BufferFlags_EndOfStream )
                 SetAudioDecodeComplete();
        }
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::AudioFormatChangedEvent()
{
    movie::Status movieStatus = movie::Status_Success;
    movie::MediaData audioFormat;
    movieStatus = m_AudioDecoder->GetOutputFormat(&audioFormat);
    if( movieStatus == movie::Status_Success )
    {
        int32_t numChannels = 0;
        int32_t sampleRate = 0;
        audioFormat.FindInt32("channel-count", &numChannels);
        audioFormat.FindInt32("sample-rate", &sampleRate);
        if( numChannels > 0 )
            m_NumChannels = numChannels;
        if( sampleRate > 0 )
            m_SampleRate = sampleRate;
    }
    audioFormat.Clear();
    NN_SDK_LOG( "\n AudioFormatChangedEvent \n" );
    m_AudioFormatChangeEventReceived = true;
    CheckAndSignalFormatChangeWaiter();
    return movieStatus;
}

void AudioVideoWorkerThreadFunction(void *arg)
{
    MovieDecoderPlayer *movieDecoderPlayer = (MovieDecoderPlayer*) arg;
    // Thread is starting, link all MultiWait holders
    nn::os::LinkMultiWaitHolder( &movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioVideoWorkerThreadExitEventHolder );
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.errorEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
    nn::os::LinkMultiWaitHolder(&movieDecoderPlayer->m_AudioVideoWorkerMultiWait, &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);
    // Wait for events and process them.
    for (;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny( &movieDecoderPlayer->m_AudioVideoWorkerMultiWait );
        if( holder == &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder )
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_VideoDecoderEvents.inputBufferAvailableEvent);
            movieDecoderPlayer->VideoInputBufferAvailableEvent();
        }
        else if ( holder == &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder )
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_VideoDecoderEvents.outputBufferAvailableEvent);
            movieDecoderPlayer->VideoOutputBufferAvailableEvent();
        }
        else if ( holder == &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder )
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_VideoDecoderEvents.formatChangedEvent);
            movieDecoderPlayer->VideoFormatChangedEvent();
        }
        else if( holder == &movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.errorEventHolder )
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_VideoDecoderEvents.errorEvent);
            NN_LOG("\n Error event received from Video Decoder \n");
            movie::Status movieError = movieDecoderPlayer->m_VideoDecoder->GetLastError();
            if( movieError != movie::Status_Success )
            {
                NN_LOG("\n Error = 0x%x \"%s\" \n", movieError, movie::StatusToString(movieError));
                NN_LOG("\n Terminate Playback \n");
                movieDecoderPlayer->SetPlaybackComplete();
            }
        }
        if( holder == &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder )
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_AudioDecoderEvents.inputBufferAvailableEvent);
            movieDecoderPlayer->AudioInputBufferAvailableEvent();
        }
        else if ( holder == &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder )
        {
            nn::os::ClearEvent( movieDecoderPlayer->m_AudioDecoderEvents.outputBufferAvailableEvent);
            movieDecoderPlayer->AudioOutputBufferAvailableEvent();
        }
        else if ( holder == &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder )
        {
            nn::os::ClearEvent( movieDecoderPlayer->m_AudioDecoderEvents.formatChangedEvent);
            movieDecoderPlayer->AudioFormatChangedEvent();
        }
        else if( holder == &movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.errorEventHolder )
        {
            nn::os::ClearEvent(movieDecoderPlayer->m_AudioDecoderEvents.errorEvent);
            NN_LOG("\n Error event received from Audio Decoder \n");
            movie::Status movieError = movieDecoderPlayer->m_AudioDecoder->GetLastError();
            if( movieError != movie::Status_Success )
            {
                NN_LOG("\n Error = 0x%x \"%s\" \n", movieError, movie::StatusToString(movieError));
                NN_LOG("\n Terminate Playback \n");
                movieDecoderPlayer->SetPlaybackComplete();
            }
        }
        else if ( holder == &movieDecoderPlayer->m_AudioVideoWorkerThreadExitEventHolder )
        {
            nn::os::ClearEvent( &movieDecoderPlayer->m_AudioVideoWorkerThreadExitEvent );
            // Thread is exiting, unlink VideoDecoder MultiWait holders
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.errorEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.formatChangedEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_VideoDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);
            // Thread is exiting, unlink AudioDecoder MultiWait holders
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.errorEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.formatChangedEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.inputBufferAvailableEventHolder);
            nn::os::UnlinkMultiWaitHolder(&movieDecoderPlayer->m_AudioDecoderMultiWaitHolderTypes.outputBufferAvailableEventHolder);
            // Thread is exiting, unlink AudioVideoWorker MultiWait holder
            nn::os::UnlinkMultiWaitHolder( &movieDecoderPlayer->m_AudioVideoWorkerThreadExitEventHolder );
            break;
        }
    }
}//NOLINT(impl/function_size)

void MovieDecoderPlayer::Reset()
{
    movie::Status movieStatus = movie::Status_Success;
    if( m_AudioDecoder != NULL )
    {
        m_AudioDecoder->Reset();
        nn::os::LockMutex(&m_AudioListMutex);
        m_AudioBufferIndexList.clear();
        m_MovieAudioOutputHandler->Stop();
        nn::os::UnlockMutex(&m_AudioListMutex);
    }
    if( m_VideoDecoder != NULL )
    {
        m_VideoDecoder->Reset();
        nn::os::LockMutex(&m_VideoListMutex);
        m_VideoBufferIndexList.clear();
        m_MovieVideoOutputHandler->Stop();
        nn::os::UnlockMutex(&m_VideoListMutex);
    }
    if(m_VideoTrackIndex >= 0)
    {
        movieStatus = ConfigureDecoder(m_VideoDecoder, &m_VideoDecoderEvents, &m_TrackFormat[m_VideoTrackIndex]);
        if( movieStatus == movie::Status_Success )
        {
            StartDecoder(m_VideoDecoder);
            NN_SDK_LOG( "\n Failed to reconfigure video decoder \n" );
        }
    }
    if(m_AudioTrackIndex >= 0)
    {
        movieStatus = ConfigureDecoder(m_AudioDecoder, &m_AudioDecoderEvents, &m_TrackFormat[m_AudioTrackIndex]);
        if( movieStatus == movie::Status_Success )
        {
            StartDecoder(m_AudioDecoder);
            NN_SDK_LOG("\n Failed to reconfigure audio decoder \n");
        }
    }
}

movie::Status MovieDecoderPlayer::HandleLoopBack()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    ResetPlaybackComplete();
    nn::os::LockMutex(&m_AudioListMutex);
    m_AudioBufferIndexList.clear();
    nn::os::UnlockMutex(&m_AudioListMutex);
    nn::os::LockMutex(&m_VideoListMutex);
    m_VideoBufferIndexList.clear();
    nn::os::UnlockMutex(&m_VideoListMutex);
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Pause();
        m_MovieVideoOutputHandler->Stop();
        m_MovieVideoOutputHandler->Finalize();
    }
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Pause();
        m_MovieAudioOutputHandler->Stop();
        m_MovieAudioOutputHandler->Close();
        m_MovieAudioOutputHandler->Finalize();
    }
    if( m_VideoDecoder != nullptr )
        m_VideoDecoder->Flush();
    if( m_AudioDecoder != nullptr )
        m_AudioDecoder->Flush();
    if( m_Extractor != nullptr )
        m_Extractor->SeekTo(0);
    if( m_ByteStreamReader != NULL )
    {
        movieStatus = m_ByteStreamReader->PrepareForLooping();
        if( movieStatus != movie::Status_Success )
            return movieStatus;
    }
    m_MediaClock->ClearAnchorTime();
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Initialize(m_MediaClock, nullptr);
        m_MovieVideoOutputHandler->Start(m_Width, m_Height);
    }
    if( m_MovieAudioOutputHandler != nullptr )
    {
        m_MovieAudioOutputHandler->Initialize(m_MediaClock);
        m_MovieAudioOutputHandler->Open(m_AudioDecoder, m_SampleRate, m_NumChannels, m_VariableSizeAudioFrames);
        m_MovieAudioOutputHandler->Start();
    }
    if( m_VideoDecoder != nullptr )
        m_VideoDecoder->Start();
    if( m_AudioDecoder != nullptr )
        m_AudioDecoder->Start();
    m_LoopCount++;
    NN_SDK_LOG("\n Loop Count : %d\n", m_LoopCount);
    return movie::Status_Success;
}//NOLINT(impl/function_size)

bool MovieDecoderPlayer::IsAudioDecoder(movie::DecoderType decoderType)
{
    if( ( decoderType == movie::DecoderType_AudioAac ) ||
        ( decoderType == movie::DecoderType_AudioVorbis ) )
    {
        return true;
    }
    else
        return false;
}

bool MovieDecoderPlayer::IsVideoDecoder(movie::DecoderType decoderType)
{
    if( ( decoderType == movie::DecoderType_VideoAvc ) ||
        ( decoderType == movie::DecoderType_VideoVp8 ) ||
        ( decoderType == movie::DecoderType_VideoVp9 ) )
    {
        return true;
    }
    else
        return false;
}

void MovieDecoderPlayer::CheckAndSignalFormatChangeWaiter()
{
    if( ( m_AudioFormatChangeEventReceived == true ) && ( m_VideoFormatChangeEventReceived == true ) )
    {
        nn::os::SignalEvent(&m_FormatChangedPlayerEvent);
        m_AudioFormatChangeEventReceived = false;
        m_VideoFormatChangeEventReceived = false;
        if( m_AudioDecoder == nullptr )
            m_AudioFormatChangeEventReceived = true;
        if( m_VideoDecoder == nullptr )
            m_VideoFormatChangeEventReceived = true;
    }
}

void MovieDecoderPlayer::WaitForFormatChange()
{
    nn::os::TimedWaitEvent(&m_FormatChangedPlayerEvent, nn::TimeSpan::FromMilliSeconds(5000));
    nn::os::ClearEvent(&m_FormatChangedPlayerEvent);
}
