﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @brief
 * NvnVideoRenderer class for rendering of video frames using NVN APIs.
 *
 * @details
 *
 * Purpose:
 *     @class NvnVideoRenderer uses NVN APIs for playing back of video frames. Video frames can be submitted either in
 * NVN buffer or NVN texture. Up to four videos can be rendered simultaneously. If the video frame is in YUV format, it
 * will be converted to RGB format using shaders. There are no threads, client need to drive rendering.
 *
 * Setup:
 *     A client using @class NvnVideoRenderer needs to create an instance by passing nn::vi::NativeWindowHandle
 * and NvnOutputFormat. If nn::vi::NativeWindowHandle is null, NvnVideoRenderer creates one. NvnOutputFormat can
 * be either NvnOutputFormat_YuvNv12NvnBuffer or NvnOutputFormat_AbgrNvnTexture. This needs to map with movie video decoder
 * output format. When Initialization() API is called, memory needed for rendering using NVN is reserved. Based on decoder
 * output format shader is compiled using Glsl compiler and loaded to NVN shader memory. Resources needed
 * for rendering using NVN is allocated and initialized. NVNcommandBuffer is setup and will be used later
 * in draw call.

 * Main Processing:
 *     Client need to call  GetNvnVideoBuffer() or GetNvnVideoTexture() to get either a video buffer or a texture
 * handle. This will be sent to video decoder. Once decoder completes decoding the frame, the client needs to submit the
 * frame to NvnVideoRenderer using AddVideoFrameToRender() or AddTextureToRender() API. This will add video frame to a list
 * which will be used by Draw() API. Draw API will acquire a texture from NVN. If the decoder uses CPU buffers, YUV data
 * is copied from NVN buffer to NVN texture. If the decoder output is texture, a texture copy is done. In case of
 * NVN texture, client can pass NVN sync point to the decoder. In this case renderer will wait on the sync point before
 * presenting the texture for display. This ensures decoder completing its processing on the texture.
 *
 * Teardown:
 *     When client is done submitting video frames for rendering, Finalize() API need to be called. All resources allocated
 * by NvnVideoRenderer will be released.
 *
 */

#include "NvnVideoRenderer.h"
#include <cstdlib>
#include <nvnTool/nvnTool_GlslcInterface.h>
#include <nvn/nvn_FuncPtrImpl.h>
#include <nn/nn_Assert.h>
#include <nn/nn_SdkLog.h>
#include <nv/nv_MemoryManagement.h>
#include <nvn/nvn_FuncPtrInline.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_Vector.h>
#include <nn/nn_Log.h>

static const char g_yuv2rgbVertexShader[] =
"#version 450\n"
"layout(location = 0) in vec4 a_position;"
"layout(location = 1) in vec2 a_texCoord;"
"out vec2 v_texCoord;"
"void main() {"
"   gl_Position = a_position;"
"   v_texCoord = a_texCoord;"
"   v_texCoord.y = v_texCoord.y;"
"}";

static const char g_yuv2rgbFragmentShader[] ="#version 450\n"
"in vec2 v_texCoord;"
"layout(binding = 0) uniform sampler2D u_textureY;"
"layout(binding = 1) uniform sampler2D u_textureUV;"
"layout(binding = 2) uniform UBO {"
"    vec3 src_bias;"
"    mat3 src_xform;"
"};"
"out vec4 colorOut;"
"void main() {"
"    float y = texture(u_textureY, v_texCoord).r;"
"    vec2 cbcr = texture(u_textureUV, v_texCoord).rg;"
"    vec3 yuv = vec3(y, cbcr);"
"    yuv -= src_bias;"
"    yuv *= src_xform;"
"    colorOut = vec4(yuv, 1.0);"
"}";

static const char g_rgbFragmentShader[] =
"#version 450\n"
"in vec2 v_texCoord;"
"layout(binding = 0) uniform sampler2D rgb_texture;"
"out vec4 colorOut;"
"void main() {"
"    colorOut = texture(rgb_texture, v_texCoord).rgba;"
"}";

unsigned Align(unsigned value, unsigned alignment)
{
    return value + (alignment - value % alignment) % alignment;
}

char                        g_ShaderHeapBuffer[32 << 20];
nn::mem::StandardAllocator  g_ShaderAllocator(g_ShaderHeapBuffer, sizeof(g_ShaderHeapBuffer));

static void *shaderAllocate(size_t size, size_t alignment, void *userPtr)
{
    return g_ShaderAllocator.Allocate(size, alignment);
}

static void shaderDeallocate(void *block, void *userPtr)
{
    g_ShaderAllocator.Free(block);
}

static void *shaderReallocate(void *block, size_t size, void *userPtr)
{
    return g_ShaderAllocator.Reallocate(block, size);
}

GlslCompiler::GlslCompiler() : m_Init(false)
{
}

GlslCompiler::~GlslCompiler()
{
    Destroy();
}

bool GlslCompiler::Init()
{
    glslcSetAllocator(shaderAllocate, shaderDeallocate, shaderReallocate, NULL);
    uint8_t status = glslcInitialize(&m_Compiler);
    if(status == 0)
        return false;
    m_Init = true;
    return true;
}

void GlslCompiler::Destroy()
{
    if(m_Init)
    {
        glslcFinalize(&m_Compiler);
        m_Init = false;
    }
}

bool GlslCompiler::CompileShader(const char *src, NVNshaderStage stage)
{
    m_Compiler.input.sources = &src;
    m_Compiler.input.stages = &stage;
    m_Compiler.input.count = 1;

    if(!glslcCompile(&m_Compiler))
    {
        NN_SDK_LOG("glslcCompile failed:\n%s\n", m_Compiler.lastCompiledResults->compilationStatus->infoLog);
        return false;
    }
    GLSLCoutput *output = m_Compiler.lastCompiledResults->glslcOutput;
    m_OutputData.data = output;
    m_OutputData.size = output->size;
    return true;
}

DataSection GlslCompiler::GetOutput() const
{
    return m_OutputData;
}

NvnShaderSerialize::NvnShaderSerialize() : m_ShaderData(NULL), m_ScratchSize(0)
{
}

NvnShaderSerialize::~NvnShaderSerialize()
{
    std::free(m_ShaderData);
}

void NvnShaderSerialize::LoadNvnShaderMemory(const uint8_t *shaderBinary)
{
    const GLSLCoutput *output = reinterpret_cast<const GLSLCoutput*>(shaderBinary);
    m_ShaderData = std::malloc(output->size);
    memcpy(m_ShaderData, output, output->size);
    for(int i = 0; i < output->numSections; ++i)
    {
        GLSLCsectionTypeEnum type = output->headers[i].genericHeader.common.type;
        if(type == GLSLC_SECTION_TYPE_GPU_CODE)
        {
            const GLSLCgpuCodeHeader *gpuSection = &output->headers[i].gpuCodeHeader;
            const uint8_t *data = reinterpret_cast<const uint8_t*>(m_ShaderData) + gpuSection->common.dataOffset;
            m_DataSection.size = gpuSection->dataSize;
            m_DataSection.data = data + gpuSection->dataOffset;
            m_ControlSection.size = gpuSection->controlSize;
            m_ControlSection.data = data + gpuSection->controlOffset;
            m_ScratchSize = gpuSection->scratchMemBytesRecommended;
        }
    }
}

DataSection NvnShaderSerialize::GetDataSection()
{
    return m_DataSection;
}

DataSection NvnShaderSerialize::GetControlSection()
{
    return m_ControlSection;
}

unsigned NvnShaderSerialize::GetScratchSize()
{
    return m_ScratchSize;
}

NvnMemoryPool::NvnMemoryPool() : /*m_Allocator(NULL),*/ m_Size(0), m_pMemory(NULL), m_Offset(0)
{
}

NvnMemoryPool::~NvnMemoryPool()
{
    Destroy();
}

bool NvnMemoryPool::Init(NVNdevice &device, int32_t flags, unsigned size, void *pMemory)
{
    if(!pMemory)
    {
        NN_SDK_LOG("NvnMemoryPool::Init: pMemory needs to be a valid pointer\n");
        return false;
    }
    if(size % NVN_MEMORY_POOL_STORAGE_GRANULARITY)
    {
        NN_SDK_LOG("NvnMemoryPool::Init: size needs to be aligned to NVN_MEMORY_POOL_STORAGE_GRANULARITY (%i)\n", NVN_MEMORY_POOL_STORAGE_GRANULARITY);
        return false;
    }
    m_Size = size;
    m_pMemory = pMemory;

    NVNmemoryPoolBuilder poolBuilder;
    nvnMemoryPoolBuilderSetDefaults(&poolBuilder);
    nvnMemoryPoolBuilderSetDevice(&poolBuilder, &device);
    nvnMemoryPoolBuilderSetFlags(&poolBuilder, flags);
    nvnMemoryPoolBuilderSetStorage(&poolBuilder, m_pMemory, m_Size);

    return nvnMemoryPoolInitialize(&m_MemoryPool, &poolBuilder) == NVN_TRUE;
}

void NvnMemoryPool::Destroy()
{
    if(m_pMemory)
    {
        nvnMemoryPoolFinalize(&m_MemoryPool);
        m_pMemory = NULL;
        m_Size = 0;
        m_Offset = 0;
    }
}

ptrdiff_t NvnMemoryPool::GetNewMemoryChunkOffset(unsigned size, unsigned alignment)
{
    NN_ASSERT(m_Offset + size <= m_Size, "Memory pool out of memory.");
    m_Offset = Align(m_Offset, alignment);
    ptrdiff_t dataOffset = m_Offset;
    m_Offset += size;
    return dataOffset;
}

NVNmemoryPool &NvnMemoryPool::GetMemoryPool()
{
    return m_MemoryPool;
}

NvnBuffer::NvnBuffer() : m_pDevice(NULL), m_pMappedData(NULL)
{
}

NvnBuffer::~NvnBuffer()
{
    Destroy();
}

bool NvnBuffer::Init(NVNdevice &device, NVNmemoryPool &pPool, unsigned offset, unsigned size)
{
    m_pDevice = &device;

    NVNbufferBuilder bufferBuilder;
    nvnBufferBuilderSetDefaults(&bufferBuilder);
    nvnBufferBuilderSetDevice(&bufferBuilder, &device);
    nvnBufferBuilderSetStorage(&bufferBuilder, &pPool, offset, size);
    if(nvnBufferInitialize(&m_Buffer, &bufferBuilder) == NVN_FALSE)
        return false;
    m_pMappedData = nvnBufferMap(&m_Buffer);
    return true;
}

void NvnBuffer::Destroy()
{
    if(m_pDevice)
    {
        nvnBufferFinalize(&m_Buffer);
        m_pDevice = NULL;
    }
}

NVNbufferAddress NvnBuffer::GetAddress() const
{
    return nvnBufferGetAddress(&m_Buffer);
}

NVNbuffer &NvnBuffer::GetBuffer()
{
    return m_Buffer;
}

void *NvnBuffer::Map()
{
    return m_pMappedData;
}

NvnCommandBuffer::NvnCommandBuffer() : m_pDevice(NULL), m_pAllocator(NULL), m_pMemoryPool(NULL), m_CommandOffset(0), m_pControlPool(NULL)
{
}

NvnCommandBuffer::~NvnCommandBuffer()
{
    Destroy();
}

bool NvnCommandBuffer::Init(NvnDevice &device, nn::mem::StandardAllocator &allocator, NvnMemoryPool &memoryPool)
{
    m_pDevice = &device;
    m_pAllocator = &allocator;
    m_pMemoryPool = &memoryPool;

    if(nvnCommandBufferInitialize(&m_CommandBuffer, &device.GetDevice()) == NVN_FALSE)
        return false;

    int commandBufferCommandAlignment = device.GetDeviceInfo(NVN_DEVICE_INFO_COMMAND_BUFFER_COMMAND_ALIGNMENT);
    int commandBufferControlAlignment = device.GetDeviceInfo(NVN_DEVICE_INFO_COMMAND_BUFFER_CONTROL_ALIGNMENT);

    m_CommandOffset = memoryPool.GetNewMemoryChunkOffset(m_CommandSize, commandBufferCommandAlignment);
    m_pControlPool = allocator.Allocate(m_ControlSize, commandBufferControlAlignment);

    Reset();
    return true;
}

void NvnCommandBuffer::Destroy()
{
    if(m_pDevice)
    {
        nvnCommandBufferFinalize(&m_CommandBuffer);
        m_pAllocator->Free(m_pControlPool);
        m_pControlPool = NULL;
        m_CommandOffset = 0;
        m_pMemoryPool = NULL;
        m_pAllocator = NULL;
        m_pDevice = NULL;
    }
}

void NvnCommandBuffer::Reset()
{
    nvnCommandBufferAddCommandMemory(&m_CommandBuffer, &m_pMemoryPool->GetMemoryPool(), m_CommandOffset, m_CommandSize);
    nvnCommandBufferAddControlMemory(&m_CommandBuffer, m_pControlPool, m_ControlSize);
}

size_t NvnCommandBuffer::GetCommandMemoryUsed() const
{
    return nvnCommandBufferGetCommandMemoryUsed(&m_CommandBuffer);
}

size_t NvnCommandBuffer::GetControlMemoryUsed() const
{
    return nvnCommandBufferGetControlMemoryUsed(&m_CommandBuffer);
}

NVNcommandBuffer &NvnCommandBuffer::GetCommandBuffer()
{
    return m_CommandBuffer;
}

//in the Windows demos nvnBootstrapLoader needs to be loaded dynamically, but on Horizon it seems to exist in libnvn.a
extern "C"
{
    PFNNVNGENERICFUNCPTRPROC NVNAPIENTRY nvnBootstrapLoader(const char * name);
}

NvnDevice::NvnDevice() : m_pAllocator(NULL), m_pRenderTargetMemory(NULL), m_FirstResize(true)
{
}

NvnDevice::~NvnDevice()
{
    Destroy();
}

bool NvnDevice::Init(NVNnativeWindow nativeWindow, nn::mem::StandardAllocator &allocator)
{
    pfnc_nvnDeviceInitialize = reinterpret_cast<PFNNVNDEVICEINITIALIZEPROC>(nvnBootstrapLoader("nvnDeviceInitialize"));
    if(!pfnc_nvnDeviceInitialize)
    {
        NN_SDK_LOG("NvnDevice::Init: BootstrapLoader failed to find nvnDeviceInitialize\n");
        return false;
    }
    pfnc_nvnDeviceGetProcAddress = reinterpret_cast<PFNNVNDEVICEGETPROCADDRESSPROC>(nvnBootstrapLoader("nvnDeviceGetProcAddress"));
    if(!pfnc_nvnDeviceGetProcAddress)
    {
        NN_SDK_LOG("NvnDevice::Init: BootstrapLoader failed to find nvnDeviceGetProcAddress\n");
        return false;
    }

    nvnLoadCProcs(nullptr, pfnc_nvnDeviceGetProcAddress);

    int deviceFlags = 0;
    NVNdeviceBuilder deviceData;
    nvnDeviceBuilderSetDefaults(&deviceData);
    nvnDeviceBuilderSetFlags(&deviceData, deviceFlags);

    if(pfnc_nvnDeviceInitialize(&m_Device, &deviceData) == NVN_FALSE)
    {
        NN_SDK_LOG("NvnDevice::Init: nvnDeviceInitialize failed\n");
        return false;
    }

    nvnLoadCProcs(&m_Device, pfnc_nvnDeviceGetProcAddress);

    int MajorVersion = GetDeviceInfo(NVN_DEVICE_INFO_API_MAJOR_VERSION);
    int MinorVersion = GetDeviceInfo(NVN_DEVICE_INFO_API_MINOR_VERSION);

    NN_ASSERT(MajorVersion == NVN_API_MAJOR_VERSION && MinorVersion == NVN_API_MINOR_VERSION, "NVN SDK not supported by current driver.");

    /*
     * Debug Layer Callback
     * --------------------
     * Install the debug layer callback if the debug layer was enabled during
     * device initialization. It is possible to pass a pointer to the NVN API
     * to remember and pass back through the debug callback.
     */
    if (deviceFlags & NVN_DEVICE_FLAG_DEBUG_ENABLE_LEVEL_2_BIT)
    {
        nvnDeviceInstallDebugCallback(
            &m_Device,
            reinterpret_cast<PFNNVNDEBUGCALLBACKPROC>(&DebugLayerCallback),
            NULL, // For testing purposes; any pointer is OK here.
            NVN_TRUE // NVN_TRUE = Enable the callback.
            );
    }

    m_pAllocator = &allocator;

    nvnTextureBuilderSetDevice(&m_RenderTargetBuilder, &m_Device);
    nvnTextureBuilderSetDefaults(&m_RenderTargetBuilder);

    /* Render targets that need to be displayed to the screen need both the render target and display access bits. */
    nvnTextureBuilderSetFlags (&m_RenderTargetBuilder, NVN_TEXTURE_FLAGS_DISPLAY_BIT);
    nvnTextureBuilderSetTarget(&m_RenderTargetBuilder, NVN_TEXTURE_TARGET_2D);
    nvnTextureBuilderSetFormat(&m_RenderTargetBuilder, NVN_FORMAT_RGBA8);
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, 1920, 1080);
    m_ColorTargetSize = nvnTextureBuilderGetStorageSize(&m_RenderTargetBuilder);

    /* Setup the render target memory pool. (See MemoryPool.cpp/.h) */
    unsigned memorySize = Align(m_ColorTargetSize * m_NumColorBuffers, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pRenderTargetMemory = allocator.Allocate(memorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if(!m_RenderTargetMemoryPool.Init(m_Device, NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, memorySize, m_pRenderTargetMemory))
    {
        NN_SDK_LOG("NvnDevice::Init: m_RenderTargetMemoryPool.Init() failed\n");
        return false;
    }

   /*
    * NVN Window
    * ----------
    * The NVNwindow is used to present a render target to the screen. It manages
    * an array of render targets and associates them with an NVNnativeWindow. On
    * Horizon NVNnativeWindow defined as ANativeWindow* and on Windows it's defined
    * as HWND. Each frame, the NVNwindow needs to be queried for the active render
    * target for that frame.
    *
    * Window Builder Settings
    * -----------------------
    * SetTextures     - Sets the list of render targets to be used by the window.
    *                   Textures used by the NVNwindow must be initialized with the
    *                   display access bit. A window can be given a max of 6 targets.
    *
    * SetNativeWindow - Sets the handle to the window.
    *
    */
    nvnWindowBuilderSetDefaults(&m_WindowBuilder);
    nvnWindowBuilderSetDevice(&m_WindowBuilder, &m_Device);
    nvnWindowBuilderSetNativeWindow(&m_WindowBuilder, nativeWindow);
    return true;
}

void NvnDevice::Destroy()
{
    if(m_pAllocator)
    {
        nvnWindowFinalize(&m_pWindow);

        for( int i = 0; i < sizeof(m_RenderTargets) / sizeof(m_RenderTargets[0]); ++i )
            nvnTextureFinalize(m_RenderTargets + i);

        m_RenderTargetMemoryPool.Destroy();
        m_pAllocator->Free(m_pRenderTargetMemory);
        nvnDeviceFinalize(&m_Device);
        m_pAllocator = NULL;
        m_pRenderTargetMemory = NULL;
        m_FirstResize = true;
    }
}

void NvnDevice::ResizeRenderTargets(unsigned width, unsigned height)
{
    // Check for the window being minimized or having no visible surface.
    if (width == 0 || height == 0)
    {
        return;
    }

    // Set the size of the texture to be created by the texture builder.
    // Implicitly sets depth to one.
    nvnTextureBuilderSetSize2D(&m_RenderTargetBuilder, width, height);

    for(int i = 0; i < m_NumColorBuffers; ++i)
    {
        if(!m_FirstResize)
        {
            nvnTextureFinalize(m_RenderTargets + i);
        }

        // Set the memory pool the texture is to be allocated from and offset to where it will be located.
        nvnTextureBuilderSetStorage(&m_RenderTargetBuilder, &m_RenderTargetMemoryPool.GetMemoryPool(), m_ColorTargetSize * i);

        // Create the texture using the current state of the texture builder.
        nvnTextureInitialize(m_RenderTargets + i, &m_RenderTargetBuilder);
    }

    // Recreate the NVNwindow with the newly allocated/created render targets
    // for the new resolution.
    if(!m_FirstResize)
    {
        nvnWindowFinalize(&m_pWindow);
    }

    // Sets the number of render targets that the NVNwindow will have available and
    //  gives a pointer to an array of those targets.
    NVNtexture *renderTargets[m_NumColorBuffers];
    for(int i = 0; i < m_NumColorBuffers; ++i)
        renderTargets[i] = m_RenderTargets + i;
    nvnWindowBuilderSetTextures(&m_WindowBuilder, m_NumColorBuffers, renderTargets);
    nvnWindowInitialize(&m_pWindow, &m_WindowBuilder);

    m_FirstResize = false;
}

NVNdevice &NvnDevice::GetDevice()
{
    return m_Device;
}

NVNwindow &NvnDevice::GetWindow()
{
    return m_pWindow;
}

NVNtexture &NvnDevice::GetRenderTarget(int index)
{
    NN_ASSERT(index >= 0 && index < m_NumColorBuffers, "NvnDevice::GetRenderTarget index out of range");
    return m_RenderTargets[index];
}

int NvnDevice::GetDeviceInfo(NVNdeviceInfo name)
{
    int i;
    nvnDeviceGetInteger(&m_Device, name, &i);
    return i;
}

void NvnDevice::DebugLayerCallback(
    NVNdebugCallbackSource source,
    NVNdebugCallbackType type,
    int id,
    NVNdebugCallbackSeverity severity,
    const char * message,
    void* pUser
    )
{
    NN_ASSERT(pUser == NULL);

    NN_SDK_LOG("NVN Debug Layer Callback:\n");
    NN_SDK_LOG("  source:       0x%08x\n", source);
    NN_SDK_LOG("  type:         0x%08x\n", type);
    NN_SDK_LOG("  id:           0x%08x\n", id);
    NN_SDK_LOG("  severity:     0x%08x\n", severity);
    NN_SDK_LOG("  message:      %s\n",     message);

    NN_ASSERT(0, "Debug layer callback hit");
}

NvnProgram::NvnProgram() : m_pDevice(NULL)
{
}

NvnProgram::~NvnProgram()
{
    Destroy();
}

bool NvnProgram::Init(NVNdevice &device, const NVNshaderData *stageData, unsigned stageCount)
{
    m_pDevice = &device;
    if(nvnProgramInitialize(&m_Program, &device) == NVN_FALSE)
        return false;
    return nvnProgramSetShaders(&m_Program, stageCount, stageData) == NVN_TRUE;
}

void NvnProgram::Destroy()
{
    if(m_pDevice)
    {
        nvnProgramFinalize(&m_Program);
        m_pDevice = NULL;
    }
}

NVNprogram &NvnProgram::GetProgram()
{
    return m_Program;
}

NvnQueue::NvnQueue() : m_pDevice(NULL)
{
}

NvnQueue::~NvnQueue()
{
    Destroy();
}

bool NvnQueue::Init(NVNdevice &device)
{
    m_pDevice = &device;
    NVNqueueBuilder queueBuilder;
    nvnQueueBuilderSetDefaults(&queueBuilder);
    nvnQueueBuilderSetDevice(&queueBuilder, &device);
    if(nvnQueueInitialize(&m_Queue, &queueBuilder) == NVN_FALSE)
        return false;
    return nvnSyncInitialize(&m_CommandBufferSync, &device) == NVN_TRUE;
}

void NvnQueue::Destroy()
{
    if(m_pDevice)
    {
        nvnQueueFinalize(&m_Queue);
        nvnSyncFinalize(&m_CommandBufferSync);
        m_pDevice = NULL;
    }
}

void NvnQueue::WaitForAllCommands()
{
    nvnQueueFenceSync(&m_Queue, &m_CommandBufferSync, NVN_SYNC_CONDITION_ALL_GPU_COMMANDS_COMPLETE, NVN_SYNC_FLAG_FLUSH_FOR_CPU_BIT);
    nvnQueueFlush(&m_Queue);
    nvnSyncWait(&m_CommandBufferSync, NVN_WAIT_TIMEOUT_MAXIMUM);
}

NVNqueue &NvnQueue::GetQueue()
{
    return m_Queue;
}

NVNsync& NvnQueue::GetNvnSync()
{
    return m_CommandBufferSync;
}

NvnSampler::NvnSampler() : m_pDevice(NULL)
{
}

NvnSampler::~NvnSampler()
{
    Destroy();
}

bool NvnSampler::Init(NVNdevice &device)
{
    m_pDevice = &device;
    NVNsamplerBuilder builder;
    nvnSamplerBuilderSetDefaults(&builder);
    nvnSamplerBuilderSetDevice(&builder, &device);
    return nvnSamplerInitialize(&m_Sampler, &builder) == NVN_TRUE;
}

void NvnSampler::Destroy()
{
    if(m_pDevice)
    {
        nvnSamplerFinalize(&m_Sampler);
        m_pDevice = NULL;
    }
}

NVNsampler &NvnSampler::GetSampler()
{
    return m_Sampler;
}

NvnTexture::NvnTexture() : m_pDevice(NULL)
{
}

NvnTexture::~NvnTexture()
{
    Destroy();
}

bool NvnTexture::Init(NVNdevice &device,
    NvnMemoryPool &memoryPool,
    unsigned width,
    unsigned height,
    NVNformat format,
    int32_t flags,
    unsigned stride,
    bool nvnVideoInteropMode)
{
    m_pDevice = &device;

    NVNtextureBuilder builder;
    nvnTextureBuilderSetDefaults(&builder);
    nvnTextureBuilderSetDevice(&builder, &device);
    nvnTextureBuilderSetSize2D(&builder, width, height);
    nvnTextureBuilderSetFlags(&builder, flags);
    nvnTextureBuilderSetFormat(&builder, format);
    nvnTextureBuilderSetTarget(&builder, NVN_TEXTURE_TARGET_2D);
    if( nvnVideoInteropMode == true )
    {
        nvnTextureBuilderSetFlags(&builder, NVN_TEXTURE_FLAGS_VIDEO_DECODE_BIT);
    }
    nvnTextureBuilderSetStride(&builder, stride);
    size_t size = nvnTextureBuilderGetStorageSize(&builder);
    size_t storageAlignment = nvnTextureBuilderGetStorageAlignment(&builder);
    nvnTextureBuilderSetStorage(&builder, &memoryPool.GetMemoryPool(), memoryPool.GetNewMemoryChunkOffset(size, storageAlignment));
    return nvnTextureInitialize(&m_Texture, &builder) == NVN_TRUE;
}

void NvnTexture::Destroy()
{
    if(m_pDevice)
    {
        nvnTextureFinalize(&m_Texture);
        m_pDevice = NULL;
    }
}

NVNtexture &NvnTexture::GetTexture()
{
    return m_Texture;
}

NvnTextureSamplerPool::NvnTextureSamplerPool() : m_pMemory(NULL), m_pDevice(NULL), m_Id(256)
{
}

NvnTextureSamplerPool::~NvnTextureSamplerPool()
{
    Destroy();
}

bool NvnTextureSamplerPool::Init(NVNdevice &device, nn::mem::StandardAllocator &allocator)
{
    m_pDevice = &device;
    m_pAllocator = &allocator;

    int maxSamplerPoolSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_MAX_SAMPLER_POOL_SIZE, &maxSamplerPoolSize);
    int samplerDescriptorSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_SAMPLER_DESCRIPTOR_SIZE, &samplerDescriptorSize);
    int textureDescriptorSize;
    nvnDeviceGetInteger(&device, NVN_DEVICE_INFO_TEXTURE_DESCRIPTOR_SIZE, &textureDescriptorSize);
    const unsigned poolSize = Align(maxSamplerPoolSize * (samplerDescriptorSize + textureDescriptorSize), NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pMemory = allocator.Allocate(poolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_MemoryPool.Init(device, NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, poolSize, m_pMemory);
    if(nvnSamplerPoolInitialize(&m_SamplerPool, &m_MemoryPool.GetMemoryPool(), 0, maxSamplerPoolSize) == NVN_FALSE)
        return false;
    int samplerPoolOffset = Align(maxSamplerPoolSize * samplerDescriptorSize, textureDescriptorSize);
    return nvnTexturePoolInitialize(&m_TexturePool, &m_MemoryPool.GetMemoryPool(), samplerPoolOffset, maxSamplerPoolSize) == NVN_TRUE;
}

void NvnTextureSamplerPool::Destroy()
{
    if(m_pDevice)
    {
        nvnTexturePoolFinalize(&m_TexturePool);
        nvnSamplerPoolFinalize(&m_SamplerPool);
        m_MemoryPool.Destroy();
        m_pAllocator->Free(m_pMemory);
        m_pMemory = NULL;
        m_pAllocator = NULL;
        m_pDevice = NULL;
        m_Id = 256;
    }
}

NVNtexturePool &NvnTextureSamplerPool::GetTexturePool()
{
    return m_TexturePool;
}

NVNsamplerPool &NvnTextureSamplerPool::GetSamplerPool()
{
    return m_SamplerPool;
}

unsigned NvnTextureSamplerPool::Register(NVNtexture &pTexture, NVNsampler &pSampler)
{
    NVNtextureView view;
    nvnTextureViewSetDefaults(&view);
    nvnTexturePoolRegisterTexture(&m_TexturePool, m_Id, &pTexture, &view);
    nvnSamplerPoolRegisterSampler(&m_SamplerPool, m_Id, &pSampler);
    return m_Id++;
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON ) && defined( NN_BUILD_CONFIG_SPEC_NX )
void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

NvnWindow::NvnWindow()
    : m_NativeWindow(NULL),
      m_pDisplay(NULL),
      m_pLayer(NULL),
      m_WindowWidth(1280),
      m_WindowHeight(720),
      m_NativeWindowCreated(false)
{
}

NvnWindow::~NvnWindow()
{
}

bool NvnWindow::Init(nn::vi::NativeWindowHandle nativeWindow)
{
    bool retValue = false;
    InitMemory();
    if( nativeWindow == nullptr )
    {
        if( true == InitWindow() )
        {
            m_NativeWindowCreated = true;
            retValue = true;
        }
    }
    else
    {
        m_NativeWindow = nativeWindow;
        retValue = true;
    }
    return retValue;
}

void NvnWindow::Destroy()
{
    if( m_NativeWindowCreated == true )
    {
        DestroyLayer(m_pLayer);
        CloseDisplay(m_pDisplay);
        nn::vi::Finalize();
        m_NativeWindowCreated = false;
    }
}

nn::vi::NativeWindowHandle NvnWindow::GetNativeWindow()
{
    return m_NativeWindow;
}

void NvnWindow::InitMemory()
{
}

bool NvnWindow::InitWindow()
{
    // Initialize Visual Interface (VI) system to display
    // to the target's screen
    //
    nn::vi::Initialize();

    nn::Result result = nn::vi::OpenDefaultDisplay(&m_pDisplay);
    if( !result.IsSuccess() )
    {
        return false;
    }
    result = nn::vi::CreateLayer(&m_pLayer, m_pDisplay);
    if( !result.IsSuccess() )
    {
        return false;
    }
    nn::vi::SetLayerScalingMode(m_pLayer, nn::vi::ScalingMode_FitToLayer);
    result = nn::vi::GetNativeWindow(&m_NativeWindow, m_pLayer);
    if( !result.IsSuccess() )
    {
        return false;
    }
    return true;
}

int NvnWindow::GetWidth() const
{
    return m_WindowWidth;
}

int NvnWindow::GetHeight() const
{
    return m_WindowHeight;
}

NvnVideoRenderer::NvnVideoRenderer(nn::vi::NativeWindowHandle nativeWindow, NvnOutputFormat nvnOutputFormat)
    : m_pAllocatorMemory(nullptr),
    m_pShaderPoolMemory(nullptr),
    m_pVertexPoolMemory(nullptr),
    m_pCommandMemory(nullptr),
    m_pShaderScratchMemory(nullptr),
    m_NativeWindow(nativeWindow),
    m_NvnOutputFormat(nvnOutputFormat),
    m_NumberOfVideosToRender(1),
    m_VideoId(0),
    m_MaxNvnFramesPerDecoder(3),
    m_Lock {false}
{
}

movie::Status NvnVideoRenderer::Initialize(int numberOfVideosToRender)
{
    const unsigned nvnMemorySize = 272 << 20;
    m_pAllocatorMemory = std::malloc(nvnMemorySize);
    if( m_pAllocatorMemory == NULL )
        return movie::Status_OutOfMemory;
    m_allocator.Initialize(m_pAllocatorMemory, nvnMemorySize);

    GlslCompiler compiler;
    compiler.Init();
    compiler.CompileShader(g_yuv2rgbVertexShader, NVN_SHADER_STAGE_VERTEX);
    m_vertexShaderFile.LoadNvnShaderMemory(reinterpret_cast<const uint8_t*>( compiler.GetOutput().data ));
    if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
    {
        if( !compiler.CompileShader(g_yuv2rgbFragmentShader, NVN_SHADER_STAGE_FRAGMENT) )
            return movie::Status_UnknownError;
    }
    else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
    {
        if( !compiler.CompileShader(g_rgbFragmentShader, NVN_SHADER_STAGE_FRAGMENT) )
            return movie::Status_UnknownError;
    }
    else
        return movie::Status_NotSupported;

    m_fragmentShaderFile.LoadNvnShaderMemory(reinterpret_cast<const uint8_t*>( compiler.GetOutput().data ));
    compiler.Destroy();
    m_vertexDataSection = m_vertexShaderFile.GetDataSection();
    m_vertexControlSection = m_vertexShaderFile.GetControlSection();
    m_fragmentDataSection = m_fragmentShaderFile.GetDataSection();
    m_fragmentControlSection = m_fragmentShaderFile.GetControlSection();

    m_window.Init(m_NativeWindow);
    m_device.Init(m_window.GetNativeWindow(), m_allocator);
    m_queue.Init(m_device.GetDevice());

    m_pCommandMemory = m_allocator.Allocate(4 * NVN_MEMORY_POOL_STORAGE_GRANULARITY, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pCommandMemory == NULL )
        return movie::Status_OutOfMemory;
    m_commandPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_NO_ACCESS_BIT, 4 * NVN_MEMORY_POOL_STORAGE_GRANULARITY, m_pCommandMemory);
    m_renderTargetCommand.Init(m_device, m_allocator, m_commandPool);
    m_texturePool.Init(m_device.GetDevice(), m_allocator);

    m_ShaderScratchGranularity = m_device.GetDeviceInfo(NVN_DEVICE_INFO_SHADER_SCRATCH_MEMORY_GRANULARITY);
    m_pShaderScratchMemory = m_allocator.Allocate(m_ShaderScratchGranularity, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pShaderScratchMemory == NULL )
        return movie::Status_OutOfMemory;
    m_shaderScratchPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, m_ShaderScratchGranularity, m_pShaderScratchMemory);
    unsigned fragmentShaderOffset = Align(m_vertexDataSection.size + 1024, 256);
    unsigned shaderMemorySize = Align(fragmentShaderOffset + m_fragmentDataSection.size + 1024, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pShaderPoolMemory = m_allocator.Allocate(shaderMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    if( m_pShaderPoolMemory == NULL )
        return movie::Status_OutOfMemory;
    memcpy(m_pShaderPoolMemory, m_vertexDataSection.data, m_vertexDataSection.size);
    memcpy(reinterpret_cast<char*>( m_pShaderPoolMemory ) + fragmentShaderOffset, m_fragmentDataSection.data, m_fragmentDataSection.size);
    m_shaderPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT | NVN_MEMORY_POOL_FLAGS_SHADER_CODE_BIT, shaderMemorySize, m_pShaderPoolMemory);

    m_vertexDataBuffer.Init(m_device.GetDevice(), m_shaderPool.GetMemoryPool(), m_shaderPool.GetNewMemoryChunkOffset(fragmentShaderOffset, 1), m_vertexDataSection.size);
    m_fragmentDataBuffer.Init(m_device.GetDevice(), m_shaderPool.GetMemoryPool(), m_shaderPool.GetNewMemoryChunkOffset(m_fragmentDataSection.size + 1024, 256), m_fragmentDataSection.size);

    NVNshaderData stageData[ 2 ];
    stageData[ 0 ].control = m_vertexControlSection.data;
    stageData[ 0 ].data = m_vertexDataBuffer.GetAddress();
    stageData[ 1 ].control = m_fragmentControlSection.data;
    stageData[ 1 ].data = m_fragmentDataBuffer.GetAddress();
    m_program.Init(m_device.GetDevice(), stageData, sizeof(stageData) / sizeof(*stageData));

    const float vertices[] =
    {
        -1.0f,  1.00f, 0, 1, 0, 0,
        -1.0f, -1.00f, 0, 1, 0, 1,
        1.0f,  1.00f, 0, 1, 1, 0,
        1.0f, -1.00f, 0, 1, 1, 1,
    };
    unsigned vertexPoolSize = Align(sizeof(vertices) * numberOfVideosToRender, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    m_pVertexPoolMemory = m_allocator.Allocate(vertexPoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
    m_vertexPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, vertexPoolSize, m_pVertexPoolMemory);
    m_squareMeshes = new NvnBuffer[ numberOfVideosToRender ];
    for( int i = 0; i < numberOfVideosToRender; ++i )
        m_squareMeshes[ i ].Init(m_device.GetDevice(), m_vertexPool.GetMemoryPool(), sizeof(vertices) * i, sizeof(vertices));
    m_NumberOfVideosToRender = numberOfVideosToRender;
    nvnBlendStateSetDefaults(&m_blendState);
    nvnChannelMaskStateSetDefaults(&m_channelMaskState);
    nvnColorStateSetDefaults(&m_colorState);
    nvnMultisampleStateSetDefaults(&m_multisampleState);
    nvnMultisampleStateSetMultisampleEnable(&m_multisampleState, NVN_FALSE);
    nvnPolygonStateSetDefaults(&m_polygonState);
    nvnPolygonStateSetCullFace(&m_polygonState, NVN_FACE_BACK);
    nvnDepthStencilStateSetDefaults(&m_depthStencilState);
    m_videoSampler.Init(m_device.GetDevice());

    m_queue.WaitForAllCommands();
    m_device.ResizeRenderTargets(m_window.GetWidth(), m_window.GetHeight());
    return movie::Status_Success;
}

void NvnVideoRenderer::CreateVideoBufferAndTextures(movie::Decoder* decoder, int width, int height)
{
    const float vertices[] =
    {
        -1.0f,  1.00f, 0, 1, 0, 0,
        -1.0f, -1.00f, 0, 1, 0, 1,
        -1.0f,  1.00f, 0, 1, 1, 0,
        -1.0f, -1.00f, 0, 1, 1, 1,
    };
    m_SizeOfVertices = sizeof(vertices);
    float aspectRatioDisplay = static_cast<float>( m_window.GetWidth() ) / m_window.GetHeight();
    float aspectRatioVideo = static_cast<float>( width / m_NumberOfVideosToRender ) / height;
    float ndcWidth = aspectRatioVideo / aspectRatioDisplay;
    if( m_NumberOfVideosToRender == 4 )
    {
        for( int i = 0; i < m_NumberOfVideosToRender; ++i )
        {
            float xOff = 0.0;
            float yOff = 0.0;
            switch( i )
            {
                case 0: xOff = -1.0f; yOff = -1.0f; break;
                case 1: xOff = 0.0f; yOff = -1.0f; break;
                case 2: xOff = 0.0f; yOff = 0.0f; break;
                case 3: xOff = -1.0f; yOff = 0.0f; break;
                default:
                    break;
            }
            float realVertices[] =
            {
                xOff, yOff + 1.0f, 0, 1, 0, 0,
                xOff, yOff, 0, 1, 0, 1,
                xOff + 1.0f, yOff + 1.0f, 0, 1, 1, 0,
                xOff + 1.0f, yOff, 0, 1, 1, 1,
            };
            memcpy(static_cast<char*>( m_pVertexPoolMemory ) + sizeof(realVertices) * i, realVertices, sizeof(realVertices));
        }
    }
    else
    {
        for( int i = 0; i < m_NumberOfVideosToRender; ++i )
        {
            float realVertices[] =
            {
                -1.0f + 2.0f * i / m_NumberOfVideosToRender,  ndcWidth * 1.00f, 0, 1, 0, 0,
                -1.0f + 2.0f * i / m_NumberOfVideosToRender, ndcWidth * -1.00f, 0, 1, 0, 1,
                -1.0f + 2.0f * ( i + 1 ) / m_NumberOfVideosToRender,  ndcWidth * 1.00f, 0, 1, 1, 0,
                -1.0f + 2.0f * ( i + 1 ) / m_NumberOfVideosToRender, -ndcWidth * 1.00f, 0, 1, 1, 1,
            };
            memcpy(static_cast<char*>( m_pVertexPoolMemory ) + sizeof(realVertices) * i, realVertices, sizeof(realVertices));
        }
    }
    int textureStrideAlignment = m_device.GetDeviceInfo(NVN_DEVICE_INFO_LINEAR_TEXTURE_STRIDE_ALIGNMENT);
    int videoStride = Align(width, textureStrideAlignment);
    unsigned texturePoolSize = Align(videoStride * height * 2, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
        texturePoolSize = Align(videoStride * height * 8, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    unsigned videoBufferSize = width * height * 3 / 2;
    if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
        videoBufferSize = width * height * 4;
    const unsigned bufferMemorySize = Align(videoBufferSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);
    NvnRenderContext *renderContext = nullptr;
    if( m_NvnRenderContextMap.find(decoder) != m_NvnRenderContextMap.end() )
        renderContext = m_NvnRenderContextMap.find(decoder)->second;
    if( renderContext != nullptr )
    {
        for( int i = 0; i < m_MaxNvnFramesPerDecoder; ++i )
        {
            NvnVideoFrameContext* nvnVideoFrameContext = new NvnVideoFrameContext(decoder);
            nvnVideoFrameContext->index = i;
            nvnVideoFrameContext->pTextureMemory = m_allocator.Allocate(texturePoolSize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
            nvnVideoFrameContext->textureMemoryPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_NO_ACCESS_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, texturePoolSize, nvnVideoFrameContext->pTextureMemory);
            if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
            {
                nvnVideoFrameContext->pBufferMemory = m_allocator.Allocate(bufferMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
                nvnVideoFrameContext->bufferMemory.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, bufferMemorySize, nvnVideoFrameContext->pBufferMemory);
                int videoStride = Align(width, textureStrideAlignment);
                NvnMemoryPool &textureMemoryPool = nvnVideoFrameContext->textureMemoryPool;
                nvnVideoFrameContext->videoTextureY.Init(m_device.GetDevice(), textureMemoryPool, width, height, NVN_FORMAT_R8, NVN_TEXTURE_FLAGS_LINEAR_BIT, videoStride);
                nvnVideoFrameContext->videoTextureUV.Init(m_device.GetDevice(), textureMemoryPool, width / 2, height / 2, NVN_FORMAT_RG8, NVN_TEXTURE_FLAGS_LINEAR_BIT, videoStride);
                nvnVideoFrameContext->videoTextureIdY = m_texturePool.Register(nvnVideoFrameContext->videoTextureY.GetTexture(), m_videoSampler.GetSampler());
                nvnVideoFrameContext->videoTextureIdUV = m_texturePool.Register(nvnVideoFrameContext->videoTextureUV.GetTexture(), m_videoSampler.GetSampler());
                nvnVideoFrameContext->videoTextureHandleY = nvnDeviceGetTextureHandle(&m_device.GetDevice(), nvnVideoFrameContext->videoTextureIdY, nvnVideoFrameContext->videoTextureIdY);
                nvnVideoFrameContext->videoTextureHandleUV = nvnDeviceGetTextureHandle(&m_device.GetDevice(), nvnVideoFrameContext->videoTextureIdUV, nvnVideoFrameContext->videoTextureIdUV);
                nvnVideoFrameContext->videoBufferYUV.Init(m_device.GetDevice(), nvnVideoFrameContext->bufferMemory.GetMemoryPool(), nvnVideoFrameContext->bufferMemory.GetNewMemoryChunkOffset(videoBufferSize, 1), videoBufferSize);
                void *buff = reinterpret_cast< char* >( nvnBufferMap(&nvnVideoFrameContext->videoBufferYUV.GetBuffer()) );
                nvnVideoFrameContext->handle = buff;
                nvnVideoFrameContext->yuvBufferSize = videoBufferSize;
            }
            else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
            {
                bool nvnVideoInteropMode = true;
                nvnVideoFrameContext->rgbTexture.Init(m_device.GetDevice(),
                    nvnVideoFrameContext->textureMemoryPool,
                    width,
                    height,
                    NVN_FORMAT_RGBA8,
                    NVN_TEXTURE_FLAGS_LINEAR_BIT, Align(videoStride * 4, textureStrideAlignment),
                    nvnVideoInteropMode);
                nvnVideoFrameContext->rgbTextureId = m_texturePool.Register(nvnVideoFrameContext->rgbTexture.GetTexture(), m_videoSampler.GetSampler());
                nvnVideoFrameContext->rgbTextureHandle = nvnDeviceGetTextureHandle(&m_device.GetDevice(), nvnVideoFrameContext->rgbTextureId, nvnVideoFrameContext->rgbTextureId);
                nvnVideoFrameContext->yuvBufferSize = videoBufferSize;
                nvnVideoFrameContext->handle = &nvnVideoFrameContext->rgbTexture.GetTexture();
            }
            renderContext->m_NvnVideoFrameList.push_back(nvnVideoFrameContext);
            renderContext->m_DecoderList.push_back(NvnFrameContext(decoder, nvnVideoFrameContext->handle, i, 0,0,0,0,0,0, videoBufferSize));
            renderContext->m_AllocList.push_back(NvnFrameContext(decoder, nvnVideoFrameContext->handle, i, 0, 0, 0, 0, 0, 0, videoBufferSize));
        }
        renderContext->uniformUsefulSize = sizeof(float) * 4 * 4;
        const unsigned uniformMemorySize = Align(renderContext->uniformUsefulSize, NVN_MEMORY_POOL_STORAGE_GRANULARITY);;
        renderContext->pUniformMemory = m_allocator.Allocate(uniformMemorySize, NVN_MEMORY_POOL_STORAGE_ALIGNMENT);
        renderContext->uniformPool.Init(m_device.GetDevice(), NVN_MEMORY_POOL_FLAGS_CPU_UNCACHED_BIT | NVN_MEMORY_POOL_FLAGS_GPU_CACHED_BIT, uniformMemorySize, renderContext->pUniformMemory);
        renderContext->uniformBuffer.Init(m_device.GetDevice(), renderContext->uniformPool.GetMemoryPool(), renderContext->uniformPool.GetNewMemoryChunkOffset(renderContext->uniformUsefulSize, 1), renderContext->uniformUsefulSize);
    }
}//NOLINT(impl/function_size)

movie::Status NvnVideoRenderer::RegisterDecoder(movie::Decoder* decoder, int width, int height, NvnOutputFormat nvnOutputFormat)
{
    NvnRenderContext *nvnRenderContext = new NvnRenderContext(decoder, width, height, m_VideoId);
    m_NvnRenderContextMap.insert(std::pair<movie::Decoder *, NvnRenderContext *>(decoder, nvnRenderContext));
    m_VideoId++;
    return movie::Status_Success;
}

void NvnVideoRenderer::Finalize()
{
    NvnRenderContext *nvnRenderContext = nullptr;
    for( auto &element : m_NvnRenderContextMap )
    {
        nvnRenderContext = element.second;
        for( int i = 0; i < m_MaxNvnFramesPerDecoder; ++i )
        {
            if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
            {
                nvnRenderContext->m_NvnVideoFrameList.at(i)->videoBufferYUV.Destroy();
                nvnRenderContext->m_NvnVideoFrameList.at(i)->videoTextureY.Destroy();
                nvnRenderContext->m_NvnVideoFrameList.at(i)->videoTextureUV.Destroy();
                m_allocator.Free(nvnRenderContext->m_NvnVideoFrameList.at(i)->pBufferMemory);
            }
            else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
                nvnRenderContext->m_NvnVideoFrameList.at(i)->rgbTexture.Destroy();

            nvnRenderContext->m_NvnVideoFrameList.at(i)->bufferMemory.Destroy();
            nvnRenderContext->m_NvnVideoFrameList.at(i)->textureMemoryPool.Destroy();
            m_allocator.Free(nvnRenderContext->m_NvnVideoFrameList.at(i)->pTextureMemory);
            delete nvnRenderContext->m_NvnVideoFrameList.at(i);
        }
        nvnRenderContext->uniformBuffer.Destroy();
        nvnRenderContext->m_NvnVideoFrameList.clear();
        delete nvnRenderContext;
    }
    m_fragmentDataBuffer.Destroy();
    m_vertexDataBuffer.Destroy();
    m_videoSampler.Destroy();
    m_texturePool.Destroy();
    m_program.Destroy();
    m_shaderScratchPool.Destroy();
    m_uniformPool.Destroy();
    m_renderTargetCommand.Destroy();
    m_commandPool.Destroy();
    m_vertexPool.Destroy();
    m_shaderPool.Destroy();
    m_queue.Destroy();
    m_device.Destroy();
    m_window.Destroy();
    if( m_pShaderPoolMemory != NULL )
    {
        m_allocator.Free(m_pShaderPoolMemory);
        m_pShaderPoolMemory = nullptr;
    }
    if( m_pVertexPoolMemory != NULL )
    {
        m_allocator.Free(m_pVertexPoolMemory);
        m_pVertexPoolMemory = nullptr;
    }
    m_squareMesh.Destroy();
    delete[] m_squareMeshes;
    if( m_pCommandMemory != NULL )
    {
        m_allocator.Free(m_pCommandMemory);
        m_pCommandMemory = nullptr;
    }
    if( m_pShaderScratchMemory != NULL )
    {
        m_allocator.Free(m_pShaderScratchMemory);
        m_pShaderScratchMemory = nullptr;
    }
    m_allocator.Finalize();
    if( m_pAllocatorMemory != NULL )
    {
        std::free(m_pAllocatorMemory);
        m_pAllocatorMemory = nullptr;
    }
}

void NvnVideoRenderer::GetNvnVideoTexture(movie::Decoder* decoder, int *index, void **textureHandle, size_t *size)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( textureHandle != nullptr ) && ( size != nullptr ) )
    {
        NvnRenderContext *renderContext = nullptr;
        if( m_NvnRenderContextMap.find(decoder) != m_NvnRenderContextMap.end() )
            renderContext = m_NvnRenderContextMap.find(decoder)->second;
        if( renderContext != nullptr )
        {
            if( renderContext->m_DecoderList.size() > 0 )
            {
                *index = renderContext->m_DecoderList.begin()->index;
                *textureHandle = renderContext->m_DecoderList.begin()->handle;
                *size = sizeof(void*);
                renderContext->m_DecoderList.erase(renderContext->m_DecoderList.begin());
            }
        }
    }
}

void NvnVideoRenderer::GetNvnVideoBuffer(movie::Decoder* decoder, int *index, void **bufferMemory, size_t *yuvBufferSize)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( bufferMemory != nullptr ) && ( yuvBufferSize != nullptr ) )
    {
        NvnRenderContext *renderContext = nullptr;
        if( m_NvnRenderContextMap.find(decoder) != m_NvnRenderContextMap.end() )
            renderContext = m_NvnRenderContextMap.find(decoder)->second;
        if( renderContext != nullptr )
        {
            if( renderContext->m_DecoderList.size() > 0 )
            {
                *bufferMemory = renderContext->m_DecoderList.begin()->handle;
                *yuvBufferSize = renderContext->m_DecoderList.begin()->yuvBufferSize;
                *index = renderContext->m_DecoderList.begin()->index;
                renderContext->m_DecoderList.erase(renderContext->m_DecoderList.begin());
            }
        }
    }
}

void NvnVideoRenderer::GetNvnSyncInfo(void **deviceHandle, void **syncHandle)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( deviceHandle != nullptr ) && ( syncHandle != nullptr ) )
    {
        *deviceHandle = static_cast<void*>(&m_device.GetDevice());
        *syncHandle = static_cast<void*>(&m_queue.GetNvnSync());
    }
}

void NvnVideoRenderer::ReleaseVideoFrame()
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    for( auto &&element : m_NvnRenderContextMap )
    {
        if( element.second->m_RenderList.size() > 1  )
        {
            element.second->m_DecoderList.push_back(*element.second->m_RenderList.begin());
            element.second->m_RenderList.erase(element.second->m_RenderList.begin());
        }
    }
}

void NvnVideoRenderer::ReturnBufferToRenderer(movie::Decoder* decoder, void *handle, int allocIndex)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( decoder != nullptr ) && ( handle != nullptr ) )
    {
        for( auto &&element : m_NvnRenderContextMap )
        {
            if( element.second->decoder == decoder )
            {
                for( auto &&ele : element.second->m_AllocList )
                {
                    if( ele.handle == handle )
                    {
                        element.second->m_DecoderList.push_back(element.second->m_AllocList.at(allocIndex));
                        break;
                    }
                }
            }
        }
    }
}

void NvnVideoRenderer::AddTextureToRender(movie::Decoder* decoder, void *textureHandle, int allocIndex, int width, int height)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( decoder != nullptr ) && ( textureHandle != nullptr ) )
    {
        for( auto &&element : m_NvnRenderContextMap )
        {
            if( element.second->decoder == decoder )
            {
                for( auto &&ele : element.second->m_AllocList )
                {
                    if( ele.handle == textureHandle )
                    {
                        element.second->m_RenderList.push_back(NvnFrameContext(decoder, ele.handle, allocIndex, width, height, 0, 0, 0, 0, 0));
                        break;
                    }
                }
            }
        }
    }
}

void NvnVideoRenderer::AddVideoFrameToRender(movie::Decoder* decoder, void *buffer, int index, int width, int height, int yOffset, int yStride, int uvOffset, int colorSpace, size_t bufferSize)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( decoder != nullptr ) && ( buffer != nullptr ) )
    {
        for( auto &&element : m_NvnRenderContextMap )
        {
            if( element.second->decoder == decoder )
            {
                for( auto &&ele : element.second->m_AllocList )
                {
                    if( ele.handle == buffer )
                    {
                        element.second->m_RenderList.push_back(NvnFrameContext(decoder, ele.handle, index, width, height, yOffset, yStride, uvOffset, colorSpace, bufferSize));
                        break;
                    }
                }
            }
        }
    }
}

void NvnVideoRenderer::GetNvnTextureInfo(movie::Decoder* decoder, int index, NvnTexture** rgbTexture, NVNtextureHandle** rgbTextureHandle)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( decoder != nullptr ) && ( rgbTexture != nullptr ) && ( rgbTextureHandle != nullptr ) )
    {
        for( auto &&element : m_NvnRenderContextMap )
        {
            if( element.second->decoder == decoder )
            {
                auto nvnTexture = element.second->m_NvnVideoFrameList.at(index);
                *rgbTexture = &nvnTexture->rgbTexture;
                *rgbTextureHandle = &nvnTexture->rgbTextureHandle;
                break;
            }
        }
    }
}

void NvnVideoRenderer::GetNvnVideoFrameInfo(movie::Decoder* decoder, int index, NvnBuffer** nvnBufferYUV,
    NVNtextureHandle**  textureHandleY, NVNtextureHandle**  textureHandleUV, NvnTexture** textureY, NvnTexture** textureUV)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( decoder != nullptr ) && ( nvnBufferYUV != nullptr ) && ( textureHandleY != nullptr ) &&
        ( textureHandleUV != nullptr ) && ( textureY != nullptr ) && ( textureUV != nullptr )  )
    {
        for( auto &&element : m_NvnRenderContextMap )
        {
            if( element.second->decoder == decoder )
            {
                auto nvnVideoFrame = element.second->m_NvnVideoFrameList.at(index);
                *nvnBufferYUV = &nvnVideoFrame->videoBufferYUV;
                *textureHandleY = &nvnVideoFrame->videoTextureHandleY;
                *textureHandleUV = &nvnVideoFrame->videoTextureHandleUV;
                *textureY = &nvnVideoFrame->videoTextureY;
                *textureUV = &nvnVideoFrame->videoTextureUV;
                break;
            }
        }
    }
}

void NvnVideoRenderer::GetDecodeVideoFrameInfo(movie::Decoder* decoder, int *index, int *width, int *height, int *yOffset, int *yStride, int *uvOffset, int *colorSpace)
{
    std::lock_guard<nn::os::Mutex> lock { m_Lock };
    if( ( decoder != nullptr ) && ( index != nullptr ) && ( width != nullptr ) &&
        ( height != nullptr ) && ( yOffset != nullptr ) && ( yStride != nullptr ) &&
        ( uvOffset != nullptr ) && ( colorSpace != nullptr ) )
    {
        for( auto &&element : m_NvnRenderContextMap )
        {
            if( element.second->decoder == decoder )
            {
                if( element.second->m_RenderList.size() > 0 )
                {
                    auto nvnVideoFrame = element.second->m_RenderList.begin();
                    *index = nvnVideoFrame->index;
                    *width = nvnVideoFrame->width;
                    *height = nvnVideoFrame->height;
                    *yOffset = nvnVideoFrame->yOffset;
                    *yStride = nvnVideoFrame->yStride;
                    *uvOffset = nvnVideoFrame->uvOffset;
                    *colorSpace = nvnVideoFrame->colorSpace;
                    break;
                }
            }
        }
    }
}

bool NvnVideoRenderer::CanStartRendering()
{
    bool tryRender = true;
    for( auto element : m_NvnRenderContextMap )
    {
        if( element.second->m_RenderList.size() < 1 )
        {
            tryRender = false;
            break;
        }
    }
    return tryRender;
}

bool NvnVideoRenderer::Draw(int64_t *timeToRenderUs)
{
    if( timeToRenderUs == nullptr )
        return false;
    int64_t nowStartUs = Clock::GetNowUs();
    *timeToRenderUs = 0;
    if( false == CanStartRendering() )
        return false;
    bool anyFrameToRender = false;
    for( auto element : m_NvnRenderContextMap )
    {
        if( element.second->m_RenderList.size() > 0 )
        {
            anyFrameToRender = true;
            break;
        }
    }
    if( anyFrameToRender == false )
    {
        ReleaseVideoFrame();
        return false;
    }
    int bufferIndex;
    nvnQueueAcquireTexture(&m_queue.GetQueue(), &m_device.GetWindow(), &bufferIndex);
    m_queue.WaitForAllCommands();
    m_renderTargetCommand.Reset();
    NVNcommandBuffer *targetCommandBuffer = &m_renderTargetCommand.GetCommandBuffer();
    nvnCommandBufferBeginRecording(targetCommandBuffer);
    {
        NVNtexture *renderTarget = &m_device.GetRenderTarget(bufferIndex);
        nvnCommandBufferSetRenderTargets(targetCommandBuffer, 1, &renderTarget, NULL, NULL, NULL);
        nvnCommandBufferSetScissor(targetCommandBuffer, 0, 0, m_window.GetWidth(), m_window.GetHeight());
        nvnCommandBufferSetViewport(targetCommandBuffer, 0, 0, m_window.GetWidth(), m_window.GetHeight());
        nvnCommandBufferBindBlendState(targetCommandBuffer, &m_blendState);
        nvnCommandBufferBindChannelMaskState(targetCommandBuffer, &m_channelMaskState);
        nvnCommandBufferBindColorState(targetCommandBuffer, &m_colorState);
        nvnCommandBufferBindMultisampleState(targetCommandBuffer, &m_multisampleState);
        nvnCommandBufferBindPolygonState(targetCommandBuffer, &m_polygonState);
        nvnCommandBufferBindDepthStencilState(targetCommandBuffer, &m_depthStencilState);
        nvnCommandBufferSetShaderScratchMemory(targetCommandBuffer, &m_shaderScratchPool.GetMemoryPool(), 0, m_ShaderScratchGranularity);
        float black[ 4 ] = { 0.0f, 0.0f, 0.0f, 1.0f };
        nvnCommandBufferClearColor(targetCommandBuffer, 0, black, NVN_CLEAR_COLOR_MASK_RGBA);
        nvnCommandBufferSetSampleMask(targetCommandBuffer, static_cast<int>( ~0 ));
        NVNvertexAttribState vertexAttrib[ 2 ];
        for( int i = 0; i < sizeof(vertexAttrib) / sizeof(*vertexAttrib); ++i )
        {
            nvnVertexAttribStateSetDefaults(vertexAttrib + i);
            nvnVertexAttribStateSetStreamIndex(vertexAttrib + i, i);
        }
        nvnVertexAttribStateSetFormat(vertexAttrib + 0, NVN_FORMAT_RGBA32F, 0);
        nvnVertexAttribStateSetFormat(vertexAttrib + 1, NVN_FORMAT_RG32F, sizeof(float) * 4);
        nvnCommandBufferBindVertexAttribState(targetCommandBuffer, sizeof(vertexAttrib) / sizeof(*vertexAttrib), vertexAttrib);
        NVNvertexStreamState vertexStream[ 2 ];
        for( int i = 0; i < sizeof(vertexStream) / sizeof(*vertexStream); ++i )
        {
            nvnVertexStreamStateSetDefaults(vertexStream + i);
            nvnVertexStreamStateSetStride(vertexStream + i, sizeof(float) * 6);
        }
        nvnCommandBufferBindVertexStreamState(targetCommandBuffer, sizeof(vertexAttrib) / sizeof(*vertexAttrib), vertexStream);
        nvnCommandBufferBindProgram(targetCommandBuffer, &m_program.GetProgram(), NVN_SHADER_STAGE_VERTEX_BIT | NVN_SHADER_STAGE_FRAGMENT_BIT);
        nvnCommandBufferSetTexturePool(targetCommandBuffer, &m_texturePool.GetTexturePool());
        nvnCommandBufferSetSamplerPool(targetCommandBuffer, &m_texturePool.GetSamplerPool());
        for( auto element : m_NvnRenderContextMap )
        {
            nvnCommandBufferSetCopyRowStride(targetCommandBuffer, 0);
            int index = 0; int width = 0; int height = 0; int colorSpace = 0;
            int yOffset = 0; int yStride = 0; int uvOffset = 0;
            GetDecodeVideoFrameInfo(element.second->decoder, &index, &width, &height, &yOffset, &yStride, &uvOffset, &colorSpace);
            if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
            {
                NvnBuffer* nvnBufferYUV;
                NvnTexture* textureY;
                NvnTexture* textureUV;
                NVNtextureHandle *textureHandleY;
                NVNtextureHandle *textureHandleUV;
                GetNvnVideoFrameInfo(element.second->decoder, index, &nvnBufferYUV, &textureHandleY, &textureHandleUV, &textureY, &textureUV);
                NVNcopyRegion copyRegion;
                copyRegion.depth = 1;
                copyRegion.height = height;
                copyRegion.width = width;
                copyRegion.xoffset = 0;
                copyRegion.yoffset = 0;
                copyRegion.zoffset = 0;
                nvnCommandBufferCopyBufferToTexture(targetCommandBuffer, nvnBufferYUV->GetAddress() + yOffset, &textureY->GetTexture(), NULL, &copyRegion, NVN_COPY_FLAGS_NONE);
                copyRegion.height = height / 2;
                copyRegion.width = width / 2;
                nvnCommandBufferCopyBufferToTexture(targetCommandBuffer, nvnBufferYUV->GetAddress() + uvOffset, &textureUV->GetTexture(), NULL, &copyRegion, NVN_COPY_FLAGS_NONE);
                nvnCommandBufferBindTexture(targetCommandBuffer, NVN_SHADER_STAGE_FRAGMENT, 0, *textureHandleY);
                nvnCommandBufferBindTexture(targetCommandBuffer, NVN_SHADER_STAGE_FRAGMENT, 1, *textureHandleUV);
            }
            else if( m_NvnOutputFormat == NvnOutputFormat_AbgrNvnTexture )
            {
                NvnTexture* rgbTexture;
                NVNtextureHandle* rgbTextureHandle;
                NVNcopyRegion srcCopyRegion;
                srcCopyRegion.depth = 0;
                srcCopyRegion.height = height;
                srcCopyRegion.width = width;
                srcCopyRegion.xoffset = 0;
                srcCopyRegion.yoffset = 0;
                srcCopyRegion.zoffset = 0;

                NVNcopyRegion dstCopyRegion;
                dstCopyRegion.height = height / m_NumberOfVideosToRender;
                dstCopyRegion.width = width / m_NumberOfVideosToRender;
                dstCopyRegion.height = height;
                dstCopyRegion.width = width;
                dstCopyRegion.depth = 0;
                dstCopyRegion.xoffset = 0;
                dstCopyRegion.yoffset = 0;
                dstCopyRegion.zoffset = 0;

                GetNvnTextureInfo(element.second->decoder, index, &rgbTexture, &rgbTextureHandle);
                nvnCommandBufferCopyTextureToTexture(targetCommandBuffer, &rgbTexture->GetTexture(), NULL, &srcCopyRegion, renderTarget, NULL, &dstCopyRegion, NVN_COPY_FLAGS_NONE);
                nvnCommandBufferBindTexture(targetCommandBuffer, NVN_SHADER_STAGE_FRAGMENT, 0, *rgbTextureHandle);
            }

            NvnBuffer &m_squareMesh = m_squareMeshes[ element.second->videoId ];
            nvnCommandBufferBindVertexBuffer(targetCommandBuffer, 0, m_squareMesh.GetAddress(), m_SizeOfVertices);
            nvnCommandBufferBindVertexBuffer(targetCommandBuffer, 1, m_squareMesh.GetAddress(), m_SizeOfVertices);
            nvnCommandBufferBindUniformBuffer(targetCommandBuffer, NVN_SHADER_STAGE_FRAGMENT, 2, element.second->uniformBuffer.GetAddress(), element.second->uniformUsefulSize);
            nvnCommandBufferDrawArrays(targetCommandBuffer, NVN_DRAW_PRIMITIVE_TRIANGLE_STRIP, 0, 4);
            if( m_NvnOutputFormat == NvnOutputFormat_YuvNv12NvnBuffer )
            {
                switch( colorSpace )
                {
                    case movie::ColorSpace_YCbCr2020:
                    case movie::ColorSpace_YCbCr601:
                    {
                        const float x = 1.164383f;
                        const float y = 1.138393f;
                        const float z = 1.138393f;
                        const float uniforms[] = {
                            16.0f / 255, 128.0f / 255, 128.0f / 255, 0,
                            1.00000000f * x,  0.00000000f * y,  1.40200000f * z, 0,
                            1.00000000f * x, -0.34413629f * y, -0.71413629f * z, 0,
                            1.00000000f * x,  1.77200000f * y,  0.00000000f * z, 0
                        };
                        memcpy(element.second->pUniformMemory, uniforms, sizeof(uniforms));
                        break;
                    }
                    case movie::ColorSpace_YCbCr601_ER:
                    {
                        const float uniforms[] = {
                            0, 128.0f / 255, 128.0f / 255, 0,
                            1.00000000f,  0.00000000f,  1.40200000f, 0,
                            1.00000000f, -0.34413629f, -0.71413629f, 0,
                            1.00000000f,  1.77200000f,  0.00000000f, 0
                        };
                        memcpy(element.second->pUniformMemory, uniforms, sizeof(uniforms));
                        break;
                    }
                    case movie::ColorSpace_YCbCr709:
                    {
                        const float x = 1.164383f;
                        const float y = 1.138393f;
                        const float z = 1.138393f;
                        const float uniforms[] = {
                            16.0f / 255, 128.0f / 255, 128.0f / 255, 0,
                            1.00000000f * x,  0.00000000f * y,  1.57480000f * z, 0,
                            1.00000000f * x, -0.18732427f * y, -0.46812427f * z, 0,
                            1.00000000f * x,  1.85560000f * y,  0.00000000f * z, 0
                        };
                        memcpy(element.second->pUniformMemory, uniforms, sizeof(uniforms));
                        break;
                    }
                    case movie::ColorSpace_YCbCr709_ER:
                    {
                        const float uniforms[] = {
                            0, 128.0f / 255, 128.0f / 255, 0,
                            1.00000000f,  0.00000000f,  1.57480000f, 0,
                            1.00000000f, -0.18732427f, -0.46812427f, 0,
                            1.00000000f,  1.85560000f,  0.00000000f, 0
                        };
                        memcpy(element.second->pUniformMemory, uniforms, sizeof(uniforms));
                        break;
                    }
                    default:
                        break;
                }
            }
        }
    }
    NVNcommandHandle commandHandleTarget = nvnCommandBufferEndRecording(targetCommandBuffer);
    nvnQueueSubmitCommands(&m_queue.GetQueue(), 1, &commandHandleTarget);
    m_queue.WaitForAllCommands();
    nvnQueuePresentTexture(&m_queue.GetQueue(), &m_device.GetWindow(), bufferIndex);
    ReleaseVideoFrame();
    *timeToRenderUs = Clock::GetNowUs() - nowStartUs;
    return true;
}//NOLINT(impl/function_size)
