﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "MovieVideoInputHandler.h"

/**
 * @brief
 * MovieVideoInputHandler class for handling video decoder input events and feed input data to the decoder
 *
 * @details
 *
 * Purpose:
 *     The purpose of @class MovieVideoInputHandler is to handle video decoder input events. @class MovieDecoderEventHandler
 * registers events with decoder. When any input buffer related events are signaled from the decoder, @class MovieDecoderEventHandler
 * will sent a message to @class MovieVideoInputHandler message queue. When the @class MovieVideoInputHandler receives
 * input buffer available message, it will check whether the extractor has data for video track. If extractor has data
 * for video track it will get input buffer from decoder and reads input video data from the extractor. The filled input buffer is
 * sent for decoding. It also handles flush message, by clearing all input buffers from its queue.
 *
 * Setup:
 *     @class MovieDecoderPlayer will create @class MovieVideoInputHandler instance. During initialization a thread is created.
 * The thread will wait for messages in this message queue.
 *
 * Main Processing:
 *     When @class MovieDecoderEventHandler receives input buffer available event from video decoder, it sends a message to
 * @class MovieVideoInputHandler message queue. When the message is serviced, extractor is checked for video data availability.
 * If video data is available, video decoder input buffer is requested from decoder. This will be filled with compressed
 * video data. Filled input buffer is sent to decoder for decoding. This process continues as long as input data is available
 * from extractor. When end of stream is received from extractor, an input buffer with EOS flag is sent to the decoder.
 *
 * Teardown:
 *    When @class MovieDecoderEventHandler receives EOS buffer from video decoder, media playback will be stopped. At the end
 * MovieDecoderPlayer will delete @class MovieVideoInputHandler instance. This will result in sending thread exit message
 * to @class MovieVideoInputHandler thread. The thread will exit and resources are released.
 *
 */

MovieVideoInputHandler::MovieVideoInputHandler()
    : m_ThreadStackSize(1024 * 256),
      m_ThreadStack(nullptr),
      m_MessageQueueBufferSize(256),
      m_MessageQueueBuffer(nullptr)
{
}

MovieVideoInputHandler::~MovieVideoInputHandler()
{
    nn::os::SendMessageQueue(&m_MessageQueue, MovieVideoInputHandler::MoviePlayerMessage_VideoInputThreadExit);
    nn::os::WaitThread(&m_ThreadType);
    nn::os::DestroyThread(&m_ThreadType);
    nn::os::FinalizeMessageQueue(&m_MessageQueue);
    for( auto &element : m_VideoInputBufferInfoMap )
        delete element.second;
    if( m_MessageQueueBuffer != nullptr )
        delete[] m_MessageQueueBuffer;
    if( m_ThreadStack != nullptr )
        free(m_ThreadStack);
}
movie::Status MovieVideoInputHandler::Initialize(uint64_t coreMask)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::Result nnResult = nn::ResultSuccess();
    if( coreMask )
        movie::SetCoreMask(coreMask);
    m_MessageQueueBuffer = new uintptr_t[ m_MessageQueueBufferSize ];
    if( m_MessageQueueBuffer == nullptr )
        return movie::Status_OutOfMemory;
    nn::os::InitializeMessageQueue(&m_MessageQueue, m_MessageQueueBuffer, m_MessageQueueBufferSize);
    nn::os::InitializeMutex(&m_VideoListMutex, false, 0);
    m_ThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
    if( m_ThreadStack == nullptr )
        return movie::Status_OutOfMemory;
    nnResult = nn::os::CreateThread(&m_ThreadType,
        &MovieVideoInputHandlerThreadFunction,
        ( void* )this,
        m_ThreadStack,
        m_ThreadStackSize,
        nn::os::DefaultThreadPriority);
    if( nnResult.IsSuccess() )
    {
        nn::os::SetThreadName(&m_ThreadType, "MovieVideoInputReadThread");
        nn::os::StartThread(&m_ThreadType);
    }
    else
        movieStatus = movie::Status_FailedToCreateThread;
    return movieStatus;
}

movie::Status MovieVideoInputHandler::RegisterExtractorAndDecoder(movie::Extractor* extractor, movie::Decoder *decoder)
{
    VideoInputBufferInfo *inputBufferInfo = new VideoInputBufferInfo(extractor, decoder);
    if( inputBufferInfo == nullptr )
        return movie::Status_OutOfMemory;
    m_VideoInputBufferInfoMap.insert(std::pair<movie::Decoder *, VideoInputBufferInfo* >(decoder, inputBufferInfo));
    return movie::Status_Success;
}

movie::Status MovieVideoInputHandler::AddBufferToVideoIndexList(movie::Decoder *decoder, int32_t bufferIndex)
{
    movie::Status movieStatus = movie::Status_Success;
    nn::os::LockMutex( &m_VideoListMutex );
    VideoInputBufferInfo *inputBufferInfo = nullptr;
    if( m_VideoInputBufferInfoMap.find(decoder) != m_VideoInputBufferInfoMap.end() )
        inputBufferInfo = m_VideoInputBufferInfoMap.find(decoder)->second;
    if( inputBufferInfo != nullptr )
        inputBufferInfo->m_IndexList.push_back(bufferIndex);
    nn::os::UnlockMutex( &m_VideoListMutex );
    return movieStatus;
}

movie::Status MovieVideoInputHandler::VideoInputBufferAvailableEvent(movie::Extractor* extractor, movie::Decoder* decoder)
{
    SignalVideoInputBufferAvailable(extractor, decoder);
    return movie::Status_Success;;
}

movie::Status MovieVideoInputHandler::SignalVideoInputBufferAvailable(movie::Extractor* extractor, movie::Decoder* decoder)
{
    nn::os::SendMessageQueue(&m_MessageQueue, MovieVideoInputHandler::MoviePlayerMessage_VideoInputAvailable);
    return movie::Status_Success;;
}

movie::Status MovieVideoInputHandler::RemoveBufferFromVideoIndexList(movie::Decoder *decoder, int32_t* bufferIndex)
{
    nn::os::LockMutex( &m_VideoListMutex );
    VideoInputBufferInfo *inputBufferInfo = nullptr;
    if( m_VideoInputBufferInfoMap.find(decoder) != m_VideoInputBufferInfoMap.end() )
        inputBufferInfo = m_VideoInputBufferInfoMap.find(decoder)->second;
    int32_t index = -1;
    if( ( inputBufferInfo != nullptr ) && ( inputBufferInfo->m_IndexList.size() > 0 ) )
    {
        index = inputBufferInfo->m_IndexList.front();
        inputBufferInfo->m_IndexList.erase(inputBufferInfo->m_IndexList.begin());
    }
    *bufferIndex = index;
    nn::os::UnlockMutex( &m_VideoListMutex );
    return movie::Status_Success;;
}

movie::Status MovieVideoInputHandler::ReadInputDataFromExtractorSendTodecoder(movie::Extractor* extractor, movie::Decoder* decoder, int index)
{
    movie::Status movieStatus = movie::Status_Success;
    movie::Buffer buffer;
    uint32_t flags = movie::BufferFlags_None;
    int64_t presentationTimeUs = 0;
    movieStatus = decoder->GetInputBuffer(index, &buffer);
    if( movieStatus == movie::Status_Success )
    {
        buffer.SetRange(0, 0);
        movieStatus = extractor->Read(&buffer);
        if( ( movieStatus != movie::Status_Success ) ||
            ( buffer.Size() <= 0 ) ||
            ( movieStatus == movie::Status_EndOfStream ) )
            flags = movie::BufferFlags_EndOfStream;
        extractor->GetSampleTime(&presentationTimeUs);
        decoder->SendInputBufferForDecode(index, buffer.Offset(), buffer.Size(), presentationTimeUs, flags);
        extractor->Advance();
    }
    return movieStatus;
}

movie::Status MovieVideoInputHandler::CheckForInputBuffersAndReadVideoInputData()
{
    movie::Status movieStatus = movie::Status_Success;
    size_t currentTrackIndex = -1;
    movie::Extractor* extractor = nullptr;
    movie::Decoder* decoder = nullptr;
    VideoInputBufferInfo*  inputBufferInfo = nullptr;
    for( std::map<movie::Decoder*, VideoInputBufferInfo*>::iterator it = m_VideoInputBufferInfoMap.begin(); it != m_VideoInputBufferInfoMap.end(); ++it )
    {
        inputBufferInfo = it->second;
        extractor = inputBufferInfo->m_Extractor;
        decoder = inputBufferInfo->m_Decoder;
        if( extractor != NULL )
        {
            movieStatus = extractor->GetTrackIndexForAvailableData(&currentTrackIndex);
            if( movieStatus == movie::Status_EndOfStream )
                currentTrackIndex = m_VideoTrackIndex;
            else if( movieStatus != movie::Status_Success )
                return movieStatus;
        }
        if( currentTrackIndex == m_VideoTrackIndex )
        {
            int index = -1;
            RemoveBufferFromVideoIndexList(decoder, &index);
            if( index != -1 )
                ReadInputDataFromExtractorSendTodecoder(extractor, decoder, index);
        }
    }
    return movieStatus;
}

void MovieVideoInputHandler::Flush(movie::Decoder* decoder)
{
    nn::os::LockMutex(&m_VideoListMutex);
    if( m_VideoInputBufferInfoMap.find(decoder) != m_VideoInputBufferInfoMap.end() )
        m_VideoInputBufferInfoMap.find(decoder)->second->m_IndexList.clear();
    nn::os::UnlockMutex(&m_VideoListMutex);
}

void MovieVideoInputHandler::Flush()
{
    nn::os::LockMutex(&m_VideoListMutex);
    for( auto element : m_VideoInputBufferInfoMap )
        element.second->m_IndexList.clear();
    nn::os::UnlockMutex(&m_VideoListMutex);
}

void MovieVideoInputHandler::SignalFlush()
{
    nn::os::SendMessageQueue(&m_MessageQueue, MovieVideoInputHandler::MoviePlayerMessage_VideoInputInputFlush);
}

void MovieVideoInputHandlerThreadFunction(void *arg)
{
    MovieVideoInputHandler *movieVideoInputHandler = ( MovieVideoInputHandler*) arg;
    if( movieVideoInputHandler != nullptr )
    {
        for( ;; )
        {
            uintptr_t data = 0;
            nn::os::TimedReceiveMessageQueue(&data, &movieVideoInputHandler->m_MessageQueue, nn::TimeSpan::FromMilliSeconds(10));
            if( data == MovieVideoInputHandler::MoviePlayerMessage_VideoInputAvailable )
                movieVideoInputHandler->CheckForInputBuffersAndReadVideoInputData();
            else if( data == MovieVideoInputHandler::MoviePlayerMessage_VideoInputThreadExit )
                break;
            else if( data == MovieVideoInputHandler::MoviePlayerMessage_VideoInputInputFlush )
                movieVideoInputHandler->Flush();
            else
                movieVideoInputHandler->CheckForInputBuffersAndReadVideoInputData();
        }
    }
}
