﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @file
 * @brief MovieDecoderPlayer API implementation.
 */

#include "MovieDecoderPlayer.h"

MovieDecoderPlayer::MovieDecoderPlayer()
    : m_State(State_NotInitialized),
      m_ErrorEvent(nullptr),
      m_StateChangedEvent(nullptr),
      m_NativeWindow(nullptr),
      m_Loop(false),
      m_MovieVideoInputHandler(nullptr),
      m_MovieVideoOutputHandler(nullptr),
      m_MovieDecoderEventHandler(nullptr),
      m_CoreMask(0),
      m_ThreadStackSize(1024 * 128),
      m_ThreadStack(nullptr),
      m_LastError(movie::Status_Success),
      m_EventsInitialized(false),
      m_ThreadCreated(false),
      m_ThreadStarted(false),
      m_StateLock {false},
      m_ApiLock { false },
      m_MediaDurationUs(0),
      m_TrackIndex(0),
      m_DecoderMode(movie::DecoderMode_NativeTexture){}

MovieDecoderPlayer::~MovieDecoderPlayer()
{
    State currentState = State_NotInitialized;
    GetState(&currentState);
    if( currentState != State_NotInitialized )
    {
        if( m_ThreadCreated == true )
        {
            if( m_ThreadStarted == true )
            {
                nn::os::SignalEvent(&m_ThreadExitEvent);
                nn::os::WaitThread(&m_ThreadType);
            }
            nn::os::DestroyThread(&m_ThreadType);
        }
        for( auto&& decoderInfo : m_DecoderInfoList )
        {
            nn::os::UnlinkMultiWaitHolder(&decoderInfo.renderCompleteEventHolder);
            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&decoderInfo.renderCompleteEventHolder));
            nn::os::FinalizeMultiWaitHolder(&decoderInfo.renderCompleteEventHolder);
            nn::os::FinalizeEvent(&decoderInfo.renderCompleteEvent);
        }
        if( m_EventsInitialized == true )
        {
            nn::os::UnlinkMultiWaitHolder(&m_PrepareEventHolder);
            nn::os::UnlinkMultiWaitHolder(&m_ThreadExitEventHolder);
            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_PrepareEventHolder));
            DeleteUserData(( EventUserData* ) GetMultiWaitHolderUserData(&m_ThreadExitEventHolder));
            nn::os::FinalizeMultiWaitHolder(&m_PrepareEventHolder);
            nn::os::FinalizeMultiWaitHolder(&m_ThreadExitEventHolder);
            nn::os::FinalizeEvent(&m_ThreadExitEvent);
            nn::os::FinalizeMultiWait(&m_PlayerMultiWait);
        }
        if( m_ThreadStack != nullptr )
        {
            free(m_ThreadStack);
            m_ThreadStack = nullptr;
        }
        if( m_MovieDecoderEventHandler != nullptr )
        {
            delete m_MovieDecoderEventHandler;
            m_MovieDecoderEventHandler = nullptr;
        }
        if( m_MovieVideoOutputHandler != nullptr )
        {
            delete m_MovieVideoOutputHandler;
            m_MovieVideoOutputHandler = nullptr;
        }
        DeletePlayerOwnedObjects();
    }
}

movie::Status MovieDecoderPlayer::Initialize()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( true == IsInRequestedState(State::State_Initialized) )
        return movie::Status_Success;
    if( m_EventsInitialized == false )
    {
        nn::os::InitializeMultiWait(&m_PlayerMultiWait);
        movieStatus = RegisterPlayerEvent(PlayerEventType_Prepare, &m_PrepareEvent, &m_PrepareEventHolder);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = RegisterPlayerEvent(PlayerEventType_ThreadExit, &m_ThreadExitEvent, &m_ThreadExitEventHolder);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        m_EventsInitialized = true;
    }
    if( m_ThreadCreated == false )
    {
        m_ThreadStackSize = 1024 * 128;
        m_ThreadStack = aligned_alloc(nn::os::ThreadStackAlignment, m_ThreadStackSize);
        if( m_ThreadStack == nullptr )
            movieStatus = movie::Status_OutOfMemory;
        else
        {
            nn::Result nnResult = nn::os::CreateThread(&m_ThreadType,
                &MoviePlayerThreadFunction,
                ( void* )this,
                m_ThreadStack,
                m_ThreadStackSize,
                nn::os::DefaultThreadPriority);
            if( nnResult.IsSuccess() )
            {
                m_ThreadCreated = true;
                nn::os::SetThreadName(&m_ThreadType, "MoviePlayerThread");
            }
            else
                movieStatus = movie::Status_FailedToCreateThread;
        }
    }
    movieStatus = ValidateStateTransition(State::State_Initialized);
    if( movieStatus == movie::Status_Success )
        SetState(State::State_Initialized);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SetViNativeWindowHandle(nn::vi::NativeWindowHandle nativeWindowHandle)
{
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( nativeWindowHandle == nullptr )
        return movie::Status_ErrorBadValue;
    else
        m_NativeWindow = nativeWindowHandle;
    return movie::Status_Success;
}

movie::Status MovieDecoderPlayer::RegisterEvent(EventType eventType, nn::os::EventType *event)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    switch( eventType )
    {
        case EventType_ErrorEvent:
            m_ErrorEvent = event;
            break;
        case EventType_StateChangedEvent:
            m_StateChangedEvent = event;
            break;
        default:
            break;
    }
    movieStatus = movie::Status_Success;
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SetDataSource(movie::Extractor** xtractor, std::string mediaPath, int *trackIndex, movie::DecoderMode decoderMode)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    m_DecoderMode = decoderMode;
    movie::Extractor* extractor = nullptr;
    movieStatus = CreateExtractorForPlayer(mediaPath, &extractor);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    movieStatus = extractor->SetDataSource(mediaPath.c_str());
    if( movieStatus == movie::Status_Success )
    {
        int trackIndex = -1;
        SelectVideoTrack(extractor, &trackIndex);
        DecoderInfo decoderInfo;
        GetDecoderType(extractor, trackIndex, &decoderInfo.decoderType);
        decoderInfo.extractor = extractor;
        decoderInfo.trackIndex = trackIndex;
        decoderInfo.decoderMode = decoderMode;
        decoderInfo.decoderOutputFormat = movie::DecoderOutputFormat_VideoColorNv12;
        if( decoderMode == movie::DecoderMode_NativeTexture )
            decoderInfo.decoderOutputFormat = movie::DecoderOutputFormat_VideoColorAbgr;
        movieStatus = extractor->GetTrackConfiguration(trackIndex, &decoderInfo.trackFormat);
        m_DecoderInfoList.push_back(decoderInfo);
        m_TrackIndex++;
    }
    *xtractor = extractor;
    return movieStatus;
}

movie::Status MovieDecoderPlayer::Prepare()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( true == IsInRequestedState(State::State_Preparing) )
        return movie::Status_Success;
    movieStatus = ValidateStateTransition(State::State_Preparing);
    if( movieStatus == movie::Status_Success )
        SetState(State::State_Preparing);
    else
        return movieStatus;
    m_MovieDecoderEventHandler = new MovieDecoderEventHandler();
    if( m_MovieDecoderEventHandler == nullptr )
        return movie::Status_OutOfMemory;
    uint64_t coreAffinityMask = 0;
    movieStatus = m_MovieDecoderEventHandler->Initialize(coreAffinityMask);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    m_MovieVideoInputHandler = new MovieVideoInputHandler();
    if( m_MovieVideoInputHandler == nullptr )
        return movie::Status_OutOfMemory;
    movieStatus = m_MovieVideoInputHandler->Initialize(coreAffinityMask);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    m_MovieVideoOutputHandler = new MovieVideoOutputHandler(m_DecoderInfoList.size(), m_DecoderMode);
    if( m_MovieVideoOutputHandler == nullptr )
        return movie::Status_OutOfMemory;
    movieStatus = m_MovieVideoOutputHandler->Initialize();
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.decoderType == movie::DecoderType_Unknown )
            continue;
        decoderInfo.decoder = new movie::Decoder(decoderInfo.decoderType, decoderInfo.decoderOutputFormat, decoderInfo.decoderMode);
        if( decoderInfo.decoder == nullptr )
            return movie::Status_OutOfMemory;
        RegisterPlayerEvent(PlayerEventType_VideoRenderComplete, &decoderInfo.renderCompleteEvent, &decoderInfo.renderCompleteEventHolder, decoderInfo.decoder);
        decoderInfo.mediaClock = new  MediaClock();
        if( decoderInfo.mediaClock == nullptr )
            return movie::Status_OutOfMemory;
        movieStatus = m_MovieVideoOutputHandler->RegisterDecoder(decoderInfo.decoder, decoderInfo.width, decoderInfo.height, decoderInfo.decoderOutputFormat, decoderInfo.mediaClock, &decoderInfo.renderCompleteEvent);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = m_MovieDecoderEventHandler->RegisterExtractorAndDecoder(decoderInfo.extractor, decoderInfo.decoder, &decoderInfo.decoderEvents);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = m_MovieDecoderEventHandler->RegisterInputHandler(m_MovieVideoInputHandler);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = m_MovieDecoderEventHandler->RegisterOutputHandler(m_MovieVideoOutputHandler);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = m_MovieVideoInputHandler->RegisterExtractorAndDecoder(decoderInfo.extractor, decoderInfo.decoder);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        movieStatus = decoderInfo.decoder->RegisterClientEvents(&decoderInfo.decoderEvents);
        if( movieStatus != movie::Status_Success )
            return movieStatus;
        if( decoderInfo.extractor != nullptr )
            movieStatus = decoderInfo.decoder->Configure(&decoderInfo.trackFormat);
        else
            return movie::Status_OutOfMemory;
    }

    if( m_ThreadStarted  == false)
    {
        nn::os::StartThread(&m_ThreadType);
        m_ThreadStarted = true;
    }
    m_MovieDecoderEventHandler->Start();
    movieStatus = ValidateStateTransition(State::State_Prepared);
    if( movieStatus == movie::Status_Success )
        SetState(State::State_Prepared);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::Start()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( true == IsInRequestedState(State::State_Started) )
        return movie::Status_Success;
    movieStatus = ValidateStateTransition(State::State_Started);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    State currentState;
    movieStatus = GetState(&currentState);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    for( auto&& decoderInfo : m_DecoderInfoList )
        decoderInfo.decoder->Start();
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        movie::MediaData videoFormat;
        int width = 0;
        int height = 0;
        movieStatus = decoderInfo.decoder->GetOutputFormat(&videoFormat);
        if( movieStatus == movie::Status_Success )
        {
            videoFormat.FindInt32("width", &width);
            videoFormat.FindInt32("height", &height);
            if( width > 0 )
                decoderInfo.width = width;
            if( height > 0 )
                decoderInfo.height = height;
        }
        videoFormat.Clear();
        if( m_MovieVideoOutputHandler != nullptr )
            m_MovieVideoOutputHandler->UpdateDecoderProperties(decoderInfo.decoder, decoderInfo.width, decoderInfo.height);
    }
    if( m_MovieVideoOutputHandler != nullptr )
        m_MovieVideoOutputHandler->Open();
    // Wait untill video renderer has data to render or maximum of 500ms (50 x 10) to start playback
    int waitTimeLoopCount = 0;
    int waitTimeLoopCountMax = 50;
    while( waitTimeLoopCount < waitTimeLoopCountMax )
    {
        bool canVideoStartRendering = true;
        waitTimeLoopCount ++;
        if( m_MovieVideoOutputHandler != nullptr )
            canVideoStartRendering = m_MovieVideoOutputHandler->CanStartRendering();
        if( canVideoStartRendering == true )
            break;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
    if( m_MovieVideoOutputHandler != nullptr )
        m_MovieVideoOutputHandler->Start();
    SetState(State::State_Started);
    movieStatus = movie::Status_Success;
    return movieStatus;
}

movie::Status MovieDecoderPlayer::Stop()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( true == IsInRequestedState(State::State_Stopped) )
        return movie::Status_Success;
    movieStatus = ValidateStateTransition(State::State_Stopped);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    for( auto&& decoderInfo : m_DecoderInfoList )
        decoderInfo.decoder->Stop();
    if( m_MovieDecoderEventHandler != nullptr )
    {
        delete m_MovieDecoderEventHandler;
        m_MovieDecoderEventHandler = nullptr;
    }
    if( m_MovieVideoOutputHandler != nullptr )
    {
        m_MovieVideoOutputHandler->Pause();
        m_MovieVideoOutputHandler->Stop();
        m_MovieVideoOutputHandler->Close();
        m_MovieVideoOutputHandler->Finalize();
    }
    DeletePlayerOwnedObjects();
    SetState(State::State_Stopped);
    movieStatus = movie::Status_Success;
    return movieStatus;
}

movie::Status MovieDecoderPlayer::Pause()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( true == IsInRequestedState(State::State_Paused) )
        return movie::Status_Success;
    movieStatus = ValidateStateTransition(State::State_Paused);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    if( m_MovieVideoOutputHandler != nullptr )
        m_MovieVideoOutputHandler->Pause();
    SetState(State::State_Paused);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::Resume()
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( true == IsInRequestedState(State::State_Started) )
        return movie::Status_Success;
    movieStatus = ValidateStateTransition(State::State_Started);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    m_MovieDecoderEventHandler->SignalOutputBufferAvailableEvent();
    m_MovieDecoderEventHandler->SignalInputBufferAvailableEvent();
    int waitTimeLoopCount = 0;
    int waitTimeLoopCountMax = 10;
    while( waitTimeLoopCount < waitTimeLoopCountMax )
    {
        bool canVideoStartRendering = true;
        waitTimeLoopCount++;
        if( m_MovieVideoOutputHandler != nullptr )
            canVideoStartRendering = m_MovieVideoOutputHandler->CanStartRendering();
        if( canVideoStartRendering == true )
            break;
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(10));
    }
    if( m_MovieVideoOutputHandler != nullptr )
        m_MovieVideoOutputHandler->Resume();
    SetState(State::State_Started);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::GetPlaybackPosition(movie::Extractor* extractor, int64_t *playbackPositionUs)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( playbackPositionUs == nullptr )
        return movie::Status_ErrorBadValue;
    *playbackPositionUs = 0;
    movie::Decoder* decoder = nullptr;
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.extractor == extractor )
        {
            decoder = decoderInfo.decoder;
            break;
        }
    }
    if( ( decoder != nullptr ) && ( m_MovieVideoOutputHandler != nullptr ) )
        movieStatus = m_MovieVideoOutputHandler->GetPlaybackPosition(decoder, playbackPositionUs);
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SelectVideoTrack(movie::Extractor* extractor, int *trackNumber)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    movie::MediaData trackFormat;
    if( trackNumber == nullptr )
        return movie::Status_ErrorBadValue;
    if( extractor == nullptr )
        return movie::Status_NullExtractor;
    int numberOfTracks = 0;
    extractor->GetTrackCount(&numberOfTracks);
    for( int i = 0; i < numberOfTracks; i++ )
    {
        extractor->GetTrackConfiguration(i, &trackFormat);
        const char *mime;
        if( trackFormat.FindString("mime", &mime) )
        {
            if( !strncasecmp("video/", mime, 6) )
            {
                movieStatus = extractor->SelectTrack(i);
                *trackNumber = i;
                break;
            }
        }
    }
    trackFormat.Clear();
    return movieStatus;
}

movie::Status MovieDecoderPlayer::GetTrackCount(movie::Extractor* extractor, int *trackCount) const
{
    movie::Status movieStatus = movie::Status_UnknownError;
    int numberOfTracks = 0;
    if( trackCount == nullptr )
        return movie::Status_ErrorBadValue;
    if( extractor == nullptr )
        return movie::Status_NullExtractor;
    movieStatus = extractor->GetTrackCount(&numberOfTracks);
    if( movieStatus == movie::Status_Success )
    {
        if( numberOfTracks <= 0 )
            movieStatus = movie::Status_TrackNotFound;
    }
    *trackCount = numberOfTracks;
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SelectTrack(movie::Extractor* extractor, int trackNumber)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    movieStatus = ValidateTrackNumber(extractor, trackNumber);
    if( movieStatus == movie::Status_Success )
    {
        if( extractor != NULL )
            movieStatus = extractor->SelectTrack(trackNumber);
        else
            movieStatus = movie::Status_NullExtractor;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::UnSelectTrack(movie::Extractor* extractor, int trackNumber)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    movieStatus = ValidateTrackNumber(extractor, trackNumber);
    if( movieStatus == movie::Status_Success )
    {
        if( extractor != NULL )
            movieStatus = extractor->UnselectTrack(trackNumber);
        else
            movieStatus = movie::Status_NullExtractor;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::GetTrackType(movie::Extractor* extractor, int trackNumber, TrackType *trackType) const
{
    movie::Status movieStatus = movie::Status_UnknownError;
    if( trackType == nullptr )
        return movie::Status_ErrorBadValue;
    *trackType = TrackType::TrackType_Unknown;
    int trackCount = 0;
    movieStatus = GetTrackCount(extractor, &trackCount);
    if( movieStatus == movie::Status_Success )
    {
        if( ( trackNumber < trackCount ) && ( trackNumber >= 0 ) )
        {
            movie::MediaData trackFormat;
            if( extractor != nullptr )
            {
                extractor->GetTrackConfiguration(trackNumber, &trackFormat);
                const char *mime;
                if( trackFormat.FindString("mime", &mime) )
                {
                    if( !strncasecmp("audio/", mime, 6) )
                        *trackType = TrackType::TrackType_Audio;
                    else if( !strncasecmp("video/", mime, 6) )
                        *trackType = TrackType::TrackType_Video;
                }
            }
            else
                movieStatus = movie::Status_NullExtractor;
            trackFormat.Clear();
        }
        else
            movieStatus = movie::Status_TrackNotFound;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::SetLooping(bool loop)
{
    movie::Status status = movie::Status_Success;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    m_Loop = loop;
    return status;
}

movie::Status MovieDecoderPlayer::GetLooping(bool *loop) const
{
    if( loop == nullptr )
        return movie::Status_ErrorBadValue;
    else
        *loop = m_Loop;
    return movie::Status_Success;
}

movie::Status MovieDecoderPlayer::GetState(State *state)
{
    std::lock_guard<nn::os::Mutex> lock { m_StateLock };
    if( state == nullptr )
        return movie::Status_ErrorBadValue;
    else
        *state = m_State;
    return movie::Status_Success;
}

movie::Status MovieDecoderPlayer::GetTrackInfo(movie::Extractor* extractor, int trackNumber, TrackInfo* trackInfo)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_ApiLock };
    if( trackInfo == nullptr )
        return movie::Status_ErrorBadValue;
    movieStatus = ValidateTrackNumber(extractor, trackNumber);
    if( movieStatus == movie::Status_Success )
    {
        movie::MediaData trackFormat;
        if( extractor != NULL )
        {
            movieStatus = extractor->GetTrackConfiguration(trackNumber, &trackFormat);
            if( movieStatus == movie::Status_Success )
            {
                const char *mime;
                if( trackFormat.FindString("mime", &mime) )
                {
                    int32_t avgBitRate = 0;
                    trackFormat.FindInt32("bit-rate", &avgBitRate);
                    if( !strncasecmp("video/", mime, 6) )
                    {
                        int32_t width = 0;
                        int32_t height = 0;
                        int32_t videoFrameRate = 0;
                        trackInfo->trackInfoVideo.decoderType = movie::DecoderType_Unknown;
                        trackInfo->trackInfoVideo.mime = std::string(mime);
                        trackFormat.FindInt32("frame-rate", &videoFrameRate);
                        trackFormat.FindInt32("width", &width);
                        trackFormat.FindInt32("height", &height);
                        trackInfo->trackInfoVideo.width = width;
                        trackInfo->trackInfoVideo.height = height;
                        trackInfo->trackInfoVideo.frameRate = videoFrameRate;
                        trackInfo->trackInfoVideo.bitRate = avgBitRate;
                        if( std::strstr(mime, "avc") )
                            trackInfo->trackInfoVideo.decoderType = movie::DecoderType_VideoAvc;
                        else if( std::strstr(mime, "x-vnd.on2.vp8") )
                            trackInfo->trackInfoVideo.decoderType = movie::DecoderType_VideoVp8;
                        else if( std::strstr(mime, "x-vnd.on2.vp9") )
                            trackInfo->trackInfoVideo.decoderType = movie::DecoderType_VideoVp9;
                    }
                    else if( !strncasecmp("audio/", mime, 6) )
                    {
                        trackInfo->trackInfoAudio.decoderType = movie::DecoderType_Unknown;
                        trackInfo->trackInfoAudio.mime = std::string(mime);
                        int32_t numChannels = 0;
                        int32_t sampleRate = 0;
                        trackFormat.FindInt32("channel-count", &numChannels);
                        trackFormat.FindInt32("sample-rate", &sampleRate);
                        trackInfo->trackInfoAudio.bitRate = avgBitRate;
                        trackInfo->trackInfoAudio.sampleRate = sampleRate;
                        trackInfo->trackInfoAudio.channels = numChannels;
                        if( std::strstr(mime, "mp4a") )
                            trackInfo->trackInfoAudio.decoderType = movie::DecoderType_AudioAac;
                        else if( std::strstr(mime, "vorbis") )
                            trackInfo->trackInfoAudio.decoderType = movie::DecoderType_AudioVorbis;
                    }
                }
            }
        }
        else
            movieStatus = movie::Status_NullExtractor;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::GetLastError()
{
    return m_LastError;
}

const char* MovieDecoderPlayer::StateToString(State state)
{
    switch( state )
    {
    case State::State_NotInitialized:
        return "State_NotInitialized";
    case State::State_Initialized:
        return "State_Initialized";
    case State::State_Preparing:
        return "State_Preparing";
    case State::State_Prepared:
        return "State_Prepared";
    case State::State_Started:
        return "State_Started";
    case State::State_Stopped:
        return "State_Stopped";
    case State::State_Paused:
        return "State_Paused";
    case State::State_PlaybackCompleted:
        return "State_PlaybackCompleted";
    case State::State_Error:
        return "State_Error";
    default:
        return "Unknown_State";
    }
}

void MovieDecoderPlayer::UpdateLastErrorAndSignalClient(movie::Status lastError)
{
    SetLastError(lastError);
    if( m_ErrorEvent != nullptr )
        nn::os::SignalEvent(m_ErrorEvent);
}

movie::Status MovieDecoderPlayer::GetContainerType(std::string mediaPath, movie::ContainerType *containertype)
{
    if( containertype == nullptr )
        return movie::Status_ErrorBadValue;
    movie::ContainerType container = movie::ContainerType_Unknown;
    std::size_t dotLocation = mediaPath.rfind(".");
    std::string mediaExtension = mediaPath.substr(dotLocation, mediaPath.size() - dotLocation);
    for( int i = 0; i<mediaExtension.length(); i++ )
        mediaExtension[ i ] = tolower(mediaExtension[ i ]);
    if( mediaExtension.compare(std::string(".mp4")) == 0 )
        container = movie::ContainerType_Mpeg4;
    else if( mediaExtension.compare(std::string(".mkv")) == 0 )
        container = movie::ContainerType_Matroska;
    else if( mediaExtension.compare(std::string(".webm")) == 0 )
        container = movie::ContainerType_WebM;
    else
        container = movie::ContainerType_Unknown;
    *containertype = container;
    return movie::Status_Success;
}

movie::Status MovieDecoderPlayer::ValidateStateTransition(State newState)
{
    if( newState == State_NotInitialized )
        return movie::Status_Success;
    movie::Status movieStatus = movie::Status_InvalidStateTransition;
    State currentState = State_NotInitialized;
    movieStatus = GetState(&currentState);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    movieStatus = movie::Status_InvalidStateTransition;
    switch( currentState )
    {
        case State_NotInitialized:
        {
            if( newState == State_Initialized )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_Initialized:
        {
            if( newState == State_Preparing )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_Preparing:
        {
            if( ( newState == State_Prepared ) || ( newState == State_Initialized ) )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_Prepared:
        {
            if( newState == State_Started )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_Started:
        {
            if( ( newState == State_Stopped ) || ( newState == State_Paused ) )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_Stopped:
        {
            if( newState == State_Preparing )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_Paused:
        {
            if( ( newState == State_Stopped ) || ( newState == State_Started ) ||
                ( newState == State_Paused ) )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_PlaybackCompleted:
        {
            if( ( newState == State_Stopped ) || ( newState == State_Started ) )
                movieStatus = movie::Status_Success;
            break;
        }
        case State_Error:
        {
            if( newState == State_NotInitialized )
                movieStatus = movie::Status_Success;
            break;
        }
        default:
            break;
    }
    return movieStatus;
}//NOLINT(impl/function_size)

movie::Status MovieDecoderPlayer::SetState(State state)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    std::lock_guard<nn::os::Mutex> lock { m_StateLock };
    if( ( state >= State::State_NotInitialized ) && ( state <= State::State_Error ) )
    {
        m_State = state;
        if( m_StateChangedEvent != nullptr )
            nn::os::SignalEvent(m_StateChangedEvent);
        movieStatus = movie::Status_Success;
    }
    return movieStatus;
}

movie::Status MovieDecoderPlayer::RegisterPlayerEvent(PlayerEventType eventType, nn::os::EventType *event, nn::os::MultiWaitHolderType *holder, movie::Decoder* decoder)
{
    nn::os::InitializeEvent(event, false, nn::os::EventClearMode_ManualClear);
    EventUserData *userData = new EventUserData(event, eventType, decoder);
    if( userData == nullptr )
        return movie::Status_OutOfMemory;
    nn::os::InitializeMultiWaitHolder(holder, event);
    nn::os::SetMultiWaitHolderUserData(holder, ( uintptr_t ) userData);
    nn::os::LinkMultiWaitHolder(&m_PlayerMultiWait, holder);
    return  movie::Status_Success;
}

void MovieDecoderPlayer::DeleteUserData(EventUserData* userData)
{
    if( userData != nullptr )
        delete userData;
}

void MoviePlayerThreadFunction(void *arg)
{
    MovieDecoderPlayer *moviePlayer = ( MovieDecoderPlayer* ) arg;
    bool exitThread = false;
    for( ;;)
    {
        nn::os::MultiWaitHolderType* holder = nn::os::WaitAny(&moviePlayer->m_PlayerMultiWait);
        uintptr_t userData = GetMultiWaitHolderUserData(holder);
        MovieDecoderPlayer::EventUserData *eventUserData = ( MovieDecoderPlayer::EventUserData* ) userData;
        MovieDecoderPlayer::PlayerEventType eventType = eventUserData->m_EventType;
        nn::os::EventType *event = eventUserData->m_Event;
        nn::os::ClearEvent(event);
        switch( eventType )
        {
            case MovieDecoderPlayer::PlayerEventType_VideoRenderComplete:
            {
                bool loopBack = false;
                movie::Decoder* decoder = eventUserData->m_Decoder;
                moviePlayer->GetLooping(&loopBack);
                if( loopBack == true )
                    moviePlayer->PrepareForLoopBack(decoder);
                else
                {
                    moviePlayer->SetVideoPlaybackComplete(decoder);
                    if( moviePlayer->IsMediaPlaybackComplete() == true )
                        moviePlayer->SetState(MovieDecoderPlayer::State_PlaybackCompleted);
                }
                break;
            }
            case MovieDecoderPlayer::PlayerEventType_ThreadExit:
                exitThread = true;
                break;
            default:
                break;
        }
        if( exitThread == true )
            break;
    }
}

void MovieDecoderPlayer::SetVideoPlaybackComplete(movie::Decoder* decoder)
{
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.decoder == decoder )
            decoderInfo.videoPlaybackComplete = true;
    }
}

void MovieDecoderPlayer::ReSetVideoPlaybackComplete(movie::Decoder* decoder)
{
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.decoder == decoder )
            decoderInfo.videoPlaybackComplete = false;
    }
}

bool MovieDecoderPlayer::IsMediaPlaybackComplete()
{
    bool videoPlaybackComplete = true;
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.videoPlaybackComplete == false )
            videoPlaybackComplete = false;
    }
    return videoPlaybackComplete;
}

movie::Status MovieDecoderPlayer::ValidateTrackNumber(movie::Extractor* extractor, int trackNumber)
{
    movie::Status movieStatus = movie::Status_TrackNotFound;
    int trackCount = 0;
    movieStatus = GetTrackCount(extractor, &trackCount);
    if( movieStatus == movie::Status_Success )
    {
        if( ( trackNumber >= trackCount ) || ( trackNumber < 0 ) )
            movieStatus = movie::Status_TrackNotFound;
    }
    return movieStatus;
}

void MovieDecoderPlayer::SetLastError(movie::Status movieStatus)
{
    m_LastError = movieStatus;
}

movie::Status MovieDecoderPlayer::CreateExtractorForPlayer(std::string mediaPath, movie::Extractor** extractor)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    movie::ContainerType containertype = movie::ContainerType_Unknown;
    movieStatus = GetContainerType(mediaPath, &containertype);
    if( movieStatus != movie::Status_Success )
        return movieStatus;
    if( containertype == movie::ContainerType_Unknown )
        return movie::Status_UnsupportedMediaType;
    *extractor = new movie::Extractor(containertype);
    if( *extractor == nullptr )
        return movie::Status_OutOfMemory;
    return movieStatus;
}

movie::Status MovieDecoderPlayer::PrepareForLoopBack(movie::Decoder* decoder)
{
    movie::Status movieStatus = movie::Status_UnknownError;

    if( m_MovieVideoOutputHandler != nullptr )
        m_MovieVideoOutputHandler->Stop(decoder);

    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.decoder == decoder )
        {
            decoderInfo.decoder->Flush();
            decoderInfo.extractor->SeekTo(0);
            decoderInfo.mediaClock->ClearAnchorTime();
        }
    }
    if( m_MovieVideoInputHandler != nullptr )
        m_MovieVideoInputHandler->Flush(decoder);
    if( m_MovieVideoOutputHandler != nullptr )
        m_MovieVideoOutputHandler->Start();
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.decoder == decoder )
            decoderInfo.decoder->Start();
    }
    return movieStatus;
}

bool MovieDecoderPlayer::IsInRequestedState(State state)
{
    State currentState;
    movie::Status movieStatus = GetState(&currentState);
    if( movieStatus != movie::Status_Success )
        return false;
    if( currentState == state )
        return true;
    else
        return false;
}

void MovieDecoderPlayer::DeletePlayerOwnedObjects()
{
    if( m_MovieVideoInputHandler != nullptr )
    {
        delete m_MovieVideoInputHandler;
        m_MovieVideoInputHandler = nullptr;
    }
    for( auto &decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.decoder != nullptr )
        {
            delete decoderInfo.decoder;
            decoderInfo.decoder = nullptr;
        }
        if( decoderInfo.extractor != nullptr )
        {
            delete decoderInfo.extractor;
            decoderInfo.extractor = nullptr;
        }
        if( decoderInfo.mediaClock != nullptr )
        {
            delete decoderInfo.mediaClock;
            decoderInfo.mediaClock = nullptr;
        }
    }
}

void MovieDecoderPlayer::FlushInputAndOutputHandlers()
{
    if( m_MovieVideoOutputHandler != nullptr )
        m_MovieVideoOutputHandler->Flush();
    if( m_MovieVideoInputHandler != nullptr )
        m_MovieVideoInputHandler->SignalFlush();
}

void MovieDecoderPlayer::FlushDecoders()
{
    for( auto&& decoderInfo : m_DecoderInfoList )
    {
        if( decoderInfo.decoder != nullptr )
            decoderInfo.decoder->Flush();
    }
}

movie::Status MovieDecoderPlayer::GetDecoderType(movie::Extractor* extractor, int trackNumber, movie::DecoderType *decoderType)
{
    movie::Status movieStatus = movie::Status_UnknownError;
    *decoderType = movie::DecoderType_Unknown;
    movieStatus = ValidateTrackNumber(extractor, trackNumber);
    if( movieStatus == movie::Status_Success )
    {
        movie::MediaData trackFormat;
        if( extractor != NULL )
        {
            movieStatus = extractor->GetTrackConfiguration(trackNumber, &trackFormat);
            if( movieStatus == movie::Status_Success )
            {
                const char *mime;
                if( trackFormat.FindString("mime", &mime) )
                {
                    int32_t avgBitRate = 0;
                    trackFormat.FindInt32("bit-rate", &avgBitRate);
                    if( !strncasecmp("video/", mime, 6) )
                    {
                        if( std::strstr(mime, "avc") )
                            *decoderType = movie::DecoderType_VideoAvc;
                        else if( std::strstr(mime, "x-vnd.on2.vp8") )
                            *decoderType = movie::DecoderType_VideoVp8;
                        else if( std::strstr(mime, "x-vnd.on2.vp9") )
                            *decoderType = movie::DecoderType_VideoVp9;
                    }
                    else if( !strncasecmp("audio/", mime, 6) )
                    {
                        if( std::strstr(mime, "mp4a") )
                            *decoderType = movie::DecoderType_AudioAac;
                        else if( std::strstr(mime, "vorbis") )
                            *decoderType = movie::DecoderType_AudioVorbis;
                    }
                }
            }
        }
    }
    return movieStatus;
}
