﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#include "MemoryHandler.h"

const int g_FsHeapSize = 512 * 1024;
const int g_MmHeapSize = 640 << 20;

uint8_t                     g_FsHeapBuffer[ g_FsHeapSize ];
nn::lmem::HeapHandle        g_FsHeap;
char                        g_MmHeapBuffer[ g_MmHeapSize ];

const size_t  g_GraphicsSystemMemorySize = 8 * 1024 * 1024;
uint8_t g_GraphicsHeap[ g_GraphicsSystemMemorySize ] __attribute__(( aligned(4096) ));

void GetGraphicsHeap(uint8_t **graphicsHeap, size_t *graphicsHeapSize)
{
    *graphicsHeap = &g_GraphicsHeap[0];
    *graphicsHeapSize = g_GraphicsSystemMemorySize;
}

void FsInitHeap()
{
    g_FsHeap = nn::lmem::CreateExpHeap(g_FsHeapBuffer, g_FsHeapSize, nn::lmem::CreationOption_DebugFill);
}

void* FsAllocate(size_t size)
{
    return nn::lmem::AllocateFromExpHeap(g_FsHeap, size);
}

void FsDeallocate(void* address, size_t size)
{
    NN_UNUSED(size);
    return nn::lmem::FreeToExpHeap(g_FsHeap, address);
}

MemoryTracker &MovieMemoryTracker()
{
    static MemoryTracker g_MovieMemoryTracker;
    return g_MovieMemoryTracker;
}

MemoryTracker &CoreMemoryTracker()
{
    static MemoryTracker g_CoreMemoryTracker;
    return g_CoreMemoryTracker;
}

MemoryTracker &MallocMemoryTracker()
{
    static MemoryTracker g_MallocMemoryTracker;
    return g_MallocMemoryTracker;
}

MemoryTracker &NewMemoryTracker()
{
    static MemoryTracker g_NewMemoryTracker;
    return g_NewMemoryTracker;
}

nn::mem::StandardAllocator &GlobalAllocator()
{
    static nn::mem::StandardAllocator g_MultimediaAllocator(g_MmHeapBuffer, sizeof(g_MmHeapBuffer));
    return g_MultimediaAllocator;
}

void* MovieAllocate(size_t size, size_t alignment, void *userPtr)
{
    void *address = GlobalAllocator().Allocate(size, alignment);
    static size_t totalAllocatedSize = 0;
    totalAllocatedSize += size;
    const char* threadName = nullptr;
    if( g_TrackMemoryAllocation )
    {
        threadName = ::nn::os::GetThreadNamePointer(::nn::os::GetCurrentThread());
        static_cast<MemoryTracker*>( userPtr )->Track(address, size, threadName);
    }
    return address;
}

void MovieDeallocate(void *address, void *userPtr)
{
    GlobalAllocator().Free(address);
    if( g_TrackMemoryAllocation )
        static_cast<MemoryTracker*>( userPtr )->Untrack(address);
}

void *MovieReallocate(void* address, size_t newSize, void *userPtr)
{
    if( g_TrackMemoryAllocation )
    {
        if( address != nullptr )
            static_cast< MemoryTracker* >( userPtr )->Untrack(address);
    }
    void *memory = GlobalAllocator().Reallocate(address, newSize);
    const char* threadName = nullptr;
    if( g_TrackMemoryAllocation )
    {
        threadName = ::nn::os::GetThreadNamePointer(::nn::os::GetCurrentThread());
        static_cast<MemoryTracker*>( userPtr )->Track(memory, newSize, threadName);
    }
    return memory;
}

extern "C" void* malloc(size_t size)
{
    void *address = MovieAllocate(size, 8, &MallocMemoryTracker());
    return address;
}

extern "C" void free(void* address)
{
    if( address != nullptr )
        MovieDeallocate(address, &MallocMemoryTracker());
}

extern "C" void* calloc(size_t num, size_t size)
{
    const size_t sum = num * size;
    void* address = malloc(sum);
    if( address != nullptr )
        ::std::memset(address, 0, sum);
    return address;
}

extern "C" void* realloc(void* address, size_t newSize)
{
    return MovieReallocate(address, newSize, &MallocMemoryTracker());
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    return MovieAllocate(size, alignment, &MallocMemoryTracker());
}

void* operator new( ::std::size_t size, const ::std::nothrow_t& ) throw( )
{
    void *address =  MovieAllocate(static_cast<int>( size ), 8, &NewMemoryTracker());
    return address;
}

void* operator new[](::std::size_t size, const ::std::nothrow_t&) throw( )
{
    return operator new( size, ::std::nothrow_t() );
}

void* operator new( ::std::size_t size ) throw( ::std::bad_alloc )
{
    return operator new( size, ::std::nothrow_t() );
}

void* operator new[](::std::size_t size) throw( ::std::bad_alloc )
{
    return operator new( size, ::std::nothrow_t() );
}

void operator delete( void* address ) throw( )
{
    if( address != nullptr )
    {
        MovieDeallocate(address, &NewMemoryTracker());
    }
}

void operator delete[](void* address) throw( )
{
    operator delete( address );
}
