﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
/**
 * @examplesource{MiiG3dSimple.cpp,PageSampleMiiG3dSimple}
 *
 * @brief
 * nn::g3d::ResModel への Mii モデルアサイン機能のサンプルプログラム
 */

/**
 * @page PageSampleMiiG3dSimple MiiG3dSimple
 * @tableofcontents
 *
 * @brief
 * nn::g3d::ResModel への Mii モデルアサイン機能のサンプルプログラムの解説です。
 *
 * @section PageSampleMiiG3dSimple_SectionBrief 概要
 * nn::g3d::mii を使用して、Mii モデルを nn::g3d::ResModel へアサインし、表示するサンプルプログラムの説明をします。
 * 本サンプルはボーンの計算が最も軽くなるようにモデルデータを最適化しています。大量の Mii を描画するのに適していますが
 * Mii の描画タイプ（通常モデル、被り物モデル）の切り替えはできません。
 *
 * @section PageSampleMiiG3dSimple_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/MiiG3dSimple
 * Samples/Sources/Applications/MiiG3dSimple @endlink 以下にあります。
 *
 * @section PageSampleMiiG3dSimple_SectionNecessaryEnvironment 必要な環境
 * 特にありません。
 *
 * @section PageSampleMiiG3dSimple_SectionHowToOperate 操作方法
 * 特にありません。
 *
 * @section PageSampleMiiG3dSimple_SectionPrecaution 注意事項
 * 動作には Mii ライブラリが必要です。
 *
 * @section PageSampleMiiG3dSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleMiiG3dSimple_SectionDetail 解説
 * このサンプルプログラムは、nn::g3d::ResFile へ Mii モデルをアサインし、アニメーションを適用し、
 * Mii モデルを表示します。サンプルプログラムの全体像を説明します。
 *
 * @subsection PageSampleMiiG3dSimple_SectionDetail_ModelConvert モデルコンバート処理
 * 下記のリソースをコンバートします。
 * - Samples/Sources/Applications/MiiG3dSimple/Resources/Model/Mii.fmdb
 *
 *   頂点やインデクスを含まないデータ構造のみが記述された空のモデルデータです。
 *   ランタイムで Mii モデルをアサインします。
 *
 * - Samples/Sources/Applications/MiiG3dSimple/Resources/Model/MiiSimpleFaceExpressionChange.fmab
 *
 *   Mii の表情を切り替えるためのマテリアルアニメーションです。
 *   ランタイムで Mii モデルがアサインされた nn::g3d::ResModel に適用します。
 *
 * - Samples/Sources/Applications/MiiG3dSimple/Resources/Shader/SampleShader.glsl
 *
 *   Mii を表示するためのシェーダです。
 *
 * @subsection PageSampleMiiG3dSimple_SectionDetail_RuntimeFlow サンプルプログラム実行時の処理
 * サンプルプログラム実行時の処理の流れを説明します。
 *
 * - Mii モデルの構築
 * - Mii モデルの nn::g3d::ResModel へのアサイン
 * - nn::g3d モデルインスタンスの作成
 * - nn::g3d マテリアルアニメーションインスタンスの作成
 * - レンダーループ開始
 * - アニメーション更新
 * - モデル更新
 * - コマンドリスト実行
 * - スキャンバッファをスワップ
 * - レンダーループ開始に戻る
 *
 */

#include <cstdlib>
#include <cstring>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/os.h>
#include <nn/fs.h>
#include <nn/init.h>
#include <nn/vi.h>
#include <nn/gfx.h>
#include <nn/mii.h>
#include <nn/util/util_Matrix.h>
#include <nn/g3d/g3d_ResFile.h>
#include <nn/g3d/g3d_ResShader.h>
#include <nn/g3d/g3d_ModelObj.h>
#include <nn/g3d/g3d_MaterialAnimObj.h>
#include <nn/g3d/mii/g3d_MiiConverter.h>
#include <nns/gfx/gfx_GraphicsFramework.h>

namespace {

nns::gfx::GraphicsFramework g_GfxFramework;

const size_t MemoryHeapSize = 16 * 1024 * 1024;

/// CharInfoのパス
const char* CharInfoPath = "Contents:/Mii/charinfo/SampleMiiG3d.dat";
/// TextureShaderのパス
const char* TextureShaderPath = "Contents:/Mii/shader/TextureShader.bnsh";
#if NN_BUILD_CONFIG_SPEC_NX
#if NN_BUILD_CONFIG_OS_HORIZON
/// NX用テクスチャリソースのパス
const char* TextureResourcePath = "Contents:/Mii/resource/NXTextureMidSRGB.dat";
#else
/// WinNvn用テクスチャリソースのパス
const char* TextureResourcePath = "Contents:/Mii/resource/WinNvnTextureMidSRGB.dat";
#endif    // #if NN_BUILD_CONFIG_SPEC_NX
# else
/// OpenGl用リソースのパス
const char* TextureResourcePath = "Contents:/Mii/resource/WinGenericTextureMidSRGB.dat";
#endif
/// シェイプリソースのパス
const char* ShapeResourcePath = "Contents:/Mii/resource/ShapeMid.dat";
/// Mii描画用簡易シェーダー
const char* SampleShaderPath = "Contents:/SampleShader.bnsh";

/// マスクテクスチャとFacelineテクスチャの解像度
const size_t TextureResolution = 512;

/// マスクテクスチャとFacelineテクスチャのミップマップ数
const int TextureMipCount = 1;

/// マスクテクスチャとFacelineテクスチャのフォーマット
const nn::gfx::ImageFormat TextureFormat = nn::gfx::ImageFormat_R8_G8_B8_A8_Unorm;

/// 生成する表情のフラグ
const int ExpressionFlags = nn::mii::ExpressionFlag_Normal | nn::mii::ExpressionFlag_Anger;

/// 生成するModelの設定
const int CreateFlag = nn::mii::CreateFlag_Normal | nn::mii::CreateFlag_NoseNormal;

/// SampleShaderの行列コンスタントバッファ
struct ConstantBufferMatrix
{
    nn::util::FloatColumnMajor4x3 modelView;
    nn::util::FloatColumnMajor4x4 projection;
};

/// SampleShaderのModulateコンスタントバッファ
struct ConstantBufferModulate
{
    int32_t modulateType; ///< @ref ModulateType
    int32_t pad[3];
    nn::util::Float4 constantColor[3]; ///< z値は利用されていない。
};

nn::g3d::ModelObj* g_pModelObj;
bool g_IsVerticalFlip = false;

//-----------------------------------------------------------------------------
// メモリ
nn::util::BytePtr g_pMemoryHeap( NULL );
nn::util::BytePtr g_pMemory( NULL );

char* g_MountRomCacheBuffer = NULL;

// ビューポートシザーを初期化
nn::gfx::ViewportScissorState g_ViewportScissor;
void InitializeViewport()
{
    auto pDevice = g_GfxFramework.GetDevice();

    nn::gfx::ViewportScissorState::InfoType info;
    info.SetDefault();
    info.SetScissorEnabled( true );
    nn::gfx::ViewportStateInfo viewportInfo;
    {
        viewportInfo.SetDefault();
        viewportInfo.SetWidth( static_cast<float>( g_GfxFramework.GetDisplayWidth() ) );
        viewportInfo.SetHeight( static_cast<float>( g_GfxFramework.GetDisplayHeight() ) );
    }
    nn::gfx::ScissorStateInfo scissorInfo;
    {
        scissorInfo.SetDefault();
        scissorInfo.SetWidth( g_GfxFramework.GetDisplayWidth() );
        scissorInfo.SetHeight( g_GfxFramework.GetDisplayHeight() );
    }
    info.SetViewportStateInfoArray( &viewportInfo, 1 );
    info.SetScissorStateInfoArray( &scissorInfo, 1 );
    g_ViewportScissor.Initialize( pDevice, info );
}

void InitializeResources()
{
    // メモリ
    g_pMemoryHeap.Reset( malloc( MemoryHeapSize ) );
    g_pMemory = g_pMemoryHeap;

    InitializeViewport();
}

void FinalizeResources()
{
    g_ViewportScissor.Finalize( g_GfxFramework.GetDevice() );
    g_GfxFramework.Finalize();
    free( g_pMemoryHeap.Get() );
}

void* Allocate( size_t size )
{
    return malloc( size );
}

void Deallocate( void* ptr, size_t )
{
    free( ptr );
}

//------------------------------------------------------------------------------
// グラフィックスシステム用メモリ割り当て・破棄関数
//------------------------------------------------------------------------------
void InitializeFs()
{
    nn::Result result;

    nn::fs::SetAllocator( Allocate, Deallocate );

    size_t cacheSize = 0;
    result = nn::fs::QueryMountRomCacheSize(&cacheSize);
    NN_ASSERT( result.IsSuccess() );

    g_MountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ASSERT_NOT_NULL(g_MountRomCacheBuffer);

    result = nn::fs::MountRom("Contents", g_MountRomCacheBuffer, cacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS( result );
}

void FinalizeFs()
{
    nn::fs::Unmount("Contents");
    delete[] g_MountRomCacheBuffer;
    g_MountRomCacheBuffer = NULL;
}

nn::gfx::Sampler g_Sampler;
void InitializeSampler()
{
    nn::gfx::Sampler::InfoType info;

    /// Miiで利用できるようにサンプラを設定する
    nn::mii::GetSamplerInfo(&info);

    /// サンプラー初期化
    g_Sampler.Initialize( g_GfxFramework.GetDevice(), info );
}

nn::mii::CharInfo g_CharInfo;
void LoadCharInfo()
{
    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle,CharInfoPath,nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    result = nn::fs::ReadFile(handle, 0, &g_CharInfo, sizeof(g_CharInfo));
    NN_ASSERT(result.IsSuccess());

    nn::fs::CloseFile(handle);

    /// 中身をチェックする
    NN_ASSERT(nn::mii::CharInfoAccessor(g_CharInfo).IsValid());
}
nn::gfx::ShaderCodeType GetShaderCodeType(const nn::gfx::ResShaderContainer* pContainer)
{
    for( int idxShaderCodeType = 0; idxShaderCodeType < nn::gfx::ShaderCodeType_End; ++idxShaderCodeType )
    {
        auto codeType = nn::gfx::ShaderCodeType( idxShaderCodeType );
        if( pContainer->GetResShaderVariation(0)->GetResShaderProgram( codeType ) != nullptr )
        {
            return codeType;
        }
    }
    NN_ASSERT( 0 );
    return nn::gfx::ShaderCodeType_Source;
}

nn::gfx::ResShaderFile* InitializeResShaderFile(nn::gfx::ShaderCodeType* pCodeType,const char* pPath)
{
    void* pFile;
    {
        /// シェーダーファイルを読み込む
        nn::fs::FileHandle handle;
        nn::Result result = nn::fs::OpenFile(&handle,pPath,nn::fs::OpenMode_Read);
        NN_ASSERT(result.IsSuccess());

        /// ファイルサイズ取得
        int64_t size;
        result = nn::fs::GetFileSize(&size, handle);
        NN_ASSERT(result.IsSuccess());

        /// アライメントを取得する
        size_t alignment;
        nn::util::BinaryFileHeader header;
        result = nn::fs::ReadFile(handle, 0, &header,sizeof(header));
        NN_ASSERT(result.IsSuccess());
        alignment = header.GetAlignment();

        /// リソース用メモリ確保
        g_pMemory.AlignUp(alignment);
        pFile = g_pMemory.Get();
        g_pMemory.Advance(ptrdiff_t(size));

        /// リソースを読み込む
        result = nn::fs::ReadFile(handle, 0, pFile,size_t(size));
        NN_ASSERT(result.IsSuccess());
        NN_UNUSED( result );
        nn::fs::CloseFile(handle);
    }

    /// シェーダーの初期化
    nn::gfx::ResShaderFile* pShaderFile = nn::gfx::ResShaderFile::ResCast(pFile);
    nn::gfx::ResShaderContainer* pContainer = pShaderFile->GetShaderContainer();
    *pCodeType = GetShaderCodeType(pContainer);
    pContainer->Initialize( g_GfxFramework.GetDevice() );
    for( int idxShaderVariation = 0; idxShaderVariation < pContainer->GetShaderVariationCount(); ++idxShaderVariation )
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxShaderVariation );
        nn::gfx::ShaderInitializeResult result = pVariation->GetResShaderProgram(*pCodeType)->Initialize( g_GfxFramework.GetDevice() );
        NN_ASSERT( result == nn::gfx::ShaderInitializeResult_Success );
        NN_UNUSED( result );
    }

    return pShaderFile;
}


nn::gfx::ResShaderFile* g_pTextureShaderFile;
nn::gfx::ShaderCodeType g_TextureShaderCodeType;
nn::mii::TextureShader g_TextureShader;
nn::mii::TextureShaderInfo g_TextureShaderInfo;
void InitializeTextureShader()
{
    auto pDevice = g_GfxFramework.GetDevice();
    g_pTextureShaderFile = InitializeResShaderFile(&g_TextureShaderCodeType, TextureShaderPath);

    /// フォーマット設定
    g_TextureShaderInfo.SetFacelineImageFormat(TextureFormat);
    g_TextureShaderInfo.SetMaskImageFormat(TextureFormat);

    /// アライメントとサイズを取得
    size_t alignment = nn::mii::TextureShader::CalculateMemoryAlignment(*g_pTextureShaderFile, g_TextureShaderCodeType, g_TextureShaderInfo, pDevice);
    size_t size = nn::mii::TextureShader::CalculateMemorySize(*g_pTextureShaderFile, g_TextureShaderCodeType, g_TextureShaderInfo, pDevice);

    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    nn::Result result = g_TextureShader.Initialize( pMemory,size, pDevice,*g_pTextureShaderFile, g_TextureShaderCodeType, g_TextureShaderInfo );
    NN_ASSERT(result.IsSuccess());
}

void* ReadFile(size_t* pSize, const char* filePath, size_t alignement)
{
    int64_t size;

    nn::fs::FileHandle handle;
    nn::Result result = nn::fs::OpenFile(&handle, filePath, nn::fs::OpenMode_Read);
    NN_ASSERT(result.IsSuccess());

    /// ファイルサイズ取得
    result = nn::fs::GetFileSize(&size, handle);
    NN_ASSERT(result.IsSuccess());

    /// メモリ確保
    g_pMemory.AlignUp(alignement);
    void* ptr = g_pMemory.Get();
    g_pMemory.Advance(ptrdiff_t(size));

    /// ファイルを読み込む
    result = nn::fs::ReadFile(handle, 0, ptr, size_t(size));
    NN_ASSERT(result.IsSuccess());

    /// 閉じる
    nn::fs::CloseFile(handle);

    *pSize = size_t(size);
    return ptr;
}

nn::mii::Resource g_Resource;
void* g_pShapeResourceFile;
void* g_pTextureResourceFile;
void InitializeMiiResource()
{
    /// テクスチャリソースファイルの読み込み
    size_t textureFileSize;
    g_pTextureResourceFile = ReadFile( &textureFileSize, TextureResourcePath, nn::mii::ResourceMemoryAlignment );
    /// シェイプリソースファイルの読み込み
    size_t shapeFileSize;
    g_pShapeResourceFile = ReadFile( &shapeFileSize, ShapeResourcePath, nn::mii::ResourceMemoryAlignment );

    nn::mii::ResourceInfo info;
    info.SetDefault();
    info.SetGammaType(nn::mii::GammaType_Srgb);

    /// サイズとアライメント取得
    size_t alignment =
        nn::mii::Resource::CalculateMemoryAlignment( g_pTextureResourceFile, textureFileSize,
                                                    g_pShapeResourceFile, shapeFileSize, info );

    size_t size =
        nn::mii::Resource::CalculateMemorySize( g_pTextureResourceFile, textureFileSize,
                                                g_pShapeResourceFile, shapeFileSize, info );

    /// メモリ確保
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// リソースを初期化する
    nn::Result result = g_Resource.Initialize( pMemory, size, g_pTextureResourceFile, textureFileSize,
                                                g_pShapeResourceFile, shapeFileSize, info, g_GfxFramework.GetDevice() );
    NN_ASSERT(result.IsSuccess());
}

/// 指定フラグのindex番目の表情を取得する
nn::mii::Expression GetExpression(int flag,int index)
{
    int count = 0;
    for( int idxExpression = 0; idxExpression < nn::mii::Expression_End; ++idxExpression )
    {
        if( ( flag & ( 1 << idxExpression ) ) != 0 )
        {
            if( count == index )
            {
                return nn::mii::Expression( idxExpression );
            }
            ++count;
        }
    }
    /// ここには到達しない。
    return nn::mii::Expression_Normal;
}

nn::mii::CharModel g_CharModel;
void InitializeCharModel()
{
    auto pDevice = g_GfxFramework.GetDevice();
    nn::mii::CharModelInfo info;
    info.SetDefault();
    info.SetDynamicTextureResolution( TextureResolution,TextureResolution );
    info.SetCreateFlag( CreateFlag );
    info.SetDynamicTextureFormat( TextureFormat,TextureFormat );
    info.SetMaskCount( nn::mii::GetExpressionCount(ExpressionFlags ) );
    info.SetDynamicTextureMipCount( TextureMipCount, TextureMipCount );

    /// メモリサイズとアライメント取得
    size_t alignment = nn::mii::CharModel::CalculateMemoryAlignment( info );
    size_t size = nn::mii::CharModel::CalculateMemorySize( info );

    /// 一時メモリのサイズ取得
    size_t tempSize = nn::mii::CharModel::CalculateMemorySize( info );

    /// メモリと一時メモリを確保する
    g_pMemory.AlignUp( alignment );
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance( size );

    /// 一時メモリはCharModel::Initialize()後破棄できます。
    nn::mii::TemporaryBuffer* pTempMemory = g_pMemory.Get<nn::mii::TemporaryBuffer>();
    g_pMemory.Advance(tempSize);

    /// メモリプールとサイズを取得
    size_t dynamicMemoeryPoolSize = nn::mii::CharModel::CalculateDynamicMemoryPoolSize( pDevice, info );
    size_t dynamicMemoryPoolAlignment = nn::mii::CharModel::CalculateDynamicMemoryPoolAlignment( pDevice, info );
    size_t staticMemoeryPoolSize = nn::mii::CharModel::CalculateStaticMemoryPoolSize( g_Resource );
    size_t staticMemoryPoolAlignment = nn::mii::CharModel::CalculateStaticMemoryPoolAlignment( g_Resource );
    ptrdiff_t dynamicMemoryPoolOffset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget,
                                                                            dynamicMemoeryPoolSize, dynamicMemoryPoolAlignment );
    ptrdiff_t staticMemoryPoolOffset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_Data,
                                                                            staticMemoeryPoolSize, staticMemoryPoolAlignment );

    /// CharModelを初期化する
    nn::Result result = g_CharModel.Initialize( pMemory, size
                                                , pDevice
                                                , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget )
                                                , dynamicMemoryPoolOffset, dynamicMemoeryPoolSize
                                                , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_Data )
                                                , staticMemoryPoolOffset, staticMemoeryPoolSize
                                                , g_Resource
                                                , pTempMemory
                                                , info, g_CharInfo );
    NN_ASSERT(result.IsSuccess());
}

nn::mii::Faceline g_Faceline;
void InitializeFaceline()
{
    /// メモリサイズとアライメント取得
    size_t alignment = nn::mii::Faceline::CalculateMemoryAlignment();
    size_t size = nn::mii::Faceline::CalculateMemorySize();

    /// 一時メモリのサイズ取得
    size_t tempSize = nn::mii::TemporaryBufferSize;

    /// メモリと一時メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// 一時メモリはFaceline::Initialize()後破棄できます。
    nn::mii::TemporaryBuffer* pTempMemory = g_pMemory.Get<nn::mii::TemporaryBuffer>();
    g_pMemory.Advance(tempSize);


    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::Faceline::CalculateMemoryPoolAlignment( pDevice, g_Resource );
    size_t memoryPoolSize = nn::mii::Faceline::CalculateMemoryPoolSize( pDevice, g_Resource );

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, memoryPoolAlignment );

    g_Faceline.Initialize( pMemory, size, pDevice
                        , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_Data )
                        , offset, memoryPoolSize, pTempMemory
                        , g_Resource, g_CharInfo, g_IsVerticalFlip );
}

nn::mii::Mask g_Mask;
void InitializeMask()
{
    /// メモリサイズとアライメント取得
    size_t alignment = nn::mii::Mask::CalculateMemoryAlignment();
    size_t size = nn::mii::Mask::CalculateMemorySize();

    /// 一時メモリのサイズ取得
    size_t tempSize = nn::mii::TemporaryBufferSize;

    /// メモリと一時メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// 一時メモリはMask::Initialize()後破棄できます。
    nn::mii::TemporaryBuffer* pTempMemory = g_pMemory.Get<nn::mii::TemporaryBuffer>();
    g_pMemory.Advance(tempSize);

    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::Mask::CalculateMemoryPoolAlignment( pDevice, g_Resource, ExpressionFlags);
    size_t memoryPoolSize = nn::mii::Mask::CalculateMemoryPoolSize( pDevice, g_Resource, ExpressionFlags);

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_Data, memoryPoolSize, memoryPoolAlignment );

    g_Mask.Initialize( pMemory, size, pDevice,
                        g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_Data ), offset,
                        memoryPoolSize,
                        pTempMemory, g_Resource, g_CharInfo, ExpressionFlags, g_IsVerticalFlip );
}

nn::gfx::DescriptorSlot g_SamplerSlot;
void BindDescriptorSlot()
{
    /// サンプラのバインド
    {
        auto pSamplerDescriptorPool = g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler );
        pSamplerDescriptorPool->BeginUpdate();
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler, 1 );
        pSamplerDescriptorPool->SetSampler( slotIndex, &g_Sampler);
        pSamplerDescriptorPool->GetDescriptorSlot( &g_SamplerSlot, slotIndex );
        pSamplerDescriptorPool->EndUpdate();
    }

    /// テクスチャービューのバインド
    auto pTextureDescriptorPool = g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView );
    pTextureDescriptorPool->BeginUpdate();

    /// CharModelのMask以外のテクスチャをバインドする
    for( int idxTextureType = 0; idxTextureType < nn::mii::CharModel::TextureType_End; ++idxTextureType )
    {
        int mask = 0;
        nn::mii::CharModel::TextureType type = nn::mii::CharModel::TextureType( idxTextureType );
        const nn::gfx::TextureView* pView = g_CharModel.GetTextureView( type, mask );

        if( pView == nullptr )
        {
            continue;
        }

        if( type == nn::mii::CharModel::TextureType_Mask )
        {
            continue;
        }

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_CharModel.SetTextureDescriptorSlot( type, slot, mask );
    }

    /// CharModelのMaskのテクスチャをバインドする
    for( int idxExpression = 0; idxExpression < nn::mii::GetExpressionCount( ExpressionFlags ); ++idxExpression )
    {
        nn::mii::CharModel::TextureType type = nn::mii::CharModel::TextureType_Mask;
        const nn::gfx::TextureView* pView = g_CharModel.GetTextureView( type, idxExpression );

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_CharModel.SetTextureDescriptorSlot( type, slot, idxExpression );
    }

    /// Facelineのテクスチャをバインドする
    for( int idxTextureType = 0; idxTextureType < nn::mii::Faceline::TextureType_End; ++idxTextureType )
    {
        nn::mii::Faceline::TextureType type = nn::mii::Faceline::TextureType( idxTextureType );
        const nn::gfx::TextureView* pView = g_Faceline.GetTextureView( type );

        if( pView == nullptr )
        {
            continue;
        }

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_Faceline.SetTextureDescriptorSlot( type, slot );
    }

    /// Maskのテクスチャをバインドする
    for( int idxMaskTextureType = 0; idxMaskTextureType < nn::mii::Mask::TextureType_End; ++idxMaskTextureType )
    {
        nn::mii::Mask::TextureType type = nn::mii::Mask::TextureType( idxMaskTextureType );
        const nn::gfx::TextureView* pView = g_Mask.GetTextureView( type );

        if( pView == nullptr )
        {
            continue;
        }

        nn::gfx::DescriptorSlot slot;
        const int slotIndex = g_GfxFramework.AllocateDescriptorSlot( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView, 1 );
        pTextureDescriptorPool->SetTextureView( slotIndex, pView );
        pTextureDescriptorPool->GetDescriptorSlot( &slot, slotIndex );

        /// 取得したスロットをCharModelに設定する
        g_Mask.SetTextureDescriptorSlot( type, slot );
    }

    pTextureDescriptorPool->EndUpdate();
}

nn::mii::FacelineGpuBuffer g_FacelineGpuBuffer;
void InitializeFacelineGpuBuffer()
{
    size_t alignment = nn::mii::FacelineGpuBuffer::CalculateMemoryAlignment();
    size_t size = nn::mii::FacelineGpuBuffer::CalculateMemorySize();

    /// メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance( size );

    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::FacelineGpuBuffer::CalculateMemoryPoolAlignment( pDevice );
    size_t memoryPoolSize = nn::mii::FacelineGpuBuffer::CalculateMemoryPoolSize( pDevice );

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, memoryPoolSize, memoryPoolAlignment );
    g_FacelineGpuBuffer.Initialize( pMemory, size, pDevice, g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, memoryPoolSize );
}

nn::mii::MaskGpuBuffer g_MaskGpuBuffer;
void InitializeMaskGpuBuffer()
{
    size_t alignment = nn::mii::MaskGpuBuffer::CalculateMemoryAlignment();
    size_t size = nn::mii::MaskGpuBuffer::CalculateMemorySize();

    /// メモリを確保する
    g_pMemory.AlignUp(alignment);
    void* pMemory = g_pMemory.Get();
    g_pMemory.Advance(size);

    /// メモリプールとサイズを取得
    auto pDevice = g_GfxFramework.GetDevice();
    size_t memoryPoolAlignment = nn::mii::MaskGpuBuffer::CalculateMemoryPoolAlignment( pDevice );
    size_t memoryPoolSize = nn::mii::MaskGpuBuffer::CalculateMemoryPoolSize( pDevice );

    /// メモリプールアライメント調整
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, memoryPoolSize, memoryPoolAlignment );
    g_MaskGpuBuffer.Initialize( pMemory, size, pDevice, g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, memoryPoolSize );
}

void RenderFacelineTexture()
{
    /// GPUバッファを設定
    g_FacelineGpuBuffer.SetColorTarget( g_GfxFramework.GetDevice(), g_CharModel.GetFacelineTexture()
                                       ,g_TextureShaderInfo
                                       ,TextureResolution / 2,TextureResolution, TextureMipCount );
    g_FacelineGpuBuffer.SetFaceline( g_Faceline );

    /// GPUを使って描画する
    const int cmdBufferIndex = 0;
    auto pCommandBuffer = g_GfxFramework.GetRootCommandBuffer( cmdBufferIndex );
    g_GfxFramework.BeginFrame( cmdBufferIndex );
    pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView ) );
    pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler ) );
    pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
                                    | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );
    g_TextureShader.DrawFaceline( pCommandBuffer, &g_FacelineGpuBuffer, g_SamplerSlot );
    g_GfxFramework.EndFrame( cmdBufferIndex );

    /// コマンドの実行と完了待ち
    auto pQueue = g_GfxFramework.GetQueue();
    pQueue->ExecuteCommand( pCommandBuffer, 0 );
    pQueue->Sync();
}

void RenderMaskTexture()
{
    for( int idxExpression = 0; idxExpression < nn::mii::GetExpressionCount( ExpressionFlags ); ++idxExpression )
    {
        /// 表情を取得
        nn::mii::Expression expression = GetExpression( ExpressionFlags, idxExpression );

        /// GPUバッファを設定
        g_MaskGpuBuffer.SetColorTarget( g_GfxFramework.GetDevice(), g_CharModel.GetMaskTexture( idxExpression )
                                        ,g_TextureShaderInfo,TextureResolution, TextureMipCount );
        g_MaskGpuBuffer.SetMask( g_Mask, expression );

        /// GPUを使って描画する
        const int cmdBufferIdx = 0;
        auto pCommandBuffer = g_GfxFramework.GetRootCommandBuffer( cmdBufferIdx );
        g_GfxFramework.BeginFrame( cmdBufferIdx );
        pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_TextureView ) );
        pCommandBuffer->SetDescriptorPool( g_GfxFramework.GetDescriptorPool( nn::gfx::DescriptorPoolType::DescriptorPoolType_Sampler ) );
        pCommandBuffer->InvalidateMemory( nn::gfx::GpuAccess_Texture | nn::gfx::GpuAccess_IndexBuffer
                                        | nn::gfx::GpuAccess_ConstantBuffer | nn::gfx::GpuAccess_VertexBuffer );
        g_TextureShader.DrawMask( pCommandBuffer, &g_MaskGpuBuffer, g_SamplerSlot );
        g_GfxFramework.EndFrame( cmdBufferIdx );

        /// コマンドの実行と完了待ち
        auto pQueue = g_GfxFramework.GetQueue();
        pQueue->ExecuteCommand( pCommandBuffer, 0 );
        pQueue->Sync();
    }
}

nn::gfx::ResShaderFile* g_SampleShaderFile;
nn::gfx::ShaderCodeType g_SampleShaderCodeType;
int g_SampleShaderMatrixIndex;
int g_SampleShaderModulateIndex;
int g_SampleShaderSamplerIndex;
void InitializeSampleShader()
{
    g_SampleShaderFile = InitializeResShaderFile(&g_SampleShaderCodeType, SampleShaderPath);

    const nn::gfx::ResShaderVariation* pVariation
        = g_SampleShaderFile->GetShaderContainer()->GetResShaderVariation(0);
    const nn::gfx::ResShaderProgram* pProgram
        = pVariation->GetResShaderProgram(g_SampleShaderCodeType);
    const nn::gfx::Shader* pShader = pProgram->GetShader();

    g_SampleShaderMatrixIndex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Vertex, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_Matrix" );
    g_SampleShaderModulateIndex = pShader->GetInterfaceSlot (nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_ConstantBuffer, "u_Modulate" );
    g_SampleShaderSamplerIndex = pShader->GetInterfaceSlot( nn::gfx::ShaderStage_Pixel, nn::gfx::ShaderInterfaceType_Sampler, "s_Tex" );

    NN_ASSERT( g_SampleShaderMatrixIndex >= 0 );
    NN_ASSERT( g_SampleShaderModulateIndex >= 0 );
    NN_ASSERT( g_SampleShaderSamplerIndex >= 0 );
}

nn::gfx::RasterizerState g_RasterizerState[nn::gfx::CullMode_End];
void InitializeRasterizerState()
{
    nn::gfx::RasterizerStateInfo info;
    info.SetDefault();
    info.SetPrimitiveTopologyType( nn::gfx::PrimitiveTopologyType_Triangle );
    info.SetScissorEnabled( true );
    info.SetDepthClipEnabled( false );
    for( int idxCullMode = 0; idxCullMode < nn::gfx::CullMode_End; ++idxCullMode )
    {
        nn::gfx::CullMode cullMode = nn::gfx::CullMode( idxCullMode );
        info.SetCullMode( cullMode );
        g_RasterizerState[idxCullMode].Initialize( g_GfxFramework.GetDevice(), info );
    }
}

enum DrawMode
{
    DrawMode_Opa,
    DrawMode_Xlu,
    DrawMode_End
};

nn::gfx::BlendState g_BlendState[DrawMode_End];
void InitializeBlendState()
{
    for( int idxDrawMode = 0; idxDrawMode < DrawMode_End; ++idxDrawMode )
    {
        DrawMode drawMode = DrawMode( idxDrawMode );
        nn::gfx::BlendTargetStateInfo targetInfo;
        {
            targetInfo.SetDefault();
        }

        if( drawMode == DrawMode_Xlu )
        {
            targetInfo.SetBlendEnabled( true );
            targetInfo.SetColorBlendFunction( nn::gfx::BlendFunction_Add );
            targetInfo.SetSourceColorBlendFactor( nn::gfx::BlendFactor_SourceAlpha );
            targetInfo.SetDestinationColorBlendFactor( nn::gfx::BlendFactor_OneMinusSourceAlpha );
        }

        nn::gfx::BlendState::InfoType info;
        info.SetDefault();
        info.SetBlendTargetStateInfoArray( &targetInfo, 1 );

        size_t size = nn::gfx::BlendState::GetRequiredMemorySize( info );
        g_pMemory.AlignUp( nn::gfx::BlendState::RequiredMemoryInfo_Alignment );
        g_BlendState[idxDrawMode].SetMemory( g_pMemory.Get(), size );
        g_pMemory.Advance( size );
        g_BlendState[idxDrawMode].Initialize( g_GfxFramework.GetDevice(), info );
    }
}

nn::gfx::DepthStencilState g_DepthStencilState[DrawMode_End];
void InitializeDepthStencilState()
{
    for( int idxDrawMode = 0; idxDrawMode < DrawMode_End; ++idxDrawMode )
    {
        nn::gfx::DepthStencilStateInfo info;
        info.SetDefault();
        info.SetDepthTestEnabled( true );
        info.SetDepthWriteEnabled( idxDrawMode == DrawMode_Opa );

        g_DepthStencilState[idxDrawMode].Initialize( g_GfxFramework.GetDevice(), info );
    }
}

nn::gfx::VertexState g_VertexState;
void InitializeVertexState( nn::g3d::ResFile* pResFile )
{
    // 有効な mii のシェイプで頂点ステートを初期化します。
    int validMiiShapeIdx = -1;
    for( int idxShape = 0; idxShape < g_pModelObj->GetShapeCount(); ++idxShape )
    {
        if( g_pModelObj->IsShapeVisible( idxShape ) )
        {
            validMiiShapeIdx = idxShape;
            break;
        }
    }
    NN_ASSERT( validMiiShapeIdx != -1 );
    auto pResVertex = pResFile->GetModel( 0 )->GetShape( validMiiShapeIdx )->GetVertex();

    NN_ASSERT( pResVertex->GetVertexAttrCount() == nn::mii::DrawParam::AttributeType_End );
    NN_ASSERT( pResVertex->GetVertexBufferCount() == nn::mii::DrawParam::AttributeType_End );

    nn::gfx::VertexAttributeStateInfo attribs[nn::mii::DrawParam::AttributeType_End];
    nn::gfx::VertexBufferStateInfo buffers[nn::mii::DrawParam::AttributeType_End];
    for( int idxAttrib = 0; idxAttrib < pResVertex->GetVertexAttrCount(); ++idxAttrib )
    {
        auto pResVertexAttr = pResVertex->GetVertexAttr( idxAttrib );
        attribs[idxAttrib].SetDefault();
        attribs[idxAttrib].SetBufferIndex( idxAttrib );                    // mii のリソースは 1 attribute 1 buffer です。
        attribs[idxAttrib].SetFormat( pResVertexAttr->GetFormat() );
        attribs[idxAttrib].SetOffset( pResVertexAttr->GetOffset() );
        attribs[idxAttrib].SetShaderSlot( idxAttrib );                    // スロットはシェーダでロケーションを指定しています。
    }

    for( int idxBuff = 0; idxBuff < pResVertex->GetVertexBufferCount(); ++idxBuff )
    {
        buffers[idxBuff].SetDefault();
        buffers[idxBuff].SetStride( pResVertex->GetVertexBufferStride( idxBuff ) );
    }

    nn::gfx::VertexState::InfoType info;
    info.SetDefault();
    info.SetVertexAttributeStateInfoArray(attribs, nn::mii::DrawParam::AttributeType_End);
    info.SetVertexBufferStateInfoArray(buffers, nn::mii::DrawParam::AttributeType_End);

    size_t size = nn::gfx::VertexState::GetRequiredMemorySize( info );
    g_pMemory.AlignUp( nn::gfx::VertexState::RequiredMemoryInfo_Alignment );
    g_VertexState.SetMemory( g_pMemory.Get(), size );
    g_pMemory.Advance( size );
    g_VertexState.Initialize( g_GfxFramework.GetDevice(), info, NULL );
}

nn::gfx::Buffer g_ConstantBufferMatrix;
void InitializeConstantBuffers()
{
    auto pDevice = g_GfxFramework.GetDevice();
    nn::gfx::BufferInfo info;
    info.SetDefault();
    info.SetGpuAccessFlags( nn::gfx::GpuAccess_ConstantBuffer );
    info.SetSize( sizeof( ConstantBufferMatrix ) );

    size_t alignment = nn::gfx::Buffer::GetBufferAlignment( pDevice, info );
    ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, info.GetSize(), alignment );

    g_ConstantBufferMatrix.Initialize( pDevice, info
                                        , g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer )
                                        , offset
                                        , info.GetSize() );
}

void SetupConstantBuffers()
{
    ConstantBufferMatrix* pBuffer = g_ConstantBufferMatrix.Map<ConstantBufferMatrix>();

    /// 投影行列を設定
    float fovy = 3.14f / 4.0f;
    float aspect = static_cast<float>( g_GfxFramework.GetDisplayWidth() ) / static_cast<float>( g_GfxFramework.GetDisplayHeight() );
    nn::util::Matrix4x4fType projection = {};
    nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projection, fovy, aspect, 0.01f, 10000.0f);
    nn::util::MatrixStore(&pBuffer->projection, projection);

    /// LookAt設定
    nn::util::Vector3fType camPos = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f,30.0f,115.0f);
    nn::util::Vector3fType camTarget = NN_UTIL_VECTOR_3F_INITIALIZER(0.0f,30.0f,0.0f);
    nn::util::Matrix4x3fType modelView;
    nn::util::MatrixLookAtRightHanded(&modelView, camPos, camTarget, 0.0f);
    nn::util::MatrixStore(&pBuffer->modelView, modelView);

    g_ConstantBufferMatrix.Unmap();
}

void FinalizeMii()
{
    auto pDevice = g_GfxFramework.GetDevice();
    g_ConstantBufferMatrix.Finalize( pDevice );
    g_VertexState.Finalize( pDevice );
    for(int idxDrawMode = 0; idxDrawMode < DrawMode_End; ++idxDrawMode )
    {
        g_BlendState[idxDrawMode].Finalize( pDevice );
        g_DepthStencilState[idxDrawMode].Finalize( pDevice );
    }
    for(int idxCullMode = 0; idxCullMode < nn::gfx::CullMode_End; ++idxCullMode )
    {
        g_RasterizerState[idxCullMode].Finalize( pDevice );
    }

    nn::gfx::ResShaderContainer* pContainer = g_SampleShaderFile->GetShaderContainer();
    for( int idxShaderVariation = 0; idxShaderVariation < pContainer->GetShaderVariationCount(); ++idxShaderVariation )
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxShaderVariation );
        pVariation->GetResShaderProgram(g_SampleShaderCodeType)->Finalize( pDevice );
    }
    pContainer->Finalize( pDevice );

    g_MaskGpuBuffer.Finalize( pDevice );
    g_FacelineGpuBuffer.Finalize( pDevice );
    g_Mask.Finalize( pDevice );
    g_Faceline.Finalize( pDevice );
    g_CharModel.Finalize( pDevice );
    g_Resource.Finalize( pDevice );
    g_TextureShader.Finalize( pDevice );

    pContainer = g_pTextureShaderFile->GetShaderContainer();
    for(int idxShaderVariation = 0; idxShaderVariation < pContainer->GetShaderVariationCount(); ++idxShaderVariation )
    {
        nn::gfx::ResShaderVariation* pVariation = pContainer->GetResShaderVariation( idxShaderVariation );
        pVariation->GetResShaderProgram(g_TextureShaderCodeType)->Finalize( pDevice );
    }
    g_pTextureShaderFile->GetShaderContainer()->Finalize( pDevice );
    g_Sampler.Finalize( pDevice );
    g_pModelObj->CleanupBlockBuffer( pDevice );
}

}

template <typename TType1, typename TType2, typename TType3, typename TType4 > struct MiiBindTable
{
    nn::util::string_view    name;
    TType1    type1;
    TType2    type2;
    TType3    type3;
    TType4    type4;
};

typedef MiiBindTable<nn::mii::CharModel::DrawType, nn::mii::CreateModelType, nn::mii::CreateNoseType, int> MaterialBindTable;

template<typename TType1, typename TType2, typename TType3, typename TType4, size_t TTableSize>
int FindMiiBindTableIndex( const char* pTargetName, const MiiBindTable<TType1, TType2, TType3, TType4>( &miiBindTableArray )[TTableSize] )
{
    for( int idxTbl = 0; idxTbl  < static_cast<int>( TTableSize ); ++idxTbl  )
    {
        auto& miiBindTable = miiBindTableArray[idxTbl];
        if( miiBindTable.name == pTargetName )
        {
            return idxTbl;
        }
    }
    return -1;
};
template<typename TType1, typename TType2, typename TType3, typename TType4>
int FindMiiBindTableIndex( const char* pTargetName, const MiiBindTable<TType1, TType2, TType3, TType4>* miiBindTableArray, size_t miiBindTableArraySize )
{
    for( int idxTbl = 0; idxTbl  < static_cast<int>( miiBindTableArraySize ); ++idxTbl  )
    {
        auto& miiBindTable = miiBindTableArray[idxTbl];
        if( miiBindTable.name == pTargetName )
        {
            return idxTbl;
        }
    }
    return -1;
};

// render_info, shader_param のデフォルト値登録用コールバックパラメータ
class MaterialCallbackParam
{
public:
    MaterialCallbackParam() :
        renderInfoCullingName( "culling" ),
        shaderParamModulateTypeId( "mii_modulate_type" ),
        shaderParamConstColorId( "mii_const_color" )
    {
    }

    const char* renderInfoCullingName;
    const char* shaderParamModulateTypeId;
    const char* shaderParamConstColorId;

};

class DrawParamCallbckparam
{
public:
    DrawParamCallbckparam( const MaterialBindTable* materialBindTableArray, size_t materialBindTableArraySize )
        : charModelMaterialBindTableArray( materialBindTableArray )
        , charModelMaterialBindTableArraySize( materialBindTableArraySize )
    {
    }
    const MaterialBindTable* charModelMaterialBindTableArray;
    size_t charModelMaterialBindTableArraySize;
};

// マテリアル名と mii のパーツを紐づけるテーブル
const MaterialBindTable g_CharModelMaterialBindTableArray[] =
{
    { "beard",      nn::mii::CharModel::DrawType_Beard,        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "faceline",   nn::mii::CharModel::DrawType_Faceline,    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "forehead",   nn::mii::CharModel::DrawType_Forehead,    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "hair",       nn::mii::CharModel::DrawType_Hair,        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "hat",        nn::mii::CharModel::DrawType_Hat,        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "nose",       nn::mii::CharModel::DrawType_Nose,        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "noseline",   nn::mii::CharModel::DrawType_Noseline,    nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "glass",        nn::mii::CharModel::DrawType_Glass,        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
    { "mask",        nn::mii::CharModel::DrawType_Mask,        nn::mii::CreateModelType_Normal, nn::mii::CreateNoseType_Normal, 0 },
};
const size_t g_CharModelMaterialBindTableArraySize = sizeof( g_CharModelMaterialBindTableArray ) / sizeof( MaterialBindTable );

// ビジブルにするボーン
const char* g_VisibleMiiBoneNameArray[] =
{
    "mii_all",
};
const size_t g_VisibleMiiBoneNameArraySize = sizeof( g_VisibleMiiBoneNameArray ) / sizeof( char* );

// DrawParam 取得コールバック
void DrawParamCallback( nn::g3d::mii::DrawParamCallbackOutput* pDrawParamCallbackOutput, const nn::g3d::ResModel* pModel,
                        const int shapeIndex, const nn::mii::CharModel* pCharModel, void* pCallbackParam )
{
    auto materialBindTableArray = reinterpret_cast<DrawParamCallbckparam*>( pCallbackParam )->charModelMaterialBindTableArray;
    size_t materialBindTableArraySize = reinterpret_cast<DrawParamCallbckparam*>( pCallbackParam )->charModelMaterialBindTableArraySize;
    auto pShape = pModel->GetShape( shapeIndex );
    const char* pMatName = pModel->GetMaterial( pShape->GetMaterialIndex() )->GetName();
    int idxCharModelMaterialBindTable = FindMiiBindTableIndex( pMatName, materialBindTableArray, materialBindTableArraySize );

    // Mii のパーツではない場合
    if( idxCharModelMaterialBindTable == -1 )
    {
        pDrawParamCallbackOutput->isMiiFace = false;
        pDrawParamCallbackOutput->pDrawParam = nullptr;
        return;
    }

    // Mii のパーツの場合、DrawParam を取得します。
    auto& matBindTbl = materialBindTableArray[idxCharModelMaterialBindTable];
    pDrawParamCallbackOutput->isMiiFace = true;
    pDrawParamCallbackOutput->pDrawParam = pCharModel->GetDrawParam( matBindTbl.type1, matBindTbl.type2, matBindTbl.type3, matBindTbl.type4 );
    pDrawParamCallbackOutput->pBoundingBox = &pCharModel->GetBoundingBox( matBindTbl.type2, matBindTbl.type3 );
}

// render_info, shader_param のデフォルト値登録用のコールバック関数
void MaterialCallback( nn::g3d::ResMaterial* pMaterial, const nn::mii::DrawParam* pDrawParam, void* pUserParam )
{
    auto pCbParam = static_cast<MaterialCallbackParam*>( pUserParam );

    const int ShaderParamCount = pMaterial->GetShaderParamCount();
    for( int idxShaderParam = 0; idxShaderParam < ShaderParamCount; ++idxShaderParam )
    {
        auto pShaderParam = pMaterial->GetShaderParam( idxShaderParam );
        const char* pShaderParamId = pShaderParam->GetId();

        if ( pCbParam->shaderParamModulateTypeId == std::string( pShaderParamId ) )
        {
            NN_SDK_ASSERT( pShaderParam->GetSrcSize() == sizeof( int32_t ), "Modulate type param size is not consistent." );
            void* pSrcParamBuffer = pMaterial->GetSrcParam( idxShaderParam );
            *static_cast<int32_t*>(pSrcParamBuffer) = static_cast<int32_t>(pDrawParam->GetModulateType());
        }
        else if ( pCbParam->shaderParamConstColorId == std::string( pShaderParamId ) )
        {
            const size_t ConstColorSize = sizeof(float) * 9;    // vec3 * 3
            NN_SDK_ASSERT( pShaderParam->GetSrcSize() == ConstColorSize, "Mii const color size is not consistent." );
            NN_UNUSED( ConstColorSize );
            char* pSrcParamBuffer = reinterpret_cast<char*>( pMaterial->GetSrcParam(idxShaderParam) );
            for ( int idxConstColor = 0; idxConstColor < nn::mii::DrawParam::ConstantColorNum; ++idxConstColor, pSrcParamBuffer += sizeof( float ) * 4 )
            {
                const nn::mii::Color3* pConstColor = pDrawParam->GetConstantColor( idxConstColor );
                if ( pConstColor )
                {
                    memcpy( pSrcParamBuffer, pConstColor, sizeof( nn::mii::Color3 ) );
                }
            }
        }
    }
    const int RenderInfoCount = pMaterial->GetRenderInfoCount();
    for( int idxRenderInfo = 0; idxRenderInfo < RenderInfoCount; ++idxRenderInfo )
    {
        auto pRenderInfo = pMaterial->GetRenderInfo( idxRenderInfo );
        const char* pRenderInfoName = pRenderInfo->GetName();
        if( pRenderInfoName == std::string( pCbParam->renderInfoCullingName ) )
        {
            NN_SDK_ASSERT( nn::g3d::ResRenderInfo::Type_Int == pRenderInfo->GetType(), "RenderInfo culling type mismatch." );
            nn::gfx::CullMode cullMode = pDrawParam->GetCullMode();    // none, front, back
            *pRenderInfo->ToData().intValueArray.Get() = static_cast<int>(cullMode);
        }
    }
}

nn::g3d::TextureRef TextureBindCallback( const char* name, void* pUserData )
{
    auto pCharModel = static_cast<nn::mii::CharModel*>( pUserData );

    const char* texNameArray[] =
    {
        "faceline",
        "hat",
        "mask",
        "noseline",
        "glass",
    };
    const int TexCount = sizeof( texNameArray ) / sizeof( char* );

    nn::g3d::TextureRef textureRef;
    for ( int idxTex = 0; idxTex < TexCount; ++idxTex )
    {
        if ( name == std::string(texNameArray[idxTex]) )
        {
            int MaskSlot = nn::mii::Expression_Min;    // "mask" 以外では MaskSlot は無効なので適当な値を入れます

            if ( name == std::string( texNameArray[ 2 ] ) )
            {
                MaskSlot = 1;    // "mask" を切り替えるアニメーションの適用で上書きされます
            }
            nn::mii::CharModel::TextureType miiTexType = static_cast<nn::mii::CharModel::TextureType>( idxTex );
            auto* pTextureView = pCharModel->GetTextureView( miiTexType, MaskSlot );
            nn::gfx::DescriptorSlot descSlot = pCharModel->GetTextureDescriptorSlot( miiTexType, MaskSlot );
            textureRef.SetTextureView( pTextureView );
            textureRef.SetDescriptorSlot( descSlot );
            return textureRef;
        }
    }
    return textureRef;
}

nn::g3d::ResFile*    g_pResFile = nullptr;
void InitializeResFile()
{
    // Mii 用のテンプレート ResFile をロード
    {
        nn::fs::FileHandle handle;
        nn::Result result = nn::fs::OpenFile( &handle, "Contents:/Mii.bfres", nn::fs::OpenMode_Read );
        NN_ASSERT( result.IsSuccess() );

        int64_t fileSize = 0;
        nn::fs::GetFileSize( &fileSize, handle );
        g_pMemory.AlignUp( 4 * 1024 );    // ResFile の最大アライメントにアラインアップ
        void* pData = g_pMemory.Get();
        g_pMemory.Advance( static_cast<ptrdiff_t>( fileSize ) );
        result = nn::fs::ReadFile( handle, 0, pData, static_cast<size_t>( fileSize ) );
        NN_ASSERT( result.IsSuccess() );

        g_pResFile = nn::g3d::ResFile::ResCast( pData );
        NN_ASSERT( g_pResFile->IsValid( g_pResFile ) );
        g_pResFile->Setup( g_GfxFramework.GetDevice() );

        g_pResFile->BindTexture(TextureBindCallback, &g_CharModel );

        // gfx::SamplerDescriptorSlot, gfx::Sampler の登録
        for( int idxModel = 0; idxModel < g_pResFile->GetModelCount(); ++idxModel )
        {
            auto pModel = g_pResFile->GetModel( idxModel );
            for( int idxMaterial = 0; idxMaterial < pModel->GetMaterialCount(); ++idxMaterial )
            {
                auto pMaterial = pModel->GetMaterial( idxMaterial );
                if( pMaterial->GetSamplerCount() )
                {
                    const int IdxSampler = 0;    // SimpleMii の shader では uniform sampler は 1 つ
                    pMaterial->SetSamplerDescriptorSlot( IdxSampler, g_SamplerSlot );
                }
                pMaterial->SetRawParamSize( sizeof( ConstantBufferModulate ) );
            }
        }
        nn::fs::CloseFile(handle);
    }

    // CharModel を ResFile に登録します。
    const int IdxModel = 0;
    NN_ASSERT( IdxModel  < g_pResFile->GetModelCount(), "ResFile can have %d models but IdxModel is %d\n", g_pResFile->GetModelCount(), IdxModel );
    for( int idxModel = 0; idxModel < g_pResFile->GetModelCount(); ++idxModel )
    {
        auto pModel = g_pResFile->GetModel( idxModel );
        MaterialCallbackParam renderInfoShaderParamCbFuncParam;
        DrawParamCallbckparam drawParamCallbackParam( g_CharModelMaterialBindTableArray, g_CharModelMaterialBindTableArraySize );
        nn::g3d::mii::AssignCharModel( pModel, &g_CharModel, MaterialCallback, &renderInfoShaderParamCbFuncParam, DrawParamCallback, &drawParamCallbackParam );
    }
}

void FinalizeResFile()
{
    // nn::g3d::mii::AssignCharModel() により頂点、インデックスバッファが一部差し替えられているので元に戻す。
    for( int idxModel = 0; idxModel < g_pResFile->GetModelCount(); ++idxModel )
    {
        auto pResModel = g_pResFile->GetModel( idxModel );
        pResModel->Reset();
    }
    g_pResFile->Cleanup( g_GfxFramework.GetDevice() );
    g_pResFile = nullptr;
}

void InitializeModelObj( nn::g3d::ResFile* pFile )
{
    for( int idxModel = 0; idxModel < pFile->GetModelCount(); ++idxModel )
    {
        auto pModel = pFile->GetModel( idxModel );
        nn::g3d::ModelObj::Builder builder( pModel );

        builder.CalculateMemorySize();
        size_t alignment = nn::g3d::ModelObj::Alignment_Buffer;
        size_t bufferSize = builder.GetWorkMemorySize();
        size_t objectSize = nn::util::align_up(sizeof(nn::g3d::ModelObj), alignment);
        g_pMemory.AlignUp( alignment );
        g_pModelObj = reinterpret_cast<nn::g3d::ModelObj*>( g_pMemory.Get() );
        g_pMemory.Advance( objectSize + bufferSize );
        NN_ASSERT_NOT_NULL( g_pModelObj );
        new( g_pModelObj ) nn::g3d::ModelObj();

        bool success = false;
        success = builder.Build(g_pModelObj, nn::g3d::AddOffset( g_pModelObj, objectSize ), bufferSize);
        ( void ) success;
        NN_ASSERT( success == true );

        auto pDevice = g_GfxFramework.GetDevice();
        size_t size = g_pModelObj->CalculateBlockBufferSize( pDevice );
        size_t memoryPoolAlignment = g_pModelObj->GetBlockBufferAlignment( g_GfxFramework.GetDevice() );
        ptrdiff_t offset = g_GfxFramework.AllocatePoolMemory( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer, size, memoryPoolAlignment );
        success = g_pModelObj->SetupBlockBuffer(pDevice, g_GfxFramework.GetMemoryPool( nns::gfx::GraphicsFramework::MemoryPoolType_ConstantBuffer ), offset, size );
        ( void ) success;
        NN_ASSERT( success == true );
    }

    // 描画するパーツのボーンをビジブルにします
    {
        for( int idxBone = 0; idxBone < g_pModelObj->GetSkeleton()->GetBoneCount(); ++idxBone )
        {
            g_pModelObj->SetBoneVisible( idxBone, false );
        }

        for( int idxVisibleBone = 0; idxVisibleBone < g_VisibleMiiBoneNameArraySize; ++idxVisibleBone )
        {
            int idxBone = g_pModelObj->GetSkeleton()->FindBoneIndex( g_VisibleMiiBoneNameArray[idxVisibleBone] );
            NN_ASSERT( idxBone != -1 );
            g_pModelObj->SetBoneVisible( idxBone, true );
        }
    }

    // GPU が参照する Material UniformBlock の値を更新します。
    for( int idxMat = 0, materialCount = g_pModelObj->GetMaterialCount(); idxMat < materialCount; ++idxMat )
    {
        nn::g3d::MaterialObj* pMaterialObj = g_pModelObj->GetMaterial( idxMat );
        if( pMaterialObj->GetShaderParamCount() )
        {
            ConstantBufferModulate* pMaterialBlock = pMaterialObj->GetMaterialBlock(0)->Map<ConstantBufferModulate>();
            memcpy( &pMaterialBlock->modulateType, pMaterialObj->GetShaderParam( 0 ), sizeof( int ) );
            memcpy( &pMaterialBlock->constantColor, pMaterialObj->GetShaderParam( 1 ), sizeof( float ) * 9 );
            pMaterialObj->GetMaterialBlock(0)->Unmap();
        }
    }
}

void InitializeMaterialAnimObj( nn::g3d::MaterialAnimObj& materialAnimObj, nn::g3d::ResFile* pFile )
{
    nn::g3d::MaterialAnimObj::Builder builder;
    builder.Reserve( g_pModelObj->GetResource() );
    builder.Reserve( pFile->GetMaterialAnim( 0 ) );
    builder.CalculateMemorySize();
    size_t bufferSize = builder.GetWorkMemorySize();

    g_pMemory.AlignUp( nn::g3d::MaterialAnimObj::Alignment_Buffer );
    void* pBuffer = g_pMemory.Get();
    NN_ASSERT_NOT_NULL( pBuffer );
    g_pMemory.Advance( bufferSize );
    bool success = builder.Build( &materialAnimObj, pBuffer, bufferSize );
    NN_UNUSED( success );
    NN_ASSERT( success );

    nn::g3d::ResMaterialAnim* pMaterialAnim = nullptr;
    pMaterialAnim = pFile->FindMaterialAnim( "MiiSimpleFaceExpressionChange" );
    NN_ASSERT_NOT_NULL( pMaterialAnim );
    materialAnimObj.SetResource( pMaterialAnim );
    materialAnimObj.Bind( g_pModelObj );

    // MaterialanimObj へテクスチャの登録をします
    NN_ASSERT( materialAnimObj.GetTextureCount() <= nn::mii::GetExpressionCount( ExpressionFlags ) );
    for( int idxTex = 0; idxTex < materialAnimObj.GetTextureCount(); ++idxTex )
    {
        nn::g3d::TextureRef textureRef;
        auto pTextureView = g_CharModel.GetTextureView( nn::mii::CharModel::TextureType::TextureType_Mask, idxTex );
        NN_ASSERT_NOT_NULL( pTextureView );
        textureRef.SetTextureView( pTextureView );
        textureRef.SetDescriptorSlot( const_cast<nn::gfx::DescriptorSlot&>( g_CharModel.GetTextureDescriptorSlot( nn::mii::CharModel::TextureType::TextureType_Mask, idxTex ) ) );
        materialAnimObj.SetTexture( idxTex, textureRef );
    }
}

// モデル内のシェイプインデクスを不透明から順に並べます。
void SortMaterial( std::vector<int>* pShapeIndexArray, const nn::g3d::ModelObj* pModelObj )
{
    NN_ASSERT_NOT_NULL( pShapeIndexArray );
    NN_ASSERT_NOT_NULL( pModelObj );

    auto& shapeIdxArray = *pShapeIndexArray;
    shapeIdxArray.reserve( pModelObj->GetShapeCount() );
    for( int idxShape = 0; idxShape < pModelObj->GetShapeCount(); ++idxShape )
    {
        auto pShapeObj = pModelObj->GetShape( idxShape );
        auto pMaterialObj = pModelObj->GetMaterial( pShapeObj->GetMaterialIndex() );
        auto pDrawMode = pMaterialObj->GetResource()->FindRenderInfo( "opacity" )->GetInt();
        if( ( pDrawMode == nullptr ) || ( ( pDrawMode != nullptr ) && ( *pDrawMode == DrawMode_Opa ) ) ) // "opacity"を持っていない場合は不透明とします。
        {
            shapeIdxArray.insert( shapeIdxArray.begin(), idxShape );
        }
        else
        {
            shapeIdxArray.emplace_back( idxShape );
        }
    }
}

//  計算処理コールバック
void CalculateCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    nn::g3d::MaterialAnimObj* pMaterialAnimObj = reinterpret_cast<nn::g3d::MaterialAnimObj*>(pUserData);

    // CPU 処理
    if (pMaterialAnimObj->GetResource())
    {
        pMaterialAnimObj->Calculate();
        pMaterialAnimObj->ApplyTo(g_pModelObj);
        pMaterialAnimObj->GetFrameCtrl().UpdateFrame();
    }
}

//  コマンド生成コールバック
void MakeCommandCallback(nns::gfx::GraphicsFramework* pGraphicsFramework, int bufferIndex, void* pUserData)
{
    NN_UNUSED(pGraphicsFramework);
    std::vector<int>* pShapeIdxArray = reinterpret_cast<std::vector<int>*>(pUserData);

    g_GfxFramework.BeginFrame(bufferIndex);
    auto pCommandBuffer = g_GfxFramework.GetRootCommandBuffer(bufferIndex);

    /// クリア
    nn::gfx::ColorTargetView* pColorTarget = g_GfxFramework.GetColorTargetView();
    pCommandBuffer->ClearColor(pColorTarget, 0.3f, 0.1f, 0.1f, 0.1f, NULL);
    auto pDepthStencilView = g_GfxFramework.GetDepthStencilView();
    pCommandBuffer->ClearDepthStencil(pDepthStencilView, 1.0f, 255, nn::gfx::DepthStencilClearMode_DepthStencil, NULL);

    pCommandBuffer->SetRenderTargets(1, &pColorTarget, pDepthStencilView);
    pCommandBuffer->SetViewportScissorState(g_GfxFramework.GetViewportScissorState());

    for (auto idxShape : *pShapeIdxArray)
    {
        if (!g_pModelObj->IsShapeVisible(idxShape))
        {
            continue;
        }
        auto pShapeObj = g_pModelObj->GetShape(idxShape);
        auto pMaterialObj = g_pModelObj->GetMaterial(pShapeObj->GetMaterialIndex());

        /// 描画ステート設定
        const nn::gfx::CullMode cullMode = static_cast<nn::gfx::CullMode>(*pMaterialObj->GetResource()->FindRenderInfo("culling")->GetInt());
        DrawMode drawMode = static_cast<DrawMode>(*pMaterialObj->GetResource()->FindRenderInfo("opacity")->GetInt());
        pCommandBuffer->SetRasterizerState(&g_RasterizerState[cullMode]);
        pCommandBuffer->SetBlendState(&g_BlendState[drawMode]);
        pCommandBuffer->SetDepthStencilState(&g_DepthStencilState[drawMode]);
        pCommandBuffer->SetVertexState(&g_VertexState);
        pCommandBuffer->SetShader(g_SampleShaderFile->GetShaderContainer()->GetResShaderVariation(0)->GetResShaderProgram(g_SampleShaderCodeType)->GetShader(), nn::gfx::ShaderStageBit_All);

        /// MVP 行列のセット
        nn::gfx::GpuAddress gpuAddress;
        g_ConstantBufferMatrix.GetGpuAddress(&gpuAddress);
        pCommandBuffer->SetConstantBuffer(g_SampleShaderMatrixIndex, nn::gfx::ShaderStage_Vertex, gpuAddress, sizeof(ConstantBufferMatrix));

        /// 描画する
        // Material Uniform Block の設定
        if (pMaterialObj->GetMaterialBlockSize() != 0)
        {
            pMaterialObj->GetMaterialBlock(0)->GetGpuAddress(&gpuAddress);
            pCommandBuffer->SetConstantBuffer(g_SampleShaderModulateIndex, nn::gfx::ShaderStage_Pixel, gpuAddress, pMaterialObj->GetMaterialBlockSize());
        }

        for (int idxSampler = 0; idxSampler < pMaterialObj->GetSamplerCount(); ++idxSampler)
        {
            pCommandBuffer->SetTextureAndSampler(g_SampleShaderSamplerIndex, nn::gfx::ShaderStage_Pixel
                , pMaterialObj->GetTexture(idxSampler).GetDescriptorSlot()
                , pMaterialObj->GetSampler(idxSampler).GetDescriptorSlot());
        }

        NN_ASSERT(pShapeObj->GetVertexAttrCount() == nn::mii::DrawParam::AttributeType_End);
        NN_ASSERT(pShapeObj->GetVertexBufferCount() == nn::mii::DrawParam::AttributeType_End);

        for (int idxAttrib = nn::mii::DrawParam::AttributeType_Position; idxAttrib < nn::mii::DrawParam::AttributeType_End; ++idxAttrib)
        {
            auto pBuffer = pShapeObj->GetVertexBuffer(idxAttrib);
            if (pBuffer == nullptr)
            {
                continue;
            }

            auto pResVtx = pShapeObj->GetResVertex();
            pBuffer->GetGpuAddress(&gpuAddress);
            pCommandBuffer->SetVertexBuffer(idxAttrib, gpuAddress, pResVtx->GetVertexBufferStride(idxAttrib), pResVtx->GetVertexBufferInfo(idxAttrib)->GetSize());
        }
        pShapeObj->GetResMesh()->Draw(pCommandBuffer, 1);
    }

    g_GfxFramework.EndFrame(bufferIndex);
}

#if defined( NN_BUILD_CONFIG_OS_SUPPORTS_HORIZON )
//-----------------------------------------------------------------------------
// nninitStartup() is invoked before calling nnMain().
//
extern "C" void nninitStartup()
{
    const size_t ApplicationMemorySize = 512 * 1024 * 1024;
    const size_t MallocMemorySize = 256 * 1024 * 1024;
    nn::Result result = nn::os::SetMemoryHeapSize(ApplicationMemorySize);
    NN_ASSERT(result.IsSuccess());
    uintptr_t address;
    result = nn::os::AllocateMemoryBlock(&address, MallocMemorySize);
    NN_ASSERT(result.IsSuccess());
    nn::init::InitializeAllocator(reinterpret_cast<void*>(address), MallocMemorySize);
}
#endif

//
//  Main Function
//  メイン関数です。
//

extern "C" void nnMain()
{
    InitializeFs();

    // フレームワーク初期化
    nns::gfx::GraphicsFramework::InitializeGraphicsSystem( 1024 * 1024 * 32 );
    nns::gfx::GraphicsFramework::FrameworkInfo fwInfo;
    fwInfo.SetDefault();
    fwInfo.SetDisplayHeight(720);
    fwInfo.SetDisplayWidth(1280);
    fwInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_Data, 1024 * 1024 * 32 );
    fwInfo.SetMemoryPoolSize( nns::gfx::GraphicsFramework::MemoryPoolType_RenderTarget, 1024 * 1024 * 32 );
    fwInfo.SetBufferCount(2);
    fwInfo.SetSwapChainBufferCount(2);
    g_GfxFramework.Initialize( fwInfo );

    InitializeResources();

    /// Mii用のサンプラを準備する
    InitializeSampler();

    /// CharInfoをファイルから読み込む
    LoadCharInfo();

    /// Faceline,Maskテクスチャ作成用シェーダーを初期化
    InitializeTextureShader();

    //// Miiリソースを初期化
    InitializeMiiResource();

    /// CharModelを初期化
    InitializeCharModel();

    /// Facelineを初期化
    InitializeFaceline();

    /// Maskを初期化
    InitializeMask();

    /// デスクリプタスロットの割り当て
    BindDescriptorSlot();

    /// Faceline,Maskを描画する
    InitializeFacelineGpuBuffer();
    InitializeMaskGpuBuffer();
    /// テクスチャへ描画後はg_Faceline,g_Maskを破棄できます。
    RenderFacelineTexture();
    RenderMaskTexture();

    // ResFile、ModelObj の構築
    InitializeResFile();
    NN_ASSERT_NOT_NULL( g_pResFile );
    nn::g3d::ResFile* pFile = g_pResFile;
    InitializeModelObj( pFile );

    // MaterialAnimObj の構築
    nn::g3d::MaterialAnimObj    materialAnimObj;
    InitializeMaterialAnimObj( materialAnimObj, pFile );

    /// Mii描画用のシェーダーを初期化
    InitializeSampleShader();

    /// Mii描画用の処理を初期化
    InitializeRasterizerState();
    InitializeBlendState();
    InitializeDepthStencilState();
    InitializeVertexState( pFile );
    InitializeConstantBuffers();

    /// 描画のためにコンスタントバッファを設定する
    SetupConstantBuffers();

    /// バッファーオーバーフローチェック
    NN_ASSERT( g_pMemoryHeap.Distance( g_pMemory.Get() ) < MemoryHeapSize );

    // マテリアルソート
    std::vector<int> shapeIdxArray;
    SortMaterial( &shapeIdxArray, g_pModelObj );

    // フレームワーク設定
    g_GfxFramework.SetCalculateCallback(CalculateCallback, &materialAnimObj);
    g_GfxFramework.SetMakeCommandCallback(MakeCommandCallback, &shapeIdxArray);

    // 毎フレームのレンダリング
    for( int frame = 0; frame < 60 * 5; ++frame )
    {
        g_GfxFramework.ProcessFrame();
    }
    g_GfxFramework.QueueFinish();

    FinalizeMii();
    FinalizeResFile();
    FinalizeResources();
    FinalizeFs();
} //NOLINT(impl/function_size)
