﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cinttypes>
#include "HidHandler.h"
#include <nn/nn_SdkLog.h>
#include "MediaPlayerMseSample.h"

void HidHandler::Initialize()
{
    nn::hid::InitializeDebugPad();
    nn::hid::InitializeNpad();
    nn::hid::SetSupportedNpadIdType(npadIds, npadIdCount);
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleJoyDual::Mask);
    current_playback_value = playback_1x;
}

void HidHandler::Update()
{
    movie::Status movieStatus = movie::Status_UnknownError;

    if(m_player == nullptr || m_mediaPlayerObserver == nullptr)
    {
        return;
    }

    nn::hid::GetDebugPadStates(debugPadState, nn::hid::DebugPadStateCountMax);

    allCurrentNpadButtonsPressed.Reset();
    allCurrentNpadButtonsHeld.Reset();

    int LStickX = 0;
    int LStickY = 0;
    int RStickX = 0;

    for (int i = 0; i < npadIdCount; i++)
    {
        oldNpadJoyDualState[i] = currentNpadJoyDualState[i];

        nn::hid::GetNpadState(&(currentNpadJoyDualState[i]), npadIds[i]);

        if ((currentNpadJoyDualState[i].buttons ^ oldNpadJoyDualState[i].buttons & currentNpadJoyDualState[i].buttons).IsAnyOn())
        {
            //NN_LOG("NpadJoyDual (%d) ", i);
            allCurrentNpadButtonsPressed |= currentNpadJoyDualState[i].buttons ^ oldNpadJoyDualState[i].buttons;
        }
        if ((currentNpadJoyDualState[i].buttons & oldNpadJoyDualState[i].buttons).IsAnyOn()) {
            allCurrentNpadButtonsHeld |= currentNpadJoyDualState[i].buttons & oldNpadJoyDualState[i].buttons;
        }

        if (i == 0 && currentNpadJoyDualState[i].analogStickR.x != 0)
        {
            auto clamp
                = [](double v, double low, double high) { if (v > high) { return high; } else
                                                          if (v < low) { return low; } else return v; };
            auto normalize_range_zero_one
                = [=](double v, double low, double high) { return (clamp(v,low,high) - low) / (high - low); };

            auto normalize_range_neg_one_one
                = [=](double v, double low, double high) { return 2.0 * (normalize_range_zero_one(v,low,high) - 0.5); };

            auto split_range
                = [](double value, double low, double mid, double high) {
                                   if (value < 0.5) {
                                       value *= 2.0f;
                                       return (low * (1.0f - value)) + (mid * value);
                                   } else {
                                       value -= 0.5f;
                                       value *= 2.0f;
                                       return (mid * (1.0f - value)) + (high * value);
                                   }};

            RStickX = currentNpadJoyDualState[i].analogStickR.x;
            float accel = normalize_range_neg_one_one(RStickX, -std::numeric_limits<int16_t>::max(),
                                                                std::numeric_limits<int16_t>::max());
            playback_slide = clamp(playback_slide + accel, -1000.0, 1000.0);
            double new_rate
                = split_range(normalize_range_zero_one(playback_slide, -1000.0, 1000.0)
                              , 0.05   // Lowest playback rate
                              , 1.00   // Middle
                              , 4.00); // Highest

            // float last_rate = m_player->GetPlaybackRate();
            m_player->SetPlaybackRate(new_rate);
            // float where_is_the_slide = normalize_range_neg_one_one(playback_slide, -1000.0, 1000.0);
            // NN_SDK_LOG("RStickX == %d, playback_slide == %f, where_is_the_slide == %f, accel == %f, new rate == %f\n",
            //             RStickX, playback_slide, where_is_the_slide, accel, new_rate);
        }
    }

    int64_t padSamplingCount = debugPadState[0].samplingNumber - prevPadSamplingNumber;
    if (padSamplingCount >= nn::hid::DebugPadStateCountMax)
    {
        padSamplingCount = nn::hid::DebugPadStateCountMax - 1;
    }
    prevPadSamplingNumber = debugPadState[0].samplingNumber;

    nn::hid::DebugPadButtonSet debugPadButtonDown(debugPadState[0].buttons & ~debugPadState[padSamplingCount].buttons);

    nn::hid::DebugPadButtonSet debugPadButtonHeld(debugPadState[0].buttons);

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::L >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::L>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: L button pressed!\n");
        int64_t pos = 0;
        movieStatus = m_player->GetCurrentPlaybackPosition(&pos);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
        }
        pos -= (padHoldScrollSpeed * 15);
        if( pos >= lastSeekPosition )
        {
            pos -= (padHoldScrollSpeed * 15);
        }
        if( pos < 0ll )
        {
            pos = 0ll;
            lastSeekPosition = 0ll;
        }
        lastSeekPosition = pos;

        NN_SDK_LOG("\n MediaPlayerMseSample:: SeekTo %" PRId64 "\n", pos);
        movieStatus = m_player->SeekTo(pos);
        if( movieStatus != movie::Status_Success )
        {
            if( movieStatus == movie::Status_EndOfStream && !loop )
            {
                NN_SDK_LOG("\n MediaPlayerMseSample:: Seek resulted in End of Stream \n");
                m_mediaPlayerObserver->SetPlaybackComplete();
            }
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to SeekTo: Error : 0x%x \n", movieStatus );
        }
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::R >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::R>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: R button pressed!\n");
        int64_t pos = 0;
        movieStatus = m_player->GetCurrentPlaybackPosition(&pos);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
        }
        pos += (padHoldScrollSpeed * 10);

        if( pos <= lastSeekPosition )
        {
            pos += (padHoldScrollSpeed * 10);
        }
        if( pos < 0ll )
        {
            pos = 0ll;
            lastSeekPosition = 0ll;
        }
        lastSeekPosition = pos;
        NN_SDK_LOG("\n MediaPlayerMseSample:: SeekTo %" PRId64 "\n", pos);
        int64_t trackDuration = 0;
        m_player->GetPlaybackDuration(&trackDuration);
        if(loop && pos >= trackDuration)
            pos = 0;
        movieStatus = m_player->SeekTo(pos);
        if( movieStatus != movie::Status_Success )
        {
            if( movieStatus == movie::Status_EndOfStream && !loop )
            {
                NN_SDK_LOG("\n MediaPlayerMseSample:: Seek resulted in End of Stream \n");
                m_mediaPlayerObserver->SetPlaybackComplete();
            }
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to SeekTo: Error : 0x%x \n", movieStatus );
        }
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::A >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::A>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: A button pressed!\n");

        auto state = movie::PlayerState::PlayerState_Error;
        paused = false;
        movieStatus = m_player->GetState(&state);
        if( movieStatus == movie::Status_Success )
        {
            if( state == movie::PlayerState::PlayerState_Paused )
                paused = true;

            paused = !paused;
            if( paused )
                NN_SDK_LOG("\n MediaPlayerMseSample:: Pause \n");
            else
                NN_SDK_LOG("\n MediaPlayerMseSample:: Resume \n");
            movieStatus = m_player->Pause(paused);
            if( movieStatus != movie::Status_Success )
                NN_SDK_LOG("\n MediaPlayerMseSample:: Failed to Pause: Error : 0x%x \n", movieStatus);
        }
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::B >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::B>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: B button pressed!\n");
        NN_SDK_LOG("\n MediaPlayerMseSample:: Stop \n");
        movieStatus = m_player->Stop();
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to Stop: Error : 0x%x   - Terminating PlayerSample \n", movieStatus );
            //goto Exit_PlayerSample;
        }
        m_mediaPlayerObserver->SetPlaybackComplete();

        return;
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::X >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::X>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: X button pressed, setting playback rate back to 1x. \n");
        playback_slide = 0.0;
        current_playback_value = playback_1x;
        m_player->SetPlaybackRate(1.0f);
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::Y >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::Y>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: Y button pressed!\n");
        NN_SDK_LOG("\n MediaPlayerMseSample:: current playback rate == %f\n", m_player->GetPlaybackRate());
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::Up >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::Up>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: Up button pressed, setting volume to 1.0!\n");
        m_player->SetVolume(1.0f);
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::Down >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::Down>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: Down button pressed, setting volume to 0.0!\n");
        m_player->SetVolume(0.0f);
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::Left >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::Left>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: Left button pressed!\n");
    } else if (debugPadButtonHeld.Test< ::nn::hid::DebugPadButton::Left >() ||
            allCurrentNpadButtonsHeld.Test<nn::hid::NpadJoyButton::Left>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: Left button held!\n");
        int64_t pos = 0;
        movieStatus = m_player->GetCurrentPlaybackPosition(&pos);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
        }
        pos -= (padHoldScrollSpeed * 2);
        if( pos >= lastSeekPosition )
        {
            pos -= (padHoldScrollSpeed * 2);
        }
        if( pos < 0ll )
        {
            pos = 0ll;
            lastSeekPosition = 0ll;
        }
        lastSeekPosition = pos;

        NN_SDK_LOG("\n MediaPlayerMseSample:: SeekTo %" PRId64 "\n", pos);
        movieStatus = m_player->SeekTo(pos);
        if( movieStatus != movie::Status_Success )
        {
            if( movieStatus == movie::Status_EndOfStream && !loop )
            {
                NN_SDK_LOG("\n MediaPlayerMseSample:: Seek resulted in End of Stream \n");
                m_mediaPlayerObserver->SetPlaybackComplete();
            }
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to SeekTo: Error : 0x%x \n", movieStatus );
        }
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::Right >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::Right>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: Right button pressed!\n");
    } else if (debugPadButtonHeld.Test< ::nn::hid::DebugPadButton::Right >() ||
            allCurrentNpadButtonsHeld.Test<nn::hid::NpadJoyButton::Right>()) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: Right button held!\n");
        int64_t pos = 0;
        movieStatus = m_player->GetCurrentPlaybackPosition(&pos);
        if( movieStatus != movie::Status_Success )
        {
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
        }
        pos += padHoldScrollSpeed;
        if( pos <= lastSeekPosition )
        {
            pos += padHoldScrollSpeed;
        }
        if( pos < 0ll )
        {
            pos = 0ll;
            lastSeekPosition = 0ll;
        }
        NN_SDK_LOG("\n MediaPlayerMseSample:: SeekTo %" PRId64 "\n", pos);
        lastSeekPosition = pos;
        int64_t trackDuration = 0;
        m_player->GetPlaybackDuration(&trackDuration);
        if(loop && pos >= trackDuration)
            pos = 0;
        movieStatus = m_player->SeekTo(pos);
        if( movieStatus != movie::Status_Success )
        {
            if( movieStatus == movie::Status_EndOfStream && !loop )
            {
                NN_SDK_LOG("\n MediaPlayerMseSample:: Seek resulted in End of Stream \n");
                m_mediaPlayerObserver->SetPlaybackComplete();
            }
            NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to SeekTo: Error : 0x%x \n", movieStatus );
        }
    }

    if (debugPadButtonDown.Test< ::nn::hid::DebugPadButton::ZR >() ||
            allCurrentNpadButtonsPressed.Test<nn::hid::NpadJoyButton::ZR>())
    {
        NN_SDK_LOG("\n MediaPlayerMseSample:: ZR button pressed, current !\n");
        playback_slide = 0.0;

        if (++current_playback_value == end(playback_values)) {
            current_playback_value = begin(playback_values);
        }
        NN_SDK_LOG("\n SetPlaybackRate(%f)\n", *current_playback_value);
        m_player->SetPlaybackRate(*current_playback_value);
    }

    if (debugPadState[0].analogStickR.x != 0 || debugPadState[0].analogStickR.y != 0) {
        NN_SDK_LOG("\n MediaPlayerMseSample:: R Stick is %d %d!\n", debugPadState[0].analogStickR.x, debugPadState[0].analogStickR.y);
    }

    if (debugPadState[0].analogStickL.x != 0 || debugPadState[0].analogStickL.y != 0 ||
            LStickX != 0 || LStickY != 0) {
        if (LStickX == 0) {
            LStickX = debugPadState[0].analogStickL.x;
        }
        if (LStickY == 0) {
            LStickY = debugPadState[0].analogStickL.y;
        }

        NN_SDK_LOG("\n MediaPlayerMseSample:: L Stick is %d %d!\n", LStickX, LStickY);
        if (abs(LStickX) > abs(LStickY))
        {
            if (LStickX > 0)
            {
                int64_t pos = 0;
                movieStatus = m_player->GetCurrentPlaybackPosition(&pos);
                if( movieStatus != movie::Status_Success )
                {
                    NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
                }
                pos += padHoldScrollSpeed;
                if( pos <= lastSeekPosition )
                {
                    pos += padHoldScrollSpeed;
                }
                if( pos < 0ll )
                {
                    pos = 0ll;
                    lastSeekPosition = 0ll;
                }
                lastSeekPosition = pos;

                NN_SDK_LOG("\n MediaPlayerMseSample:: SeekTo %" PRId64 "\n", pos);
                int64_t trackDuration = 0;
                m_player->GetPlaybackDuration(&trackDuration);
                if(loop && pos >= trackDuration)
                    pos = 0;
                movieStatus = m_player->SeekTo(pos);
                if( movieStatus != movie::Status_Success )
                {
                    if( movieStatus == movie::Status_EndOfStream && !loop )
                    {
                        NN_SDK_LOG("\n MediaPlayerMseSample:: Seek resulted in End of Stream \n");
                        m_mediaPlayerObserver->SetPlaybackComplete();
                    }
                    NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to SeekTo: Error : 0x%x \n", movieStatus );
                }
            }
            else
            {
                int64_t pos = 0;
                movieStatus = m_player->GetCurrentPlaybackPosition(&pos);
                if( movieStatus != movie::Status_Success )
                {
                    NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to GetCurrentPlaybackPosition: Error : 0x%x \n", movieStatus );
                }
                pos -= padHoldScrollSpeed;
                if( pos >= lastSeekPosition )
                {
                    pos -= padHoldScrollSpeed;
                }
                if( pos < 0ll )
                {
                    pos = 0ll;
                    lastSeekPosition = 0ll;
                }
                lastSeekPosition = pos;
                NN_SDK_LOG("\n MediaPlayerMseSample:: SeekTo %" PRId64 "\n", pos);
                movieStatus = m_player->SeekTo(pos);
                if( movieStatus != movie::Status_Success )
                {
                    if( movieStatus == movie::Status_EndOfStream && !loop )
                    {
                        NN_SDK_LOG("\n MediaPlayerMseSample:: Seek resulted in End of Stream \n");
                        m_mediaPlayerObserver->SetPlaybackComplete();
                    }
                    NN_SDK_LOG( "\n MediaPlayerMseSample:: Failed to SeekTo: Error : 0x%x \n", movieStatus );
                }
            }
        }
    }
} //NOLINT(impl/function_size)
