﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_KeyboardKey.h>
#include <nn/settings/settings_DebugPad.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nns/hid.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include <vector>
#include <string>

#include "PlayerUI_ApplicationHeap.h"
#include "PlayerUI_Color.h"
#include "PlayerUI_FontSystem.h"
#include "PlayerUI_GraphicsSystem.h"
#include "PlayerUI_WindowMessage.h"

#include "UpnpDeviceGame.h"
#include "UpnpDeviceGamePlayer.h"

namespace nns {
namespace libupnp {

enum UpnpDeviceGameUIValueTouched
{
    GameUIValueUnset = 0,
    GameUIValueA     = 1,
    GameUIValueB     = 2,
    GameUIValueC     = 3
};

class UpnpDeviceGameUI
{
    static const size_t              ApplicationHeapSize;
    static const int                 FrameBufferWidth;
    static const int                 FrameBufferHeight;
    static const int                 FrameRate;
    static const size_t              GraphicsMemorySize;

    static const float               AValueX;
    static const float               AValueY;
    static const float               BValueX;
    static const float               BValueY;
    static const float               CValueX;
    static const float               CValueY;

    static const float               ExitValueX;
    static const float               ExitValueY;
    static const nn::util::Unorm8x4  ExitValueColor;

    static const float               TouchBoxInitialX;
    static const float               TouchBoxInitialY;
    static const nn::util::Unorm8x4  TouchBoxColor;

    static const float               UpnpInfoX;
    static const float               UpnpInfoY;
    static const nn::util::Unorm8x4  UpnpInfoColor;

    static const float               UpnpInstructX;
    static const float               UpnpInstructY;
    static const nn::util::Unorm8x4  UpnpInstructColor;

    static const float               UIMessagesX;
    static const float               UIMessagesY;
    static const nn::util::Unorm8x4  UIMessagesColor;

private:
    hid::ControllerManager*          m_pControllerManager;
    ApplicationHeap*                 m_pApplicationHeap;
    GraphicsSystem*                  m_pGraphicsSystem;
    FontSystem*                      m_pFontSystem;
    bool                             m_GameUIShouldRun;

    float                            m_TouchBoxX;
    float                            m_TouchBoxY;
    bool                             m_DisplayTouchBox;

    UpnpDeviceGameUIValueTouched     m_LastTouchedValue;
    int                              m_AGameValue;
    nn::util::Unorm8x4               m_AValueColor;
    int                              m_BGameValue;
    nn::util::Unorm8x4               m_BValueColor;
    int                              m_CGameValue;
    nn::util::Unorm8x4               m_CValueColor;

    std::vector<std::string>           m_Messages;
    std::vector<std::string>::iterator m_MessagesIter;

    static void*                     NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT;
    static void                      NvFree(void* addr, void* userPtr) NN_NOEXCEPT;
    static void*                     NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT;
    static void                      PrintBox( nn::gfx::util::DebugFontTextWriter* pTextWriter, const char* pGameValue,
                                               const float offsetX,const float offsetY) NN_NOEXCEPT;
    static bool                      BoxIsTouched(const float srcX, const float srcY, const float touchX, const float touchY) NN_NOEXCEPT;
    static int                       ValueToColor(const int inValue, nn::util::Unorm8x4& outColor) NN_NOEXCEPT;
    static int                       NextValue(const int inValue) NN_NOEXCEPT;

    void                             ProcessTouchScreenState(const nns::hid::TouchScreen* const pTouchScreen) NN_NOEXCEPT;
    void                             ProcessDebugPadState(const nns::hid::DebugPad* const pDebugPad) NN_NOEXCEPT;
    int                              DrawBoxes(nn::gfx::util::DebugFontTextWriter* pTextWriter, UpnpDeviceGamePlayer* pGamePlayer) NN_NOEXCEPT;

public:
    UpnpDeviceGameUI() NN_NOEXCEPT;
    ~UpnpDeviceGameUI() NN_NOEXCEPT;

    int                              MainLoop(UpnpDeviceGamePlayer* pGamePlayer) NN_NOEXCEPT;
    void                             SetTouchBoxDisplayed(const bool isDisplayed) NN_NOEXCEPT;
};

}} // namespace nns / libupnp
