﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include "UpnpDeviceGameStateXML.h"
#include "UpnpDeviceGameState.h"

#include <cstring>

namespace nns {
namespace libupnp {

//***************************************************************
//*
//*                    G L O B A L s
//*
//***************************************************************

const char*  UpnpDeviceGameState::WebFileNintendoGame = "/nintendogame.xml";
const char*  UpnpDeviceGameState::WebFileNintendoGameControlScpd = "/nintendogamecontrolSCPD.xml";
const char*  UpnpDeviceGameState::WebFileNintendoGamePresentationHtml = "/nintendogamepres.html";

//***************************************************************
//*
//* UpnpDeviceGameState
//*
//***************************************************************

UpnpDeviceGameState::UpnpDeviceGameState() NN_NOEXCEPT: m_UpnpDeviceHandle(-1),
                                                        m_UpnpAdvertisementExpireTime(100),
                                                        m_IsUpnpRegistered(false)
{
    nn::os::InitializeMutex(&m_MutexLock, false, 0);

    memset(m_Udn, 0, sizeof(m_Udn));
    memset(m_ServiceId, 0, sizeof(m_ServiceId));
    memset(m_EventUrl, 0, sizeof(m_EventUrl));
    memset(m_ControlUrl, 0, sizeof(m_ControlUrl));
    memset(m_VariableValuesDef, 0, sizeof(m_VariableValuesDef));

    for(int idx = 0; idx < UpnpDeviceGameMaxVars; idx++)
    {
        m_VariableValues[idx] = m_VariableValuesDef[idx];
    }

    strcpy(m_VariableValues[0], "1");  // Green
    strcpy(m_VariableValues[1], "2");  // Orange
    strcpy(m_VariableValues[2], "3");  // Red
}

UpnpDeviceGameState::~UpnpDeviceGameState() NN_NOEXCEPT
{
    // If object is (still) Registered to UPNP..
    if (m_IsUpnpRegistered == true)
    {
        Stop();
    }

    nn::os::LockMutex(&m_MutexLock);

    memset(m_Udn, 0, sizeof(m_Udn));
    memset(m_ServiceId, 0, sizeof(m_ServiceId));
    memset(m_EventUrl, 0, sizeof(m_EventUrl));
    memset(m_ControlUrl, 0, sizeof(m_ControlUrl));

    nn::os::UnlockMutex(&m_MutexLock);

    // Destroy the Mutex
    nn::os::FinalizeMutex(&m_MutexLock);
}

int UpnpDeviceGameState::CreateVirtualFiles(nn::libupnp::VirtualDir& virtualDir) NN_NOEXCEPT
{
    UpnpWebFileHandle  fds = nullptr;
    int                ret = 0;
    int                rc = 0;

    do
    {
        // Open File: nintendogame.xml
        fds = virtualDir.Open(WebFileNintendoGame, UPNP_WRITE);
        if (fds == 0)
        {
            NN_LOG( "Failed to OPEN [%s] on the Virtual File System\n", WebFileNintendoGame);
            ret = -1;
            break;
        }

        // Write File: nintendogame.xml
        rc = virtualDir.Write(fds, g_pUpnpDeviceNintendoGameXml, strlen(g_pUpnpDeviceNintendoGameXml));
        if (rc != (int) strlen(g_pUpnpDeviceNintendoGameXml))
        {
            NN_LOG( "Failed to WRITE to [%s] on the Virtual File System\n", WebFileNintendoGame);
            ret = -1;
            break;
        }

        // Close file: nintendogame.xml
        virtualDir.Close(fds);
        fds = nullptr;

        // Open File: nintendogamecontrolSCPD.xml
        fds = virtualDir.Open(WebFileNintendoGameControlScpd, UPNP_WRITE);
        if (fds == 0)
        {
            NN_LOG( "Failed to OPEN [%s] on the Virtual File System\n", WebFileNintendoGameControlScpd);
            ret = -1;
            break;
        }

        // Write File: nintendogamecontrolSCPD.xml
        rc = virtualDir.Write(fds, g_pUpnpDeviceNintendoGameControlXml, strlen(g_pUpnpDeviceNintendoGameControlXml));
        if (rc != (int) strlen(g_pUpnpDeviceNintendoGameControlXml))
        {
            NN_LOG( "Failed to WRITE to [%s] on the Virtual File System\n", WebFileNintendoGameControlScpd);
            ret = -1;
            break;
        }

        // Close file: nintendogamecontrolSCPD.xml
        virtualDir.Close(fds);
        fds = nullptr;

        // Open File: nintendogamepres.html
        fds = virtualDir.Open(WebFileNintendoGamePresentationHtml, UPNP_WRITE);
        if (fds == 0)
        {
            NN_LOG( "Failed to OPEN [%s] on the Virtual File System\n", WebFileNintendoGamePresentationHtml);
            ret = -1;
            break;
        }

        // Write File: nintendogamepres.html
        rc = virtualDir.Write(fds, g_pUpnpDeviceNintendoGamePresentationHtml, strlen(g_pUpnpDeviceNintendoGamePresentationHtml));
        if (rc != (int) strlen(g_pUpnpDeviceNintendoGamePresentationHtml))
        {
            NN_LOG( "Failed to WRITE to [%s] on the Virtual File System\n", WebFileNintendoGamePresentationHtml);
            ret = -1;
            break;
        }

        // Close file: nintendogamepres.html
        virtualDir.Close(fds);
        fds = nullptr;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    // If a file is open - close it
    if (fds != nullptr)
    {
        virtualDir.Close(fds);
        fds = nullptr;
    }

    return(ret);
}

int UpnpDeviceGameState::Start(nn::libupnp::VirtualDir& virtualDir) NN_NOEXCEPT
{
    int         rc = 0;
    int         ret = 0;
    int         len = 0;
    char        descDocUrl[256];
    const char* pIpAddress = nullptr;
    uint16_t    port = 0;

    nn::os::LockMutex(&m_MutexLock);

    do
    {
        // If already registered?
        if (m_IsUpnpRegistered == true)
        {
            NN_LOG( "[UpnpDeviceGameState] is already initialized!\n");
            ret = -1;
            break;
        }

        pIpAddress = UpnpGetServerIpAddress();
        if (pIpAddress == nullptr)
        {
            NN_LOG( "Failed to get IP Address from libupnp\n" );
            ret = -1;
            break;
        }

        port = UpnpGetServerPort();
        if (port < 1)
        {
            NN_LOG( "Failed to get IP Port from libupnp\n" );
            ret = -1;
            break;
        }
        NN_LOG ("[UpnpDeviceGameState] Initializing on Network Interface / Port [%s:%d]\n", pIpAddress, port);

        // Create Virtual File System
        rc = CreateVirtualFiles(virtualDir);
        if (rc != 0)
        {
            NN_LOG ("[UpnpDeviceGameState] Failed to add web server files to the Virtual File System\n" );
            ret = -1;
            break;
        }

        len = snprintf(descDocUrl, sizeof(descDocUrl), "http://%s:%d%s", pIpAddress, port, WebFileNintendoGame);
        if (len >= sizeof(descDocUrl))
        {
            NN_LOG( "Content of Finished Document Description URL has length [%ld], but supporting buffer is size: [%ld]\n",
                                len, sizeof(descDocUrl));
            ret = -1;
            break;
        }

        NN_LOG( "[UpnpDeviceGameState] Registering the RootDevice\n\t with: %s\n", descDocUrl);

        rc = UpnpRegisterRootDevice(descDocUrl, &UpnpCallbackEventHandler, reinterpret_cast<void *>(this), &m_UpnpDeviceHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR  UpnpRegisterRootDevice(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            ret = -1;
            break;
        }
        m_IsUpnpRegistered = true;

        rc = FindAndParseService(descDocUrl);
        if (rc != 0)
        {
            NN_LOG("Failed calling FindAndParseService()!\n");
            ret = -1;
            break;
        }

        rc = UpnpSendAdvertisement(m_UpnpDeviceHandle, m_UpnpAdvertisementExpireTime);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR UpnpSendAdvertisement(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            ret = -1;
            break;
        }

        {
            NN_LOG ( "[UpnpDeviceGameState] Initialized\n" );
            NN_LOG( "m_Udn        : %s\n", m_Udn);
            NN_LOG( "m_ServiceId  : %s\n", m_ServiceId);
            NN_LOG( "m_EventUrl   : %s\n", m_EventUrl);
            NN_LOG( "m_ControlUrl : %s\n", m_ControlUrl);
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (ret != 0)
    {
        if (m_IsUpnpRegistered == true)
        {
            rc = UpnpUnRegisterRootDevice(m_UpnpDeviceHandle);
            if (rc != UPNP_E_SUCCESS)
            {
                NN_LOG ("** ERROR UpnpUnRegisterRootDevice(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
                // Fall Thru
            }
            m_IsUpnpRegistered = false;
        }
    }

    nn::os::UnlockMutex(&m_MutexLock);

    return(ret);
}

int UpnpDeviceGameState::Stop() NN_NOEXCEPT
{
    int   rc = -1;
    int   ret = 0;

    nn::os::LockMutex(&m_MutexLock);

    if (m_IsUpnpRegistered == true)
    {
        rc = UpnpUnRegisterRootDevice(m_UpnpDeviceHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR UpnpUnRegisterRootDevice(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
            ret = -1;
            // Fall Thru
        }
        m_IsUpnpRegistered = false;
    }

    nn::os::UnlockMutex(&m_MutexLock);

    return(ret);
}

int UpnpDeviceGameState::FindAndParseService(const char* pLocation) NN_NOEXCEPT
{
    IXML_Document*  pDesc = nullptr;
    IXML_NodeList*  pServiceList = nullptr;
    IXML_Element*   pService = nullptr;
    int             idx = 0;
    int             ret = 0;
    int             len = 0;
    char            tempValue[200];

    do
    {
        ret = UpnpDownloadXmlDoc(pLocation, &pDesc);
        if (ret != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR UpnpDownloadXmlDoc(): %d/%s\n", ret, UpnpGetErrorMessage(ret));
            ret = -1;
            break;
        }

        ret = GetFirstDocumentItem(m_Udn, sizeof(m_Udn), pDesc, "UDN");
        if (ret < 0)
        {
            NN_LOG( "Failed to get <UDN> from XML!\n");
            ret = -1;
            break;
        }

        pServiceList = GetNthServiceList(pDesc, 0);
        if (pServiceList == nullptr)
        {
            NN_LOG("Failed to get first <serviceList> from XML!\n");
            ret = -1;
            break;
        }

        len = ixmlNodeList_length(pServiceList);
        if (len != 1)
        {
            NN_LOG("<serviceList> XML has length: [%d] - but I expected [1] length!\n", len);
            ret = -1;
            break;
        }

        pService = reinterpret_cast<IXML_Element *>(ixmlNodeList_item(pServiceList, idx));
        if (pService == nullptr)
        {
            NN_LOG( "Failed to get first element from <serviceList> XML!\n");
            ret = -1;
            break;
        }

        ret = GetFirstElementItem(tempValue, sizeof(tempValue), pService, "serviceType");
        if (ret < 0)
        {
            NN_LOG( "Failed to get <serviceType> from XML!\n");
            ret = -1;
            break;
        }

        // Does this service type match our service?
        if ( (strlen(tempValue) == strlen(ServiceType))  &&
             (strcmp(tempValue, ServiceType) == 0)       )
        {
            ;   // Yes
        }
        else
        {
            NN_LOG( "<serviceType> from XML is [%s], but expected: [%s]!\n", tempValue, ServiceType);
            ret = -1;
            break;
        }
        NN_LOG( "Found service: [%s]\n", tempValue);

        ret = GetFirstElementItem(m_ServiceId, sizeof(m_ServiceId), pService, "serviceId");
        if (ret < 0)
        {
            NN_LOG( "Failed to get <serviceId> from XML <serviceList>!\n");
            ret = -1;
            break;
        }

        ret = GetFirstElementItem(tempValue, sizeof(tempValue), pService, "controlURL");
        if (ret < 0)
        {
            NN_LOG( "Failed to get <controlURL> from XML <serviceList>!\n");
            ret = -1;
            break;
        }

        ret = UpnpResolveURL(pLocation, tempValue, m_ControlUrl);
        if (ret < 0)
        {
            NN_LOG( "Failed calling UpnpResolveURL() - Generating UPnP Service Control URL for: [%s]\n", ServiceType);
            ret = -1;
            break;
        }

        ret = GetFirstElementItem(tempValue, sizeof(tempValue), pService, "eventSubURL");
        if (ret < 0)
        {
            NN_LOG( "Failed to get <controlURL> from XML <serviceList>!\n");
            ret = -1;
            break;
        }

        ret = UpnpResolveURL(pLocation, tempValue, m_EventUrl);
        if (ret < 0)
        {
            NN_LOG( "Failed calling UpnpResolveURL() - Generating UPnP Service Event URL for: [%s]\n", ServiceType);
            ret = -1;
            break;
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (pServiceList != nullptr)
    {
        ixmlNodeList_free(pServiceList);
        pServiceList = nullptr;
    }

    if (pDesc != nullptr)
    {
        ixmlDocument_free(pDesc);
        pDesc = nullptr;
    }

    return(ret);

}    // NOLINT(impl/function_size)

int UpnpDeviceGameState::NotifySubscribers() NN_NOEXCEPT
{
    int ret = 0;
    int rc = -1;

    NN_LOG( "NotifySubscribers() - Entry\n");

    do
    {
        rc = UpnpNotify( m_UpnpDeviceHandle,
                         m_Udn,
                         m_ServiceId,
                         (const char **) VariableNames,
                         (const char **) m_VariableValues,
                         UpnpDeviceGameMaxVars);

         if (rc != UPNP_E_SUCCESS)
         {
             NN_LOG( "** ERROR UpnpAcceptSubscription(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
             ret = -1;
             break;
         }

         // All Done
         break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG( "NotifySubscribers() - Exit\n");

    return(ret);
}

int UpnpDeviceGameState::DeviceSubscriptionRequest(struct Upnp_Subscription_Request* pEvent) NN_NOEXCEPT
{
    int     ret = 0;
    int     rc = -1;
    char*   pServiceId = nullptr;
    char*   pUdn = nullptr;
    char*   pSid = nullptr;

    NN_LOG( "DeviceSubscriptionRequest() - Entry\n");

    do
    {
        pServiceId = pEvent->ServiceId;
        pUdn       = pEvent->UDN;
        pSid       = pEvent->Sid;

        // Is this the right Device and Service?
        if ( (strcmp(pUdn, m_Udn) == 0)               &&
             (strcmp(pServiceId, m_ServiceId) == 0)   )
        {
            rc = UpnpAcceptSubscription( m_UpnpDeviceHandle,
                                         m_Udn,
                                         m_ServiceId,
                                         (const char **) VariableNames,
                                         (const char **) m_VariableValues,
                                         UpnpDeviceGameMaxVars,
                                         pSid);
            if (rc != UPNP_E_SUCCESS)
            {
                NN_LOG( "Error in UPNP_EVENT_SUBSCRIPTION_REQUEST callback:\n"
                        "   Error processing Subscription, Device: [%s], Service: [%s], Subscription: [%s]\n", pUdn, pServiceId, pSid);
                NN_LOG( "** ERROR UpnpAcceptSubscription(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
                ret = -1;
                break;
            }
        }
        else
        {
            NN_LOG( "Error in UPNP_EVENT_SUBSCRIPTION_REQUEST callback:\n"
                    "   Unknown Subscription: Device: [%s], Service: [%s[\n", pUdn, pServiceId);
            ret = -1;
            break;
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG( "DeviceSubscriptionRequest() - Exit\n");

    return(ret);
}

int UpnpDeviceGameState::DeviceGetVarRequest(struct Upnp_State_Var_Request* pEvent) NN_NOEXCEPT
{
    int     ret = -1;
    int     idx = 0;
    char*   pServiceId = nullptr;
    char*   pUdn = nullptr;
    char*   pStateVarName = nullptr;
    bool    foundVar = false;

    NN_LOG( "DeviceGetVarRequest() - Entry\n");

    do
    {
        pServiceId    = pEvent->ServiceID;
        pUdn          = pEvent->DevUDN;
        pStateVarName = pEvent->StateVarName;

        // Is this the right Device and Service?
        if ( (strcmp(pUdn, m_Udn) == 0)               &&
             (strcmp(pServiceId, m_ServiceId) == 0)   )
        {
            for (idx = 0; idx < UpnpDeviceGameMaxVars; idx++)
            {
                if ( (strlen(pStateVarName) == strlen(VariableNames[idx]))                           &&
                     (strncmp(pStateVarName, VariableNames[idx], strlen(VariableNames[idx])) == 0) )
                {
                    foundVar = true;
                    pEvent->CurrentVal = ixmlCloneDOMString(m_VariableValues[idx]);
                    break;
                }
            }
        }

        if (foundVar == true)
        {
            pEvent->ErrCode = UPNP_E_SUCCESS;
        }
        else
        {
            NN_LOG( "Error in UPNP_CONTROL_GET_VAR_REQUEST callback:\n"
                    "   Unknown variable name = %s\n", pStateVarName);
            pEvent->ErrCode = 404;
            strcpy(pEvent->ErrStr, "Invalid Variable");
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG( "DeviceGetVarRequest() - Exit\n");

    return(ret);
}

int UpnpDeviceGameState::DeviceActionRequest(struct Upnp_Action_Request* pEvent) NN_NOEXCEPT
{
    int     ret = 0;
    int     rc = -1;
    int     idx = -1;
    int     value = -1;
    char*   pServiceId = nullptr;
    char*   pUdn = nullptr;
    char*   pActionName = nullptr;
    char    newValue[UpnpDeviceGameVarLength];
    bool    actionFound = false;

    NN_LOG( "DeviceActionRequest() - Entry\n");

    do
    {
        pEvent->ErrCode = 0;
        pEvent->ActionResult = nullptr;

        pServiceId  = pEvent->ServiceID;
        pUdn        = pEvent->DevUDN;
        pActionName = pEvent->ActionName;

        // Is this the right Device and Service?
        if (  (strcmp(pUdn, m_Udn) == 0)               &&
              (strcmp(pServiceId, m_ServiceId) == 0)   )
        {
            NN_LOG( "UPNP_CONTROL_ACTION_REQUEST callback:\n"
                    "   Device: [%s], Service: [%s], Action: [%s]\n", pUdn, pServiceId, pActionName);
        }
        else
        {
            NN_LOG("Error in UPNP_CONTROL_ACTION_REQUEST callback:\n"
                    "   Unknown Device: [%s], Service: [%s]\n", pUdn, pServiceId);
            ret = -1;
            break;
        }

        NN_LOG( "Looking for Action Name: [%s]\n", pActionName );
        for (idx = 0; idx < UpnpDeviceGameMaxVars; idx++)
        {
            if (strcmp(pActionName, ActionSetNames[idx]) == 0)
            {
                actionFound = true;
                NN_LOG("MATCH: [%s], Existing Value [%s]\n", VariableNames[idx], m_VariableValues[idx]);

                ret = GetFirstDocumentItem(newValue, sizeof(newValue), pEvent->ActionRequest, VariableNames[idx]);
                if (ret != 0)
                {
                    NN_LOG("Failed calling 'GetFirstDocumentItem()' for ActionName [%s], Variable [%s]\n",
                                               pActionName, VariableNames[idx]);
                    break;
                }

                value = atoi(newValue);
                if ( (value >= UpnpDeviceGameColor_Min)   &&
                     (value <= UpnpDeviceGameColor_Max)   )
                {
                    ;   // Legal value
                }
                else
                {
                    NN_LOG("Ilegal value [%d] for [%s] received!  I only support values [%d] and [%d]\n",
                                    value, VariableNames[idx], UpnpDeviceGameColor_Min, UpnpDeviceGameColor_Max);
                    rc = UPNP_E_INVALID_PARAM;
                    ret = -1;
                    break;
                }

                // Save Legal Value
                memset(m_VariableValues[idx], 0, UpnpDeviceGameVarLength);
                memcpy(m_VariableValues[idx], newValue, strlen(newValue));

                // Display Changed Value
                NN_LOG("==> CHANGE: [%s], to new Value [%s]\n", VariableNames[idx], m_VariableValues[idx]);

                // Return Changed Value
                rc = UpnpAddToActionResponse(&pEvent->ActionResult, pActionName, ServiceType,
                                             VariableNames[idx], m_VariableValues[idx]);
                if (rc !=  UPNP_E_SUCCESS)
                {
                    NN_LOG( "** ERROR UpnpAddToActionResponse(): %d/%s", rc, UpnpGetErrorMessage(rc));
                    ret = -1;
                    break;
                }

                // Notify Subscribers that the value was changed
                rc = NotifySubscribers();
                if (rc != 0)
                {
                    NN_LOG( "Failed calling 'NotifySubscribers()' notifying change for [%s]\n", pActionName);
                    ret = -1;
                    break;
                }
                break;
            }

            NN_LOG( "Comparing to: [%s]\n", ActionGetNames[idx]);
            if (strcmp(pActionName, ActionGetNames[idx]) == 0)
            {
                actionFound = true;
                NN_LOG("==> MATCH FOUND: [%s], Value [%s]\n", VariableNames[idx], m_VariableValues[idx]);

                rc = UpnpAddToActionResponse(&pEvent->ActionResult, pActionName, ServiceType,
                                             VariableNames[idx], m_VariableValues[idx]);
                if (rc != UPNP_E_SUCCESS)
                {
                    NN_LOG( "** ERROR UpnpAddToActionResponse(): %d/%s", rc, UpnpGetErrorMessage(rc));
                    ret = -1;
                    break;
                }
                break;
            }
        }

        if (actionFound == true)
        {
            switch(rc)
            {
            case UPNP_E_SUCCESS:        pEvent->ErrCode = UPNP_E_SUCCESS;
                                        memset(pEvent->ErrStr, 0, sizeof(pEvent->ErrStr));
                                        break;
            case UPNP_E_INVALID_PARAM:  pEvent->ErrCode = 402;
                                        strcpy(pEvent->ErrStr, "Variable Invalid");
                                        break;
            default:                    pEvent->ErrCode = 501;
                                        strcpy(pEvent->ErrStr, "Internal Error");
                                        break;
            }
            // Fall Thru
        }
        else
        {
            NN_LOG("I never found ActionName [%s]!\n", pActionName);
            pEvent->ActionResult = nullptr;
            strcpy(pEvent->ErrStr, "Invalid Action");
            pEvent->ErrCode = 401;
            // Fall Thru
        }

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG( "DeviceActionRequest() - Exit\n");

    return(ret);

}    // NOLINT(impl/function_size)

int UpnpDeviceGameState::UpnpCallbackEventHandler(Upnp_EventType eventType, void* pEvent, void* pCookie) NN_NOEXCEPT
{
    UpnpDeviceGameState * pThisObj = reinterpret_cast<UpnpDeviceGameState *>(pCookie);

    switch (eventType)
    {
       case UPNP_EVENT_SUBSCRIPTION_REQUEST:
           {
               struct Upnp_Subscription_Request* pUpnpEvent = reinterpret_cast<Upnp_Subscription_Request *>(pEvent);
               int ret = -1;

               nn::os::LockMutex(&pThisObj->m_MutexLock);

               do
               {
                   ret = pThisObj->DeviceSubscriptionRequest(pUpnpEvent);
                   if (ret < 0)
                   {
                       NN_LOG("Failed calling 'DeviceSubscriptionRequest()'\n" );
                       break;
                   }

                   // All done
                   break;

               } while (NN_STATIC_CONDITION(false));

               nn::os::UnlockMutex(&pThisObj->m_MutexLock);
           }
           break;

       case UPNP_CONTROL_GET_VAR_REQUEST:
           {
               struct Upnp_State_Var_Request* pUpnpEvent = reinterpret_cast<Upnp_State_Var_Request*>(pEvent);
               int ret = -1;

               nn::os::LockMutex(&pThisObj->m_MutexLock);

               do
               {
                   ret = pThisObj->DeviceGetVarRequest(pUpnpEvent);
                   if (ret < 0)
                   {
                       NN_LOG("Failed calling 'DeviceGetVarRequest()'\n" );
                       break;
                   }

                   // All done
                   break;

               } while (NN_STATIC_CONDITION(false));

               nn::os::UnlockMutex(&pThisObj->m_MutexLock);
           }
           break;

       case UPNP_CONTROL_ACTION_REQUEST:
           {
               struct Upnp_Action_Request* pUpnpEvent = reinterpret_cast<Upnp_Action_Request*>(pEvent);
               int ret = -1;

               nn::os::LockMutex(&pThisObj->m_MutexLock);

               do
               {
                   ret = pThisObj->DeviceActionRequest(pUpnpEvent);
                   if (ret < 0)
                   {
                       NN_LOG("Failed calling 'DeviceActionRequest()'\n" );
                       break;
                   }

                   // All done
                   break;

               } while (NN_STATIC_CONDITION(false));

               nn::os::UnlockMutex(&pThisObj->m_MutexLock);
           }
           break;

        case UPNP_DISCOVERY_ADVERTISEMENT_ALIVE:
        case UPNP_DISCOVERY_SEARCH_RESULT:
        case UPNP_DISCOVERY_SEARCH_TIMEOUT:
        case UPNP_DISCOVERY_ADVERTISEMENT_BYEBYE:
        case UPNP_CONTROL_ACTION_COMPLETE:
        case UPNP_CONTROL_GET_VAR_COMPLETE:
        case UPNP_EVENT_RECEIVED:
        case UPNP_EVENT_RENEWAL_COMPLETE:
        case UPNP_EVENT_SUBSCRIBE_COMPLETE:
        case UPNP_EVENT_UNSUBSCRIBE_COMPLETE:
                break;

        default:
                NN_LOG( "Error in UpnpCallbackEventHandler: Unknown event type %d\n", eventType);
    }

    return 0;

}    // NOLINT(impl/function_size)

}}  // Namespace nns / libupnp
