﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <cstring>

#include "UpnpDeviceGame.h"

namespace nns {
namespace libupnp {

//***************************************************************
//*
//*                    G L O B A L s
//*
//***************************************************************

const char*  UpnpDeviceGame::DeviceType = "urn:schemas-upnp-org:device:nintendogame:1";
const char*  UpnpDeviceGame::ServiceType = "urn:schemas-upnp-org:service:nintendogamecontrol:1";

const char*  UpnpDeviceGame::VariableNames[]  = { "A", "B", "C" };
const char*  UpnpDeviceGame::ActionSetNames[] = { "SetA", "SetB", "SetC" };
const char*  UpnpDeviceGame::ActionGetNames[] = { "GetA", "GetB", "GetC" };

//***************************************************************
//*
//* UpnpDeviceGame
//*
//***************************************************************

IXML_NodeList* UpnpDeviceGame::GetNthServiceList(IXML_Document* pDesc, unsigned int inServicePos) NN_NOEXCEPT
{
    IXML_NodeList* pServiceList = nullptr;
    IXML_NodeList* pServlistnodelist = nullptr;
    IXML_Node*     pServlistnode = nullptr;

    pServlistnodelist = ixmlDocument_getElementsByTagName(pDesc, "serviceList");

    if ( (pServlistnodelist != nullptr)                             &&
         (ixmlNodeList_length(pServlistnodelist) != 0)              &&
         (inServicePos < ixmlNodeList_length(pServlistnodelist))    )
    {
        pServlistnode = ixmlNodeList_item(pServlistnodelist, inServicePos);
        if (pServlistnode != nullptr)
        {
            pServiceList = ixmlElement_getElementsByTagName(reinterpret_cast<IXML_Element*>(pServlistnode), "service");
        }
        else
        {
            pServiceList = nullptr;
        }
    }

    if (pServlistnodelist != nullptr)
    {
        ixmlNodeList_free(pServlistnodelist);
        pServlistnodelist = nullptr;
    }

    return(pServiceList);
}

int UpnpDeviceGame::GetFirstElementItem(char* pOutItem, size_t outSize, IXML_Element* pElement, const char* pItem) NN_NOEXCEPT
{
    IXML_NodeList* pNodeList = nullptr;
    IXML_Node*     pTextNode = nullptr;
    IXML_Node*     pTmpNode = nullptr;
    size_t         itemSize = 0;
    int            ret = 0;

    do
    {
        pNodeList = ixmlElement_getElementsByTagName(pElement, reinterpret_cast<const DOMString>(pItem));
        if (pNodeList == nullptr)
        {
            NN_LOG("Error finding [%s] in XML Node\n", pItem);
            ret = -1;
            break;
        }

        pTmpNode = ixmlNodeList_item(pNodeList, 0);
        if (pTmpNode == nullptr)
        {
            NN_LOG("Error finding [%s] value in XML Node\n", pItem);
            ret = -1;
            break;
        }

        pTextNode = ixmlNode_getFirstChild(pTmpNode);
        if (pTextNode == nullptr)
        {
            NN_LOG("Error failed to get 'textnode' from IXML_Node\n" );
            ret = -1;
            break;
        }

        itemSize = strlen(ixmlNode_getNodeValue(pTextNode));
        if (itemSize > outSize - 1)
        {
            NN_LOG( "ixmlNode_getNodeValue(): Error text length: [%ld] exceeds (out variable) max size: [%ld]\n", itemSize, outSize - 1);
            ret = -1;
            break;
        }

        memset(pOutItem, 0, outSize);
        memcpy(pOutItem, ixmlNode_getNodeValue(pTextNode), itemSize);

        // Free the Nodelist
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (pNodeList != nullptr)
    {
        ixmlNodeList_free(pNodeList);
    }

    return(ret);
}

int UpnpDeviceGame::GetFirstDocumentItem(char* pOutItem, size_t outSize, IXML_Document* pDesc, const char* pItem) NN_NOEXCEPT
{
    IXML_NodeList* pNodeList = nullptr;
    IXML_Node*     pTextNode = nullptr;
    IXML_Node*     pTmpNode = nullptr;
    size_t         itemSize = 0;
    int            ret = 0;

    do
    {
        if (pOutItem == nullptr || outSize < 2)
        {
            NN_LOG("Return value should be (NOT) be NULL before use, or outSize [%ld] is too small\n", outSize);
            ret = -1;
            break;
        }

        pNodeList = ixmlDocument_getElementsByTagName(pDesc, reinterpret_cast<const DOMString>(pItem));
        if (pNodeList == nullptr)
        {
            NN_LOG( "ixmlDocument_getElementsByTagName(): Error finding [%s] in XML Node\n", pItem);
            ret = -1;
            break;
        }

        pTmpNode = ixmlNodeList_item(pNodeList, 0);
        if (pTmpNode == nullptr)
        {
            NN_LOG( "ixmlNodeList_item(): Error getting Node from Nodelist\n" );
            ret = -1;
            break;
        }

        pTextNode = ixmlNode_getFirstChild(pTmpNode);
        if (pTextNode == nullptr)
        {
            NN_LOG( "ixmlNode_getFirstChild(): Error getting First Child from Node\n" );
            ret = -1;
            break;
        }

        itemSize = strlen(ixmlNode_getNodeValue(pTextNode));
        if (itemSize > outSize - 1)
        {
            NN_LOG( "ixmlNode_getNodeValue(): Error text length: [%ld] exceeds (out variable) max size: [%ld]\n", itemSize, outSize - 1);
            ret = -1;
            break;
        }

        memset(pOutItem, 0, outSize);
        memcpy(pOutItem, ixmlNode_getNodeValue(pTextNode), itemSize);

        // Free the Nodelist
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    if (pNodeList != nullptr)
    {
        ixmlNodeList_free(pNodeList);
        pNodeList = nullptr;
    }

    return(ret);
}

int UpnpDeviceGame::GetElementValue(char* pOutItem, size_t outSize, IXML_Element* pElement) NN_NOEXCEPT
{
    IXML_Node * pChild = nullptr;
    size_t      itemSize = 0;
    int         ret = 0;

    do
    {
        if (pOutItem == nullptr || outSize < 2)
        {
            NN_LOG("Return value should be (NOT) be NULL before use\n");
            ret = -1;
            break;
        }

        pChild = ixmlNode_getFirstChild((IXML_Node *) pElement);
        if (pChild == nullptr)
        {
            NN_LOG( "ixmlNode_getFirstChild(): Error getting First Child from Node\n" );
            ret = -1;
            break;
        }

        if (ixmlNode_getNodeType(pChild) != eTEXT_NODE)
        {
            NN_LOG( "ixmlNode_getNodeType(): Error Node type (IS NOT) 'eTEXT_NODE' - Value is: [%d]\n", ixmlNode_getNodeType(pChild));
            ret = -1;
            break;
        }

        itemSize = strlen(ixmlNode_getNodeValue(pChild));
        if (itemSize > outSize)
        {
            NN_LOG( "ixmlNode_getNodeValue(): Error text length: [%ld] exceeds (out variable) max size: [%ld]\n", itemSize, outSize);
            ret = -1;
            break;
        }

        memset(pOutItem, 0, outSize);
        memcpy(pOutItem, ixmlNode_getNodeValue(pChild), itemSize);

        // All Done
        break;

    } while (NN_STATIC_CONDITION(false));

    return(ret);
}

}} // namespace nns / libupnp
