﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/init.h>
#include <nn/os.h>

#include <nn/socket.h>
#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>          // nn::nifm::GetCurrentIpConfigInfo

#include <upnp/upnp.h>
#include <upnp/upnpdebug.h>
#include <upnp/upnptools.h>
#include <upnp/nnlibupnp_VirtualDir.h>

#include <cstdlib>

#include "nintendoswitch.h"

/**
 * @examplesource{UpnpAdvertise.cpp,PageSampleUpnpAdvertise}
 *
 * @brief
 * Use libupnp to adverterise the sample on a LAN
 */

/**
 * @page PageSampleUpnpAdvertise UpnpAdvertise
 * @tableofcontents
 *
 * @brief
 *  Documentation about the Sample program which uses libupnp to adverterise the sample on a LAN
 *
 * @section PageSampleUpnpAdvertise_SectionBrief Overview
 *  First this sample initializes network connectivity by using the NIFM library.  Then the libupnp library is
 *  initialized.  A virtual file system used to support the libupnp web server is then created and assigned to
 *  the libupnp library.  The sample then creates a new file called '/nintendoswitch.xml' on the virtual file
 *  system.  The '/nintendoswitch.xml' file describes the advertised UPnP device in XML format as defined by
 *  http://www.upnp.org/specs/arch/UPnP-arch-DeviceArchitecture-v1.0-20081015.pdf - Section 2.1.  The sample
 *  then begins advertising itself to the LAN for the next 60 seconds, pretending to be a Nintendo Switch device.
 *  If you open Windows Network Neighborhood during the 60 second interval while the sample is running you will
 *  find the sample listed in the output as "Nintendo Switch" under "Network Infrastructure" (Windows 10).
 *  Windows Network Neighborhood documentation can be found here: https://en.wikipedia.org/wiki/My_Network_Places
 *
 * @section UpnpAdvertise_SectionFileStructure File Structure
 *  This sample program can be found at
 *  @link ../../../Samples/Sources/Applications/Libupnp/UpnpAdvertise Samples/Sources/Applications/Libupnp/UpnpAdvertise @endlink.
 *
 * @section PageSampleUpnpAdvertise_SectionNecessaryEnvironment System Requirements
 *  It is required to import network setting by SettingManager prior to running this sample.
 *  Please refer @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} for further information.
 *
 * @section PageSampleUpnpAdvertise_SectionHowToOperate Operation Procedure
 *  Run the sample.  For the next 60 seconds the sample will advertise itself using UPnP to the LAN.  Within 60 seconds
 *  of launching the program, look for "Nintendo Switch" under "Network Infrastructure" in the Windows Network
 *  Neighborhood (Windows 10).  The sample will exit after 60 seconds.
 *
 *  NOTE: By default the sample will use the Nifm negotiated network interface to send and receive libupnp generated
 *  multicast network traffic.  If the sample is running on a host that supports multiple network interfaces the
 *  desired network interface to use can be selected by passing the command line argument "-interface <Ipv4 Address>",
 *  where <Ipv4 Address> is assigned to the host network interface that the sample should use.

 * @section PageSampleUpnpAdvertise_SectionPrecaution Precautions
 *  None.
 *
 * @section PageSampleUpnpAdvertise_SectionHowToExecute Execution Procedure
 *  Build the Visual Studio Solution in the desired configuration and run it.
 *
 * @section PageSampleUpnpAdvertise_SectionDetail Detail
 *  Here is the sequence of this sample program.
 *
 *  - Sends a request to use the network to the NIFM library
 *  - Initializes the socket library
 *  - The libupnp library is initialized with the host network interface to use for sending and receiving multicast traffic
 *  - Initializes a "virtual filesystem" to support the libupnp "web server".
 *  - Creates the file '/nintendoswitch.xml' on the virtual file system. The '/nintendoswitch.xml' file describes the advertised UPnP
 *    network device in XML format as defined by http://www.upnp.org/specs/arch/UPnP-arch-DeviceArchitecture-v1.0-20081015.pdf
 *    Section 2.1.
 *    Enables the libupnp "web server" functionality.
 *  - Sample advertises itself as a "Nintendo Switch" to the LAN using libupnp.
 *  - Sample sleeps for 60 seconds - other background threads continue to run in the sample which continuosly advertise the sample
 *    to the LAN.
 *  - Sample stops advertising itself to the local LAN using libupnp.
 *  - Sample finishes.
 *
 *  Please enable "#define DETAILED_LIBUPNP_DEBUGGING" if you are interested in seeing debugging output from libupnp.
 *  The libupnp library supports the displaying of detailed debugging information to NN_LOG.  By default only CRITICAL
 *  error information from the libupnp library is displayed.
 *
 * @subsection PageSampleUpnpAdvertise_SectionSampleProgram Sample Program
 *  Below is the source code for the main tutorial file for this sample.
 *
 *  UpnpAdvertise.cpp
 *  @includelineno UpnpAdvertise.cpp

 * @subsection PageUpnpAdvertise_SectionExpectedOutput Expected Output
 * @verbinclude UpnpAdvertise_OutputExample.txt
 */

// ------------------------------------------------------------------------------------------------
// Build flags
// ------------------------------------------------------------------------------------------------
// #define DETAILED_LIBUPNP_DEBUGGING

namespace nns {
namespace libupnp {

// Constants
const size_t                        ThreadStackSize = 8192;
const int                           TimeoutOnNicConfigurationInSec = 15;
const int                           UpnpAdvertisementExpireTime = 100;
uint16_t                            UpnpServerPort = 49152;

// Socket memory
nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

// Thread Memory
NN_OS_ALIGNAS_THREAD_STACK char     g_ThreadStack[ ThreadStackSize ];

// Nifm values
struct in_addr                      g_NifmIpv4Addr = { 0 };
struct in_addr                      g_NifmIpv4Mask = { 0 };
struct in_addr                      g_NifmIpv4Gateway = { 0 };
struct in_addr                      g_NifmIpv4Dns1 = { 0 };
struct in_addr                      g_NifmIpv4Dns2 = { 0 };
bool                                g_IsNetworkInitialized = false;

// UPnP values
struct UpnpVirtualDirCallbacks      g_Callbacks;
nn::libupnp::VirtualDir             g_VirtualDirectory;
UpnpDevice_Handle                   g_UpnpDeviceHandle = -1;
bool                                g_IsUpnpInitialized = false;
bool                                g_IsUpnpRootDeviceRegistered = false;

// ------------------------------------------------------------------------------------------------
// Setup network interface
// ------------------------------------------------------------------------------------------------
int SetupNetwork() NN_NOEXCEPT
{
    nn::Result     result = nn::ResultSuccess();
    uint32_t       timeoutSec = 0;
    int            ret = 0;
    bool           isSocketLibInitialized = false;
    bool           isNifmLibInitialized = false;

    do
    {
        // If already initialized
        if (g_IsNetworkInitialized == true)
        {
            NN_LOG("Network is already initialized!\n");
            break;
        }

        // Initialize NIFM
        result = nn::nifm::Initialize();
        if(result.IsFailure())
        {
            NN_LOG("Failed calling: 'nn::nifm::Initialize()'\n");
            ret = -1;
            break;
        }

        // Submit the NIFM Network Request
        nn::nifm::SubmitNetworkRequest();

        // Wait for NIFM to go online
        while(nn::nifm::IsNetworkRequestOnHold())
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            if(timeoutSec++ > TimeoutOnNicConfigurationInSec)
            {
                NN_LOG("Failed to setup NIFM network interface.\n");
                ret = -1;
                break;
            }
        }

        // Did NIFM request timeout?
        if (ret < 0)
        {
            break;  // Yes
        }

        // Is the NIFM Network Available?
        if(nn::nifm::IsNetworkAvailable() == 0)
        {
            NN_LOG("Network is not available.\n");
            ret = -1;
            break;
        }
        isNifmLibInitialized = true;

        // Ask NIFM for our current (primary interface) IP Address
        nn::nifm::GetCurrentIpConfigInfo( &g_NifmIpv4Addr, &g_NifmIpv4Mask, &g_NifmIpv4Gateway, &g_NifmIpv4Dns1, &g_NifmIpv4Dns2 );

        NN_LOG( "Network Interface is UP!\n" );
        NN_LOG( "NIFM Address Ipv4      : %s\n", nn::socket::InetNtoa( g_NifmIpv4Addr ));
        NN_LOG( "NIFM Network Mask Ipv4 : %s\n", nn::socket::InetNtoa( g_NifmIpv4Mask ));
        NN_LOG( "NIFM Gateway Ipv4      : %s\n", nn::socket::InetNtoa( g_NifmIpv4Gateway));
        NN_LOG( "NIFM DNS1 Ipv4         : %s\n", nn::socket::InetNtoa( g_NifmIpv4Dns1 ));
        NN_LOG( "NIFM DNS2 Ipv4         : %s\n", nn::socket::InetNtoa( g_NifmIpv4Dns2 ));

        // Initialize the socket library
        result = nn::socket::Initialize(g_SocketConfigWithMemory);
        if(result.IsFailure())
        {
            NN_LOG("Failed to initialize socket library.\n");
            ret = -1;
            break;
        }
        isSocketLibInitialized = true;

        // Successfully initialized the network
        g_IsNetworkInitialized = true;

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    // If failed - disable network
    if (ret != 0)
    {
        if (isSocketLibInitialized == true)
        {
            nn::socket::Finalize();
            isSocketLibInitialized  = false;
            ret = -1;
        }

        if (isNifmLibInitialized == true)
        {
            nn::nifm::CancelNetworkRequest();
            isNifmLibInitialized = false;
            ret = -1;
        }

        g_IsNetworkInitialized = false;
    }

    return(ret);
}

// ------------------------------------------------------------------------------------------------
// Finalize Network
// ------------------------------------------------------------------------------------------------
int FinalizeNetwork() NN_NOEXCEPT
{
    nn::Result     result = nn::ResultSuccess();
    int            ret = 0;

    if (g_IsNetworkInitialized == true)
    {
        result = nn::socket::Finalize();
        if (result.IsFailure())
        {
            NN_LOG("Failed calling 'nn::socket::Finalize()'!\n");
        }

        nn::nifm::CancelNetworkRequest();
        g_IsNetworkInitialized = false;
    }

    return(ret);
}

int UpnpCallbackEventHandler(Upnp_EventType eventType, void* pEvent, void* pCookie) NN_NOEXCEPT
{
    switch (eventType)
    {
        case UPNP_EVENT_SUBSCRIPTION_REQUEST:
        case UPNP_CONTROL_GET_VAR_REQUEST:
        case UPNP_CONTROL_ACTION_REQUEST:
        case UPNP_DISCOVERY_ADVERTISEMENT_ALIVE:
        case UPNP_DISCOVERY_SEARCH_RESULT:
        case UPNP_DISCOVERY_SEARCH_TIMEOUT:
        case UPNP_DISCOVERY_ADVERTISEMENT_BYEBYE:
        case UPNP_CONTROL_ACTION_COMPLETE:
        case UPNP_CONTROL_GET_VAR_COMPLETE:
        case UPNP_EVENT_RECEIVED:
        case UPNP_EVENT_RENEWAL_COMPLETE:
        case UPNP_EVENT_SUBSCRIBE_COMPLETE:
        case UPNP_EVENT_UNSUBSCRIBE_COMPLETE:
                break;

        default:
                NN_LOG( "Error in TvUpnpCallbackEventHandler: unknown event type %d\n", eventType);
    }
    return 0;
}

void CommandLoop(void* args) NN_NOEXCEPT
{
    NN_LOG( "***********************************************************************************************\n" );
    NN_LOG( "        Sample will advertise itself as a UPnP Nintendo Switch for the next [60] seconds!\n" );
    NN_LOG( "***********************************************************************************************\n\n" );

    nn::os::SleepThread(nn::TimeSpan::FromSeconds(60));

    NN_LOG( "***********************************************************************************************\n" );
    NN_LOG( "                  Sample has stopped advertising itself to UPnP\n" );
    NN_LOG( "***********************************************************************************************\n" );

    return;
}

int MakeVirtualFileSystem() NN_NOEXCEPT
{
    UpnpWebFileHandle    fds;
    int                  rc;

    g_Callbacks.get_info = &g_VirtualDirectory.GetInfo;
    g_Callbacks.open     = &g_VirtualDirectory.Open;
    g_Callbacks.read     = &g_VirtualDirectory.Read;
    g_Callbacks.write    = &g_VirtualDirectory.Write;
    g_Callbacks.seek     = &g_VirtualDirectory.Seek;
    g_Callbacks.close    = &g_VirtualDirectory.Close;

    rc = UpnpSetVirtualDirCallbacks(&g_Callbacks);
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpSetVirtualDirCallbacks(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    rc = UpnpEnableWebserver(1);
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpEnableWebserver(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    rc = UpnpAddVirtualDir("/");
    if ( rc != UPNP_E_SUCCESS )
    {
        NN_LOG ("** ERROR UpnpEnableWebserver(): %d/%s", rc, UpnpGetErrorMessage(rc));
        return(-1);
    }

    // Open File: nintendoswitch.xml
    fds = g_VirtualDirectory.Open( "/nintendoswitch.xml", UPNP_WRITE);
    if (fds == 0)
    {
        NN_LOG( "Failed to open nintendoswitch.xml on the virtual file system!\n" );
        return(-1);
    }

    // Write File: nintendoswitch.xml
    rc = g_VirtualDirectory.Write(fds, g_pUpnpAdvertiseNintendoSwitch, strlen(g_pUpnpAdvertiseNintendoSwitch));
    if ( rc != (int) strlen(g_pUpnpAdvertiseNintendoSwitch))
    {
        NN_LOG( "Failed to write nintendoswitch.xml\n" );
        g_VirtualDirectory.Close(fds);
        return(-1);
    }

    // Close File: nintendoswitch.xml
    g_VirtualDirectory.Close(fds);

    // Open File: pres.html
    fds = g_VirtualDirectory.Open( "/pres.html", UPNP_WRITE);
    if (fds == 0)
    {
        NN_LOG( "Failed to open /pres.html on the virtual file system!\n" );
        return(-1);
    }

    // Write File: pres.html
    rc = g_VirtualDirectory.Write(fds, g_pUpnpAdvertisePresentationHtml, strlen(g_pUpnpAdvertisePresentationHtml));
    if ( rc != (int) strlen(g_pUpnpAdvertisePresentationHtml))
    {
        NN_LOG( "Failed to write /pres.html\n" );
        g_VirtualDirectory.Close(fds);
        return(-1);
    }

    // Close File: pres.html
    g_VirtualDirectory.Close(fds);

    return(0);
}

// ------------------------------------------------------------------------------------------------
// Parse Arguments
// ------------------------------------------------------------------------------------------------
bool ParseArgs(char** pInterface) NN_NOEXCEPT
{
    size_t        ifaceSize = 0;
    int           options = 0;
    int           argc = nn::os::GetHostArgc();
    char**        argv = nn::os::GetHostArgv();
    const char*   pChoosenInterface = nullptr;
    bool          ret = true;

    // Initialize passed variables
    *pInterface = nullptr;

    do
    {
        // Look for a passed command line option
        for(options = 0; options < argc; options++)
        {
            if(strcmp(argv[options], "-interface" ) == 0)  // Optional: Network Interface
            {
                options++;
                if(options >= argc)
                {
                    NN_LOG( "Arg: -interface requires an additional option, which is an IPv4 address assigned to this host\n");
                    ret = false;
                    break;
                }

                // Save choosen network interface
                pChoosenInterface = argv[options];
                break;
            }
        }

        // If parsing options failed..
        if (ret == false)
        {
            break;
        }

        // If NO network interface specified
        if (pChoosenInterface == nullptr)
        {
            break;
        }

        // Figure out size of the network Interface
        ifaceSize = strlen(pChoosenInterface) + 1;

        // Allocate Memory to hold network Interface
        *pInterface = new char[ifaceSize];
        if (*pInterface == nullptr)
        {
            NN_LOG("Failed calling `new` for network interface argument [%s], size [%d]\n", pChoosenInterface, ifaceSize);
            ret = false;
            break;
        }

        // Copy the network Interface
        memset(*pInterface, 0, ifaceSize);
        memcpy(*pInterface, pChoosenInterface, strlen(pChoosenInterface));

    } while (NN_STATIC_CONDITION(false));

    if (ret == false)
    {
        if (*pInterface != nullptr)
        {
            delete [] *pInterface;
            *pInterface = nullptr;
        }
    }

    return(ret);
}
// ------------------------------------------------------------------------------------------------
// Main
// ------------------------------------------------------------------------------------------------
void mainLoop() NN_NOEXCEPT
{
    nn::Result          result;
    nn::os::ThreadType  cmdloopThread;
    size_t              len = 0;
    int                 rc = UPNP_E_SUCCESS;
    char                descDocUrl[1024];
    char*               pNetworkIface = nullptr;
    char*               pTmpNetworkIface = nullptr;
    char*               pUpnpIpAddress = nullptr;
    uint16_t            upnpPort = 0;

    do
    {
       // Determine if caller specified a network interface
       if (ParseArgs(&pNetworkIface) == false)
        {
            NN_LOG("\nError: Invalid arguments.\n");
            NN_LOG("\nUsage:\n");
            NN_LOG("  -interface <ipv4 address> - Network interface to initialize libupnp on. (optional)\n");
            break;
        }

        // Setup network interface
        NN_LOG ( "Initializing Nifm Interface... \n");
        rc = SetupNetwork();
        if (rc != 0)
        {
            NN_LOG("Failed to enable network interface\n");
            break;
        }

        // If caller (DID NOT) specify a network interface..
        if (pNetworkIface == nullptr)
        {
            // Turn the Nifm negotiated IPv4 Address into ASCII
            pTmpNetworkIface = nn::socket::InetNtoa(g_NifmIpv4Addr);
            if (pTmpNetworkIface == nullptr)
            {
                NN_LOG( "Failed to translate IPv4 address: [0x%08x] into an ASCII IP Address\n", nn::socket::InetNtohl(g_NifmIpv4Addr.s_addr));
                break;
            }

            // Figure out size of the network Interface
            len = strlen(pTmpNetworkIface) + 1;

            // Allocate Memory to hold network Interface
            pNetworkIface = new char[len];
            if (pNetworkIface == nullptr)
            {
                NN_LOG("Failed calling `new` for network interface argument [%s], size [%d]\n", pTmpNetworkIface, len);
                break;
            }

            // Copy the network Interface
            memset(pNetworkIface, 0, len);
            memcpy(pNetworkIface, pTmpNetworkIface, strlen(pTmpNetworkIface));

            NN_LOG( "Sample will use [Nifm] assigned network interface [%s]\n", pNetworkIface);
        }
        else
        {
            NN_LOG( "Sample will use [Command Line] assigned network interface [%s]\n", pNetworkIface);
        }

    // libupnp debugging flags:  UPNP_CRITICAL, UPNP_PACKET, UPNP_INFO, UPNP_ALL
#ifdef DETAILED_LIBUPNP_DEBUGGING
        UpnpSetLogLevel(UPNP_ALL);
#else
        UpnpSetLogLevel(UPNP_CRITICAL);
#endif

        NN_LOG ("Initializing libupnp: (Requesting) IPv4 Network Interface Address:Port: [%s:%d]\n", pNetworkIface, UpnpServerPort);

        rc = UpnpInit(pNetworkIface, UpnpServerPort);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG ("** ERROR UpnpInit(): %d/%s", rc, UpnpGetErrorMessage(rc));
            break;
        }
        g_IsUpnpInitialized = true;

        pUpnpIpAddress = UpnpGetServerIpAddress();
        if (pUpnpIpAddress == nullptr)
        {
            NN_LOG( "UpnpGetServerIpAddress() returned a (NULL) Network Interface Address - Can't continue\n");
            break;
        }

        upnpPort = UpnpGetServerPort();
        if (upnpPort < 1)
        {
            NN_LOG( "UpnpGetServerPort() returned a (NULL) Server Port - Can't continue\n");
            break;
        }
        NN_LOG ("Successfully Initialized libupnp: (Actual) IPv4 Network Interface Address:Port: [%s:%d]\n", pUpnpIpAddress, upnpPort);

        rc = MakeVirtualFileSystem();
        if(rc != 0)
        {
            NN_LOG ("** ERROR: Failed calling MakeFileSystem()\n" );
            break;
        }

        len = snprintf(descDocUrl, sizeof(descDocUrl), "http://%s:%d/nintendoswitch.xml", pUpnpIpAddress, upnpPort);
        if (len >= sizeof(descDocUrl))
        {
            NN_LOG( "Content of Finished Document Description URL has length [%ld], but supporting buffer is size: [%ld]\n",
                                len, sizeof(descDocUrl));
            break;
        }

        NN_LOG( "Registering the RootDevice\n\t with: %s\n", descDocUrl);

        rc = UpnpRegisterRootDevice(descDocUrl, UpnpCallbackEventHandler, &g_UpnpDeviceHandle, &g_UpnpDeviceHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** ERROR  UpnpRegisterRootDevice(): %d/%s\n", rc, UpnpGetErrorMessage (rc));
            break;
        }
        g_IsUpnpRootDeviceRegistered = true;

        rc = UpnpSendAdvertisement(g_UpnpDeviceHandle, UpnpAdvertisementExpireTime);
        if ( rc != UPNP_E_SUCCESS )
        {
            NN_LOG ("** ERROR UpnpSendAdvertisement(): %d/%s", rc, UpnpGetErrorMessage(rc));
            break;
        }

        // Create the Device Command Loop Thread
        result = nn::os::CreateThread(&cmdloopThread, CommandLoop, NULL, g_ThreadStack, ThreadStackSize, nn::os::DefaultThreadPriority);
        if (result.IsFailure())
        {
            NN_LOG ("** ERROR Failed calling 'nn::os::CreateThread()'\n");
            break;
        }

        // Start the thread
        nn::os::StartThread(&cmdloopThread);

        // Wait for the Thread
        nn::os::WaitThread(&cmdloopThread);

        // Destroy the Thread
        nn::os::DestroyThread(&cmdloopThread);

        // All done
        break;

    } while (NN_STATIC_CONDITION(false));

    NN_LOG ("Exiting!\n");

    if (g_IsUpnpRootDeviceRegistered == true)
    {
        rc = UpnpUnRegisterRootDevice(g_UpnpDeviceHandle);
        if (rc != UPNP_E_SUCCESS)
        {
            NN_LOG("** ERROR UpnpUnRegisterRootDevice(): %d/%s\n", rc, UpnpGetErrorMessage(rc));
        }
        g_IsUpnpRootDeviceRegistered = false;
    }

    if (g_IsUpnpInitialized == true)
    {
        (void) UpnpFinish();
        g_IsUpnpInitialized = false;
    }

    if (pNetworkIface != nullptr)
    {
        delete [] pNetworkIface;
        pNetworkIface = nullptr;
    }

    FinalizeNetwork();

}   // NOLINT(impl/function_size)

}}  // namespace nns / libupnp

// ------------------------------------------------------------------------------------------------
// Startup function
// ------------------------------------------------------------------------------------------------
extern "C" void nninitStartup() NN_NOEXCEPT
{
    // Setup the size of total memory heap
    const size_t MemoryHeapSize = 16 * 1024 * 1024;
    auto result = nn::os::SetMemoryHeapSize( MemoryHeapSize );

    NN_ASSERT( result.IsSuccess() );

    // Allcate memory block from the heap for malloc
    uintptr_t address = 0;

    result = nn::os::AllocateMemoryBlock(&address, MemoryHeapSize);
    NN_ASSERT( result.IsSuccess() );

    // Set memory chunk for malloc
    nn::init::InitializeAllocator( reinterpret_cast<void*>(address), MemoryHeapSize );
}

// ------------------------------------------------------------------------------------------------
// nnMain
// ------------------------------------------------------------------------------------------------
extern "C" void nnMain() NN_NOEXCEPT
{
    nns::libupnp::mainLoop();
}

