﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LibcurlUnhIolTest.cpp,PageSampleLibcurlUnhIolTest}
 *
 * @brief
 * libcurl による HTTP(S) 通信
 */

/**
 * @page PageSampleLibcurlUnhIolTest libcurl による HTTP(S) 通信
 * @tableofcontents
 *
 * @brief
 * libcurl による HTTP(S) 通信のサンプルプログラムの解説です。
 *
 * @section PageSampleLibcurlUnhIolTest_SectionBrief 概要
 * まず、 NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * その結果、ネットワーク接続が利用可能であれば libcurl を使用して HTTP(S) リクエストを送信します。
 *
 * @section PageSampleLibcurlUnhIolTest_SectionFileStoructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/Libcurl/UnhIolTest
 * Samples/Sources/Applications/Libcurl/UnhIolTest @endlink 以下にあります。
 *
 * @section PageSampleLibcurlUnhIolTest_SectionNecessaryEnvironment 必要な環境
 * 事前に設定マネージャを使用してネットワーク接続設定をインポートする必要があります。
 *  詳細は @confluencelink{104465190,SettingsManager_network,ネットワーク接続設定の登録} をご覧ください。
 *
 * また、既定では http://www.example.com に対して HTTP リクエストを送信するため、
 * インターネットに接続できる環境が必要です。インターネットに接続できる環境を準備できない場合、
 * サンプルプログラムの URL 部分をアクセス可能な URL に置き換えてください。
 *
 * @section PageSampleLibcurlUnhIolTest_SectionHowToOperate 操作方法
 * 特にありません。
 *
 * @section PageSampleLibcurlUnhIolTest_SectionPrecaution 注意事項
 * ネットワーク接続に失敗した場合のハンドリング方法は未定です
 *
 * @section PageSampleLibcurlUnhIolTest_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLibcurlUnhIolTest_SectionDetail 解説
 * サンプルプログラムの処理の流れは以下の通りです。
 *
 * - NIFM ライブラリを使用してネットワーク接続の利用をシステムに要求します。
 * - socket ライブラリを初期化します。
 * - libcurl の easy interface を使用して HTTP(S) リクエストを送信します。
 * - HTTP(S) レスポンスの内容をログ出力します。
 *
 * このサンプルプログラムの実行結果を以下に示します。
 * ただし、 Release ビルドではログメッセージが出力されません。
 *
 * @verbinclude LibcurlUnhIolTest_Output.txt
 */

#include <cstdlib>
#include <string>

#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Log.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_ApiIpAddress.h>
#include <nn/socket.h>
#include <curl/curl.h>

#include "../Common/HttpsHelper.h"

namespace
{

    const float kGigabyte = 1024.0f * 1024.0f * 1024.0f;
    const float kMegabyte = 1024.0f * 1024.0f;
    const float kKilobyte = 1024.0f;
    const int   kMaxUrlLength = 512;

    // **** Uncomment if you need to set proxy settings different than what is set
    //      in system settings. ****
    //#define USE_PROXY
#ifdef USE_PROXY
#define PROXY_SETTINGS(curl)                    \
        curl_easy_setopt(curl,                  \
                        CURLOPT_PROXYAUTOCONFIG, \
                        0L);                     \
        curl_easy_setopt(curl,                  \
                        CURLOPT_PROXY,           \
                        "ProxyServer");          \
        curl_easy_setopt(curl,                  \
                        CURLOPT_PROXYPORT,       \
                        8080);                   \
        curl_easy_setopt(curl,                  \
                        CURLOPT_PROXYUSERPWD,    \
                        "UserName:Password");    \
        curl_easy_setopt(curl,                  \
                        CURLOPT_PROXYAUTH,       \
                        CURLAUTH_BASIC);
#endif // USE_PROXY

    typedef struct WriteData {
        const char  *pReadPos;
        size_t      sizeLeft;
    } WriteData;

    typedef struct EasyData
    {
        CURL* easyHandle;
        char pErrorBuffer[CURL_ERROR_SIZE];
    } EasyData;

    typedef struct Options
    {
        uint32_t    numNifmConnections;
        uint32_t    numConnections;
        uint32_t    timesToDownload;
        int         sndBufSize;
        int         rcvBufSize;
        int         disableTcpNagle;
        int         runDnsTest;
        int         runHttpTest;
        int         uploadSize;
        char        url[kMaxUrlLength + 1];
    } Options;

    // Socket memory
    NN_ALIGNAS(4096) uint8_t    g_SocketMemoryPoolBuffer[nn::socket::DefaultSocketMemoryPoolSize] = { 0 };
    HttpsHelperForCtxImport     g_HttpsHelper = HttpsHelperForCtxImport();


    size_t CurlSetSockOpts(void *pData, curl_socket_t curlfd, curlsocktype purpose)
    {
        int         rval        = 0;
        Options *   pOptions    = static_cast<Options *>(pData);

        if (nullptr != pOptions)
        {
            if (-1 != pOptions->sndBufSize)
            {
                rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_SndBuf, &pOptions->sndBufSize, sizeof(pOptions->sndBufSize));
                if (rval != 0)
                {
                    NN_LOG(" * nn::socket::SetSockOpt on SO_SNDBUF failed with error: %d\n\n", nn::socket::GetLastError());

                    return CURL_SOCKOPT_ERROR;
                }
                else
                {
                    NN_LOG(" * nn::socket::SetSockOpt SO_SNDBUF set to: %d\n", pOptions->sndBufSize);
                }
            }

            if (-1 != pOptions->rcvBufSize)
            {
                rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Socket, nn::socket::Option::So_RcvBuf, &pOptions->rcvBufSize, sizeof(pOptions->rcvBufSize));
                if (rval != 0)
                {
                    NN_LOG(" * nn::socket::SetSockOpt on SO_RCVBUF failed with error: %d\n\n", nn::socket::GetLastError());

                    return CURL_SOCKOPT_ERROR;
                }
                else
                {
                    NN_LOG(" * nn::socket::SetSockOpt SO_RCVBUF set to: %d\n", pOptions->rcvBufSize);
                }
            }

            rval = nn::socket::SetSockOpt(curlfd, nn::socket::Level::Sol_Tcp, nn::socket::Option::Tcp_NoDelay, &pOptions->disableTcpNagle, sizeof(pOptions->disableTcpNagle));
            if (rval != 0)
            {
                NN_LOG(" * nn::socket::SetSockOpt on nn::socket::Option::Tcp_NoDelay failed with error: %d\n\n", nn::socket::GetLastError());

                return CURL_SOCKOPT_ERROR;
            }
        }

        return CURL_SOCKOPT_OK;
    }


    size_t CurlWriteFunction(char *data, size_t blobsize, size_t blobcount, void *userdata)
    {
        size_t count = blobsize*blobcount;
        return count;
    }


    bool GetFileSize(const Options &options, uint32_t* pSize)
    {
        bool    bRet = false;
        CURL*   curl = curl_easy_init();

        if (nullptr != curl)
        {
            char pErrorBuffer[CURL_ERROR_SIZE] = { 0 };

#ifdef USE_PROXY
            PROXY_SETTINGS(curl)
#endif

                NN_LOG("url: %s\n", options.url);
            curl_easy_setopt(curl, CURLOPT_URL, options.url);
            curl_easy_setopt(curl, CURLOPT_FOLLOWLOCATION, 1);
            curl_easy_setopt(curl, CURLOPT_VERBOSE, 1);
            curl_easy_setopt(curl, CURLOPT_ERRORBUFFER, pErrorBuffer);
            curl_easy_setopt(curl, CURLOPT_SOCKOPTFUNCTION, CurlSetSockOpts);
            curl_easy_setopt(curl, CURLOPT_SOCKOPTDATA, &options);
            curl_easy_setopt(curl, CURLOPT_USERAGENT, "Nintendo");

            if (HttpsHelperBase::IsUrlHttps(options.url) == true)
            {
                /* Import the SSL context into libcurl. Because there's only one SSL context
                maintained in httpsHelper, same SSL context will be shared between CURL
                handles.
                */
                if (g_HttpsHelper.ImportSslContext(curl) < 0)
                {
                    NN_LOG("[ERROR] Failed to import the SSL context.\n");
                    goto cleanup;
                }
            }

            // This is the KEY option that tells the server not to send to body(the actual content).
            curl_easy_setopt(curl, CURLOPT_NOBODY, 1);

            CURLcode rval = curl_easy_perform(curl);
            if (rval == CURLE_OK)
            {
                double dSize = 0.0;
                if (curl_easy_getinfo(curl, CURLINFO_CONTENT_LENGTH_DOWNLOAD, &dSize) == CURLE_OK)
                {
                    if (dSize < 0)
                    {
                        NN_LOG(" * Unknown content length!\n");
                    }
                    else
                    {
                        *pSize = static_cast<uint32_t>(dSize);
                        NN_LOG("\n*******Got File size! %u\n\n", *pSize);
                        bRet = true;
                    }
                }
                else
                {
                    NN_LOG("Failed to get content length\n");
                }
            }
            else
            {
                NN_LOG("ERROR: Error fetching URL: %s. Error: %d\n", options.url, rval);
                NN_LOG("Details: %s\n", pErrorBuffer);
                if (HttpsHelperBase::IsUrlHttps(options.url) == true)
                {
                    g_HttpsHelper.PrintErrorMessage(curl, rval);
                }
            }
        }
        else
        {
            NN_LOG(" * Failed to create curl handle!\n");
        }

    cleanup:

        if (true == bRet)
        {
            curl_easy_cleanup(curl);
        }

        return bRet;
    }


#ifndef USE_PROXY
    bool EstimateServerRTT(int64_t* pServerRTT, const char* pUrl)
    {
        const char              strHttp[] = "http://";
        const char              strHttps[] = "https://";
        nn::socket::HostEnt*    pHost = nullptr;
        nn::socket::SockAddrIn  addr = nn::socket::SockAddrIn();
        char                    strHostName[255 + 1] = { '\0' };
        nn::os::Tick            tickPrev = nn::os::Tick();
        nn::os::Tick            tickNow = nn::os::Tick();

        if (strncmp(pUrl, strHttp, strlen(strHttp)) == 0)
        {
            strncpy(strHostName, &pUrl[strlen(strHttp)], sizeof(strHostName) / sizeof(strHostName[0]) - 1);
        }
        else if (strncmp(pUrl, strHttps, strlen(strHttps)) == 0)
        {
            strncpy(strHostName, &pUrl[strlen(strHttps)], sizeof(strHostName) / sizeof(strHostName[0]) - 1);
        }

        char* pIter = strHostName;
        while (*pIter != '\0')
        {
            if (*pIter == '/')
            {
                *pIter = '\0';
                break;
            }

            ++pIter;
        }

        NN_LOG(" HostName: %s\n", strHostName);
        pHost = nn::socket::GetHostEntByName(strHostName);

        if (nullptr == pHost || nullptr == pHost->h_addr_list[0])
        {
            NN_LOG("No host found!\n");
            return false;
        }

        NN_LOG("IP: %s\n", nn::socket::InetNtoa(*reinterpret_cast<nn::socket::InAddr*>(pHost->h_addr_list[0])));

        int nSocket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Stream, nn::socket::Protocol::IpProto_Tcp);
        if (nSocket == nn::socket::SocketError)
        {
            NN_LOG("Failed to create socket. Err: %d\n", nn::socket::GetLastError());
            return false;
        }

        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_addr = *(reinterpret_cast<nn::socket::InAddr*>(pHost->h_addr_list[0]));
        addr.sin_port = nn::socket::InetHtons(80);

        tickPrev = nn::os::GetSystemTick();
        if (nn::socket::Connect(nSocket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)) == nn::socket::SocketError)
        {
            NN_LOG("Could not connect! Error: %d\n", nn::socket::GetLastError());
            nn::socket::Close(nSocket);
            return false;
        }
        tickNow = nn::os::GetSystemTick();

        nn::socket::Close(nSocket);

        *pServerRTT = nn::os::ConvertToTimeSpan(tickNow - tickPrev).GetMilliSeconds();
        NN_LOG("RTT: %lld ms\n", *pServerRTT);

        return true;
    }
#endif

    const char *lookup_list[] = {
        "ctest.cdn.nintendo.net",
        "account.nintendo.net",
        "accounts.google.com",
        "amazon.hs.llnwd.net",
        "api-global.netflix.com",
        "assets.hulu.com",
        "atv-ext.amazon.com",
        "azusss-vod.hss.adaptive.level3.net",
        "ccs.cdn.wup.shop.nintendo.net",
        "ccs.wup.shop.nintendo.net",
        "cdn-0.nflximg.com",
        "cdn-1.nflximg.com",
        "cgi2.nintendo.co.jp",
        "clients1.google.com",
        "club.nintendo.com",
        "ctest.cdn.nintendo.net",
        "crashmo.nintendo.com",
        "crl.geotrust.com",
        "crl.globalsign.net",
        "crl.netflix.com",
        "ct1.addthis.com",
        "customerevents.netflix.com",
        "discovery.olv.nintendo.net",
        "drive.google.com",
        "ecs.wup.shop.nintendo.net",
        "ecx.images-amazon.com",
        "ev.ads.pointroll.com",
        "freakyforms.nintendo.com",
        "g-ec2.images-amazon.com",
        "geisha-wup.cdn.nintendo.net",
        "gtssldv-crl.geotrust.com",
        "ia.media-imdb.com",
        "ib.huluim.com",
        "idbe-wup.cdn.nintendo.net",
        "images-na.ssl-images-amazon.com",
        "img.youtube.com",
        "lb-rdv-prod06.ubisoft.com",
        "lh3.googleusercontent.com",
        "lh4.googleusercontent.com",
        "lh5.googleusercontent.com",
        "lh6.googleusercontent.com",
        "m.addthisedge.com",
        "mail.google.com",
        "maps.google.com",
        "media.nintendo.com",
        "metrics.nintendo.com",
        "mii-images.account.nintendo.net",
        "mii-images.cdn.nintendo.net",
        "netflix636.a.leastpop.nflximg.com.edgesuite.net",
        "netflix636.a.nflximg.com.edgesuite.net",
        "news.google.com",
        "newsupermariobrosu.nintendo.com",
        "ninja.wup.shop.nintendo.net",
        "ninjagaiden3.nintendo.com",
        "nintendo.nccp.netflix.com",
        "nintendojp.d1.sc.omtrdc.net",
        "nintendoland.nintendo.com",
        "nncs1.app.nintendowifi.net",
        "nncs2.app.nintendowifi.net",
        "npdi.cdn.nintendo.net",
        "nppl.app.nintendo.net",
        "npts.app.nintendo.net",
        "nus.cdn.wup.shop.nintendo.net",
        "nus.wup.shop.nintendo.net",
        "ocsp.digicert.com",
        "ocsp.verisign.com",
        "olvus.cdn.nintendo.net",
        "onlineconfigservice.ubi.com",
        "papermario.nintendo.com",
        "photoshowcase.nintendo.com",
        "picasaweb.google.com",
        "play.google.com",
        "play.hulu.com",
        "pls.wup.shop.nintendo.net",
        "plus.google.com",
        "professorlayton.nintendo.com",
        "resizer.i.tv",
        "s3.amazonaws.com",
        "s7.addthis.com",
        "samurai-wup.cdn.nintendo.net",
        "samurai.wup.shop.nintendo.net",
        "secure.hulu.com",
        "secure.netflix.com",
        "singparty.nintendo.com",
        "ssl.google-analytics.com",
        "ssl.gstatic.com",
        "static8.ubi.com",
        "store.nintendo.com",
        "support.google.com",
        "t.hulu.com",
        "t2.hulu.com",
        "tagaya-wup.cdn.nintendo.net",
        "tagaya.wup.shop.nintendo.net",
        "translate.google.com",
        "twitter.com",
        "uiboot.netflix.com",
        "wallet.google.com",
        "webcache.googleusercontent.com",
        "wiiu.app.hulu.com",
        "wsuplay.ubi.com",
        "www.blogger.com",
        "www.facebook.com",
        "www.google.com",
        "www.googleadservices.com",
        "www.gstatic.com",
        "www.meetup.com",
        "www.nintendo.com",
        "www.nintendoworldstore.com",
        "www.public-trust.com",
        "www.youtube.com",
        nullptr };

    void DnsTest(int *tests_passed, int *tests_failed, int *tests_indeterminate)
    {
        nn::os::Tick            test_start = nn::os::Tick(),
                                lookup_start = nn::os::Tick(),
                                lookup_end = nn::os::Tick();
        nn::socket::HostEnt*    hp = nullptr;
        const char**            lookup = nullptr;

        test_start = nn::os::GetSystemTick();

        for (lookup = &lookup_list[0]; *lookup; lookup++)
        {
            NN_LOG("DNS gethostbyname %s... ", *lookup);
            if (nn::os::ConvertToTimeSpan(nn::os::GetSystemTick() - test_start).GetSeconds() > 15/*DNS_MAX_SECONDS*/)
            {
                NN_LOG("SKIP\n", *lookup);
                ++*tests_indeterminate;
                continue;
            }

            lookup_start = nn::os::GetSystemTick();
            hp = nn::socket::GetHostEntByName(*lookup);
            lookup_end = nn::os::GetSystemTick();

            NN_LOG("(%6lld ms elapsed) ", nn::os::ConvertToTimeSpan(lookup_end - lookup_start).GetMilliSeconds());

            if (nullptr == hp)
            {
                ++*tests_failed;
                NN_LOG("FAIL\n", *lookup);
            }
            else
            {
                ++*tests_passed;
                unsigned int i = 0;
                while (nullptr != hp->h_addr_list[i])
                {
                    NN_LOG("%s ", nn::socket::InetNtoa(*reinterpret_cast<nn::socket::InAddr*>((hp->h_addr_list[i]))));
                    i++;
                }
                NN_LOG("PASS\n", *lookup);
            }
        }
    }

    float DownloadFile(uint32_t index, const Options &options, uint32_t fileSize)
    {
        float           fThroughput = 0.0f;
        nn::os::Tick    tickPrev = nn::os::Tick(),
            tickNow = nn::os::Tick();
        uint32_t        downloadSize = 0;
        uint32_t        chunk = 0;
        uint32_t        remainder = 0;
        uint32_t        previous = 0;
        float           fSeconds = 0.0f;

        uint32_t        curlEasyHandleCount = 0;
        CURLM*          curlMultiHandle = nullptr;

        CURLMcode       status = CURLM_OK;
        int             stillRunning = 0;

        chunk = fileSize / options.numConnections;
        remainder = fileSize % options.numConnections;

        EasyData *curlEasyHandles = static_cast<EasyData *>(std::calloc(options.numConnections, sizeof(EasyData)));

        if (nullptr == curlEasyHandles)
        {
            NN_LOG("[ERROR] Failed to allocate %d curl easy handles.\n", options.numConnections);
            return 0.0f;
        }

        for (curlEasyHandleCount = 0; curlEasyHandleCount < options.numConnections; curlEasyHandleCount++)
        {
            CURL *curlEasyHandle = curl_easy_init();
            if (nullptr == curlEasyHandle)
            {
                NN_LOG("ERROR: Could not create curl easy handle for url:%s", options.url);
                goto cleanup;
            }

            curlEasyHandles[curlEasyHandleCount].easyHandle = curlEasyHandle;
            memset(curlEasyHandles[curlEasyHandleCount].pErrorBuffer, 0, CURL_ERROR_SIZE);

            curl_easy_setopt(curlEasyHandle, CURLOPT_URL, options.url);
            curl_easy_setopt(curlEasyHandle, CURLOPT_VERBOSE, 1);
            //curl_easy_setopt(curlEasyHandle, CURLOPT_FOLLOWLOCATION, 1);
            curl_easy_setopt(curlEasyHandle, CURLOPT_ERRORBUFFER, curlEasyHandles[curlEasyHandleCount].pErrorBuffer);
            curl_easy_setopt(curlEasyHandle, CURLOPT_WRITEFUNCTION, CurlWriteFunction);
            curl_easy_setopt(curlEasyHandle, CURLOPT_USERAGENT, "Nintendo");

            if (HttpsHelperBase::IsUrlHttps(options.url) == true)
            {
                /* Import the SSL context into libcurl. Because there's only one SSL context
                maintained in g_HttpsHelper, same SSL context will be shared between CURL
                handles.
                */
                if (g_HttpsHelper.ImportSslContext(curlEasyHandle) < 0)
                {
                    NN_LOG("[ERROR] Failed to import the SSL context.\n");
                    goto cleanup;
                }
            }

            curl_easy_setopt(curlEasyHandle, CURLOPT_SOCKOPTFUNCTION, CurlSetSockOpts);
            curl_easy_setopt(curlEasyHandle, CURLOPT_SOCKOPTDATA, &options);

            if (options.numConnections > 1)
            {
                char        pBuffer[64 + 1] = { '\0' };
                uint32_t    increment = 0;

                if (remainder > 0)
                {
                    increment = chunk + 1;
                    --remainder;
                }
                else
                {
                    increment = chunk;
                }

                // snprintf() isn't supported in Visual Studio 2013 so we can "safely" use sprintf() since we
                // know the maximum length string that would ever be put into pBuffer using 32-bit values is 22
                sprintf(pBuffer, "%u-%u", previous, previous + increment - 1);
                curl_easy_setopt(curlEasyHandle, CURLOPT_RANGE, pBuffer);

                previous += increment;

                NN_LOG("Chunk: %u  %s\n", chunk, pBuffer);
            }

#ifdef USE_PROXY
            PROXY_SETTINGS(curlEasyHandle)
#endif

        }

        // init a multi stack
        curlMultiHandle = curl_multi_init();
        if (nullptr == curlMultiHandle)
        {
            NN_LOG("ERROR: Could not create libcurl multi handle.");
            goto cleanup;
        }

        // add the individual transfers
        for (uint32_t i = 0; i < curlEasyHandleCount; i++)
        {
            curl_multi_add_handle(curlMultiHandle, curlEasyHandles[i].easyHandle);
        }

        NN_LOG("Downloading...\n");
        tickPrev = nn::os::GetSystemTick();
        do
        {
            status = curl_multi_perform(curlMultiHandle, &stillRunning);
        } while (status == CURLM_CALL_MULTI_PERFORM);

        NN_LOG("Status: %u\n", status);

        while (stillRunning)
        {
            // set a suitable timeout to play around with
            nn::socket::TimeVal timeout = { 1, 0 };

            nn::socket::FdSet   fdread = nn::socket::FdSet();
            nn::socket::FdSet   fdwrite = nn::socket::FdSet();
            nn::socket::FdSet   fdexcep = nn::socket::FdSet();
            int                 maxfd = -1;
            long                curl_timeo = -1;

            nn::socket::FdSetZero(&fdread);
            nn::socket::FdSetZero(&fdwrite);
            nn::socket::FdSetZero(&fdexcep);

            curl_multi_timeout(curlMultiHandle, &curl_timeo);
            if (curl_timeo >= 0)
            {
                timeout.tv_sec = curl_timeo / 1000;
                if (timeout.tv_sec > 1)
                {
                    timeout.tv_sec = 1;
                }
                else
                {
                    timeout.tv_usec = (curl_timeo % 1000) * 1000;
                }
            }

            //Get file descriptors from the transfers
            curl_multi_fdset(curlMultiHandle, &fdread, &fdwrite, &fdexcep, &maxfd);

            int rc = nn::socket::Select(maxfd + 1, &fdread, &fdwrite, &fdexcep, &timeout);

            switch (rc)
            {
            case nn::socket::SocketError:
                // select error
                stillRunning = 0;
                NN_LOG("ERROR: select() returned error %d\n", nn::socket::GetLastError());
                break;
            case 0:
            default:
                // timeout or readable/writable sockets
                do
                {
                    status = curl_multi_perform(curlMultiHandle, &stillRunning);
                } while (status == CURLM_CALL_MULTI_PERFORM);
                break;
            }
        }

        tickNow = nn::os::GetSystemTick();

        for (uint32_t iEasyHandle = 0; iEasyHandle < options.numConnections; ++iEasyHandle)
        {
            long    responseCode = 0;
            double  val = 0.0f;

            CURLcode ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_RESPONSE_CODE, &responseCode);
            if (ret != CURLE_OK)
            {
                NN_LOG("curl_easy_getinfo failed: %d\n", ret);
            }
            else
            {
                NN_LOG("CURLINFO_RESPONSE_CODE: %d\n", responseCode);
            }

            ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_HTTP_CONNECTCODE, &responseCode);
            if (ret != CURLE_OK)
            {
                NN_LOG("curl_easy_getinfo failed: %d\n", ret);
            }
            else
            {
                NN_LOG("CURLINFO_HTTP_CONNECTCODE: %d\n", responseCode);
            }

            ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_SIZE_DOWNLOAD, &val);
            if (ret != CURLE_OK)
            {
                NN_LOG("curl_easy_getinfo failed: %d\n", ret);
            }
            else
            {
                uint32_t numBytes = static_cast<uint32_t>(val);
                downloadSize += numBytes;
                NN_LOG("CURLINFO_SIZE_DOWNLOAD: %u\n", numBytes);
            }

            /* check for total download time */
            ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_TOTAL_TIME, &val);
            if ((CURLE_OK == ret) && (val > 0))
            {
                NN_LOG("CURLINFO_TOTAL_TIME: %0.3f sec.\n", val);
            }

            /* check for average download speed */
            ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_SPEED_DOWNLOAD, &val);
            if ((CURLE_OK == ret) && (val > 0))
            {
                NN_LOG("CURLINFO_SPEED_DOWNLOAD: %0.3f kbyte/sec.\n", val / 1024);
            }

            /* check for name resolution time */
            ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_NAMELOOKUP_TIME, &val);
            if ((CURLE_OK == ret) && (val > 0))
            {
                NN_LOG("CURLINFO_NAMELOOKUP_TIME: %0.3f sec.\n", val);
            }

            /* check for connect time */
            ret = curl_easy_getinfo(curlEasyHandles[iEasyHandle].easyHandle, CURLINFO_CONNECT_TIME, &val);
            if ((CURLE_OK == ret) && (val > 0))
            {
                NN_LOG("CURLINFO_CONNECT_TIME: %0.3f sec.\n", val);
            }

            NN_LOG("Error Details %d: %s\n", iEasyHandle, curlEasyHandles[iEasyHandle].pErrorBuffer);
        }

        fSeconds = nn::os::ConvertToTimeSpan(tickNow - tickPrev).GetMilliSeconds() / 1000.0f;

        NN_LOG("\n ** Finished download %u with Status: %d\n", index, status);

        NN_LOG(" ** Duration: %.3f sec\n", fSeconds);

        if (downloadSize / kGigabyte >= 1.0f)
        {
            NN_LOG(" ** Downloaded: %.3f GB\n", downloadSize / kGigabyte);
        }
        else if (downloadSize / kMegabyte >= 1.0f)
        {
            NN_LOG(" ** Downloaded: %.3f MB\n", downloadSize / kMegabyte);
        }
        else if (downloadSize / kKilobyte >= 1.0f)
        {
            NN_LOG(" ** Downloaded: %.3f KB\n", downloadSize / kKilobyte);
        }
        else
        {
            NN_LOG(" ** Downloaded: %u Bytes\n", downloadSize);
        }

        /* bytes/ms -> Mb/s */
        fThroughput = downloadSize * 8.0f / fSeconds / kMegabyte;
        NN_LOG(" ** Speed: %.3f Mbits/sec\n\n", fThroughput);

    cleanup:

        for (uint32_t i = 0; i < curlEasyHandleCount; i++)
        {
            if (curlMultiHandle)
            {
                curl_multi_remove_handle(curlMultiHandle, curlEasyHandles[i].easyHandle);
            }

            curl_easy_cleanup(curlEasyHandles[i].easyHandle);
        }

        if (curlMultiHandle)
        {
            curl_multi_cleanup(curlMultiHandle);
        }

        free(curlEasyHandles);

        return fThroughput;
    } // NOLINT(impl/function_size)


    size_t CurlReadFunction(void *ptr, size_t size, size_t nmemb, void *userp)
    {
        WriteData   *pData = static_cast<WriteData *>(userp);
        size_t      count = size * nmemb;

        if (count < 1)
            return 0;

        if (pData->sizeLeft > 0)
        {
            memcpy(ptr, pData->pReadPos, count);
            pData->pReadPos += count;
            pData->sizeLeft -= count;
            return count;
        }

        return 0;                          /* no more data left to deliver */
    }


    float UploadFile(const Options &options)
    {
        float           fThroughput = 0.0f;
        nn::os::Tick    tickPrev = nn::os::Tick(),
                        tickNow = nn::os::Tick();
        char            *pBuf = static_cast<char *>(std::calloc(options.uploadSize, 1));
        WriteData       writeData = { 0 };
        CURLcode        res = CURLE_OK;
        float           fSeconds = 0.0f;

        writeData.pReadPos = pBuf;
        writeData.sizeLeft = options.uploadSize;

        if (nullptr == pBuf)
        {
            NN_LOG("[ERROR] Failed to allocate %d bytes for upload.\n", options.uploadSize);
            return 0.0f;
        }

        // fill buffer with randomish bytes
        memset(pBuf, rand(), options.uploadSize);

        // upload buffer to url
        CURL *curlEasyHandle = curl_easy_init();

        if (nullptr != curlEasyHandle)
        {
            curl_easy_setopt(curlEasyHandle, CURLOPT_URL, options.url);
            curl_easy_setopt(curlEasyHandle, CURLOPT_VERBOSE, 1);
            curl_easy_setopt(curlEasyHandle, CURLOPT_USERAGENT, "Nintendo NX");
            curl_easy_setopt(curlEasyHandle, CURLOPT_HTTP_VERSION, CURL_HTTP_VERSION_1_0);
            curl_easy_setopt(curlEasyHandle, CURLOPT_POST, 1L);
            curl_easy_setopt(curlEasyHandle, CURLOPT_READFUNCTION, CurlReadFunction);
            curl_easy_setopt(curlEasyHandle, CURLOPT_READDATA, &writeData);
            curl_easy_setopt(curlEasyHandle, CURLOPT_POSTFIELDSIZE, options.uploadSize);

            NN_LOG("Uploading %d bytes to %s\n", options.uploadSize, options.url);
            tickPrev = nn::os::GetSystemTick();

            res = curl_easy_perform(curlEasyHandle);
            if (CURLE_OK == res)
            {
                long    responseCode = 0;
                double  val = 0.0f;

                tickNow = nn::os::GetSystemTick();

                NN_LOG("curl_easy_perform() succeeded!\n", res);

                CURLcode ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_RESPONSE_CODE, &responseCode);
                if (ret != CURLE_OK)
                {
                    NN_LOG("curl_easy_getinfo failed: %d\n", ret);
                }
                else
                {
                    NN_LOG("CURLINFO_RESPONSE_CODE: %d\n", responseCode);
                }

                ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_HTTP_CONNECTCODE, &responseCode);
                if (ret != CURLE_OK)
                {
                    NN_LOG("curl_easy_getinfo failed: %d\n", ret);
                }
                else
                {
                    NN_LOG("CURLINFO_HTTP_CONNECTCODE: %d\n", responseCode);
                }

                /* check for total download time */
                ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_TOTAL_TIME, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    NN_LOG("CURLINFO_TOTAL_TIME: %0.3f sec.\n", val);
                }

                /* check for average download speed */
                ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_SPEED_UPLOAD, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    NN_LOG("CURLINFO_SPEED_UPLOAD: %0.3f kbyte/sec.\n", val / 1024);
                }

                /* check for name resolution time */
                ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_NAMELOOKUP_TIME, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    NN_LOG("CURLINFO_NAMELOOKUP_TIME: %0.3f sec.\n", val);
                }

                /* check for connect time */
                ret = curl_easy_getinfo(curlEasyHandle, CURLINFO_CONNECT_TIME, &val);
                if ((CURLE_OK == ret) && (val > 0))
                {
                    NN_LOG("CURLINFO_CONNECT_TIME: %0.3f sec.\n", val);
                }

                fSeconds = nn::os::ConvertToTimeSpan(tickNow - tickPrev).GetMilliSeconds() / 1000.0f;

                NN_LOG("\n ** Finished upload\n");

                NN_LOG(" ** Duration: %.3f sec\n", fSeconds);

                if (options.uploadSize / kGigabyte >= 1.0f)
                {
                    NN_LOG(" ** Uploaded: %.3f GB\n", options.uploadSize / kGigabyte);
                }
                else if (options.uploadSize / kMegabyte >= 1.0f)
                {
                    NN_LOG(" ** Uploaded: %.3f MB\n", options.uploadSize / kMegabyte);
                }
                else if (options.uploadSize / kKilobyte >= 1.0f)
                {
                    NN_LOG(" ** Uploaded: %.3f KB\n", options.uploadSize / kKilobyte);
                }
                else
                {
                    NN_LOG(" ** Uploaded: %u Bytes\n", options.uploadSize);
                }

                /* bytes/ms -> Mb/s */
                fThroughput = options.uploadSize * 8.0f / fSeconds / kMegabyte;
                NN_LOG(" ** Speed: %.3f Mbits/sec\n\n", fThroughput);
            }
            else
            {
                NN_LOG("[ERROR] curl_easy_perform() failed! curl error: %d\n", res);
            }

            curl_easy_cleanup(curlEasyHandle);
        }

        // calculate and return throughput

        free(pBuf);

        return fThroughput;
    } // NOLINT(impl/function_size)


    char* GetCmdOption(char **pBegin, char **pEnd, const std::string &option)
    {
        char **pItr = std::find(pBegin, pEnd, option);
        if (pItr != pEnd && ++pItr != pEnd)
        {
            return *pItr;
        }
        return 0;
    }


    bool CmdOptionExists(char **pBegin, char **pEnd, const std::string& option)
    {
        return std::find(pBegin, pEnd, option) != pEnd;
    }


    bool ParseCmdOptions(Options *pOptions)
    {
        const int   argc = nn::os::GetHostArgc();

        if (argc > 1)
        {
            char *option = nullptr;

            NN_LOG("Command: UnhIolTest");
            for (int i = 1; i < argc; i++)
            {
                NN_LOG(" %s", nn::os::GetHostArgv()[i]);
            }
            NN_LOG("\n");

            if (CmdOptionExists(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-h"))
            {
                NN_LOG("LibcurlUnhIolTest.nca: Run speed test using libcurl.\n");
                NN_LOG("\nUsage: RunOnTarget.exe LibcurlUnhIolTest.nca [-u <arg>] [-s <arg>] [-r <arg>]" \
                    "                                                [-p <arg>] [-t <arg>] [-n <arg>]" \
                    "                                                [-c <arg>] [-up <arg>] [-h] [-d]\n");
                NN_LOG("\nExamples:\n" \
                    "  # Run with default parameters.\n" \
                    "  RunOnTarget.exe LibcurlUnhIolTest.nca\n\n" \
                    "  # Download default url with four simultaneous connections.\n" \
                    "  RunOnTarget.exe LibcurlUnhIolTest.nca -n 4\n\n" \
                    "  # Download specific url three times.\n" \
                    "  RunOnTarget.exe LibcurlUnhIolTest.nca -u http://www.example.com -t 3\n\n" \
                    "  # Download default url with custom sized socket send and receive buffers.\n" \
                    "  RunOnTarget.exe LibcurlUnhIolTest.nca -r 131072 -s 65536\n" \
                    "  # Download default url with the TCP Nagle algorithm disabled.\n" \
                    "  RunOnTarget.exe LibcurlUnhIolTest.nca -n off\n\n");
                NN_LOG("\nAcceptable Arguments:\n" \
                    "  -c <arg>    Number of times to connect/disconnect/reconnect to NIFM.\n" \
                    "  -d          Run DNS test.\n" \
                    "  -h          Display this help.\n" \
                    "  -http <arg> on [default] - Runs the HTTP upload/download tests.\n" \
                    "              off - Does not run the HTTP upload/download tests but exits after running connection and DNS tests.\n" \
                    "  -n <arg>    on [default] - Enable TCP Nagle algorithm.\n" \
                    "              off - Disable TCP Nagle algorithm.\n" \
                    "  -p <arg>    Number of simultaneous parallel connections to make to the server.\n" \
                    "  -r <arg>    Value to set SO_RCVBUF socket option to.\n" \
                    "  -s <arg>    Value to set SO_SNDBUF socket option to.\n" \
                    "  -t <arg>    Number of times to download or upload to the url.\n" \
                    "  -u <arg>    The url to connect to.\n" \
                    "  -up <arg>   Number of random bytes to upload.\n" \
                    "\n");

                return false;
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-c");
            if (nullptr != option)
            {
                pOptions->numNifmConnections = atoi(option);
                // if numNifmConnections is 0, it could be due to the parameter being passed in incorrectly
                if (0 == pOptions->numNifmConnections)
                {
                    NN_LOG("[WARNING] The number of times to connect to network is set to 0. Was that intentional?\n");
                }
            }

            if (CmdOptionExists(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-d"))
            {
                pOptions->runDnsTest = 1;
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-http");
            if (nullptr != option)
            {
                if (std::string("off") == option)
                {
                    pOptions->runHttpTest = 0;
                    // if the user doesn't want to run any HTTP tests then we don't care about reading the rest of the args
                    return true;
                }
                else if (std::string("on") == option)
                {
                    pOptions->runHttpTest = 1;
                }
                else
                {
                    NN_LOG("[WARNING] Unknown argument, %s, used for -http option. Use \"off\" or \"on\" (default).\n", option);
                }
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-u");
            if (nullptr != option)
            {
                strncpy(pOptions->url, option, kMaxUrlLength);
                pOptions->url[kMaxUrlLength] = '\0';
                NN_LOG("setting url to %s\n", pOptions->url);
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-s");
            if (nullptr != option)
            {
                pOptions->sndBufSize = atoi(option);
                // if rcvBufSize is 0, it could be due to the parameter being passed in incorrectly
                if (0 == pOptions->sndBufSize)
                {
                    NN_LOG("[WARNING] The send buffer size is set to 0. Was that intentional?\n");
                }
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-r");
            if (nullptr != option)
            {
                pOptions->rcvBufSize = atoi(option);
                // if rcvBufSize is 0, it could be due to the parameter being passed in incorrectly
                if (0 == pOptions->rcvBufSize)
                {
                    NN_LOG("[WARNING] The receive buffer size is set to 0. Was that intentional?\n");
                }
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-p");
            if (nullptr != option)
            {
                pOptions->numConnections = atoi(option);
                // if numConnections is 0, it could be due to the parameter being passed in incorrectly
                if (0 == pOptions->numConnections)
                {
                    NN_LOG("[WARNING] The number of parallel connetions is set to 0. Was that intentional?\n");
                }
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-t");
            if (nullptr != option)
            {
                pOptions->timesToDownload = atoi(option);
                // if timesToDownload is 0, it could be due to the parameter being passed in incorrectly
                if (0 == pOptions->timesToDownload)
                {
                    NN_LOG("[WARNING] The number of times to download is set to 0. Was that intentional?\n");
                }
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-n");
            if (nullptr != option)
            {
                if (std::string("off") == option)
                {
                    pOptions->disableTcpNagle = 1;
                }
                else if (std::string("on") == option)
                {
                    pOptions->disableTcpNagle = 0;
                }
                else
                {
                    NN_LOG("[WARNING] Unknown argument, %s, used for -n option. Use \"off\" or \"on\" (default).\n", option);
                }
            }

            option = GetCmdOption(nn::os::GetHostArgv(), nn::os::GetHostArgv() + argc, "-up");
            if (nullptr != option)
            {
                pOptions->uploadSize = atoi(option);
                // if uploadSize is 0, it could be due to the parameter being passed in incorrectly
                if (0 == pOptions->uploadSize)
                {
                    NN_LOG("[WARNING] The number of upload bytes is set to 0. Was that intentional?\n");
                }
            }
        }

        return true;
    } // NOLINT(impl/function_size)


    extern "C" void nnMain()
    {
        nn::Result  ret = nn::Result();
        CURLcode    res = CURLE_OK;
        uint32_t    fileSize = 0;
        int64_t     serverRTT = -1;
        float       fAverageThroughput = 0.0f;
        Options     options = { 1,      // numNifmConnections
                                1,      // numConnections
                                1,      // timesToDownload
                                -1,     // sndBufSize
                                -1,     // rcvBufSize
                                0,      // disableTcpNagle
                                0,      // runDnsTest
                                1,      // runHttpTest
                                0,      // uploadSize
                                "http://speedtest.tele2.net/100MB.zip" };    // url
        int          dnsPass = 0;
        int          dnsFail = 0;
        int          dnsUnk = 0;

        // if this returns false, then the user only wanted to view the help
        if (ParseCmdOptions(&options) == false)
        {
            return;
        }

        NN_LOG("Started...\n");

        /* since the following will utilize the system network interface, we must initialize
        network interface manager (NIFM) */
        nn::nifm::Initialize();

        /* this application is now requesting to use network interface */
        nn::nifm::SubmitNetworkRequest();

        /* wait for network interface availability, while providing status */
        while (nn::nifm::IsNetworkRequestOnHold())
        {
            NN_LOG("Waiting for network interface availability...\n");
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }

        if (!nn::nifm::IsNetworkAvailable())
        {
            NN_LOG("\n[ERROR] network is not available.\n");
            goto CLEANUP_NIFM;
        }

        NN_LOG("Network is available\n");

        for (uint32_t i = 1; i < options.numNifmConnections; i++)
        {
            /* this application no longer requires use of network interface */
            nn::nifm::CancelNetworkRequest();

            /* this application is now requesting to use network interface */
            nn::nifm::SubmitNetworkRequest();

            /* wait for network interface availability, while providing status */
            while (nn::nifm::IsNetworkRequestOnHold())
            {
                NN_LOG("Waiting for network interface availability...\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            if (!nn::nifm::IsNetworkAvailable())
            {
                NN_LOG("\n[ERROR] network is not available.\n");
                goto CLEANUP_NIFM;
            }

            NN_LOG("Network is available\n");
        }

        /* initialize the socket library, providing application use case specific amount of memory */
        ret = nn::socket::Initialize(g_SocketMemoryPoolBuffer,
            nn::socket::DefaultSocketMemoryPoolSize,
            nn::socket::MinSocketAllocatorSize,
            nn::socket::DefaultConcurrencyLimit);

        if (ret.IsFailure())
        {
            NN_LOG("\n[ERROR] nn::socket::Initialize() failed. Err Desc: %d\n\n", ret.GetDescription());
            goto CLEANUP_NIFM;
        }

        if (options.runDnsTest == 1)
        {
            DnsTest(&dnsPass, &dnsFail, &dnsUnk);
            NN_LOG("\n\nDNS results: passed: %d failed: %d skipped/unknown: %d \n\n", dnsPass, dnsFail, dnsUnk);
        }

        if (options.runHttpTest == 0)
        {
            goto CLEANUP_SOCKET_AND_NIFM;
        }

        /* initialize using system malloc for libcurl for testing */
        res = curl_global_init(CURL_GLOBAL_DEFAULT);
        if (res != CURLE_OK)
        {
            NN_LOG("\n[ERROR] curl_global_init failed. Err: %d\n\n", res);
            goto CLEANUP_SOCKET_AND_NIFM;
        }
        NN_LOG("Network ready\n");

        /* Setup for HTTPS. The SSL context will be allocated inside g_HttpsHelper.
        The allocated SSL context will be imported into libcurl by g_HttpsHelper.ImportSslContext
        below after CURL handle is created, and when URL contains https.
        The allocated SSL context will be freed in g_HttpsHelper.Finalize after all curl handles
        which imported this context are closed at the bottom.
        */
        if (g_HttpsHelper.Initialize() < 0)
        {
            NN_LOG("[ERROR] Failed to initialize HTTPS helper util.\n");
            goto CLEANUP_CURL_AND_SOCKET_AND_NIFM;
        }

        // Can only calc RTT with no proxy
#ifndef USE_PROXY
        if (!EstimateServerRTT(&serverRTT, options.url))
        {
            NN_LOG("Failed to get server RTT!\n");
        }
#endif

        if (options.uploadSize == 0)
        {
            if (!GetFileSize(options, &fileSize))
            {
                NN_LOG("Failed to get file size!\n");
                goto CLEANUP_CURL_AND_SOCKET_AND_NIFM;
            }

            for (uint32_t iDownload = 1; iDownload <= options.timesToDownload; ++iDownload)
            {
                NN_LOG(" * Starting download number: %d\n", iDownload);
                fAverageThroughput += DownloadFile(iDownload, options, fileSize);
            }

            NN_LOG("times to download: %d\n", options.timesToDownload);
            fAverageThroughput /= options.timesToDownload;

            NN_LOG("\n******* Results *******\n");
            NN_LOG("******* URL: %s\n", options.url);
            NN_LOG("******* Average Throughput: %.3f Mbits/sec\n", fAverageThroughput);

            if (fileSize / kGigabyte >= 1.0f)
            {
                NN_LOG("******* File Size: %.3f GB\n", fileSize / kGigabyte);
            }
            else if (fileSize / kMegabyte >= 1.0f)
            {
                NN_LOG("******* File Size: %.3f MB\n", fileSize / kMegabyte);
            }
            else if (fileSize / kKilobyte >= 1.0f)
            {
                NN_LOG("******* File Size: %.3f KB\n", fileSize / kKilobyte);
            }
            else
            {
                NN_LOG("******* File Size: %u Bytes\n", fileSize);
            }

            NN_LOG("******* Number of Downloads: %u\n", options.timesToDownload);
            NN_LOG("******* Number of Connections: %u\n", options.numConnections);
        }
        else
        {
            UploadFile(options);
        }

        if (serverRTT >= 0)
        {
            NN_LOG("******* Server RTT: %lld ms\n\n", serverRTT);
        }
        else
        {
            NN_LOG("******* Server RTT: Unknown\n\n");
        }

    CLEANUP_CURL_AND_SOCKET_AND_NIFM:
        /* cleanup libcurl library */
        curl_global_cleanup();

    CLEANUP_SOCKET_AND_NIFM:
        /* cleanup socket library */
        nn::socket::Finalize();

    CLEANUP_NIFM:
        /* this application no longer requires use of network interface */
        nn::nifm::CancelNetworkRequest();

        return;
    } // NOLINT(impl/function_size)

}
