﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/os.h>
#include "Task.h"

namespace nns { namespace ldn
{
    /**
     * @brief       タスクの実行状況です。
     */
    enum TaskRunnerState
    {
        //! タスクの実行を中断しています。
        TaskRunnerState_Paused,

        //! タスクを実行中です。
        TaskRunnerState_Running,
    };

}} // namespace nns::ldn

namespace nns { namespace ldn { namespace detail
{
    // Thin Template
    class SequentialVoidTaskRunnerBase
    {
        NN_DISALLOW_COPY(SequentialVoidTaskRunnerBase);
        NN_DISALLOW_MOVE(SequentialVoidTaskRunnerBase);

    public:

        SequentialVoidTaskRunnerBase(
            void* stack, size_t stackSize, int priority,
            IVoidTask** tasks, int capacity) NN_NOEXCEPT;

        ~SequentialVoidTaskRunnerBase() NN_NOEXCEPT;

        void AddTask(IVoidTask* pTask) NN_NOEXCEPT;

        void Clear() NN_NOEXCEPT;

        void Run() NN_NOEXCEPT;

        void Cancel() NN_NOEXCEPT;

        TaskRunnerState GetState() const NN_NOEXCEPT;

        int GetTaskCount() const NN_NOEXCEPT;

        IVoidTask* GetTask(int index) NN_NOEXCEPT;

        const IVoidTask* GetTask(int index) const NN_NOEXCEPT;

        nn::os::Event& GetStateChangeEvent() NN_NOEXCEPT;

    private:

        void SetState(TaskRunnerState newState) NN_NOEXCEPT;
        static void Thread(void* pArg) NN_NOEXCEPT;

        nn::os::ThreadType m_Thread;
        nn::os::Event m_RunEvent;
        nn::os::Event m_StateChangeEvent;
        mutable nn::os::Mutex m_TaskMutex;
        IVoidTask** m_Tasks;
        TaskRunnerState m_State;
        int16_t m_TaskCapacity;
        int16_t m_TaskCount;
        int16_t m_TaskIndex;
        bool m_IsCanceled;
        bool m_IsDestroyed;
    };

}}} // namespace nns::ldn::detail

namespace nns { namespace ldn
{
    /**
     * @brief       先頭から順番にタスクを実行します。
     * @tparam      Capacity    タスクの最大登録数です。
     */
    template <int Capacity>
    class SequentialVoidTaskRunner
    {
        NN_DISALLOW_COPY(SequentialVoidTaskRunner);
        NN_DISALLOW_MOVE(SequentialVoidTaskRunner);

    public:

        /**
         * @brief       コンストラクタです。
         * @param[in]   stack           スレッドスタックです。
         * @param[in]   stackSize       スレッドスタックのバイトサイズです。
         * @param[in]   priority        スレッド優先度です。
         */
        SequentialVoidTaskRunner(
            void* stack, size_t stackSize, int priority) NN_NOEXCEPT
            : m_Impl(stack, stackSize, priority, m_Tasks, Capacity)
        {
        }

        /**
         * @brief       デストラクタです。
         */
        ~SequentialVoidTaskRunner() NN_NOEXCEPT
        {
        }

        /**
         * @brief       タスクを追加します。
         * @param[in]   pTask           実行対象のタスクです。
         * @details     タスクの実行を開始していないか全て終了している状態でなければ使えません。
         */
        void AddTask(IVoidTask* pTask) NN_NOEXCEPT
        {
            m_Impl.AddTask(pTask);
        }

        /**
         * @brief       タスクを追加します。
         * @param[in]   tasks           実行対象のタスクです。
         * @param[in]   count           タスクの数です。
         * @details     タスクの実行を開始していないか全て終了している状態でなければ使えません。
         */
        void AddTask(IVoidTask* tasks, int count) NN_NOEXCEPT
        {
            for (int i = 0; count; ++i)
            {
                m_Impl.AddTask(tasks, count);
            }
        }

        /**
         * @brief       登録されているタスクを全て消去します。
         * @details     タスクの実行を開始していないか全て終了している状態でなければ使えません。
         */
        void Clear() NN_NOEXCEPT
        {
            m_Impl.Clear();
        }

        /**
         * @brief       タスクの実行を開始あるいは再開します。
         */
        void Run() NN_NOEXCEPT
        {
            m_Impl.Run();
        }

        /**
         * @brief       タスクの実行を強制的に中断します。
         */
        void Cancel() NN_NOEXCEPT
        {
            m_Impl.Cancel();
        }

        /**
         * @brief       タスクの実行状況を取得します。
         * @return      タスクの実行状況です。
         */
        TaskRunnerState GetState() const NN_NOEXCEPT
        {
            return m_Impl.GetState();
        }

        /**
         * @brief       登録されているタスクの残数を取得します。
         * @return      登録されているタスクの残数です。実行済のタスクは含まれません。
         */
        int GetTaskCount() const NN_NOEXCEPT
        {
            return m_Impl.GetTaskCount();
        }

        /**
         * @brief       インデックスに対応するタスクを取得します。
         * @param[in]   index           タスクのインデックスです。
         * @return      登録されているタスクです。
         */
        IVoidTask* GetTask(int index) NN_NOEXCEPT
        {
            return static_cast<IVoidTask*>(m_Impl.GetTask(index));
        }

        /**
         * @brief       インデックスに対応するタスクを取得します。
         * @param[in]   index           タスクのインデックスです。
         * @return      登録されているタスクです。
         */
        const IVoidTask* GetTask(int index) const NN_NOEXCEPT
        {
            return static_cast<const IVoidTask*>(m_Impl.GetTask(index));
        }

        /**
         * @brief       状態変化を通知するイベントを取得します。
         * @return      状態変化を通知するイベントです。
         */
        nn::os::Event& GetStateChangeEvent() NN_NOEXCEPT
        {
            return m_Impl.GetStateChangeEvent();
        }

    private:

        static void Thread(void* pArg) NN_NOEXCEPT;

        detail::SequentialVoidTaskRunnerBase m_Impl;
        IVoidTask* m_Tasks[Capacity];
    };

    /**
     * @brief       先頭から順番にタスクを実行します。
     * @tparam      T           タスクの実行結果の型です。ITask<T> を実行します。
     * @tparam      Capacity    タスクの最大登録数です。
     */
    template <typename T, int Capacity>
    class SequentialTaskRunner
    {
        NN_DISALLOW_COPY(SequentialTaskRunner);
        NN_DISALLOW_MOVE(SequentialTaskRunner);

    public:

        /**
         * @brief       コンストラクタです。
         * @param[in]   stack           スレッドスタックです。
         * @param[in]   stackSize       スレッドスタックのバイトサイズです。
         * @param[in]   priority        スレッド優先度です。
         */
        SequentialTaskRunner(
            void* stack, size_t stackSize, int priority) NN_NOEXCEPT
            : m_Impl(stack, stackSize, priority, m_Tasks, Capacity)
        {
        }

        /**
         * @brief       デストラクタです。
         */
        ~SequentialTaskRunner() NN_NOEXCEPT
        {
        }

        /**
         * @brief       タスクを追加します。
         * @param[in]   pTask           実行対象のタスクです。
         * @details     タスクの実行を開始していないか全て終了している状態でなければ使えません。
         */
        void AddTask(ITask<T>* pTask) NN_NOEXCEPT
        {
            m_Impl.AddTask(pTask);
        }

        /**
         * @brief       タスクを追加します。
         * @param[in]   tasks           実行対象のタスクです。
         * @param[in]   count           タスクの数です。
         * @details     タスクの実行を開始していないか全て終了している状態でなければ使えません。
         */
        void AddTask(ITask<T>* tasks, int count) NN_NOEXCEPT
        {
            for (int i = 0; count; ++i)
            {
                m_Impl.AddTask(tasks, count);
            }
        }

        /**
         * @brief       登録されているタスクを全て消去します。
         * @details     タスクの実行を開始していないか全て終了している状態でなければ使えません。
         */
        void Clear() NN_NOEXCEPT
        {
            m_Impl.Clear();
        }

        /**
         * @brief       タスクの実行を開始あるいは再開します。
         */
        void Run() NN_NOEXCEPT
        {
            m_Impl.Run();
        }

        /**
         * @brief       タスクの実行を強制的に中断します。
         */
        void Cancel() NN_NOEXCEPT
        {
            m_Impl.Cancel();
        }

        /**
         * @brief       タスクの実行状況を取得します。
         * @return      タスクの実行状況です。
         */
        TaskRunnerState GetState() const NN_NOEXCEPT
        {
            return m_Impl.GetState();
        }

        /**
         * @brief       登録されているタスクの残数を取得します。
         * @return      登録されているタスクの残数です。実行済のタスクは含まれません。
         */
        int GetTaskCount() const NN_NOEXCEPT
        {
            return m_Impl.GetTaskCount();
        }

        /**
         * @brief       インデックスに対応するタスクを取得します。
         * @param[in]   index           タスクのインデックスです。
         * @return      登録されているタスクです。
         */
        ITask<T>* GetTask(int index) NN_NOEXCEPT
        {
            return static_cast<ITask<T>*>(m_Impl.GetTask(index));
        }

        /**
         * @brief       インデックスに対応するタスクを取得します。
         * @param[in]   index           タスクのインデックスです。
         * @return      登録されているタスクです。
         */
        const ITask<T>* GetTask(int index) const NN_NOEXCEPT
        {
            return static_cast<const ITask<T>*>(m_Impl.GetTask(index));
        }

        /**
         * @brief       状態変化を通知するイベントを取得します。
         * @return      状態変化を通知するイベントです。
         */
        nn::os::Event& GetStateChangeEvent() NN_NOEXCEPT
        {
            return m_Impl.GetStateChangeEvent();
        }

    private:

        static void Thread(void* pArg) NN_NOEXCEPT;

        detail::SequentialVoidTaskRunnerBase m_Impl;
        IVoidTask* m_Tasks[Capacity];
    };

}} // namespace nns::ldn
