﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>

namespace nns { namespace ldn
{
    /**
     * @brief       タスクの処理状況です。
     */
    enum TaskState
    {
        //! タスクの実行を開始していません。
        TaskState_None,

        //! タスクを処理中です。
        TaskState_Running,

        //! タスクの実行が完了しました。
        TaskState_Finished,

        //! タスクの実行がキャンセルされました。
        TaskState_Canceled,
    };

    /**
     * @brief           非同期に実行したいタスクのインタフェースです。
     */
    struct IVoidTask
    {
        /**
         * @brief       デストラクタです。
         */
        virtual ~IVoidTask() NN_NOEXCEPT
        {
        }

        /**
         * @brief       タスクを実行します。
         */
        virtual void Run() NN_NOEXCEPT = 0;

        /**
         * @brief       タスクの実行を中断します。
         */
        virtual void Cancel() NN_NOEXCEPT = 0;

        /**
         * @brief       タスクの実行状況を取得します。
         * @return      タスクの実行状況です。
         */
        virtual TaskState GetState() const NN_NOEXCEPT = 0;
    };

    template <typename T>
    struct ITask : public IVoidTask
    {
        /**
         * @brief       デストラクタです。
         */
        virtual ~ITask() NN_NOEXCEPT
        {
        }

        /**
         * @brief       タスクの実行結果を取得します。
         * @return      タスクの実行結果です。
         */
        virtual const T GetResult() const NN_NOEXCEPT = 0;
    };

    /**
     * @brief       タスクの処理が最後まで実行されたかを示す定数です。
     */
    enum TaskResult
    {
        //! タスクは途中で中断されました。
        TaskResult_Canceled,

        //! タスクは最後まで実行されました。
        TaskResult_Finished,
    };

    /**
     * @brief           結果を返さないタスクの基底クラスです。
     */
    class VoidTask : public IVoidTask
    {
    public:

        /**
         * @brief       コンストラクタです。
         */
        VoidTask() NN_NOEXCEPT
            : m_State(TaskState_None)
        {
        }

        /**
         * @brief       デストラクタです。
         */
        virtual ~VoidTask() NN_NOEXCEPT
        {
        }

        virtual void Run() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT_EQUAL(m_State, TaskState_None);
            m_State = TaskState_Running;
            TaskResult taskResult = RunImpl();
            m_State = taskResult == TaskResult_Finished ? TaskState_Finished : TaskState_Canceled;
        }

        virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE
        {
            CancelImpl();
        }

        virtual TaskState GetState() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_State;
        }

    private:

        virtual TaskResult RunImpl() NN_NOEXCEPT = 0;
        virtual void CancelImpl() NN_NOEXCEPT
        {
        }

        TaskState m_State;
    };

    /**
     * @brief           結果を返すタスクの基底クラスです。
     */
    template <typename T>
    class Task : public ITask<T>
    {
    public:

        /**
         * @brief       コンストラクタです。
         */
        Task() NN_NOEXCEPT
            : m_State(TaskState_None)
        {
        }

        /**
         * @brief       デストラクタです。
         */
        virtual ~Task() NN_NOEXCEPT
        {
        }

        virtual void Run() NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT_EQUAL(m_State, TaskState_None);
            m_State = TaskState_Running;
            TaskResult taskResult = RunImpl(&m_Result);
            m_State = taskResult == TaskResult_Finished ? TaskState_Finished : TaskState_Canceled;
        }

        virtual void Cancel() NN_NOEXCEPT NN_OVERRIDE
        {
            CancelImpl();
        }

        virtual TaskState GetState() const NN_NOEXCEPT NN_OVERRIDE
        {
            return m_State;
        }

        virtual const T GetResult() const NN_NOEXCEPT NN_OVERRIDE
        {
            NN_ASSERT(m_State == TaskState_Finished);
            return m_Result;
        }

    private:

        virtual TaskResult RunImpl(T* pOutResult) NN_NOEXCEPT = 0;
        virtual void CancelImpl() NN_NOEXCEPT
        {
        }

        TaskState m_State;
        T m_Result;
    };

}} // namespace nns::ldn
