﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nns { namespace ldn
{
    /**
     * @brief       コントローラの最大数です。
     */
    const int PadCountMax = 9;
    NN_STATIC_ASSERT(0 < PadCountMax);

    /**
     * @brief           コントローラのボタンの定義です。
     */
    enum Button
    {
        //! A ボタンです。
        Button_A = 1 << 0,

        //! B ボタンです。
        Button_B = 1 << 1,

        //! X ボタンです。
        Button_X = 1 << 2,

        //! Y ボタンです。
        Button_Y = 1 << 3,

        //! L ボタンです。
        Button_L = 1 << 4,

        //! R ボタンです。
        Button_R = 1 << 5,

        //! ZL ボタンです。
        Button_ZL = 1 << 6,

        //! ZR ボタンです。
        Button_ZR = 1 << 7,

        //! + ボタンです。
        Button_Plus = 1 << 8,

        //! - ボタンです。
        Button_Minus = 1 << 9,

        //! 十字ボタンの左ボタンです。
        Button_Left = 1 << 10,

        //! 十字ボタンの上ボタンです。
        Button_Up = 1 << 11,

        //! 十字ボタンの右ボタンです。
        Button_Right = 1 << 12,

        //! 十字ボタンの下ボタンです。
        Button_Down = 1 << 13,
    };

    /**
     * @brief           ボタンの入力状態です。
     */
    struct ButtonState
    {
        //! 押下されているボタンです。
        uint16_t down;

        //! このフレームで新しく押上されたボタンです。
        uint16_t trigger;
    };

    /**
     * @brief           コントローラの入力状態です。
     */
    struct PadState
    {
        //! ボタンの入力状態です。
        ButtonState button;
    };

    /**
     * @brief           コントローラのインタフェースです。
     */
    struct IPad
    {
        /**
         * @brief           デストラクタです。
         */
        virtual ~IPad()
        {
        }

        /**
         * @brief           コントローラを初期化します。
         */
        virtual void Initialize() NN_NOEXCEPT = 0;

        /**
         * @brief           コントローラの使用を終了します。
         */
        virtual void Finalize() NN_NOEXCEPT = 0;

        /**
         * @brief           コントローラの状態を更新します。
         * @param[out]      pOutState   コントローラの状態の出力先です。
         */
        virtual void Update(PadState* pOutState) NN_NOEXCEPT = 0;
    };

    /**
     * @brief           いずれかのボタンがトリガー入力されたか否かを判定するユーティリティです。
     * @param[in]       pad             コントローラの入力状態です。
     * @param[in]       button          判定対象のボタンです。
     * @return          いずれか 1 つでもトリガー入力があれば true です。
     */
    inline bool IsTriggeredAny(const ButtonState& pad, Button button) NN_NOEXCEPT
    {
        return (pad.trigger & button) != 0;
    }

    /**
     * @brief           いずれかのボタンがトリガー入力されたか否かを判定するユーティリティです。
     * @param[in]       pad             コントローラの入力状態です。
     * @param[in]       button          判定対象のボタンです。
     * @return          いずれか 1 つでもトリガー入力があれば true です。
     */
    inline bool IsTriggeredAny(const PadState& pad, Button button) NN_NOEXCEPT
    {
        return IsTriggeredAny(pad.button, button);
    }

}} // namespace nns::ldn


