﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_Log.h>
#include <nn/socket.h>
#include "NetworkTask.h"

namespace nns { namespace ldn { namespace
{
    int32_t CalculateHash(const Frame& frame)
    {
        return frame.sequenceNumber ^ INT32_C(0x5C5C5C5C);
    }

}}} // namespace nns::ldn

namespace nns { namespace ldn
{
    NetworkSendTask::NetworkSendTask(
        LdnData* pData, int socket, int port, int rate) NN_NOEXCEPT
        : m_Data(*pData),
          m_Interval(nn::TimeSpan::FromMicroSeconds(1000000 / rate)),
          m_Socket(socket),
          m_Port(port),
          m_IsCanceled(false)
    {
        m_Data.initialNumber = m_Data.sequenceNumber;
    }

    NetworkSendTask::~NetworkSendTask() NN_NOEXCEPT
    {
    }

    TaskResult NetworkSendTask::RunImpl() NN_NOEXCEPT
    {
        // ブロードキャスト送信用のアドレスを生成します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(static_cast<uint16_t>(m_Port));
        addr.sin_addr.S_addr = nn::socket::InetHtonl(
            nn::ldn::MakeBroadcastAddress(m_Data.address, m_Data.mask).raw);

        // キャンセルされるまで一定時間毎にパケットを送信します。
        while (!m_IsCanceled)
        {
            // 開始時刻を取得します。
            nn::os::Tick startedAt = nn::os::GetSystemTick();

            // カウンタの値を取得し、インクリメントします。
            Frame frame;
            frame.sequenceNumber = nn::socket::InetHtonl(m_Data.sequenceNumber);
            frame.hash = CalculateHash(frame);
            ++m_Data.sequenceNumber;

            // フレームを送信します。
            auto ret = nn::socket::SendTo(
                m_Socket, &frame, sizeof(frame), nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr));
            if (ret < 0)
            {
                return TaskResult_Finished;
            }

            // パケットレート調整のためスリープします。
            auto elapsed = (nn::os::GetSystemTick() - startedAt).ToTimeSpan();
            auto diff = m_Interval - elapsed;
            if (0 < diff)
            {
                nn::os::SleepThread(diff);
            }
        }
        return TaskResult_Canceled;
    }

    void NetworkSendTask::CancelImpl() NN_NOEXCEPT
    {
        m_IsCanceled = true;
    }

    NetworkReceiveTask::NetworkReceiveTask(
        LdnData* pData, int socket, int port) NN_NOEXCEPT
        : m_Data(*pData),
          m_Socket(socket),
          m_Port(port),
          m_IsCanceled(false)
    {
        std::memset(m_Data.counters, 0, sizeof(m_Data.counters));
    }

    NetworkReceiveTask::~NetworkReceiveTask() NN_NOEXCEPT
    {
    }

    TaskResult NetworkReceiveTask::RunImpl() NN_NOEXCEPT
    {
        // キャンセルされるまで一定時間毎にパケットを送信します。
        while (!m_IsCanceled)
        {
            // 任意のステーションからパケットを受信するためのアドレスを生成します。
            nn::socket::SockAddrIn addr;
            addr.sin_family = nn::socket::Family::Af_Inet;
            addr.sin_port = nn::socket::InetHtons(static_cast<uint16_t>(m_Port));
            addr.sin_addr.S_addr = nn::socket::InAddr_Any;

            // パケットの受信まで待機します。失敗したらその時点で終了です。
            Frame frame;
            nn::socket::SockLenT length = sizeof(addr);
            auto size = nn::socket::RecvFrom(
                m_Socket, &frame, sizeof(frame), nn::socket::MsgFlag::Msg_None,
                reinterpret_cast<nn::socket::SockAddr*>(&addr), &length);
            if (size == 0 || size < 0)
            {
                return TaskResult_Finished;
            }

            // フレームの内容が正しくない場合には無視します。
            if (size != sizeof(Frame) || frame.hash != CalculateHash(frame))
            {
                continue;
            }

            // 送信元アドレスを取得します。
            nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.S_addr) };

            // 受信したカウンタ値を保存します。
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                const auto& node = m_Data.network.ldn.nodes[i];
                auto& counter = m_Data.counters[i];
                if (node.isConnected && node.ipv4Address == from)
                {
                    if (counter.count == 0)
                    {
                        counter.start = nn::socket::InetNtohl(frame.sequenceNumber);
                    }
                    counter.current = nn::socket::InetNtohl(frame.sequenceNumber);
                    ++counter.count;

                    int total = (counter.current - counter.start + 1);
                    counter.loss = 1.0f * (total - counter.count) / total;
                }
            }
        }
        return TaskResult_Canceled;
    }

    void NetworkReceiveTask::CancelImpl() NN_NOEXCEPT
    {
        m_IsCanceled = true;
    }

}} // namespace nns::ldn
