﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LdnSimple/Main.cpp,PageSampleLdnSimple}
 *
 * @brief LDN ライブラリのサンプルプログラム
 */

/**
 * @page PageSampleLdnSimple LDN ライブラリのサンプルプログラム
 *
 * @tableofcontents
 *
 * @brief LDN ライブラリのサンプルプログラムの解説です。
 *
 * @section PageSampleLdnSimple_SectionBrief 概要
 * LDN ライブラリのサンプルプログラムです。
 *
 * @section PageSampleLdnSimple_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/LdnSimple
 * Samples/Sources/Applications/LdnSimple @endlink 以下にあります。
 *
 * @section PageSampleLdnSimple_SectionHowToOperate 操作方法
 * サンプルプログラムを実行するとログメッセージで操作方法が表示されます。
 * この操作方法に従い、デバッグパッドあるいは無線コントローラで操作してください。
 *
 * @section PageSampleLdnSimple_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLdnSimple_SectionOption オプション
 * 以下に示すオプションをコマンドライン引数として与えることでサンプルプログラムの挙動を変更できます。
 *
 * - -c, --channel CHANNEL
 *    - 無線チャンネルを 1, 6, 11, 36, 40, 44, 48 の中から指定します。
 *    - デフォルトでは nn::ldn::AutoChannel を使用するため 1, 6, 11 のいずれかが選択されます。
 * - -e, --encrypto
 *    - データ通信の暗号化を有効にします。
 *    - デフォルトではデータ通信の暗号化が無効にされています。
 *    - 製品では常にデータ通信の暗号化が有効です。
 * - -n, --node NODE_COUNT_MAX
 *    - アクセスポイントとして起動したときに接続可能なノード数を 1 - 8 の整数で指定します。
 *    - デフォルトでは nn::ldn::NodeCountMax (=8) を使用するため 7 台のステーションが接続できます。
 * - -p, --port PORT_NUMBER
 *    - UDP 通信で使用するポート番号を 0 - 65535 の整数で指定してください。
 *    - デフォルトでは 12345 を使用します。
 * - -r, --rate PACKET_RATE
 *    - UDP 通信におけるパケットの送信レートを 1-255 の整数で指定してください。
 *    - デフォルトでは 30 パケット／秒です。
 * - -s, --scene SCENE_ID
 *    - シーン識別子を 0 - 65535 の整数で指定してください。
 *    - このサンプルではシーン識別子が異なるアクセスポイントはスキャン結果に表示されません。
 *    - デフォルトでは 0 を使用します。
 * - -u, --user USER_NAME
 *    - ユーザ名を 10 文字以内の ASCII コードで入力してください。
 *    - デフォルトでは Anonymous を使用します。
 * - -v, --version LOCAL_COMMUNICATION_VERSION
 *    - ローカル通信バージョンを 0 - 32767 の範囲で指定します。
 *    - ステーションはローカル通信バージョンが異なるアクセスポイントには接続できません。
 *    - デフォルトでは 0 を使用します。
 *
 * @section PageSampleLdnSimple_SectionPrecaution 注意事項
 * デフォルトではこのサンプルプログラムは 2.4GHz 帯を使用して無線通信を行いますので、
 * 同じ周波数帯を使用する無線通信機器が多数存在する環境は避けてください。
 * 電波状況の改善が困難な場合、コマンドライン引数を指定することで 5GHz 帯を使用できます。
 * -c オプションあるいは --channel に 36, 40, 44, 48 のいずれかを指定して実行してください。
 * 電波状況が悪いと以下のような問題が多発します。
 *
 * - 接続対象の LDN ネットワークが見つからない
 * - LDN ネットワークへの接続に失敗する
 * - UDP 通信のパケットロス率が高くなる
 *
 * @section PageSampleLdnSimple_SectionDetail 解説
 * LDN ライブラリは、中継器やインターネットを介さずに複数の端末を接続し、
 * それぞれの端末上で動作するアプリケーション間で通信する機能を提供します。
 * LDN ライブラリを使用すると最大 8 台の端末を接続して通信できます。
 * 最初に 1 台の端末が「アクセスポイント」となり、LDN ネットワークを構築し、
 * 他の端末は「ステーション」として LDN ネットワークを探索して接続します。
 *
 * このサンプルプログラムを実行するには複数の端末を準備する必要があります。
 * まず、1 台の端末でサンプルプログラムを起動し、アクセスポイントの役割を与えます。
 * ログメッセージに出力される操作方法に従ってネットワークを構築してください。
 *
 * 続いて、2 台目以降の端末でサンプルプログラムを起動し、
 * ログメッセージに出力される操作方法に従って周囲のネットワークを探索してください。
 * ネットワークの探索が終了すると、見つかったネットワークの一覧が出力されますので、
 * 先ほど別の端末で構築したネットワークを選択して接続してください。
 * 複数のネットワークが出力された場合、IP アドレスを照合することもできますが、
 * サンプルプログラムの実行時に -u, --user オプションでユーザ名を指定しておくと、
 * 自分が構築したネットワークを簡単に見つけることができます。
 *
 * ネットワークに参加したアクセスポイントとステーションは、
 * SOCKET ライブラリを使用して UDP 通信を開始します。
 * 全ての端末は独自にカウンタをもっており、ネットワークに参加した時点で 0 に初期化します。
 * 現在のカウンタの値をブロードキャストで送信し、その後で自分のカウンタをインクリメントします。
 * パケットの送信間隔は -r, --rate オプションで指定できます。
 * このとき、各端末から最後に受信したカウンタの値と、
 * そのカウンタ値を受信した時点のパケット損失率が定期的にログメッセージとして出力されます。
 */

#include <new>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/socket.h>
#include "ApplicationStatus.h"
#include "CommandLineParser.h"
#include "DebugPad.h"
#include "LdnDrawer.h"
#include "LdnUpdater.h"
#include "Npad.h"

namespace
{
    nns::ldn::IUpdater* g_pUpdater;
    nns::ldn::IDrawer* g_pDrawer;

    // 通信バッファです。
    NN_OS_ALIGNAS_GUARDED_STACK nn::socket::ConfigDefaultWithMemory g_SocketConfig;

    // シーンの生成に使用するバッファです。
    NN_OS_ALIGNAS_GUARDED_STACK char m_UpdaterBuffer[256 * 1024];
    NN_OS_ALIGNAS_GUARDED_STACK char m_DrawerBuffer[256 * 1024];

    void SetNextScene(nns::ldn::ApplicationStatus* pApp, nns::ldn::Scene scene) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_EQUAL(scene, nns::ldn::Scene_None);

        // 直前のシーンの終了処理です。
        // Drawer が Updater の処理に依存している場合もあるので、先に Drawer を解放します。
        if (g_pDrawer != nullptr)
        {
            g_pDrawer->Cleanup(*pApp);
            g_pDrawer->~IDrawer();
            g_pDrawer = nullptr;
        }
        if (g_pUpdater != nullptr)
        {
            g_pUpdater->Cleanup(*pApp);
            g_pUpdater->~IUpdater();
            g_pUpdater = nullptr;
        }

        // 次のシーンを生成します。
        switch (scene)
        {
        case nns::ldn::Scene_Ldn:
            g_pUpdater = new (m_UpdaterBuffer) nns::ldn::LdnUpdater();
            g_pDrawer = new (m_DrawerBuffer) nns::ldn::LdnDrawer();
            break;
        default:
            break;
        }

        // シーンの開始処理です。
        pApp->scene = static_cast<int16_t>(scene);
        pApp->sceneFrame = 0;
        if (g_pUpdater != nullptr)
        {
            g_pUpdater->Setup(*pApp);
        }
        if (g_pDrawer != nullptr)
        {
            g_pDrawer->Setup(*pApp);
        }
    }

    void UpdateAndDraw(
        nns::ldn::ApplicationStatus* pApp,
        const nns::ldn::PadState (&padStates)[nns::ldn::PadCountMax]) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(g_pUpdater);

        // 次のシーンに遷移します。
        auto nextScene = g_pUpdater->GetNextScene();
        if (nextScene != nns::ldn::Scene_None)
        {
            SetNextScene(pApp, g_pUpdater->GetNextScene());
        }

        // シーンの更新処理です。
        if (g_pUpdater != nullptr)
        {
            g_pUpdater->Update(*pApp, padStates);
            pApp->pSceneData = g_pUpdater->GetSceneData();
        }
        if (g_pDrawer != nullptr)
        {
            g_pDrawer->Draw(*pApp);
        }

        // フレームカウントをインクリメントします。
        ++pApp->totalFrame;
        ++pApp->sceneFrame;
    }
}

extern "C" void nnMain()
{
    nn::Result result;

    // デバッグのため、スレッドに名前を付けておきます。
    nn::os::SetThreadName(nn::os::GetCurrentThread(), "MainThread");

    // コマンドライン引数を解析します。
    nns::ldn::ApplicationStatus applicationStatus = { };
    nns::ldn::Parse(&applicationStatus.config, nn::os::GetHostArgc(), nn::os::GetHostArgv());

    // socket ライブラリを初期化します。
    nn::socket::Initialize(g_SocketConfig);

    // アプリケーションの操作に使用するコントローラを取得します。
    nns::ldn::DebugPad debugPad;
    debugPad.Initialize();
    nns::ldn::Npad npads[nns::ldn::NpadCountMax];
    for (int i = 0; i < nns::ldn::NpadCountMax; ++i)
    {
        npads[i].SetIndex(i);
        npads[i].Initialize();
    }

    // メインループです。
    SetNextScene(&applicationStatus, nns::ldn::Scene_Ldn);
    nns::ldn::PadState padStates[nns::ldn::PadCountMax] = { };
    while (applicationStatus.scene != nns::ldn::Scene_Exit)
    {
        // メインループの開始時刻を取得します。
        nn::os::Tick startedAt = nn::os::GetSystemTick();

        // コントローラの入力を取得します。
        debugPad.Update(&padStates[0]);
        for (int i = 0; i < nns::ldn::NpadCountMax; ++i)
        {
            npads[i].Update(&padStates[i + 1]);
        }

        // 状態に応じた処理です。
        UpdateAndDraw(&applicationStatus, padStates);

        // 概ね 60FPS になるように調整します。
        auto elapsed = (nn::os::GetSystemTick() - startedAt).ToTimeSpan();
        auto diff = nn::TimeSpan::FromMicroSeconds(16666) - elapsed;
        if (0 < diff)
        {
            nn::os::SleepThread(diff);
        }
    }

    // socket ライブラリの使用を終了します。
    nn::socket::Finalize();

    // コントローラの使用を終了します。
    debugPad.Finalize();
    for (int i = 0; i < nns::ldn::NpadCountMax; ++i)
    {
        npads[i].Finalize();
    }
}
