﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{LdnAccessPoint.cpp,PageSampleLdnAccessPoint}
 *
 * @brief LDN ライブラリのサンプルプログラム（アクセスポイント）
 */

/**
 * @page PageSampleLdnAccessPoint アクセスポイントのサンプルプログラム（削除予定）
 *
 * @tableofcontents
 *
 * @brief LDN ライブラリのサンプルプログラム（アクセスポイント）の解説です。
 *
 * @section PageSampleLdnAccessPoint_SectionBrief 概要
 * LDN ライブラリのアクセスポイントのサンプルです。
 * ステーション側のサンプルは @link PageSampleLdnStation LdnStation @endlink です。
 *
 * LdnAccessPoint と LdnStation は削除が予定されています。
 * 代わりに新しく追加された @link PageSampleLdnStation LdnSimple @endlink を参照してください。
 *
 * @section PageSampleLdnAccessPoint_SectionFileStructure ファイル構成
 * 本サンプルプログラムは @link ../../../Samples/Sources/Applications/LdnBasic/LdnAccessPoint
 * Samples/Sources/Applications/LdnBasic/LdnAccessPoint @endlink 以下にあります。
 *
 * @section PageSampleLdnAccessPoint_SectionHowToOperate 操作方法
 * サンプルプログラムを実行するとログメッセージで操作方法が表示されます。
 * この操作方法に従い、デバッグパッドで操作してください。
 *
 * @section PageSampleLdnAccessPoint_SectionPrecaution 注意事項
 * このサンプルプログラムは 2.4GHz 帯を使用して無線通信を行いますので、
 * 同じ周波数帯を使用する無線通信機器が多数存在する環境は避けてください。
 * 電波状況が悪いと以下のような問題が多発します。
 *
 * - 接続対象の LDN ネットワークが見つからない
 * - LDN ネットワークへの接続に失敗する
 * - UDP 通信のパケットロス率が高くなる
 *
 * 事情があって電波状況を改善できない場合、サンプルプログラムを修正することで 5GHz 帯を使用できます。
 * nn::ldn::CreateNetwork() の引数として渡す nn::ldn::NetworkConfig 構造体の
 * channel メンバを nn::ldn::AutoChannel から 36, 40, 44, 48 のいずれかに変更してください。
 * 併せて @link PageSampleLdnStation LdnStation @endlink サンプルも変更が必要になります。
 * 詳細は @link PageSampleLdnStation LdnStation @endlink サンプルの解説を参照してください。
 *
 * @section PageSampleLdnAccessPoint_SectionHowToExecute 実行手順
 * サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleLdnAccessPoint_SectionDetail 解説
 * LDN ライブラリは、中継器やインターネットを介さずに複数のターゲットを接続し、
 * それぞれのターゲット上で動作するアプリケーション間で通信する機能を提供します。
 *
 * LDN ライブラリを使用すると最大 8 台のターゲットを接続して通信できます。
 * 最初に 1 台のターゲットが「アクセスポイント」となり、LDN ネットワークを構築し、
 * 他のターゲットは「ステーション」として LDN ネットワークを探索して接続します。
 *
 * このサンプルプログラムではアクセスポイントとして LDN ネットワークを構築します。
 * ネットワークを構築した後、@link PageSampleLdnStation LdnStation @endlink
 * サンプルを使用してステーションを LDN ネットワークに接続してください。
 * LDN ネットワークを構築した後は SOCKET ライブラリを用いて UDP 通信を行います。
 * 全てのターゲットは独自にカウンタをもっており、そのカウンタの値を全員で共有します。
 * 具体的には、ステーションは自身のカウンタの値をユニキャストでアクセスポイントに送信し、
 * アクセスポイントは自身を含む全員のカウンタの値をブロードキャストでステーションに配信します。
 */

#include <cstring>
#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/hid.h>
#include <nn/ldn.h>
#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/socket/socket_Api.h>
#include <nn/socket/socket_Types.h>
#include "../Common/Definition.h"
#include "../Common/Util.h"

namespace
{
    nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;
}

NN_ALIGNAS(4096) char g_SendThreadStack[8192];
NN_ALIGNAS(4096) char g_ReceiveThreadStack[8192];
NN_ALIGNAS(4096) char g_LdnThreadStack[8192];
NN_ALIGNAS(4096) char g_MonitorThreadStack[8192];

int SendThreadPriority      = nn::os::DefaultThreadPriority + 4;
int ReceiveThreadPriority   = nn::os::DefaultThreadPriority + 3;
int LdnThreadPriority       = nn::os::DefaultThreadPriority + 2;
int MonitorThreadPriority   = nn::os::DefaultThreadPriority + 1;

nn::os::Mutex g_LogMutex(false);

enum ApplicationState
{
    ApplicationState_Initialized,
    ApplicationState_NetworkCreating,
    ApplicationState_NetworkCreated,
    ApplicationState_NetworkDestroying,
    ApplicationState_NetworkDestroyed
};

struct ApplicationResource
{
    nn::os::ThreadType          monitorThread;
    nn::os::ThreadType          sendThread;
    nn::os::ThreadType          receiveThread;
    nn::os::SystemEventType     stateChangeEvent;
    nn::os::EventType           cancelEvent;
    nn::os::MutexType           mutex;
    nn::ldn::NetworkInfo        networkInfo;
    nn::ldn::Ipv4Address        ipv4Address;
    nn::ldn::SubnetMask         subnetMask;
    uint32_t                    counter[nn::ldn::NodeCountMax];
    int                         socket;
    ApplicationState            state;
};

void MonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;
    nn::ldn::NetworkInfo networkInfo;
    nn::ldn::NodeLatestUpdate updates[nn::ldn::NodeCountMax];

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::TryWaitSystemEvent(&app.stateChangeEvent);

        // ネットワークが破棄された場合には監視を終了します。
        nn::ldn::State ldnState = nn::ldn::GetState();
        if (ldnState != nn::ldn::State_AccessPointCreated)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        // 接続中のステーションを取得します。
        result = nn::ldn::GetNetworkInfo(&networkInfo, updates, nn::ldn::NodeCountMax);
        if (nn::ldn::ResultInvalidState::Includes(result))
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        else if (nn::ldn::ResultDeviceNotAvailable::Includes(result))
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }
        NN_ABORT_UNLESS(result.IsSuccess());

        // 接続状態が変化したステーションを検索します。ノード番号 0 は AP なので無視します。
        for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
        {
            std::lock_guard<nn::os::Mutex> lock(g_LogMutex);
            const auto& oldNode = app.networkInfo.ldn.nodes[i];
            const auto& newNode = networkInfo.ldn.nodes[i];
            const auto& update = updates[i];
            if (update.stateChange == nn::ldn::NodeStateChange_Disconnect ||
                update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
            {
                NN_LOG("ステーションが切断しました。\n");
                NN_LOG("Node ID    : %d\n", oldNode.nodeId);
                NN_LOG("IP Address : %s\n", ConvertToString(oldNode.ipv4Address));
                NN_LOG("User Name  : %s\n\n", oldNode.userName);
            }
            if (update.stateChange == nn::ldn::NodeStateChange_Connect ||
                update.stateChange == nn::ldn::NodeStateChange_DisconnectAndConnect)
            {
                NN_LOG("新しいステーションの接続です。\n");
                NN_LOG("Node ID    : %d\n", newNode.nodeId);
                NN_LOG("IP Address : %s\n", ConvertToString(newNode.ipv4Address));
                NN_LOG("User Name  : %s\n\n", newNode.userName);
            }
        }
        nn::os::LockMutex(&app.mutex);
        std::memcpy(&app.networkInfo, &networkInfo, sizeof(nn::ldn::NetworkInfo));
        nn::os::UnlockMutex(&app.mutex);
    }

    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);
}

void SendThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    // ブロードキャスト送信用のアドレスを生成します。
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InetHtonl(nn::ldn::BroadcastIpv4Address.raw);

    // キャンセルされるまで一定時間毎にパケットを送信します。
    const auto timeout = nn::TimeSpan::FromMilliSeconds(PacketInterval);
    while (!nn::os::TimedWaitEvent(&app.cancelEvent, timeout))
    {
        // 自身のカウンタをインクリメントします。
        nn::os::LockMutex(&app.mutex);
        app.counter[0] += 1;

        // 送信データを生成します。
        AccesspointData data;
        for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
        {
            data.counter[i] = nn::socket::InetHtonl(app.counter[i]);
        }

        // 現在の状態をログ出力します。
        {
            std::lock_guard<nn::os::Mutex> lock(g_LogMutex);
            NN_LOG("[Counter]\n");
            for (int i = 0; i < nn::ldn::NodeCountMax; ++i)
            {
                if (app.networkInfo.ldn.nodes[i].isConnected)
                {
                    NN_LOG("Node %d : %u\n", i, app.counter[i]);
                }
            }
            NN_LOG("\n");
        }
        nn::os::UnlockMutex(&app.mutex);

        // パケットを送信します。
        nn::socket::SendTo(
            app.socket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None,
            reinterpret_cast<nn::socket::SockAddr*>(&addr),
            sizeof(addr));
    }
}

void ReceiveThread(void* arg) NN_NOEXCEPT
{
    auto& app = *static_cast<ApplicationResource*>(arg);

    // キャンセルされるまでパケットを受信し続けます。
    while (!nn::os::TryWaitEvent(&app.cancelEvent))
    {
        // 受信用のバッファです。
        StationData data;

        // 任意のステーションからパケットを受信するためのアドレスを生成します。
        nn::socket::SockAddrIn addr;
        addr.sin_family = nn::socket::Family::Af_Inet;
        addr.sin_port = nn::socket::InetHtons(UdpPort);
        addr.sin_addr.S_addr = nn::socket::InAddr_Any;

        // パケットの受信まで待機します。失敗したらその時点でスレッドを終了します。
        nn::socket::SockLenT length = sizeof(addr);
        auto size = nn::socket::RecvFrom(
            app.socket,
            &data,
            sizeof(data),
            nn::socket::MsgFlag::Msg_None,
            reinterpret_cast<nn::socket::SockAddr*>(&addr),
            &length);
        if (size == 0 || size == -1)
        {
            break;
        }
        nn::ldn::Ipv4Address from = { nn::socket::InetNtohl(addr.sin_addr.S_addr) };

        // 受信したカウンタを保存します。
        nn::os::LockMutex(&app.mutex);
        for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
        {
            const auto& node = app.networkInfo.ldn.nodes[i];
            if (node.isConnected && node.ipv4Address == from)
            {
                app.counter[i] = nn::socket::InetNtohl(data.counter);
            }
        }
        nn::os::UnlockMutex(&app.mutex);
    }
}

nn::Result CreateAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    // アクセスポイントモードとして起動します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::OpenAccessPoint());

    // Advertise で配信するデータを設定します。
    const char message[] = "Welcome to LDN network!";
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::SetAdvertiseData(message, sizeof(message)));

    // 新規に構築する LDN ネットワークのパラメータを設定します。
    nn::ldn::NetworkConfig  network  = {};
    nn::ldn::SecurityConfig security = {};
    nn::ldn::UserConfig     user     = {};
    network.intentId        = BasicSampleIntentId;
    network.channel         = nn::ldn::AutoChannel;
    network.nodeCountMax    = nn::ldn::NodeCountMax;
    security.securityMode   = static_cast<nn::Bit16>(nn::ldn::SecurityMode_Product);
    security.passphraseSize = sizeof(Passphrase);
    std::memcpy(security.passphrase, Passphrase, sizeof(Passphrase));
    std::strncpy(user.userName, "AP", nn::ldn::UserNameBytesMax);

    // アクセスポイントとして新規にネットワークを構築します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::CreateNetwork(network, security, user));

    // ネットワークの情報を取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetNetworkInfo(&pApp->networkInfo));

    // 自身の IPv4 アドレスとサブネットマスクを取得します。
    NNS_LDN_RETURN_IF_FAILED(nn::ldn::GetIpv4Address(&pApp->ipv4Address, &pApp->subnetMask));

    // ネットワークの状態を監視するスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->monitorThread, MonitorThread, pApp,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack),
        MonitorThreadPriority));
    nn::os::StartThread(&pApp->monitorThread);

    return nn::ResultSuccess();
}

void DestroyAccessPoint(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ネットワークの状態を監視するスレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->monitorThread);
    nn::os::DestroyThread(&pApp->monitorThread);

    // ネットワークを破棄し、 LDN ライブラリの使用を終了します。
    nn::ldn::DestroyNetwork();
    nn::ldn::CloseAccessPoint();
}

void CreateSocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    // データの送受信に使用する socket を生成します。
    pApp->socket = nn::socket::Socket(nn::socket::Family::Af_Inet, nn::socket::Type::Sock_Dgram, nn::socket::Protocol::IpProto_Udp);
    NN_ABORT_UNLESS_NOT_EQUAL(pApp->socket, -1);
    nn::socket::SockAddrIn addr;
    addr.sin_family = nn::socket::Family::Af_Inet;
    addr.sin_port = nn::socket::InetHtons(UdpPort);
    addr.sin_addr.S_addr = nn::socket::InAddr_Any;
    NN_ABORT_UNLESS_EQUAL(nn::socket::Bind(
        pApp->socket, reinterpret_cast<nn::socket::SockAddr*>(&addr), sizeof(addr)), 0);

    // ブロードキャスト通信を有効化します。
    int isEnabled = 1;
    nn::socket::SetSockOpt(
        pApp->socket, nn::socket::Level::Sol_Socket, nn::socket::Option::So_Broadcast, &isEnabled, sizeof(isEnabled));

    // データ送信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->sendThread, SendThread, pApp,
        g_SendThreadStack, sizeof(g_SendThreadStack),
        SendThreadPriority));
    nn::os::StartThread(&pApp->sendThread);

    // データ受信用のスレッドを生成します。
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
        &pApp->receiveThread, ReceiveThread, pApp,
        g_ReceiveThreadStack, sizeof(g_ReceiveThreadStack),
        ReceiveThreadPriority));
    nn::os::StartThread(&pApp->receiveThread);
}

void DestroySocket(ApplicationResource* pApp) NN_NOEXCEPT
{
    // ソケットを破棄します。
    nn::socket::Shutdown(pApp->socket, nn::socket::ShutdownMethod::Shut_RdWr);
    nn::socket::Close(pApp->socket);

    // スレッドの終了まで待機します。
    nn::os::WaitThread(&pApp->sendThread);
    nn::os::WaitThread(&pApp->receiveThread);
    nn::os::DestroyThread(&pApp->sendThread);
    nn::os::DestroyThread(&pApp->receiveThread);
}

void LdnAccessPoint(void* arg) NN_NOEXCEPT
{
    nn::Result result;

    // LDN ネットワークを構築します。
    auto& app = *static_cast<ApplicationResource*>(arg);
    result = CreateAccessPoint(&app);
    if (result <= nn::ldn::ResultInvalidState())
    {
        // 無線スイッチオフなど通信機能を利用できない状態です。
        // 通信機能を利用できる状態になってから再度 LDN ライブラリを初期化してください。
        return;
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);
    {
        std::lock_guard<nn::os::Mutex> lock(g_LogMutex);
        NN_LOG("LDN ネットワークの構築に成功しました。\n");
        NN_LOG("Channel    : %d\n", app.networkInfo.common.channel);
        NN_LOG("IP Address : %s/%s\n",
               ConvertToString(app.ipv4Address), ConvertToString(app.subnetMask));
        NN_LOG("User Name  : %s\n\n", app.networkInfo.ldn.nodes[0].userName);
    }
    app.state = ApplicationState_NetworkCreated;

    // データの送受信を開始します.
    CreateSocket(&app);

    // メインスレッドから停止を指示されるか、ネットワークが破棄されるまで待機します。
    nn::os::WaitEvent(&app.cancelEvent);

    // LDN ネットワークを破棄し、データの送受信を終了します。
    DestroySocket(&app);
    DestroyAccessPoint(&app);
    app.state = ApplicationState_NetworkDestroyed;
    NN_LOG("LDN ネットワークが破棄されました。\n\n");
}

extern "C" void nnMain()
{
    nn::Result result;

    // デバッグパッドから入力を取得するために HID ライブラリを取得します。
    nn::hid::InitializeDebugPad();

    // LDN ライブラリを初期化します。
    result = nn::ldn::Initialize();
    if (result <= nn::ldn::ResultDeviceOccupied())
    {
        // 他の通信機能を使用中のためローカル通信機能を利用できません。
        // 通信機能を終了してから再度 LDN ライブラリを初期化してください。
    }
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // socket ライブラリを初期化します。
    result = nn::socket::Initialize(g_SocketConfigWithMemory);

    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // サンプルの操作方法を出力します。
    NN_LOG("操作方法 (LDN ネットワークを構築していない場合)\n");
    NN_LOG("    A ボタンで LDN ネットワークを構築します。\n");
    NN_LOG("    Start ボタンでサンプルを終了します。\n");
    NN_LOG("操作方法 (LDN ネットワークを構築している場合)\n");
    NN_LOG("    B ボタンで LDN ネットワークを破棄します。\n");
    NN_LOG("    X ボタンで全ステーションを強制的に切断します。\n");
    NN_LOG("\n");

    // アプリケーションが使用するリソースを初期化します。
    ApplicationResource app;
    std::memset(&app, 0, sizeof(app));

    // LDN ネットワークの接続状態の変更を通知するイベントを取得します。
    nn::ldn::AttachStateChangeEvent(&app.stateChangeEvent);

    // メインループです。
    nn::hid::DebugPadState previousPadState = { };
    nn::hid::DebugPadState padState = { };
    nn::hid::DebugPadButtonSet trigger = { };
    nn::os::ThreadType ldnThread;
    for (;;)
    {
        // デバッグパッドの入力を取得します。
        nn::hid::GetDebugPadState(&padState);

        // トリガー入力を取得します。
        trigger = (~previousPadState.buttons) & padState.buttons;
        previousPadState = padState;

        // 各状態とボタン入力に応じた処理です。
        if (app.state == ApplicationState_Initialized)
        {
            if (trigger.Test<nn::hid::DebugPadButton::A>())
            {
                app.state = ApplicationState_NetworkCreating;
                NN_LOG("入力を受け付けました。 LDN ネットワークの構築を開始します。\n\n");
                nn::os::InitializeEvent(&app.cancelEvent, false, nn::os::EventClearMode_ManualClear);
                nn::os::InitializeMutex(&app.mutex, false, 0);
                NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(
                    &ldnThread, LdnAccessPoint, &app,
                    g_LdnThreadStack, sizeof(g_LdnThreadStack), LdnThreadPriority));
                nn::os::StartThread(&ldnThread);
            }
            else if (trigger.Test<nn::hid::DebugPadButton::Start>())
            {
                NN_LOG("入力を受け付けました。サンプルを終了します。\n\n");
                break;
            }
        }
        else if (app.state == ApplicationState_NetworkCreated)
        {
            if (trigger.Test<nn::hid::DebugPadButton::B>())
            {
                std::lock_guard<nn::os::Mutex> lock(g_LogMutex);
                app.state = ApplicationState_NetworkDestroying;
                NN_LOG("入力を受け付けました。 LDN ネットワークを破棄します。\n\n");
                nn::os::SignalEvent(&app.cancelEvent);
            }
            else if (trigger.Test<nn::hid::DebugPadButton::X>())
            {
                nn::os::LockMutex(&app.mutex);
                if (0 < app.networkInfo.ldn.nodeCount)
                {
                    std::lock_guard<nn::os::Mutex> lock(g_LogMutex);
                    NN_LOG("入力を受け付けました。全ステーションを切断します。\n\n");
                    for (int i = 1; i < nn::ldn::NodeCountMax; ++i)
                    {
                        const auto& node = app.networkInfo.ldn.nodes[i];
                        if (node.isConnected)
                        {
                            nn::ldn::Reject(node.ipv4Address);
                        }
                    }
                }
                nn::os::UnlockMutex(&app.mutex);
            }
        }
        else if (app.state == ApplicationState_NetworkDestroyed)
        {
            app.state = ApplicationState_Initialized;
            nn::os::WaitThread(&ldnThread);
            nn::os::DestroyThread(&ldnThread);
            nn::os::FinalizeMutex(&app.mutex);
            nn::os::FinalizeEvent(&app.cancelEvent);
        }

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    // socket ライブラリの使用を終了します。
    result = nn::socket::Finalize();
    NNS_LDN_ABORT_UNLESS_RESULT_SUCCESS(result);

    // LDN ライブラリの使用を終了します。
    nn::ldn::Finalize();
}
