﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
// LcsHost.cpp : コンソール アプリケーションのエントリ ポイントを定義します。
//

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>
#include <nn/ae.h>
#include <nn/err/err_Api.h>
#include <nn/fs.h>
#include <nn/fs/fs_Debug.h>
#include <nn/hid.h>
#include <nn/init.h>
#include <nn/lcs.h>
#include <nn/lcs/lcs_DebugApi.h>
#include <nn/ldn/ldn_Result.h>
#include <nn/ldn/ldn_PrivateTypes.h>
#include <nn/ldn/ldn_SystemApi.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/os.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <nn/socket.h>

#include "DrawText.h"
#include "Graphics.h"
#include "HidController.h"
#include "Types.h"

NN_ALIGNAS(4096) char g_LcsBuffer[nn::lcs::RequiredBufferSize];
NN_ALIGNAS(4096) char g_MonitorThreadStack[ThreadStackSize];
NN_ALIGNAS(4096) char g_UpdateThreadStack[ThreadStackSize];

void MonitorThread(void* arg) NN_NOEXCEPT;
void UpdateThread(void* arg) NN_NOEXCEPT;

nn::socket::ConfigDefaultWithMemory g_SocketConfigWithMemory;

void ReadSessionContex(nn::lcs::SessionContext* pOutContext)
{
    nn::Result result;
    const size_t FileSize = sizeof(nn::lcs::SessionContext);

    nn::fs::DirectoryEntryType directoryEntryType;
    result = nn::fs::GetEntryType(&directoryEntryType, "save:/context");
    if (nn::fs::ResultPathNotFound::Includes(result))
    {
        result = nn::fs::CreateFile("save:/context", FileSize);
        nn::fs::FileHandle fileHandle;
        result = nn::fs::OpenFile(&fileHandle, "save:/context", nn::fs::OpenMode_Write);
        nn::lcs::SessionContext context = {};
        result = nn::fs::WriteFile(fileHandle, 0, &context, sizeof(context), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        nn::fs::CloseFile(fileHandle);
    }
    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "save:/context", nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);

    // データを読み込みます。
    size_t readSize = 0;

    (void)nn::fs::ReadFile(&readSize, fileHandle, 0, pOutContext, sizeof(nn::lcs::SessionContext));
    NN_ASSERT_EQUAL(readSize, sizeof(nn::lcs::SessionContext));
    nn::fs::CloseFile(fileHandle);

    NN_LOG("再開用のコンテキストを Read しました。\n");
}

void WriteSessionContex(const nn::lcs::SessionContext& context)
{
    nn::Result result;
    const size_t FileSize = sizeof(nn::lcs::SessionContext);

    nn::fs::DirectoryEntryType directoryEntryType;
    result = nn::fs::GetEntryType(&directoryEntryType, "save:/context");
    if (nn::fs::ResultPathNotFound::Includes(result))
    {
        result = nn::fs::CreateFile("save:/context", FileSize);
        nn::fs::FileHandle fileHandle;
        result = nn::fs::OpenFile(&fileHandle, "save:/context", nn::fs::OpenMode_Write);
        result = nn::fs::WriteFile(fileHandle, 0, &context, sizeof(context), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
        nn::fs::CloseFile(fileHandle);
    }

    nn::fs::FileHandle fileHandle;
    result = nn::fs::OpenFile(&fileHandle, "save:/context", nn::fs::OpenMode_Read | nn::fs::OpenMode_Write);
    result = nn::fs::WriteFile(fileHandle, 0, &context, sizeof(context), nn::fs::WriteOption::MakeValue(nn::fs::WriteOptionFlag_Flush));
    nn::fs::CloseFile(fileHandle);
    (void)nn::fs::Commit("save");

    NN_LOG("再開用のコンテキストを Write しました。\n");
}

bool InitilazeLcs(ApplicationResource& app)
{
    // LCS では LDN を使用するので初期化が必要です。
    nn::Result result;
    result = nn::ldn::InitializeSystem();
    if (result.IsFailure())
    {
        if (nn::ldn::ResultDeviceNotAvailable().Includes(result))
        {
            NN_LOG("無線がOFFになっているのため、LDN の初期化に失敗しました。\n");
            nn::lcs::ShowError(result);
            return false;
        }
        else
        {
            NN_ABORT("LDN の初期化に失敗しました。状態が不正です。\n");
        }
    }

    // LCS を初期化します。
    result = nn::lcs::Initialize(g_LcsBuffer, nn::lcs::RequiredBufferSize, app.config);
    if (result.IsFailure())
    {
        if (nn::lcs::ResultDeviceNotAvailable().Includes(result))
        {
            NN_LOG("無線がOFFになっているのため、LCS の初期化に失敗しました。\n");
            nn::lcs::ShowError(result);
            return false;
        }
        else
        {
            NN_ABORT("LCS の初期化に失敗しました。状態が不正です。: %d\n", result.GetDescription());
        }
    }

    // ホストになるときの設定を AcceptPolicy_AlwaysAccept に設定します。
    nn::lcs::SetClientAcceptPolicy(nn::lcs::AcceptPolicy_AlwaysAccept);
    app.isAcceptPolicy = true;

    // 状態変化の通知を受け取るイベントを取得します。
    nn::lcs::AttachStateChangeEvent(&app.stateChangeEvent);

    app.isRunning = true;
    app.cursor = 0;
    app.cursorMax = InitCount;

    // 状態変化を監視するスレッドを開始します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&app.monitorThread, MonitorThread, &app,
        g_MonitorThreadStack, sizeof(g_MonitorThreadStack), MonitorThreadPriority));
    nn::os::StartThread(&app.monitorThread);

    return true;
}

void FinalizeLcs(ApplicationResource& app)
{
    // 状態変化を監視するスレッドを終了させます。
    nn::os::SignalEvent(&app.cancelEvent);

    // 状態変化を監視するスレッドを破棄します。
    nn::os::WaitThread(&app.monitorThread);
    nn::os::DestroyThread(&app.monitorThread);

    // LCS の終了です。
    nn::lcs::Finalize();

    // LDN の終了です。
    nn::ldn::FinalizeSystem();
}

void ListUpApplication(ApplicationResource& app)
{
    nn::ns::Initialize();

    // インストールされているアプリケーションを取得します。
    nn::ns::ApplicationRecord record[ApplicationCountMax];
    int count = nn::ns::ListApplicationRecord(record, ApplicationCountMax, 0);
    app.sharableAppCount = 0;

    nn::ns::ApplicationView view;
    for (int i = 0; i < count; ++i)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            nn::ns::GetApplicationView(&view, &record[i].id, 1));

        ::memcpy(&app.sharableAppId[app.sharableAppCount], &record[i].id, sizeof(nn::ncm::ApplicationId));
        ++app.sharableAppCount;
    }

    app.appState = ApplicationState_ListUpApplication;
    app.cursorMax = app.sharableAppCount;
    app.cursor = 0;

    nn::ns::Finalize();
}

int CursorUp(int cursor, int min)
{
    if ((cursor - 1) < 0)
    {
        return min - 1;
    }
    return --cursor;
}

int CursorDown(int cursor, int max)
{
    if ((cursor + 1) > max - 1)
    {
        return 0;
    }
    return ++cursor;
}

void Open(ApplicationResource& app)
{
    nn::Result result;
    nn::lcs::SessionSettings settings;

    settings.applications[0].value = app.sharableAppId[app.cursor].value;
    settings.applicationCount = 1;

    settings.nodeCountMax = nn::lcs::NodeCountMax;
    settings.contentsShareVersionPolicy = nn::lcs::ContentsShareVersionPolicy_Latest;

    result = nn::lcs::OpenSession(settings);
    if (result.IsSuccess())
    {
        app.appState = ApplicationState_HostOpened;
        app.cursor = 0;
        ::std::memset(app.nodeInfo, 0, sizeof(nn::lcs::NodeInfo) * nn::lcs::NodeCountMax);
        app.nodeCount = 0;
        app.cursor = 0;

        nn::lcs::GetNodes(app.nodeInfo, &app.nodeCount, nn::lcs::NodeCountMax);
        nn::lcs::GetSessionInfo(&app.sessionInfo[0]);
    }
    else
    {
        NN_LOG("ホストの生成に失敗しました。(result:%d)\n", result.GetDescription());

        if (nn::lcs::ResultDeviceNotAvailable().Includes(result))
        {
            // フライトモードや、他のアプリケーションが無線を使用しており、
            // 無線が使用できない状態です。無線の状態を確認してください。
            app.isRunning = false;
            nn::lcs::ShowError(result);
            // 無線が使用できない状態になると、LCS/LDN を終了し、再度初期化しなければなりません。
        }
        else if (nn::lcs::ResultApplicationNotFound().Includes(result))
        {
            // 指定されたアプリケーションが見つかりません。
            // ゲームカードや SD カードが取り外された可能性があります。
            // ゲームカードや、SD カードを確認してやり直してください。
            app.isRunning = false;
            nn::lcs::ShowError(result);
        }
        else if (nn::lcs::ResultNodeCountLimitation().Includes(result))
        {
            // 接続されている無線コントローラが多すぎてホストを開始できません。
            // nn::lcs::Settings の nodeCountMax を減らすか、コントローラを切断させる必要があります。
        }
        else
        {
            NN_ABORT("不正な状態が発生しました。(nn::lcs::OpenSession : %d)\n", result.GetDescription());
        }
    }
}

void SetAcceptPolicy(ApplicationResource& app)
{
    nn::Result result;
    nn::lcs::AcceptPolicy policy;
    if (app.isAcceptPolicy)
    {
        policy = nn::lcs::AcceptPolicy_AlwaysReject;
    }
    else
    {
        policy = nn::lcs::AcceptPolicy_AlwaysAccept;
    }

    result = nn::lcs::SetClientAcceptPolicy(policy);
    if (result.IsSuccess())
    {
        app.isAcceptPolicy = !app.isAcceptPolicy;
    }
    else
    {
        NN_ABORT("不正な状態が発生しました。(nn::lcs::SetClientAcceptPolicy : %d)\n", result.GetDescription());
    }
}

void Start(ApplicationResource& app)
{
    nn::Result result;
    result = nn::lcs::StartContentsShare();
    if (result.IsFailure())
    {
        NN_ABORT("不正な状態が発生しました。(nn::lcs::StartContentsShare : %d)\n", result.GetDescription());
    }
}

void Scan(ApplicationResource& app)
{
    nn::Result result;

    // サンプルでは一度のスキャン結果で表示を行っていますが、
    // 電波環境などで一度のスキャンではホストを発見できないことがあるので、
    // 複数回 (3 回以上)のスキャンし、結果を統合することを推奨します。
    result = nn::lcs::Scan(app.sessionInfo, &(app.sessionCount), ScanResultCountMax);
    if (result.IsSuccess())
    {
        // 必要に応じて、GetDetailInfo() のような形でアプリケーションの情報を取得してください。

        app.appState = ApplicationState_Scaned;
        app.cursorMax = app.sessionCount;
        app.cursor = 0;
    }
    else
    {
        NN_LOG("スキャンに失敗しました。(result : %d)\n", result.GetDescription());

        if (nn::lcs::ResultDeviceNotAvailable().Includes(result))
        {
            // フライトモードや、他のアプリケーションが無線を使用しており、
            // 無線が使用できない状態です。無線の状態を確認してください。
            app.isRunning = false;
            nn::lcs::ShowError(result);
            // 無線が使用できない状態になると、LCS/LDN を終了し、再度初期化しなければなりません。
        }
        else
        {
            NN_ABORT("不正な状態が発生しました。(nn::lcs::Scan : %d)\n", result.GetDescription());
        }
    }
}

void Join(ApplicationResource& app)
{
    if (app.sessionCount == 0)
    {
        return;
    }

    nn::Result result;
    result = nn::lcs::JoinSession(app.sessionInfo[app.cursor]);
    if (result.IsSuccess())
    {
        app.appState = ApplicationState_ClientJoined;
        //nn::lcs::GetSessionInfo(&app.sessionInfo[0]);
    }
    else
    {
        NN_LOG("セッションへの参加に失敗しました。(result : %d)\n", result.GetDescription());

        if (nn::lcs::ResultDeviceNotAvailable().Includes(result))
        {
            // フライトモードや、他のアプリケーションが無線を使用しており、
            // 無線が使用できない状態です。無線の状態を確認してください。
            // 無線が使用できない状態になると、LCS/LDN を終了し、再度初期化しなければなりません。
            nn::lcs::ShowError(result);
        }
        else if (nn::lcs::ResultNoSharableContentsSession().Includes(result))
        {
            nn::lcs::ShowError(result);
        }
        else if (nn::lcs::ResultCommunicationError().Includes(result))
        {
            // 通信で失敗しました。リトライすることで情報をすることができる可能性があります。
            // リトライを繰り返しても失敗するときは、エラーメッセージを表示してください。
            nn::lcs::ShowError(result);
        }
        else if (nn::lcs::ResultNodeCountLimitation().Includes(result))
        {
            // 参加しようとしたセッションの参加人数が制限に達しています。
            // リトライすることで、情報をすることができる可能性があります。
            // リトライを繰り返しても失敗するときは、エラーメッセージを表示してください。
            nn::lcs::ShowError(result);
        }
        else if (nn::lcs::ResultIncompatibleVersion().Includes(result))
        {
            // インターネットに接続してシステムを更新してください。
            nn::lcs::ShowError(result);
        }
        else
        {
            NN_ABORT("不正な状態が発生しました。(nn::lcs::JoinSession : %d)\n", result.GetDescription());
        }
    }
}

void Leave(ApplicationResource& app)
{
    NN_LOG("Leave / State : %d\n", nn::lcs::GetState());
    nn::Result result;
    result = nn::lcs::LeaveSession();
    if (result.IsFailure())
    {
        NN_ABORT("不正な状態が発生しました。(nn::lcs::LeaveSession : %d)\n", result.GetDescription());
    }
}

void Suspend(ApplicationResource& app)
{
    NN_LOG("Suspend / State : %d\n", nn::lcs::GetState());
    nn::Result result;
    result = nn::lcs::SuspendSession();
    if (result.IsFailure())
    {
        NN_ABORT("不正な状態が発生しました。(nn::lcs::SuspendSession : %d)\n", result.GetDescription());
    }
}

void ResumeContentsShare()
{
    nn::Result result;
    result = nn::lcs::ResumeContentsShare();
    if (result.IsFailure())
    {
        NN_ABORT("不正な状態が発生しました。(nn::lcs::LeaveSession : %d)\n", result.GetDescription());
    }
}

void GetSessionContext()
{
    nn::Result result;
    nn::lcs::SessionContext context = {};

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::lcs::GetSessionContext(&context));

    WriteSessionContex(context);
}

void ResumeSession(ApplicationResource& app)
{
    nn::Result result;
    nn::lcs::SessionContext context = {};

    ReadSessionContex(&context);

    result = nn::lcs::ResumeSession(context);
    if (result.IsFailure())
    {
        NN_LOG("再合流に失敗しました。(result:%d)\n", result.GetDescription());

        if (nn::lcs::ResultDeviceNotAvailable().Includes(result))
        {
            // フライトモードや、他のアプリケーションが無線を使用しており、
            // 無線が使用できない状態です。無線の状態を確認してください。
            // 無線が使用できない状態になると、LCS/LDN を終了し、再度初期化しなければなりません。
            nn::lcs::ShowError(result);
        }
        else if (nn::lcs::ResultCommunicationError().Includes(result))
        {
            // 通信で失敗しました。リトライすることで接続できる可能性があります。
            // リトライを繰り返しても失敗するときは、エラーメッセージを表示してください。
            nn::lcs::ShowError(result);
        }
    }

    nn::lcs::State state = nn::lcs::GetState();
    if(state == nn::lcs::State_Transferring)
    {
        app.appState = ApplicationState_Transferring;
    }
    else if (state == nn::lcs::State_Suspended)
    {
        app.appState = ApplicationState_Suspending;
    }
}

void UpdateState(ApplicationResource& app)
{
    if (app.appState == ApplicationState_Initialized)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            switch (app.cursor)
            {
            case InitCursor_Send:
                ListUpApplication(app);
                break;
            case InitCursor_Receive:
                Scan(app);
                break;
            case InitCursor_Resume:
                ResumeSession(app);
                break;
            case InitCursor_End:
                app.isRunning = false;
                break;
            default:
                break;
            }
        }
    }
    else if (app.appState == ApplicationState_ListUpApplication)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            if (app.sharableAppCount != 0)
            {
                Open(app);
            }
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            app.cursorMax = InitCount;
            app.appState = ApplicationState_Initialized;
        }
    }
    else if (app.appState == ApplicationState_HostOpened)
    {
        app.cursorMax = app.nodeCount;
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            Leave(app);
            app.cursorMax = InitCount;
            app.appState = ApplicationState_Initialized;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            Start(app);
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
        {
            SetAcceptPolicy(app);
        }
    }
    else if (app.appState == ApplicationState_Scaned)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::A::Mask))
        {
            Join(app);
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            app.cursorMax = InitCount;
            app.appState = ApplicationState_Initialized;
        }
        else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            app.cursor = 0;
            Scan(app);
            app.cursorMax = app.sessionCount;
        }
    }
    else if (app.appState == ApplicationState_ClientJoined)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            Leave(app);
            app.cursorMax = InitCount;
            app.appState = ApplicationState_Initialized;
        }
    }
    else if (app.appState == ApplicationState_Transferring)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            Leave(app);
            app.cursorMax = InitCount;
            app.appState = ApplicationState_Initialized;
        }
    }
    else if (app.appState == ApplicationState_Suspending)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            Leave(app);
            app.cursorMax = InitCount;
            app.appState = ApplicationState_Initialized;
        }
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::X::Mask))
        {
            if (app.suspendedReason == nn::lcs::SuspendedReason_RebootRequired ||
                app.suspendedReason == nn::lcs::SuspendedReason_StorageSpaceNotEnough)
            {
                GetSessionContext();
                Suspend(app);
                app.cursorMax = InitCount;
                app.appState = ApplicationState_Initialized;
            }
        }
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Y::Mask))
        {
            ResumeContentsShare();
        }
    }
    else if (app.appState == ApplicationState_Resuming ||
             app.appState == ApplicationState_SessionEnd)
    {
        if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::B::Mask))
        {
            Leave(app);
            app.cursorMax = InitCount;
            app.appState = ApplicationState_Initialized;
        }
    }

    // 上下の選択を取得します。
    if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Up::Mask))
    {
        app.cursor = CursorUp(app.cursor, app.cursorMax);
    }
    else if (HasHidControllerAnyButtonsDown(nn::hid::NpadButton::Down::Mask))
    {
        app.cursor = CursorDown(app.cursor, app.cursorMax);
    }
} // NOLINT(impl/function_size)

void MonitorThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;

    // 接続状態の変化と監視のキャンセルイベントを同時に待ち受けます。
    auto& app = *static_cast<ApplicationResource*>(arg);
    nn::os::MultiWaitType       multiWait;
    nn::os::MultiWaitHolderType stateChangeHolder;
    nn::os::MultiWaitHolderType cancelHolder;
    nn::os::InitializeMultiWait(&multiWait);
    nn::os::InitializeMultiWaitHolder(&stateChangeHolder, &app.stateChangeEvent);
    nn::os::InitializeMultiWaitHolder(&cancelHolder, &app.cancelEvent);
    nn::os::LinkMultiWaitHolder(&multiWait, &stateChangeHolder);
    nn::os::LinkMultiWaitHolder(&multiWait, &cancelHolder);

    nn::os::ClearSystemEvent(&app.stateChangeEvent);
    nn::os::ClearEvent(&app.cancelEvent);

    nn::lcs::State state = nn::lcs::GetState();

    while (nn::os::WaitAny(&multiWait) != &cancelHolder)
    {
        // WaitAny では必ず手動でシグナルをクリアしなければなりません。
        nn::os::ClearSystemEvent(&app.stateChangeEvent);

        // LCS が終了された場合には監視を終了します。
        state = nn::lcs::GetState();
        if (state == nn::lcs::State_None)
        {
            nn::os::SignalEvent(&app.cancelEvent);
            break;
        }

        app.cursor = 0;

        if (state == nn::lcs::State_Transferring)
        {
            app.appState = ApplicationState_Transferring;
        }
        else if (state == nn::lcs::State_ContentsShareFailed)
        {
            app.failureReason = nn::lcs::GetContentsShareFailureReason();
            result = nn::lcs::ConvertFailureReasonToResult(app.failureReason);
            if (!result.IsSuccess())
            {
                nn::lcs::ShowError(result);
                Leave(app);
                app.cursorMax = InitCount;
                app.appState = ApplicationState_Initialized;
            }
        }
        else if (state == nn::lcs::State_Suspended)
        {
            // 中断理由を解決できるように処理を行ってください。
            app.appState = ApplicationState_Suspending;
            app.suspendedReason = nn::lcs::GetSuspendedReason();
        }
        else if (state == nn::lcs::State_Completed)
        {
            app.appState = ApplicationState_SessionEnd;
        }

        if (state != nn::lcs::State_Initialized && state != nn::lcs::State_ContentsShareFailed)
        {
            // セッション情報を更新します。
            result = nn::lcs::GetSessionInfo(&app.sessionInfo[0]);
            if (result.IsFailure())
            {
                NN_LOG("セッション情報の取得に失敗しました。(result:%d)\n", result.GetDescription());
                nn::lcs::ShowError(result);
                // nn::lcs::GetSessionInfo のエラーハンドリングをしてください。
            }

            // 端末情報を更新します。
            result = nn::lcs::GetNodes(app.nodeInfo, &app.nodeCount, nn::lcs::NodeCountMax);
            if (result.IsFailure())
            {
                NN_LOG("端末情報の取得に失敗しました。(result:%d)\n", result.GetDescription());
                nn::lcs::ShowError(result);
                // nn::lcs::GetNodes のエラーハンドリングをしてください。
            }

            // 現在のセッションでコンテンツを受け取るのに必要なストレージサイズを確認します。
            result = nn::lcs::GetRequiredStorageSize(&app.requiredStorageSize);
            if (result.IsFailure())
            {
                NN_LOG("必要なストレージの空き容量の取得に失敗しました。(result:%d)\n", result.GetDescription());
                nn::lcs::ShowError(result);
                // nn::lcs::GetRequiredStorageSize のエラーハンドリングをしてください。
            }
        }
    }
    nn::os::UnlinkMultiWaitHolder(&stateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWaitHolder(&stateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&cancelHolder);
    nn::os::FinalizeMultiWait(&multiWait);
}

void UpdateThread(void* arg) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(arg);

    nn::Result result;

    auto& app = *static_cast<ApplicationResource*>(arg);

    while (app.isRunning)
    {
        UpdateHidController();

        UpdateState(app);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }
}

extern "C" void nninitStartup()
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}


void SystemAppletMenuMain(nn::ae::SystemAppletParameters* pParam) NN_NOEXCEPT
{
    NN_UNUSED(pParam);

    nn::Result result;
    nn::os::ThreadType updateThread;
    ApplicationResource app;

    std::memset(&app, 0, sizeof(ApplicationResource));

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountSaveDataForDebug("save"));

    InitializeGraphicSystem();
    InitializeHidController();

    // Settings からデバイスニックネームを取得して、LCS の config に設定します。
    nn::settings::system::DeviceNickName deviceNickName;
    nn::settings::system::GetDeviceNickName(&deviceNickName);
    app.config.SetName(deviceNickName.string);

    // LCS では SOCKET を使用するので初期化が必要です。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::socket::Initialize(g_SocketConfigWithMemory));

    nn::os::InitializeEvent(&app.cancelEvent, false, nn::os::EventClearMode_ManualClear);

    // LCSの開始に必要な初期化を行います。
    if (!InitilazeLcs(app))
    {
        return;
    }
    app.appState = ApplicationState_Initialized;

    // 入力を取得し、状態を変化させるスレッドを作成し、開始します。
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&updateThread, UpdateThread, &app,
        g_UpdateThreadStack, sizeof(g_UpdateThreadStack), UpdateThreadPriority));
    nn::os::StartThread(&updateThread);

    while (app.isRunning)
    {
        DrawText(app);

        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(16));
    }

    nn::os::WaitThread(&updateThread);
    nn::os::DestroyThread(&updateThread);

    FinalizeLcs(app);

    nn::os::FinalizeEvent(&app.cancelEvent);
    // SOCKET の終了です。
    nn::socket::Finalize();

    FinalizeGraphicSystem();
    FinalizeHidController();

    NN_LOG("Unmount \"save\"\n\n");
    nn::fs::Unmount("save");

    nn::ae::StartRebootSequence();
}

extern "C" void nnMain()
{
    nn::ae::InvokeSystemAppletMain(nn::ae::AppletId_SystemAppletMenu, SystemAppletMenuMain);
}

