﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{IrSensorIrLedBasic.cpp,PageSampleIrSensorIrLedBasic}

    @brief
    シンプルにモーションIRカメラの IR LED モードを使用するためのサンプルプログラム
 */

/**
    @page PageSampleIrSensorIrLedBasic シンプルなモーションIRカメラのIR LEDモードの制御
    @tableofcontents

    @brief
    シンプルにモーションIRカメラの IR LED モードを使用するためのサンプルプログラムの解説です。

    @section PageSampleIrSensorIrLedBasic_SectionBrief 概要
    モーションIRカメラの IR LED モードを使用する方法の説明をします。

    @section PageSampleIrSensorIrLedBasic_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/IrSensorIrLedBasic @endlink 以下にあります。

    @section PageSampleIrSensorIrLedBasic_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラをペアリングしてください。

    @section PageSampleIrSensorIrLedBasic_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。

    接続後、コントローラのボタンを押すと、対応した IR LED が点灯します。
    また、コンソールのログに、設定した内容と、取得した IRカメラの状態が出力されます。

    サンプルプログラムを終了させるには + ボタンまたは - ボタンを押してください。

    @section PageSampleIrSensorIrLedBasic_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。
    本サンプルプログラムでは画面描画は行いません。
    モーションIRカメラが搭載されているのは Joy-Con (R) のみですが、
    コントローラのスタイルとして、FullKey, Handheld, JoyRight, JoyLeft に対応しており、
    非搭載のコントローラに対する挙動も確認できるようになっています。
    モーションIRカメラの API には呼び出し元スレッドを長時間ブロックするものが存在するため、
    メインループとは別のスレッドから呼び出すことを推奨します。 詳細については、API リファレンスをご覧ください。

    @section PageSampleIrSensorIrLedBasic_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleIrSensorIrLedBasic_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - NpadID を設定
    - NpadID からモーションIRカメラのハンドルを取得
    - ユーザのボタン入力を元に、IR LEDプロセッサを設定
    - モーションIRカメラの状態を取得
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/irsensor.h>
#include <nn/irsensor/irsensor_IrLedProcessorApi.h>

namespace {

    struct Npad
    {
        typedef nn::hid::NpadIdType Id;
        typedef nn::hid::NpadButton Button;
    };

    Npad::Id npadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::Handheld,
    };
    const int NpadIdCountMax = sizeof(npadIds) / sizeof(nn::hid::NpadIdType);

    nn::irsensor::IrCameraHandle irCameraHandles[NpadIdCountMax];

    bool isFinished = false;

    void CheckFirmwareUpdate(const nn::irsensor::IrCameraHandle& irCameraHandle)
    {
        const int FirmwareCheckTrialCountMax = 300;

        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            bool isUpdateNeeded = false;
            nn::Result result = nn::irsensor::CheckFirmwareUpdateNecessity(&isUpdateNeeded, irCameraHandle);
            if (result.IsSuccess())
            {
                // 成功時は、ファームウェアの更新が必要かどうかのフラグが返ります。
                NN_LOG("    Controller firmware update checking succeed.\n");
                if (isUpdateNeeded)
                {
                    // コントローラの更新が必要な場合は、更新アプレットを呼び出すことを推奨します。
                    // 更新アプレットはブロック処理で、接続された全てのコントローラのアップデートを行うため、呼び出しには注意が必要です。
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                    // 選択 UI なしの強制アップデート版コンサポ呼び出し
                    // 成功しても失敗しても 1 回だけ呼び出し。
                    nn::hid::ControllerFirmwareUpdateArg arg;
                    arg.enableForceUpdate = false;
                    nn::hid::ShowControllerFirmwareUpdate(arg);
#elif defined( NN_BUILD_CONFIG_OS_WIN )
                    NN_LOG("Invoke ControllerFirmwareUpdate is not suppported on Windows environment.\n");
#else
                    #error "unsupported os"
#endif
                }
                break;
            }
            else if (::nn::irsensor::ResultIrsensorUnavailable::Includes(result))
            {
                // コントローラが使用不可状態の場合は、スキップする。
                NN_LOG("    Controller is not available.\n");
                break;
            }
            else if (::nn::irsensor::ResultIrsensorFirmwareCheckIncompleted::Includes(result))
            {
                // FirmwareUpdate のチェック中、もしくはアプレット呼び出し中なのでリトライする
                counter++;
                if (counter > FirmwareCheckTrialCountMax)
                {
                    // チェック中の状態がしばらく続いた場合は、タイムアウトすることを推奨します。
                    NN_LOG("    Firmware update checking is timeout.\n");
                    break;
                }
                else
                {
                    NN_LOG("    Firmware update checking is running.\n");
                }
            }
            else
            {
                NN_ABORT("    Firmware update checking unexpected error\n");
            }
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        }
    }

    void PrintStatus(const nn::irsensor::IrCameraHandle& irCameraHandle)
    {
        //IR カメラの状態の取得
        nn::irsensor::IrCameraStatus irCameraStatus = nn::irsensor::GetIrCameraStatus(irCameraHandle);

        //IR カメラが利用可能
        if (irCameraStatus == nn::irsensor::IrCameraStatus_Available)
        {
            NN_LOG("IR Camera Available\n");
        }
        //コントローラが未接続
        else if (irCameraStatus == nn::irsensor::IrCameraStatus_Unconnected)
        {
            NN_LOG("IR Camera Unconnected\n");
        }
        //コントローラに IR カメラが非搭載
        else if (irCameraStatus == nn::irsensor::IrCameraStatus_Unsupported)
        {
            NN_LOG("IR Camera Unsupported\n");
        }
    }

    template <typename Type>
    void UpdateIrLed(Type state, int i)
    {
        nn::irsensor::IrLedProcessorConfig config;
        nn::irsensor::GetIrLedProcessorDefaultConfig(&config);
        if (state.buttons.template Test<nn::hid::NpadButton::A>())
        {
            NN_LOG("IR LED: AllObjects\n");
            config.lightTarget = nn::irsensor::IrCameraLightTarget_AllObjects;
            nn::irsensor::RunIrLedProcessor(irCameraHandles[i], config);
        }
        else if (state.buttons.template Test<nn::hid::NpadButton::B>())
        {
            NN_LOG("IR LED: None\n");
            config.lightTarget = nn::irsensor::IrCameraLightTarget_None;
            nn::irsensor::RunIrLedProcessor(irCameraHandles[i], config);
        }
        else if (state.buttons.template Test<nn::hid::NpadButton::X>())
        {
            NN_LOG("IR LED: NearObjects\n");
            config.lightTarget = nn::irsensor::IrCameraLightTarget_NearObjects;
            nn::irsensor::RunIrLedProcessor(irCameraHandles[i], config);
        }
        else if (state.buttons.template Test<nn::hid::NpadButton::Y>())
        {
            NN_LOG("IR LED: FarObjects\n");
            config.lightTarget = nn::irsensor::IrCameraLightTarget_FarObjects;
            nn::irsensor::RunIrLedProcessor(irCameraHandles[i], config);
        }
        else
        {
            //その他のボタンが押されていたらモーションIRカメラの状態を出力
            PrintStatus(irCameraHandles[i]);
        }
    }

    void Update()
    {
        for(int i = 0; i < NpadIdCountMax; i++)
        {
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(npadIds[i]);

            if (style.Test<nn::hid::NpadStyleJoyRight>())
            {
                nn::hid::NpadJoyRightState stateJoyRight;
                nn::hid::GetNpadState(&stateJoyRight, npadIds[i]);

                // ＋ ボタンが押されたら終了
                if (stateJoyRight.buttons.Test<nn::hid::NpadButton::Plus>())
                {
                    isFinished = true;
                }
                else if (stateJoyRight.buttons.IsAnyOn())
                {
                    UpdateIrLed<nn::hid::NpadJoyRightState>(stateJoyRight, i);
                }
            }

            if (style.Test<nn::hid::NpadStyleJoyLeft>())
            {
                nn::hid::NpadJoyLeftState stateJoyLeft;
                nn::hid::GetNpadState(&stateJoyLeft, npadIds[i]);

                // - ボタンが押されたら終了
                if (stateJoyLeft.buttons.Test<nn::hid::NpadButton::Minus>())
                {
                    isFinished = true;
                }
                else if (stateJoyLeft.buttons.IsAnyOn())
                {
                    PrintStatus(irCameraHandles[i]);
                }
            }

            if (style.Test<nn::hid::NpadStyleHandheld>())
            {
                nn::hid::NpadHandheldState stateHandheld;
                nn::hid::GetNpadState(&stateHandheld, npadIds[i]);

                // ＋ ボタンが押されたら終了
                if (stateHandheld.buttons.Test<nn::hid::NpadButton::Plus>())
                {
                    isFinished = true;
                }
                else if (stateHandheld.buttons.IsAnyOn())
                {
                    UpdateIrLed<nn::hid::NpadHandheldState>(stateHandheld, i);
                }
            }

            if (style.Test<nn::hid::NpadStyleFullKey>())
            {
                nn::hid::NpadFullKeyState stateFullKey;
                nn::hid::GetNpadState(&stateFullKey, npadIds[i]);

                // ＋ ボタンが押されたら終了
                if (stateFullKey.buttons.Test<nn::hid::NpadButton::Plus>())
                {
                    isFinished = true;
                }
                else if (stateFullKey.buttons.IsAnyOn())
                {
                    UpdateIrLed<nn::hid::NpadFullKeyState>(stateFullKey, i);
                }
            }
        }
    }

}//namespace

extern "C" void nnMain()
{
    NN_LOG("IR Sensor IrLed Basic Sample(Npad) Start.\n");

    nn::hid::InitializeNpad();

    //Npadのスタイルを設定
    nn::hid::SetSupportedNpadStyleSet(
        ::nn::hid::NpadStyleJoyRight::Mask
        | ::nn::hid::NpadStyleJoyLeft::Mask
        | ::nn::hid::NpadStyleHandheld::Mask
        | ::nn::hid::NpadStyleFullKey::Mask
    );

    //使用するNpadId の設定
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        // 1本ずつ割り当て
        nn::hid::SetNpadJoyAssignmentModeSingle(npadIds[i]);
        //IR カメラのハンドルの取得
        irCameraHandles[i] = nn::irsensor::GetIrCameraHandle(npadIds[i]);
        //IR カメラの初期化
        nn::irsensor::Initialize(irCameraHandles[i]);
        NN_LOG("NpadPlayerNumber(%d)\n",npadIds[i]);
        // FirmwareUpdate のチェック
        CheckFirmwareUpdate(irCameraHandles[i]);
    }

    NN_LOG("If you push button, you can change IR Led state.\n");
    NN_LOG("    A: All   B: None   X: Near Only Y: Far Only\n");
    NN_LOG("Push (+) or (-) Button to shutdown this application.\n");

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        nn::irsensor::IrLedProcessorConfig config;
        //IR LEDプロセッサのデフォルト設定の取得
        nn::irsensor::GetIrLedProcessorDefaultConfig(&config);
        //IR LEDプロセッサの開始
        nn::irsensor::RunIrLedProcessor(irCameraHandles[i], config);
    }

    while(!isFinished)
    {
        Update();
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        //IR LEDプロセッサの停止
        nn::irsensor::StopImageProcessor(irCameraHandles[i]);
        //IR カメラの終了処理
        nn::irsensor::Finalize(irCameraHandles[i]);
    }

    NN_LOG("IR Sensor IrLed Basic Sample Done\n");
}
