﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "IrSensorModeState.h"

#include <vector>

class HeartRateModeState : public IrSensorModeState
{
public:
    HeartRateModeState(IrSensorMode* pNextProcessor, int* pMenuSelection, nn::irsensor::IrCameraHandle irCameraHandle);

    void Start() NN_OVERRIDE;
    void Update() NN_OVERRIDE;
    void Render(nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer, nn::gfx::CommandBuffer* pCommandBuffer, int index) NN_OVERRIDE;
    void Reset();

private:
    const nn::util::Unorm8x4 GetCaptureStateColor();

private:
    nn::irsensor::HeartRateProcessorWorkBuffer m_HeartRateWorkBuffer;
    nn::irsensor::HeartRateProcessorConfig m_HeartRateCaptureConfig;
    nn::irsensor::HeartRateProcessorState m_lastState;

    std::vector<float> m_pulseSignal;

    class SlidingWindowView
    {
    public:
        SlidingWindowView(std::vector<float> const& signal, size_t windowSizeInSamples, size_t delayInSamples)
            : m_signal(signal)
            , m_windowSizeInSamples(windowSizeInSamples)
            , m_delayInSamples(delayInSamples)
            , m_readCursor(0)
            , m_frameId(0)
            , m_prevScale(0)
        {}

        void Clear()
        {
            m_readCursor = 0;
            m_frameId = 0;
            m_prevScale = 0;
        }

        float const* Get(size_t *windowSize, float *scale)
        {
            size_t const signalSize = m_signal.size();
            float const* data = m_signal.data();

            if (signalSize < m_prevSignalSize)
                Clear();

            if (signalSize < m_delayInSamples)
            {
                *scale = 0.f;
                *windowSize = 0;
                m_prevSignalSize = signalSize;
                return data;
            }

            if (m_readCursor < signalSize)
            {
                size_t const skipIncEachNFrame = 4;
                if (++m_frameId % skipIncEachNFrame)
                    ++m_readCursor;
            }

            *windowSize = m_readCursor > m_windowSizeInSamples ? m_windowSizeInSamples : m_readCursor;
            size_t const offset = m_readCursor - *windowSize;

            auto GetScale = [&](size_t left, size_t right)
            {
                float min = m_signal[left];
                float max = m_signal[left];
                for (size_t i = left + 1; i < signalSize && i < right; ++i)
                {
                    float const val = m_signal[i];
                    min = std::min(min, val);
                    max = std::max(max, val);
                }

                float const scale = max > min ? 1 / (max - min) : 0;
                return scale;
            };
            auto lerp = [&](float a, float b, float f) { return a + (b - a) * f; };
            float const curScale = GetScale(offset, m_readCursor);
            *scale = m_readCursor < m_windowSizeInSamples ? curScale : lerp(curScale, m_prevScale, 0.8f);

            m_prevScale = *scale;
            m_prevSignalSize = signalSize;
            return data + offset;
        }

    private:
        std::vector<float> const& m_signal;
        size_t const m_windowSizeInSamples;
        size_t const m_delayInSamples;
        size_t m_readCursor;
        size_t m_prevSignalSize;
        size_t m_frameId;
        float m_prevScale;
    };

    SlidingWindowView m_pulseSignalSlidingWindowView;
};
