﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{IrSensorClusteringBasic.cpp,PageSampleIrSensorClusteringBasic}

    @brief
    シンプルにモーションIRカメラのクラスタリングモードを使用するためのサンプルプログラム
 */

/**
    @page PageSampleIrSensorClusteringBasic シンプルなモーションIRカメラのクラスタリングモードの制御
    @tableofcontents

    @brief
    シンプルにモーションIRカメラのクラスタリングモードを使用するためのサンプルプログラムの解説です。

    @section PageSampleIrSensorClusteringBasic_SectionBrief 概要
    モーションIRカメラのクラスタリングモードを使用する方法の説明をします。

    @section PageSampleIrSensorClusteringBasic_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/IrSensorClusteringBasic @endlink 以下にあります。

    @section PageSampleIrSensorClusteringBasic_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラをペアリングしてください。

    @section PageSampleIrSensorClusteringBasic_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。

    接続後、コントローラのボタンを押している間、そのコントローラのクラスタリングモードの状態を取得します。
    コンソールのログに、取得した状態が出力されます。

    サンプルプログラムを終了させるには + ボタンまたは - ボタンを押してください。

    @section PageSampleIrSensorClusteringBasic_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。
    本サンプルプログラムでは画面描画は行いません。
    モーションIRカメラが搭載されているのは Joy-Con (R) のみですが、
    コントローラのスタイルとして、FullKey, Handheld, JoyRight, JoyLeft に対応しており、
    非搭載のコントローラに対する挙動も確認できるようになっています。
    モーションIRカメラの API には呼び出し元スレッドを長時間ブロックするものが存在するため、
    メインループとは別のスレッドから呼び出すことを推奨します。 詳細については、API リファレンスをご覧ください。

    @section PageSampleIrSensorClusteringBasic_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleIrSensorClusteringBasic_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - NpadID を設定
    - NpadID からモーションIRカメラのハンドルを取得
    - モーションIRカメラの状態を取得
    - クラスタリングモードによる処理結果の取得
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/irsensor.h>

namespace {

    struct Npad
    {
        typedef nn::hid::NpadIdType Id;
        typedef nn::hid::NpadButton Button;
    };

    Npad::Id npadIds[] =
    {
        nn::hid::NpadId::No1,
        nn::hid::NpadId::No2,
        nn::hid::NpadId::No3,
        nn::hid::NpadId::Handheld,
    };
    const int NpadIdCountMax = sizeof(npadIds) / sizeof(nn::hid::NpadIdType);

    nn::irsensor::IrCameraHandle irCameraHandles[NpadIdCountMax];

    bool isFinished = false;

    void CheckFirmwareUpdate(const nn::irsensor::IrCameraHandle& irCameraHandle)
    {
        const int FirmwareCheckTrialCountMax = 300;

        int counter = 0;
        while (NN_STATIC_CONDITION(true))
        {
            bool isUpdateNeeded = false;
            nn::Result result = nn::irsensor::CheckFirmwareUpdateNecessity(&isUpdateNeeded, irCameraHandle);
            if (result.IsSuccess())
            {
                // 成功時は、ファームウェアの更新が必要かどうかのフラグが返ります。
                NN_LOG("    Controller firmware update checking succeed.\n");
                if (isUpdateNeeded)
                {
                    // コントローラの更新が必要な場合は、更新アプレットを呼び出すことを推奨します。
                    // 更新アプレットはブロック処理で、接続された全てのコントローラのアップデートを行うため、呼び出しには注意が必要です。
#if defined( NN_BUILD_CONFIG_OS_HORIZON )
                    // 選択 UI なしの強制アップデート版コンサポ呼び出し
                    // 成功しても失敗しても 1 回だけ呼び出し。
                    nn::hid::ControllerFirmwareUpdateArg arg;
                    arg.enableForceUpdate = false;
                    nn::hid::ShowControllerFirmwareUpdate(arg);
#elif defined( NN_BUILD_CONFIG_OS_WIN )
                    NN_LOG("Invoke ControllerFirmwareUpdate is not suppported on Windows environment.\n");
#else
                    #error "unsupported os"
#endif
                }
                break;
            }
            else if (::nn::irsensor::ResultIrsensorUnavailable::Includes(result))
            {
                // コントローラが使用不可状態の場合は、スキップする。
                NN_LOG("    Controller is not available.\n");
                break;
            }
            else if (::nn::irsensor::ResultIrsensorFirmwareCheckIncompleted::Includes(result))
            {
                // FirmwareUpdate のチェック中、もしくはアプレット呼び出し中なのでリトライする
                counter++;
                if (counter > FirmwareCheckTrialCountMax)
                {
                    // チェック中の状態がしばらく続いた場合は、タイムアウトすることを推奨します。
                    NN_LOG("    Firmware update checking is timeout.\n");
                    break;
                }
                else
                {
                    NN_LOG("    Firmware update checking is running.\n");
                }
            }
            else
            {
                NN_ABORT("    Firmware update checking unexpected error\n");
            }
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
        }
    }

    void PrintStatus(const nn::irsensor::IrCameraHandle& irCameraHandle)
    {
        //IR カメラの状態の取得
        nn::irsensor::IrCameraStatus irCameraStatus = nn::irsensor::GetIrCameraStatus(irCameraHandle);

        //IR カメラが利用可能
        if (irCameraStatus == nn::irsensor::IrCameraStatus_Available)
        {
            NN_LOG("IR Camera Available\n");
        }
        //コントローラが未接続
        else if (irCameraStatus == nn::irsensor::IrCameraStatus_Unconnected)
        {
            NN_LOG("IR Camera Unconnected\n");
        }
        //コントローラに IR カメラが非搭載
        else if (irCameraStatus == nn::irsensor::IrCameraStatus_Unsupported)
        {
            NN_LOG("IR Camera Unsupported\n");
        }

        nn::irsensor::ClusteringProcessorState state;
        //クラスタリングプロセッサの処理結果の取得
        nn::Result result = nn::irsensor::GetClusteringProcessorState(&state, irCameraHandle);
        if (nn::irsensor::ResultIrsensorNotReady::Includes(result))
        {
            // NotReady のエラーが返る場合は、データの準備中ですので、正しいデータが来るまでリトライします。
            NN_LOG("Error: Data is not ready\n");
            return;
        }
        else if (nn::irsensor::ResultIrsensorUnsupported::Includes(result))
        {
            // コントローラにモーションIRカメラが非搭載の場合は、
            // ユーザに適切なコントローラを接続するよう案内します。
            NN_LOG("Error: Controller is unsupported\n");
            return;
        }
        else if (nn::irsensor::ResultIrsensorUnconnected::Includes(result))
        {
            // コントローラが非接続の場合は、ユーザに案内し、接続されるまでリトライすることを推奨します。
            NN_LOG("Error: Controller is unconnected\n");
            return;
        }
        else if (nn::irsensor::ResultIrsensorDeviceError::Includes(result))
        {
            // DeviceError が発生した場合は、モーションIRカメラは使用できませんので、一度 プロセッサを停止します。
            // この場合、詳細なエラー内容を示したエラービューア、またはコントローラのアップデートが表示されるため、
            // ユーザは問題の解決を試みます。
            NN_LOG("Error: DeviceError occured\n");
            isFinished = true;
            return;
        }

        //サンプリング番号
        NN_LOG("SamplingNumber    : %lld\n", state.samplingNumber);
        //処理開始からの経過時間
        NN_LOG("TimeStamp         : %lld us\n", state.timeStamp.GetMicroSeconds());
        //検出されたオブジェクトの個数
        NN_LOG("ObjectCount       : %d\n", state.objectCount);

        for (int i = 0; i < state.objectCount; ++i)
        {
            //オブジェクトの平均輝度値
            float averageIntensity = state.objects[i].averageIntensity;
            //オブジェクトの外接矩形
            int boundX = state.objects[i].bound.x;
            int boundY = state.objects[i].bound.y;
            int boundWidth = state.objects[i].bound.width;
            int boundHeight = state.objects[i].bound.height;
            //オブジェクトの重心
            float centroidX = state.objects[i].centroid.x;
            float centroidY = state.objects[i].centroid.y;
            //オブジェクトの面積
            int pixelCount = state.objects[i].pixelCount;
            NN_LOG("Object(%d)\n", i);
            NN_LOG("  AverageIntensity: %3.1f\n", averageIntensity);
            NN_LOG("  Centroid        : (%3.1f, %3.1f)\n", centroidX, centroidY);
            NN_LOG("  PixelCount      : %d\n", pixelCount);
            NN_LOG("  BoundX          : %d\n", boundX);
            NN_LOG("  BoundY          : %d\n", boundY);
            NN_LOG("  BoundWidth      : %d\n", boundWidth);
            NN_LOG("  BoundHeight     : %d\n", boundHeight);
        }
        NN_LOG("\n");
    }

    void Update()
    {
        for(int i = 0; i < NpadIdCountMax; i++)
        {
            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(npadIds[i]);

            if (style.Test<nn::hid::NpadStyleJoyRight>())
            {
                nn::hid::NpadJoyRightState stateJoyRight;
                nn::hid::GetNpadState(&stateJoyRight, npadIds[i]);

                // ＋ ボタンが押されたら終了
                if (stateJoyRight.buttons.Test<nn::hid::NpadButton::Plus>())
                {
                    isFinished = true;
                }
                //ボタンが押されていたらモーションIRカメラの状態と処理結果を取得して出力
                else if (stateJoyRight.buttons.IsAnyOn())
                {
                    NN_LOG("Right Controller's buttons are Pressed\n");

                    PrintStatus(irCameraHandles[i]);
                }
            }

            if (style.Test<nn::hid::NpadStyleJoyLeft>())
            {
                nn::hid::NpadJoyLeftState stateJoyLeft;
                nn::hid::GetNpadState(&stateJoyLeft, npadIds[i]);

                // - ボタンが押されたら終了
                if (stateJoyLeft.buttons.Test<nn::hid::NpadButton::Minus>())
                {
                    isFinished = true;
                }
                //ボタンが押されていたらモーションIRカメラの状態と処理結果を取得して出力
                else if (stateJoyLeft.buttons.IsAnyOn())
                {
                    NN_LOG("Left Controller's buttons are Pressed\n");

                    PrintStatus(irCameraHandles[i]);
                }
            }

            if (style.Test<nn::hid::NpadStyleHandheld>())
            {
                nn::hid::NpadHandheldState stateHandheld;
                nn::hid::GetNpadState(&stateHandheld, npadIds[i]);

                // ＋ ボタンが押されたら終了
                if (stateHandheld.buttons.Test<nn::hid::NpadButton::Plus>())
                {
                    isFinished = true;
                }
                //ボタンが押されていたらモーションIRカメラの状態と処理結果を取得して出力
                else if (stateHandheld.buttons.IsAnyOn())
                {
                    NN_LOG("Handheld Controller's buttons are Pressed\n");

                    PrintStatus(irCameraHandles[i]);
                }
            }

            if (style.Test<nn::hid::NpadStyleFullKey>())
            {
                nn::hid::NpadFullKeyState stateFullKey;
                nn::hid::GetNpadState(&stateFullKey, npadIds[i]);

                // ＋ ボタンが押されたら終了
                if (stateFullKey.buttons.Test<nn::hid::NpadButton::Plus>())
                {
                    isFinished = true;
                }
                //ボタンが押されていたらモーションIRカメラの状態と処理結果を取得して出力
                else if (stateFullKey.buttons.IsAnyOn())
                {
                    NN_LOG("FullKey Controller's buttons are Pressed\n");

                    PrintStatus(irCameraHandles[i]);
                }
            }
        }
    }

}//namespace

extern "C" void nnMain()
{
    NN_LOG("IR Sensor Clustering Basic Sample(Npad) Start.\n");

    nn::hid::InitializeNpad();

    //Npadのスタイルを設定
    nn::hid::SetSupportedNpadStyleSet(
        ::nn::hid::NpadStyleJoyRight::Mask
        | ::nn::hid::NpadStyleJoyLeft::Mask
        | ::nn::hid::NpadStyleHandheld::Mask
        | ::nn::hid::NpadStyleFullKey::Mask
    );

    //使用するNpadId の設定
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        // 1本ずつ割り当て
        nn::hid::SetNpadJoyAssignmentModeSingle(npadIds[i]);
        //IR カメラのハンドルの取得
        irCameraHandles[i] = nn::irsensor::GetIrCameraHandle(npadIds[i]);
        //IR カメラの初期化
        nn::irsensor::Initialize(irCameraHandles[i]);
        NN_LOG("NpadPlayerNumber(%d)\n",npadIds[i]);
        // FirmwareUpdate のチェック
        CheckFirmwareUpdate(irCameraHandles[i]);
    }

    NN_LOG("If you push any button, you can get IR sensor state.\n");
    NN_LOG("Push (+) or (-) Button to shutdown this application.\n");

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        nn::irsensor::ClusteringProcessorConfig config;
        //クラスタリングプロセッサのデフォルト設定の取得
        nn::irsensor::GetClusteringProcessorDefaultConfig(&config);
        //クラスタリングプロセッサの開始
        nn::irsensor::RunClusteringProcessor(irCameraHandles[i], config);
    }

    while(!isFinished)
    {
        Update();
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }

    for (int i = 0; i < NpadIdCountMax; ++i)
    {
        //クラスタリングプロセッサの停止
        nn::irsensor::StopImageProcessor(irCameraHandles[i]);
        //IR カメラの終了処理
        nn::irsensor::Finalize(irCameraHandles[i]);
    }

    NN_LOG("IR Sensor Clustering Basic Sample Done\n");
}
