﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{InitStartupWithMalloc.cpp,PageSampleInitStartupWithMalloc}
 *
 * @brief
 *  nninitStartup() 関数を使ったサンプルプログラムその 2
 */

/**
 * @page PageSampleInitStartupWithMalloc nninitStartupWithMalloc
 * @tableofcontents
 *
 * @brief
 *  nninitStartup() 関数のサンプルプログラムその 2 の解説です。
 *
 * @section PageSampleInitStartupWithMalloc_SectionBrief 概要
 *  ここでは、アプリケーションが独自の nninitStartup() 関数を定義し、
 *  かつ malloc(), free(), calloc(), realloc(), aligned_alloc(), malloc_usable_size()
 *  も独自の実装を行なう場合のサンプルプログラムについて説明します。
 *  nninitStartup() のリファレンスもあわせて参照して下さい。
 *
 * @section PageSampleInitStartupWithMalloc_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/InitStartupWithMalloc Samples/Sources/Applications/InitStartupWithMalloc @endlink 以下にあります。
 *  本サンプルは Windows 環境では利用できません。Visual Studio 関連ファイルは
 *  開発機環境用のもののみが提供されます。
 *
 * @section PageSampleInitStartupWithMalloc_SectionNecessaryEnvironment 必要な環境
 *  とくになし
 *
 * @section PageSampleInitStartupWithMalloc_SectionHowToOperate 操作方法
 *  とくになし
 *
 * @section PageSampleInitStartupWithMalloc_SectionPrecaution 注意事項
 *  このデモは画面上に何も表示されません。実行結果はログに出力されます。
 *
 * @section PageSampleInitStartupWithMalloc_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleInitStartupWithMalloc_SectionDetail 解説
 *
 * @subsection PageSampleInitStartupWithMalloc_SectionSampleProgram サンプルプログラム
 *  以下に本サンプルプログラムのソースコードを引用します。
 *
 *  InitStartupWithMalloc.cpp
 *  @includelineno InitStartupWithMalloc.cpp
 *
 * @subsection PageSampleInitStartupWithMalloc_SectionSampleDetail サンプルプログラムの解説
 *  先のサンプルプログラムの全体像は以下の通りです。
 *
 *  - nninitStartup() で malloc() 用のメモリアロケータを初期化
 *  - malloc(), free(), calloc(), realloc(), aligned_alloc(), malloc_usable_size() 関数を独自に定義
 *  - nnMain() で上記 malloc() で獲得したメモリのアドレスをログ出力して終了
 *
 *  @ref PageSampleInitStartup と比べると、
 *  本サンプルでは SDK デフォルトの malloc() を使用せずに、
 *  独自で malloc() を定義して使用します。
 *
 *  まず最初に nninitStartup() が呼ばれ、静的に確保した g_MallocBuffer[] の
 *  配列をメモリアロケータで使用するメモリとして初期化します。アロケータには、
 *  nn::mem::StandardAllocator() を使用します。
 *
 *  一つ注意すべきことは、通常、静的オブジェクトのコンストラクタは、
 *  nninitStartup() よりも後で呼び出されるということです。
 *  ここでは nninitStartup() の中でコンストラクタを起動し、
 *  その後はもうコンストラクタが起動しないようになっているべきなので
 *  以下のように実装しています。
 *
 *  - 静的オブジェクト g_SampleAllocator を nn::util::TypedStorage を使って
 *    領域だけ確保し、自動的なコンストラクタ起動を抑制
 *  - nninitStartup() の中で g_SampleAllocator を placement new を使って
 *    明示的にコンストラクタを起動
 *
 *  その後はもうこのアロケータが使用可能になっているため、
 *  malloc() や free() の実装の中で、このアロケータを使用しています。
 *
 *  最後に nnMain() が起動します。nnMain() の中で実際に malloc() を呼び出し、
 *  獲得したアドレスをログ出力して終了しています。
 *
 *  最終的なログ表示を以下に示します。
 *
 *  @verbinclude  InitStartupWithMalloc_OutputExample.txt
 *
 *  なお、SDK がデフォルトで用意している malloc() を利用したい場合は、
 *  @ref PageSampleInitStartup を参照して下さい。
 *
 *  このサンプルは Windows 環境では利用できません。
 *  Windows 環境で malloc() や free() を独自に定義すると、シンボルの重複で
 *  リンクエラーになります。
 *
 */

#include <new>
#include <cstdlib>
#include <cstring>

#include <nn/nn_Common.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>

#include <nn/util/util_TypedStorage.h>
#include <nn/init.h>
#include <nn/os.h>
#include <nn/mem.h>


//-----------------------------------------------------------------------------
//  アロケータの定義
//-----------------------------------------------------------------------------

namespace {

    // malloc の領域として静的な領域を確保
    const size_t                MallocBufferSize = 4 * 1024 * 1024;
    NN_ALIGNAS(4096) uint8_t    g_MallocBuffer[ MallocBufferSize ];

    // 初期化は nninitStartup() から行なうため、
    // nninitStartup() 後にコンストラクタが起動しないようにしておく。
    nn::util::TypedStorage<nn::mem::StandardAllocator,sizeof(nn::mem::StandardAllocator),NN_ALIGNOF(nn::mem::StandardAllocator)>    g_SampleAllocator;

}   // anonymous

//-----------------------------------------------------------------------------
//  malloc, free, calloc, realloc, aligned_alloc, malloc_usable_size の 6 つを同時に定義する
//-----------------------------------------------------------------------------

extern "C" void* malloc(size_t size)
{
    return Get(g_SampleAllocator).Allocate(size);
}

extern "C" void free(void* p)
{
    if (p)
    {
        Get(g_SampleAllocator).Free(p);
    }
}

extern "C" void* calloc(size_t num, size_t size)
{
    size_t sum = num * size;
    void*  p   = std::malloc(sum);

    if (p)
    {
        std::memset(p, 0, sum);
    }
    return p;
}

extern "C" void* realloc(void* p, size_t newSize)
{
    // メモリブロックのサイズを変更する
    // Reallocate() の仕様は realloc() の仕様と同じため、そのまま呼ぶ
    return Get(g_SampleAllocator).Reallocate(p, newSize);
}

extern "C" void* aligned_alloc(size_t alignment, size_t size)
{
    return Get(g_SampleAllocator).Allocate(size, alignment);
}

extern "C" size_t malloc_usable_size(void* p)
{
    if (!p)
    {
        return 0;
    }
    return Get(g_SampleAllocator).GetSizeOf(p);
}

//-----------------------------------------------------------------------------
//  スタートアップ関数
//-----------------------------------------------------------------------------

extern "C" void nninitStartup()
{
    NN_LOG("step1: A nninitStartup() function is invoked.\n");

    // nninitStartup() では静的オブジェクトのコンストラクタは
    // まだ呼ばれていないため、placement new で明示的に呼ぶ。
    new( &Get(g_SampleAllocator) ) nn::mem::StandardAllocator;

    // 静的に確保したバッファを malloc 用の領域として初期化
    Get(g_SampleAllocator).Initialize(g_MallocBuffer, MallocBufferSize);

    NN_LOG("step1: MallocBuffer is 0x%p - 0x%p\n\n",
                    g_MallocBuffer, g_MallocBuffer + MallocBufferSize);
}

//-----------------------------------------------------------------------------
//  メイン関数
//-----------------------------------------------------------------------------

extern "C" void nnMain()
{
    NN_LOG("step2: nnMain() function is invoked.\n");

    void* address = malloc(1024);
    NN_LOG("step2: malloc() address = 0x%p\n", address);
    free(address);

    NN_LOG("step2: nnMain() function is finished.\n");
}
