﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

/**
    @brief FS用アロケータ
 */
void* Allocate(size_t size)
{
    return std::malloc(size);
}

/**
    @brief FS用デアロケータ
 */
void Deallocate(void* p, size_t size)
{
    NN_UNUSED(size);
    std::free(p);
}

class Buffer
{
public:
    explicit Buffer(size_t size) :
        m_data(nullptr),
        m_size(0)
    {
        m_size = size;
        m_data = (size > 0? std::malloc(m_size): nullptr);
    }
    ~Buffer()
    {
        Release();
    }
    void Release()
    {
        if (m_data != nullptr)
        {
            std::free(m_data);
            m_data = nullptr;
        }
    }
    void *GetDataPtr() const
    {
        return m_data;
    }
    size_t GetDataSize() const
    {
        return m_size;
    }

private:
    void *m_data;
    size_t m_size;
};

class FileData
{
public:
    static void Save(const char *path, const void *data, size_t size)
    {
        auto r = nn::fs::CreateFile(path, size);
        if (!(r.IsSuccess() || nn::fs::ResultPathAlreadyExists::Includes(r)))
        {
            // ファイルの作成に失敗。
            NN_LOG("[ImageJpegEncoding] Failed to create file \"%s\"\n", path);
            NN_LOG(" - Please ensure that the directory of the target file exists.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        }

        nn::fs::FileHandle fileHandle;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Write));

        nn::fs::WriteOption option;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::WriteFile(fileHandle, 0, data, size, option));

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::FlushFile(fileHandle));

        nn::fs::CloseFile(fileHandle);
    }

public:
    explicit FileData(const char *path) :
        m_data(nullptr),
        m_size(0)
    {
        nn::fs::FileHandle fileHandle;
        auto r = nn::fs::OpenFile(&fileHandle, path, nn::fs::OpenMode_Read);
        if (!r.IsSuccess())
        {
            // ファイルのオープンに失敗。
            NN_LOG("[ImageJpegEncoding] Failed to open file \"%s\"\n", path);
            NN_LOG(" - Please ensure that the target file exists.\n");
            NN_ABORT_UNLESS_RESULT_SUCCESS(r);
        }

        int64_t fileSize = 0;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::GetFileSize(&fileSize, fileHandle));
        NN_ABORT_UNLESS(fileSize <= 0xFFFFFFFF, "[ImageJpegEncoding] Cannot handle a file which size > 0x%08lx\n", 0xFFFFFFFF);

        m_size = static_cast<size_t>(fileSize);
        m_data = std::malloc(m_size);
        NN_ABORT_UNLESS(m_data != nullptr, "[ImageJpegEncoding] Failed to allocate memory for file IO\n");
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::ReadFile(fileHandle, 0, m_data, m_size));

        nn::fs::CloseFile(fileHandle);
    }
    ~FileData()
    {
        if (m_data != nullptr)
        {
            std::free(m_data);
            m_data = nullptr;
        }
    }
    const void *GetDataPtr() const
    {
        return m_data;
    }
    size_t GetDataSize() const
    {
        return m_size;
    }

private:
    void *m_data;
    size_t m_size;
};

NN_FORCEINLINE void LogIfNotNull(const char *label, const char *data)
{
    if (data != nullptr)
    {
        NN_LOG("%s: \"%s\"\n", label, data);
    }
}
