﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{ImageJpegDecoding.cpp,PageSampleImageJpegDecoding}

    @brief
    JPEG データのデコードおよび Exif 情報の抽出のサンプルプログラム
 */

/**
    @page PageSampleImageJpegDecoding JPEG デコードと Exif 解析
    @tableofcontents

    @brief
    JPEG デコードと Exif 解析のサンプルプログラムの解説です。

    @section PageSampleImageJpegDecoding_SectionBrief 概要
    JPEG ファイルを読み込み、ピクセルデータにデコードします。併せて、記録された Exif 情報を解析し抽出します。

    @section PageSampleImageJpegDecoding_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/ImageJpegDecoding
    Samples/Sources/Applications/ImageJpegDecoding @endlink 以下にあります。

    @section PageSampleImageJpegDecoding_SectionNecessaryEnvironment 必要な環境
    SDK に準拠します。

    @section PageSampleImageJpegDecoding_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると、JPEG ファイルを読み込み解析を行い、解析ログを出力します。
    読み込まれる JPEG ファイルは Samples/Sources/Applications/ImageJpegDecoding/Data/ExampleInput.jpg です。

    @section PageSampleImageJpegDecoding_SectionPrecaution 注意事項
    SDK に準拠します。

    @section PageSampleImageJpegDecoding_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleImageJpegDecoding_SectionDetail 解説
    このサンプルプログラムは、ファイルシステムから取得した JPEG データを解釈する方法を示します。
    JPEG データの解釈には、「ピクセルデータへの展開」と「Exif 情報の抽出」が含まれます。

    サンプルプログラムの処理の流れは以下の通りです。

    - JPEG ファイルの読み込み
    - アプリの許容する画像サイズに基づくピクセルバッファの取得
    - JPEG ファイルの解析
        - フォーマットの検査
        - Exif 情報の有無と場所を検査
        - 展開後の画像サイズの取得
        - JPEG デコードに要するワークメモリ量を計算
    - Exif 情報の抽出
        - Exif フォーマットの検査および解析
        - サポート対象のメタデータの取得
        - サムネイル画像の展開
    - JPEG ファイルのピクセルデータへのデコード
        - 取得したワークメモリ量に基づくワークバッファの取得
        - デコード処理

    ファイルシステムから取得した JPEG ファイルは、 nn::image::JpegDecoder クラスによってその正当性を検査されます。
    もし JPEG ファイルが正当なものであれば、 Exif 情報の場所や、ピクセルデータに関する情報を
    取得することができます。
    この手順により、 Exif 情報やピクセルデータへの展開に必要な情報が手に入ります。

    もし JPEG ファイルが Exif 情報を含むようであれば、 nn::image::ExifExtractor クラスはこれを解析し、
    Exif 情報に含まれるメタデータを取得可能にします。
    ただし Exif 情報が不正なフォーマットで構成されている場合、メタデータの一部あるいは全部は存在しないとみなされます。
    また、 nn::image::ExifExtractor がサポートしていないメタデータは存在しないとみなされます。

    ここで Exif 情報にサムネイル画像が含まれている場合、次のセクションでデコードされる JPEG データをサムネイル画像で置き換えます。
    これは nn::image::JpegDecoder のデコード対象の JPEG データを、クラスの再初期化なしに交換可能であることを示す事例です。

    nn::image::JpegDecoder に設定された JPEG ファイルは、適切な量の出力バッファとワークメモリを与えることでデコードできます。
    行末にパディング領域を設けたい場合、バッファの行アラインメント値を指定することでピクセルデータを適切に配置することが可能です。
    また、出力されるピクセルデータは CPU のエンディアンに関係なく先頭から R, G, B, A が 1 バイトずつ並んだデータになっています。
    従って単純に 4 バイトを取り出して uint32_t として参照すると、エンディアンによって値が異なって見えることに注意してください。

    このサンプルを実行した結果を以下に示します。
    なおこのサンプルプログラムでは、展開されるピクセルデータについて、画面への表示を含め何も処理を行いません。

    @verbinclude ExampleOutput.txt

    このアプリケーションは次の OSS を使用して作成されています。

    - IJG libjpeg
        - this software is based in part on the work of the Independent JPEG Group
 */

#define NN_LOG_USE_DEFAULT_LOCALE_CHARSET

#include <cstdlib>
#include <new>

#include <nn/nn_Abort.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/fs.h>
#include <nn/image/image_ExifExtractor.h>
#include <nn/image/image_JpegDecoder.h>

#include "Utilities.h"

namespace
{
/// @brief JPEG ライブラリで対応しているメタデータ
struct MetaData
{
    // 0th TIFF IFD カテゴリのメタデータ
    const char *maker;  // メーカー名
    size_t makerSize;   // 終端文字を含めたメーカー名のバイト数
    const char *model;  // 撮影機器名
    size_t modelSize;   // 終端文字を含めた撮影機器名のバイト数
    bool hasOrientation;    // 画像方向の情報があるかどうか
    nn::image::ExifOrientation orientation; // 画像方向
    const char *software;   // ソフトウェア名
    size_t softwareSize;    // 終端文字を含めたソフトウェア名のバイト数
    const char *dateTime;   // 撮影日時 (主単文字を含め20 バイト)

    // 0th Exif IFD カテゴリのメタデータ
    const void *makerNote;  // メーカーノート
    size_t makerNoteSize;   // メーカーノートのバイト数
    bool hasEffectiveDimension; // 実効画像サイズがあるかどうか
    nn::image::Dimension effectiveDim;  // 実効画像サイズ
    const char *uniqueId;   // 画像のユニーク ID (終端文字を含め 33 バイト)

    // 1st Tiff IFD カテゴリのメタデータ
    const void *thumbnail;  // サムネイルのバイト数
    size_t thumbnailSize;   // サムネイル画像のバイト数
};

/// @brief 与えられた Exif 情報の内容をコンソールに出力します。
void PrintMetaData(const MetaData &metaData)
{
    LogIfNotNull(" - メーカー名", metaData.maker);
    LogIfNotNull(" - 撮影機器名", metaData.model);
    if (metaData.hasOrientation)
    {
        const char *orientationStr;
        switch (metaData.orientation)
        {
        case nn::image::ExifOrientation_Normal:
            orientationStr = "正立";
            break;
        case nn::image::ExifOrientation_FlipHorizontal:
            orientationStr = "水平方向に反転";
            break;
        case nn::image::ExifOrientation_Rotate180:
            orientationStr = "時計回りに 180 度回転";
            break;
        case nn::image::ExifOrientation_FlipVertical:
            orientationStr = "垂直方向に反転";
            break;
        case nn::image::ExifOrientation_FlipTopRightToLeftBottom:
            orientationStr = "右下がりの対角線を軸に、上辺右端が左辺下端に来るように反転";
            break;
        case nn::image::ExifOrientation_Rotate270:
            orientationStr = "時計回りに 270 度回転";
            break;
        case nn::image::ExifOrientation_FlipTopLeftToRightBottom:
            orientationStr = "左下がりの対角線を軸に、上辺左端が右辺下端に来るように反転";
            break;
        case nn::image::ExifOrientation_Rotate90:
            orientationStr = "時計回りに 90 度回転";
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
        LogIfNotNull(" - 画像方向", orientationStr);
    }
    LogIfNotNull(" - ソフトウェア名", metaData.software);
    LogIfNotNull(" - 撮影日時", metaData.dateTime);

    if (metaData.makerNote != nullptr)
    {
        NN_LOG(" - メーカーノート: %zu バイト\n", metaData.makerNoteSize);
    }
    if (metaData.hasEffectiveDimension)
    {
        NN_LOG(" - 実効画像サイズ: 幅 %zu px, 高さ %zu px\n", metaData.effectiveDim.width, metaData.effectiveDim.height);
    }
    LogIfNotNull(" - ユニークID", metaData.uniqueId);

    if (metaData.thumbnail != nullptr)
    {
        NN_LOG(" - サムネイル: %zu バイト\n", metaData.thumbnailSize);
    }
}

/// @brief 与えられたバイト列を解析し Exif 情報を抽出します。
void ExtractExifData(MetaData *pOutMetaData, const void *exifData, size_t exifSize)
{
    // Exif 情報解析クラスのインスタンス
    Buffer workBuf(nn::image::ExifExtractor::GetWorkBufferSize());
    nn::image::ExifExtractor extractor(workBuf.GetDataPtr(), workBuf.GetDataSize());

    /* ----------------------------------------------------------------------------------------
        解析フェーズ
     */
    // Exif データを設定します。
    extractor.SetExifData(exifData, exifSize);

    // 解析します。
    nn::image::JpegStatus jpegStatus = extractor.Analyze();
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_LOG("[この Exif 情報は破損しています。]\n");
        // Exif 情報が存在しないとして処理します。
        return;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
    }

    /* ----------------------------------------------------------------------------------------
        メタデータ展開フェーズ
     */
    // 0th TIFF IFD の情報を取得します。
    pOutMetaData->maker = extractor.ExtractMaker(&pOutMetaData->makerSize);
    pOutMetaData->model = extractor.ExtractModel(&pOutMetaData->modelSize);
    pOutMetaData->hasOrientation = extractor.ExtractOrientation(&pOutMetaData->orientation);
    pOutMetaData->software = extractor.ExtractSoftware(&pOutMetaData->softwareSize);
    // 「撮影日時」の文字列は Exif 情報の仕様上、終端文字を含め 20 バイトです。
    size_t dateTimeSize;
    pOutMetaData->dateTime = extractor.ExtractDateTime(&dateTimeSize);
    NN_ASSERT(pOutMetaData->dateTime == nullptr || dateTimeSize == 20);

    // 0th Exif IFD
    pOutMetaData->makerNote = extractor.ExtractMakerNote(&pOutMetaData->makerNoteSize);
    pOutMetaData->hasEffectiveDimension = extractor.ExtractEffectiveDimension(&pOutMetaData->effectiveDim);
    // 「ユニーク ID」の文字列は Exif 情報の仕様上、終端文字を含め 33 バイトです。
    size_t uniqueIdSize;
    pOutMetaData->uniqueId = extractor.ExtractUniqueId(&uniqueIdSize);
    NN_ASSERT(pOutMetaData->uniqueId == nullptr || uniqueIdSize == 33);

    // 1st TIFF IFD
    pOutMetaData->thumbnail = extractor.ExtractThumbnail(&pOutMetaData->thumbnailSize);
}

/// @brief 与えられたバイト列からピクセルデータと Exif 情報を抽出します。
bool DecodeJpegData(
    nn::image::Dimension *pOutDim,
    Buffer *pOutBuf,
    const int alignment,
    const FileData &jpegData)
{
    /* ----------------------------------------------------------------------------------------
        デコード設定フェーズ
     */

    // デコーダのインスタンス
    nn::image::JpegDecoder decoder;

    // JPEG データを設定します。
    decoder.SetImageData(jpegData.GetDataPtr(), jpegData.GetDataSize());

    // [入力例]
    // 縦横をそれぞれ 1/2 倍してデコードする場合、分母 (2) を指定します。等倍の場合は省略可能です。
    decoder.SetResolutionDenominator(2);

    /* ----------------------------------------------------------------------------------------
        JPEG データ解析フェーズ
     */
    nn::image::JpegStatus jpegStatus = decoder.Analyze();
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_LOG("[この JPEG データは破損しています。]\n");
        return false;
    case nn::image::JpegStatus_UnsupportedFormat:
        NN_LOG("[この JPEG データはサポートしていない形式です。]\n");
        return false;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
    }

    // 画像の幅と高さを取得
    *pOutDim = decoder.GetAnalyzedDimension();
    NN_LOG("出力画像サイズ: 幅 %u px, 高さ %u px\n", pOutDim->width, pOutDim->height);

    /* ----------------------------------------------------------------------------------------
        Exif 情報解析フェーズ
        (Exif 情報を取り扱わない場合はこのセクションは読み飛ばしてください。)
     */
    // Exif データの場所を取得
    size_t exifSize;
    const void *exifData = decoder.GetAnalyzedExifData(&exifSize);
    if (exifData != nullptr)
    {
        NN_LOG("Exif 情報が見つかりました。\n");

        // Exif 情報の内容を取得し内容を表示
        MetaData metaData;
        ExtractExifData(&metaData, exifData, exifSize);
        PrintMetaData(metaData);

        if (metaData.thumbnail != nullptr)
        {
            // [入力例]
            // サムネイル画像があれば、デコード対象を置き換えます。
            NN_LOG("→ デコード対象をサムネイルに変更します。\n");
            decoder.SetImageData(metaData.thumbnail, metaData.thumbnailSize);

            // [入力例]
            // サムネイルは縮小せずにデコードします。 (↑上で 2 に設定した値を 1 に戻す。)
            decoder.SetResolutionDenominator(1);

            jpegStatus = decoder.Analyze();
            switch (jpegStatus)
            {
            case nn::image::JpegStatus_WrongFormat:
                NN_LOG("[このサムネイルは破損しています。]\n");
                return false;
            case nn::image::JpegStatus_UnsupportedFormat:
                NN_LOG("[このサムネイルはサポートしていない形式です。]\n");
                return false;
            default:
                NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
            }

            // 画像の幅と高さを取得
            *pOutDim = decoder.GetAnalyzedDimension();
            NN_LOG("出力画像サイズ: 幅 %u px, 高さ %u px\n", pOutDim->width, pOutDim->height);
        }
    }

    /* ----------------------------------------------------------------------------------------
        JPEG データ展開フェーズ
     */
    NN_LOG("JPEG データをデコードします。\n");

    // 展開に必要なメモリ量を取得
    Buffer workBuf(decoder.GetAnalyzedWorkBufferSize());
    NN_LOG(" - 必要な作業メモリ量: %zu bytes\n", workBuf.GetDataSize());

    // 出力サイズを計算
    const size_t AlignedWidth = pOutDim->width % alignment == 0? pOutDim->width: pOutDim->width - pOutDim->width % alignment + alignment;
    if (AlignedWidth * pOutDim->height * sizeof(nn::Bit32) > pOutBuf->GetDataSize())
    {
        NN_LOG(
            "[出力データ(%zu バイト)がバッファの大きさ(%zu バイト)を超過しました。]\n",
            AlignedWidth * pOutDim->height * sizeof(nn::Bit32),
            pOutBuf->GetDataSize());
        return false;
    }

    // デコード処理
    jpegStatus = decoder.Decode(
        pOutBuf->GetDataPtr(),
        pOutBuf->GetDataSize(),
        alignment,
        workBuf.GetDataPtr(),
        workBuf.GetDataSize());
    switch (jpegStatus)
    {
    case nn::image::JpegStatus_WrongFormat:
        NN_LOG("[この JPEG データは破損しています。]\n");
        return false;
    case nn::image::JpegStatus_UnsupportedFormat:
        NN_LOG("[この JPEG データはサポートしていない形式です。]\n");
        return false;
    case nn::image::JpegStatus_OutOfMemory:
        NN_LOG("[デコードに必要なワークメモリが不足しました。未知の形式のデータが入力された可能性があります。]\n");
        return false;
    default:
        NN_ASSERT(jpegStatus == nn::image::JpegStatus_Ok); // 未知のエラーコード
    }

    // この時点で、decoder オブジェクトは不要になります。
    return true;
}
} // 無名名前空間終わり


extern "C" void nnMain()
{
    // ファイルシステムの初期化
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t cacheSize = 0;
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::QueryMountRomCacheSize(&cacheSize));

    char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
    NN_ABORT_UNLESS_NOT_NULL(mountRomCacheBuffer);

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountRom("Asset", mountRomCacheBuffer, cacheSize));

    // JPEG ファイルの読み込み
    // [入力例]
    // 読み込む JPEG ファイルを適宜設定してください。
    FileData jpegData("Asset:/ExampleInput.jpg");

    // 出力バッファを確保
    // このサンプルでは 2073600 画素 (1080p) まで取り扱います。
    Buffer pixBuf(1920 * 1080 * sizeof(nn::Bit32)); // RGBA 32bit で展開されるため、ピクセル数 x4 とする。

    // JPEG デコード処理の実体
    // [入力例]
    // 幅の 4 byte アラインが必要なテクスチャとしてデコードする場合
    static const int AlignmentSize = 4;
    nn::image::Dimension dim; // dim にはアラインメント前のサイズが記録されます。
    if (!DecodeJpegData(&dim, &pixBuf, AlignmentSize, jpegData))
    {
        NN_LOG("[JPEG データのデコードに失敗しました。]\n");
    }
    else
    {
        // JPEG データの使用: アプリケーションで RGBA データとして使用してください。
        // ピクセルデータは、エンディアンに関係なく R,G,B,A の順序で格納されています。
        NN_LOG("処理は正常に完了しました。\n");
    }

    // FS 終了処理
    nn::fs::Unmount("Asset");
    delete[] mountRomCacheBuffer;
    return;
}
