﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @examplesource{IgeSimple.cpp,PageSampleIgeSimple}
 *
 * @brief
 *  ゲーム内編集クラスの使い方を示したサンプルプログラム
 */

/**
 * @page PageSampleIgeSimple IgeSimple
 * @tableofcontents
 *
 * @brief
 *  ゲーム内編集クラスの使い方を示したサンプルプログラムです。
 *
 * @section PageSampleIgeSimple_SectionBrief 概要
 *  UiService クラスを使用してゲーム内編集を行うシンプルなサンプルです。
 *
 * @section PageSampleIgeSimple_SectionFileStructure ファイル構成
 *  本サンプルプログラムは @link ../../../Samples/Sources/Applications/IgeSimple Samples/Sources/Applications/IgeSimple @endlink 以下にあります。
 *
 * @section PageSampleIgeSimple_SectionNecessaryEnvironment 必要な環境
 *  画面表示が利用可能である必要があります。
 *
 * @section PageSampleIgeSimple_SectionHowToOperate 操作方法
 *  特にありません。
 *
 * @section PageSampleIgeSimple_SectionPrecaution 注意事項
 *  サンプルプログラムの実行時に Nintendo TargetManager が起動されていなければなりません。
 *
 * @section PageSampleIgeSimple_SectionHowToExecute 実行手順
 *  サンプルプログラムをビルドし、実行してください。
 *
 * @section PageSampleIgeSimple_SectionDetail 解説
 *  UiService クラスを使用してフォント表示を行う最も基本的なデモです。
 *
 */

#include <cstdlib>
#include <nn/util/util_BitUtil.h>
#include <nn/htcs.h>
#include <nn/nn_Log.h>
#include <nn/ige/ige_UiService.h>

#include <nn/ige/ige_ValueNode.h>

#include "IgeGraphicsSystem.h"

void* AlignedAllocateFunc(size_t size, size_t alignment, void*)
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
#else
    return _aligned_malloc(size, alignment);
#endif
}

void FreeFunc(void* ptr, void*)
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    free(ptr);
#else
    _aligned_free(ptr);
#endif
}

void* AllocateFunc(size_t size)
{
    return malloc(size);
}

void DeallocateFunc(void* ptr, size_t)
{
    free(ptr);
}


int s_IntSliderValue = 0;
float s_FloatSliderValue = 0.0f;
float s_FloatSliderValue2 = 0.0f;
bool s_IsChecked = true;
bool s_IsConnected = false;
char s_Text[200];

//------------------------------------------------------------------------------
//  プリント 関数
//------------------------------------------------------------------------------
void Print(nn::gfx::util::DebugFontTextWriter* pWriter)
{
    nn::util::Color4u8Type color0 = { { 255, 255, 255, 255 } };
    nn::util::Color4u8Type color1 = { { 255, 128, 0, 255 } };

    pWriter->SetScale(1.0f, 1.0f);

    pWriter->SetTextColor(color0);
    pWriter->SetCursor(0, 0);
    pWriter->Print("IgeSimple");

    pWriter->SetTextColor(color1);
    pWriter->SetCursor(100, 100);
    pWriter->Print("IntSlider: %d\n", s_IntSliderValue);
    pWriter->Print("FloatSlider: %f\n", s_FloatSliderValue);
    pWriter->Print("IsChecked: %s\n", s_IsChecked ? "true" : "false");

    pWriter->SetCursor(100, 250);
    pWriter->Print("IsConnected: %s\n", s_IsConnected ? "true" : "false");
    pWriter->Print("TextBox: %s\n", s_Text);
}

void PrintHelloWorld(void*)
{
    NN_LOG("Clicked: Hello World!\n");
}

void Exit(void* pLoopFlag)
{
    *static_cast<bool*>(pLoopFlag) = false;
}

//------------------------------------------------------------------------------
//  メイン 関数
//------------------------------------------------------------------------------
extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)

    _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
    _CrtSetBreakAlloc(-1);

    setlocale(LC_CTYPE, "");
#endif

    // サンプル用グラフィックスの初期化
    IgeGraphicsSystem system(Print);

    // nn::htcs の初期化
    nn::htcs::Initialize(AllocateFunc, DeallocateFunc);

    {
        // UiService の初期化
        nn::ige::UiService::InitializeArg arg;
        arg.portName = "IgeSimple";
        arg.allocateFunc = AlignedAllocateFunc;
        arg.freeFunc = FreeFunc;

        nn::ige::UiService service;
        service.Initialize(arg);
        service.Start();

        // ツリー構築
        nn::ige::ButtonOption buttonOption;
        buttonOption.header = "Hello World!";
        buttonOption.tooltip = "hello world";
        buttonOption.pClickCallback = PrintHelloWorld;
        service.CreateButton(buttonOption);

        nn::ige::GroupBoxOption groupBoxOption;
        groupBoxOption.header = "GroupBox Header";
        groupBoxOption.tooltip = "Grouping";
        auto& groupBox = *service.CreateGroupBox(groupBoxOption);

        nn::ige::PanelOption panelOption;
        panelOption.tooltip = "Panel";
        panelOption.background = "LightGray";
        panelOption.orientation = nn::ige::Orientation_Horizontal;
        panelOption.panelType = nn::ige::PanelType_Wrap;
        panelOption.width = 200;
        auto& panel = *groupBox.CreatePanel(panelOption);

        nn::ige::SliderOption intSliderOption;
        intSliderOption.header = "Slider<int> Header";
        intSliderOption.tooltip = "min:0 max:100";
        auto& slider = *panel.CreateSlider(&s_IntSliderValue, 0, 100, intSliderOption);

        nn::ige::SliderOption floatSliderOption;
        floatSliderOption.header = "Slider<float> Header";
        floatSliderOption.tooltip = "min:0 max:10";
        panel.CreateSlider(&s_FloatSliderValue, 0, 10, floatSliderOption);

        nn::ige::SliderOption floatSliderOption2;
        // header を空にするとラベルが非表示になります。
        floatSliderOption2.header = "";
        floatSliderOption2.tooltip = "min:0 max:10";
        panel.CreateSlider(&s_FloatSliderValue2, 0, 10, floatSliderOption2);

        nn::ige::CheckBoxOption checkBoxOption;
        checkBoxOption.header = "CheckBox Header";
        panel.CreateCheckBox(&s_IsChecked, checkBoxOption);

        nn::ige::TextBoxOption textBoxOption;
        textBoxOption.header = "TextBox Header";
        textBoxOption.isFileDropEnabled = true;
        panel.CreateTextBox(s_Text, sizeof(s_Text), textBoxOption);

        nn::ige::LabelOption labelOption;
        labelOption.header = "Label Header";
        groupBox.CreateLabel(labelOption);

        bool loopFlag = true;

        int frame = 0;
        nn::ige::Button* pExitButton = NULL;

        // 毎フレームのレンダリング
        while (loopFlag)
        {
            system.ProcessFrame();

            // データ送受信
            service.Poll();

            // データ更新
            service.Update();

            s_IsConnected = service.IsConnected();

            // コントロールの作成・削除を繰り返す。
            if (++frame % 200 == 0)
            {
                if (pExitButton)
                {
                    service.Delete(pExitButton);
                    pExitButton = NULL;
                }
                else
                {
                    nn::ige::ButtonOption extOption;
                    extOption.header = "Runtime Exit";
                    extOption.pClickCallback = Exit;
                    extOption.pUserData = &loopFlag;
                    pExitButton = service.CreateButton(extOption);
                }

                slider.SetEnabled(true);
                slider.SetHeader("enabled");
            }
            else if (frame % 200 == 100)
            {
                slider.SetEnabled(false);
                slider.SetHeader("disabled");
            }
        }
    }

    nn::htcs::Finalize();
}
