﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Net;
using System.Net.Sockets;
using System.Threading;
using System.IO;

namespace HtcsHostApp
{
    internal class Program
    {
        // チェック用文字列を送信し、返ってきたデータが送信したものと同じであることを確認する
        private static void CheckEcho(TcpClient client)
        {
            const string CheckString = "Sending from Host to Target.";
            Console.WriteLine("Sending \"{0}\".", CheckString);
            using (var sw = new StreamWriter(client.GetStream(), Encoding.ASCII, 1024, true))
            {
                sw.Write(CheckString);
                sw.Flush();
            }
            string read;
            using (var sr = new StreamReader(client.GetStream(), Encoding.ASCII, true))
            {
                read = sr.ReadLine();
            }
            Console.WriteLine("Received \"{0}\".", read);
            if (!read.Equals(CheckString))
            {
                Console.WriteLine("%s is expected but %s was received.", CheckString, read);
            }
        }

        private static void ServerFunc(object obj)
        {
            TcpListener listener = ((TcpListener)obj);
            listener.Start();
            TcpClient client = listener.AcceptTcpClient();
            Console.WriteLine("Accepted client on target.");

            CheckEcho(client);
        }

        private const string TargetServerPortName = "ServerInTarget";
        private static readonly AutoResetEvent ServerFound = new AutoResetEvent(false);
        private static string TargetEntryAddress;

        private static void OnHtcsUpdate()
        {
            string[] htcsAddresses = Nintendo.Tm.TargetManager.GetHtcsAddresses();
            int i = 0;
            Console.WriteLine("--------");
            Console.WriteLine("{0} ports found", htcsAddresses.Length);
            foreach (string htcsAddress in htcsAddresses)
            {
                Console.Write("{0}:", Nintendo.Tm.TargetManager.GetHtcsPeerName(htcsAddress).ToString());
                Console.Write("{0}, ", Nintendo.Tm.TargetManager.GetHtcsPortName(htcsAddress).ToString());
                Console.WriteLine(htcsAddress.ToString());

                if (Nintendo.Tm.TargetManager.GetHtcsPortName(htcsAddress) == TargetServerPortName)
                {
                    break;
                }
                i++;
            }
            Console.WriteLine("--------");
            if (i < htcsAddresses.Length)
            {
                TargetEntryAddress = htcsAddresses[i];
                ServerFound.Set();
            }
        }

        private static void Main(string[] args)
        {
            // TargetManager への接続
            Nintendo.Tm.Target[] targets = Nintendo.Tm.TargetManager.GetTargets();
            Console.WriteLine("Found {0} targets.", targets.Length);
            foreach (Nintendo.Tm.Target target in targets)
            {
                Console.WriteLine("{0}\t\t{1}", target.GetName(), target.GetConnectionType());
            }

            // htcs ポートマッピングの更新を監視する
            Nintendo.Tm.TargetManager.OnHtcsPortMappingUpdate += OnHtcsUpdate;

            // TargetServerPortName で表されるサーバが見つかるまで待つ
            ServerFound.WaitOne();

            // 監視の解除
            Nintendo.Tm.TargetManager.OnHtcsPortMappingUpdate -= OnHtcsUpdate;

            // クライアントとして、サーバであるターゲットに接続する
            Console.WriteLine("Connect to {0}", TargetEntryAddress.ToString());
            string[] targetAddress = TargetEntryAddress.ToString().Split(':');
            Console.WriteLine("IPAddress = {0}", targetAddress[0]);
            Console.WriteLine("Port      = {0}", targetAddress[1]);
            IPAddress ipAddress = IPAddress.Parse(targetAddress[0]);
            int portNumber = int.Parse(targetAddress[1]);
            IPEndPoint ipEndPoint = new IPEndPoint(ipAddress, portNumber);

            // TCP ソケットを作成、EndPoint 情報を基に Connect
            TcpClient client = new TcpClient();
            client.Connect(ipEndPoint);
            Console.WriteLine("Connected to server on target.");

            CheckEcho(client);

            // ソケットをクローズ
            client.Close();
            Console.WriteLine("Closed.");

            // サーバとして、クライアントであるターゲットからの接続を待つ
            const int PortNumber = 23445;
            TcpListener listener = new TcpListener(System.Net.IPAddress.Loopback, PortNumber);
            Thread thread = new Thread(ServerFunc);
            thread.Start(listener);

            // TargetManager にサーバ情報を登録
            Nintendo.Tm.TargetManager.RegisterHtcsEntry(string.Empty, "ServerInHost", "127.0.0.1:23445");

            thread.Join();

            // 登録していたポート情報を削除します。
            Nintendo.Tm.TargetManager.UnregisterHtcsEntry(string.Empty, "ServerInHost");

            listener.Stop();
            Console.WriteLine("Closed.");

            Console.WriteLine("Sample end");
            Console.ReadLine();
        }

    }
}
