﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net.Sockets;
using System.Text;
using System.Threading;
using System.Xml.Linq;
using System.Xml.Serialization;

namespace Nintendo.Htcs
{
    public class HtcsCommunicator : IDisposable
    {
        public TargetList TargetList { get; private set; }
        public PortMap PortMap { get; private set; }
        private TcpClient client;
        private Thread receiveThread;
        private bool disposed;
        private Queue<TargetManagerAckEvent> AckQueue = new Queue<TargetManagerAckEvent>();
        public event EventHandler<HtcsInfoUpdatedEventArgs> HtcsInfoUpdated;
        private event EventHandler<NewPortEventArgs> NewPortRegistered; // TODO 消す

        public HtcsCommunicator()
        {
            this.TargetList = new TargetList();
            this.PortMap = new PortMap();
            this.client = new TcpClient();
            this.receiveThread = new Thread(ReceiveFunc);
        }

        ~HtcsCommunicator()
        {
            this.Dispose(false);
        }

        public bool Start()
        {
            try
            {
                this.client.Connect("localhost", Nintendo.Tm.TargetManager.GetHtcsControlPort());
                Logger.WriteLine("Connected to TargetManager with {0}.", client.Client.LocalEndPoint);
                this.receiveThread.Start();
            }
            catch (Nintendo.Tm.TmException te)
            {
                Logger.WriteLine(te);
                if (te.GetErrorCode() == Nintendo.Tm.Error.IncompatibleTargetManager)
                {
                    Console.WriteLine("Target Manager is old, please update.");
                }
                return false;
            }
            catch (Exception e)
            {
                Logger.WriteLine(e);
                return false;
            }
            return true;
        }

        public void Stop()
        {
            this.Dispose();
        }

        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }

        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    this.client.Close();

                    try
                    {
                        this.receiveThread.Join();
                    }
                    catch (ThreadStateException)
                    {
                        // スレッド開始前にクラスが Dispose されたとしても、問題ではない
                    }
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }

        public UInt32 RegisterPort(PortMapItem portMapItem)
        {
            using (TargetManagerAckEvent waitAckEvent = new TargetManagerAckEvent())
            {
                lock (AckQueue)
                {
                    using (var sw = new StreamWriter(client.GetStream(), new UTF8Encoding(false), 1024, true))
                    {
                        var command = new RegisterPortMapCommand(portMapItem, "Auto");
                        string xml = command.ToXElement().ToString(SaveOptions.DisableFormatting);
                        sw.WriteLine(xml);
                    }
                    AckQueue.Enqueue(waitAckEvent);
                }
                waitAckEvent.Wait();
                return waitAckEvent.GetResult().GetValue();
            }
        }

        public UInt32 RegisterPort(HtcsPeerName htcsPeerName, string htcsPortName, System.Net.IPEndPoint endPoint)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            PortMapItem portMapping = new PortMapItem(htcsPortDesc, endPoint);
            return RegisterPort(portMapping);
        }

        public UInt32 UnregisterPort(HtcsPortDescriptor htcsPortDescriptor)
        {
            using (TargetManagerAckEvent waitAckEvent = new TargetManagerAckEvent())
            {
                lock (AckQueue)
                {
                    using (var sw = new StreamWriter(client.GetStream(), new UTF8Encoding(false), 1024, true))
                    {
                        var command = new UnregisterPortMapCommand(htcsPortDescriptor, "Auto");
                        string xml = command.ToXElement().ToString(SaveOptions.DisableFormatting);
                        sw.WriteLine(xml);
                    }
                    AckQueue.Enqueue(waitAckEvent);
                }
                waitAckEvent.Wait();
                return waitAckEvent.GetResult().GetValue();
            }
        }

        public UInt32 UnregisterPort(HtcsPeerName htcsPeerName, string htcsPortName)
        {
            HtcsPortDescriptor htcsPortDesc = new HtcsPortDescriptor(htcsPeerName, htcsPortName);
            return UnregisterPort(htcsPortDesc);
        }

        private void ReceiveFunc()
        {
            var sr = new StreamReader(client.GetStream(), new UTF8Encoding(false), false, 1024, true);

            while (true)
            {
                if (!ReceiveLine(sr))
                    break;
            }
        }

        private bool ReceiveLine(StreamReader sr)
        {
            try
            {
                var line = sr.ReadLine();
                if (line == null)
                {
                    return false;
                }
                var element = XElement.Parse(line);
                if (element.Name == HtcsInfo.XElementName)
                {
                    return ReceivePortMap(element);
                }
                else if (element.Name == CommandResult.XElementName)
                {
                    return ReceiveAck(element);
                }
            }
            catch (EndOfStreamException)
            {
                Logger.WriteLine("Disconnected TargetManager.");
                return false;
            }
            catch (IOException)
            {
                Logger.WriteLine("Disconnected TargetManager.");
                return false;
            }
            catch (ObjectDisposedException)
            {
                Logger.WriteLine("Disconnected TargetManager.");
                return false;
            }

            return true;
        }

        private bool ReceivePortMap(XElement element)
        {
            var htcsInfo = new HtcsInfo(element);
            var newTargets = htcsInfo.TargetList;
            var newMap = htcsInfo.PortMap;

            List<PortMapItem> newlyRegisteredItems = newMap.Except(this.PortMap).ToList();
            foreach (PortMapItem newItem in newlyRegisteredItems)
            {
                RaiseNewPort(newItem);
            }
            this.TargetList = newTargets;
            this.PortMap = newMap;

            if (HtcsInfoUpdated != null)
            {
                HtcsInfoUpdated(this, new HtcsInfoUpdatedEventArgs(this.TargetList, this.PortMap));
            }
            return true;
        }

        private bool ReceiveAck(XElement element)
        {
            TargetManagerAckEvent signal;
            lock (AckQueue)
            {
                signal = AckQueue.Dequeue();
            }
            signal.SetResult(new CommandResult(element));
            signal.Signal();
            return true;
        }

        private void RaiseNewPort(PortMapItem portMapItem)
        {
            if (NewPortRegistered != null)
            {
                NewPortRegistered(this, new NewPortEventArgs(portMapItem));
            }
        }

        private class NewPortEventArgs : EventArgs
        {
            public NewPortEventArgs(PortMapItem portMapItem)
            {
                this.PortMapItem = portMapItem;
            }

            public PortMapItem PortMapItem { get; private set; }
        }
    }

    internal class TargetManagerAckEvent : IDisposable
    {
        private bool disposed;
        private AutoResetEvent signal = new AutoResetEvent(false);
        private CommandResult result;

        ~TargetManagerAckEvent()
        {
            this.Dispose(false);
        }
        public void Dispose()
        {
            this.Dispose(true);
            GC.SuppressFinalize(this);
        }
        protected virtual void Dispose(bool disposing)
        {
            if (!this.disposed)
            {
                if (disposing)
                {
                    // ここでマネージドリソースをDisposeする
                    signal.Dispose();
                }

                // ここでアンマネージドリソースを解放する
                this.disposed = true;
            }
        }
        public void Wait()
        {
            signal.WaitOne();
        }
        public void Signal()
        {
            signal.Set();
        }
        public void SetResult(CommandResult res)
        {
            result = res;
        }
        public CommandResult GetResult()
        {
            return result;
        }
    }

    public class HtcsInfoUpdatedEventArgs : EventArgs
    {
        public TargetList TargetList { get; private set; }
        public PortMap PortMap { get; private set; }

        public HtcsInfoUpdatedEventArgs(TargetList targetList, PortMap portMap)
        {
            this.TargetList = targetList;
            this.PortMap = portMap;
        }
    }
}

