﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidbusSimple.cpp,PageSampleHidbusSimple}

    @brief
    シンプルにコントローラーの拡張バス機能を使用するためのサンプルプログラム
 */

/**
    @page PageSampleHidbus シンプルな Hidbus 機能
    @tableofcontents

    @brief
    シンプルにコントローラーの拡張バス機能を使用するためのサンプルプログラムの解説です。

    @section PageSampleHidbusSimple_SectionBrief 概要
    コントローラーの拡張バス機能を使用する方法の説明をします。

    @section PageSampleHidbusSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidbusSimple @endlink 以下にあります。

    @section PageSampleHidbusSimple_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラーをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラーをペアリングしてください。

    @section PageSampleHidbusSimple_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラーのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラーの LED が点滅から点灯に変わります。

    接続後、 コントローラーの拡張バス機能に対応したデバイスを接続すると、
    ポーリングモード及び同期通信の 2 種類で、デバイスに指定の文字列を送信し、デバイスからデータを受け取ります。

    @section PageSampleHidbusSimple_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。
    Hidbus の API には呼び出し元スレッドを長時間ブロックするものが存在するため、
    メインループとは別のスレッドから呼び出すことを推奨します。 詳細については、API リファレンスをご覧ください。

    @section PageSampleHidbusSimple_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidbusSimple_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - NpadID を設定
    - NpadID から Hidbus のハンドルを取得 (本サンプルでは右 Joy-Con を使用)
    - デバイスとの通信を有効化
    - デバイスとのポーリングモードを有効化
    - ポーリングモードでの受信データを表示
    - 同期通信でデバイスにデータを送信し、結果をブロッキングして受け取る。
    - 上記を右 Joy-Con の + ボタンが押されるまで繰り返す。
 */

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_ControllerSupport.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/hidbus/hidbus.h>

#include <nns/nns_Log.h>

namespace {

    enum HidbusState
    {
        HidbusState_NotGotHandle = 0,
        HidbusState_GotHandle    = 1,
        HidbusState_Enabled      = 2,
        HidbusState_End          = 3,
    };

    HidbusState state = HidbusState_NotGotHandle;

    nn::hidbus::BusHandle rightJoyHandle;

    // 使用したい拡張デバイスの ID を指定してください。
    const uint64_t DeviceId = 0x00;

    struct Npad
    {
        typedef nn::hid::NpadIdType Id;
        typedef nn::hid::NpadButton Button;
    };

    Npad::Id npadIds[] =
    {
        nn::hid::NpadId::No1,
    };

    // PollingMode 時に渡すバッファ (handle ごとに別のバッファが必要です。)
    NN_ALIGNAS(nn::os::MemoryPageSize) char s_Buffer[nn::os::MemoryPageSize];

    // PollingMode 用コマンド
    const uint8_t PollingDataCommand[4] = { 0x01, 0x02, 0x03, 0x04 };

    bool IsFinished()
    {
        nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(npadIds[0]);

        if (style.Test<nn::hid::NpadStyleJoyRight>())
        {
            nn::hid::NpadJoyRightState stateJoyRight;
            nn::hid::GetNpadState(&stateJoyRight, npadIds[0]);

            // ＋ ボタンが押されたら終了
            if (stateJoyRight.buttons.Test<nn::hid::NpadButton::Plus>())
            {
                return true;
            }
        }
        return false;
    }

    HidbusState WaitSupportedJoyConConnected(nn::hidbus::BusHandle* pOutHandle)
    {
        while (NN_STATIC_CONDITION(true))
        {
            // RightJoyRail を持つコントローラーが接続されるまで待ちます。
            while (!nn::hidbus::GetBusHandle(pOutHandle, nn::hid::NpadId::No1, nn::hidbus::BusType_RightJoyRail))
            {
                NNS_LOG("Please connect supported controller.\n");
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }

            // 対応するコントローラーが接続されたら、ライブラリを初期化します。
            auto result = nn::hidbus::Initialize(*pOutHandle);
            if (result.IsSuccess())
            {
                return HidbusState_GotHandle;
            }
            // Initialize に失敗した時はコントローラーが切断される等によって handle が無効化されているため、
            // 再度 handle 取得からやり直します。
        }
    }

    HidbusState EnableDevice(nn::hidbus::BusHandle handle)
    {
        while (NN_STATIC_CONDITION(true))
        {
            auto enableResult = nn::hidbus::EnableExternalDevice(true, DeviceId, handle);
            if (enableResult.IsSuccess())
            {
                return HidbusState_Enabled;
            }
            else if (nn::hidbus::ResultFWUpdateFailed::Includes(enableResult))
            {
                // コントローラーの Firmware Update が発生したが失敗しました。
                // リトライするか、アプリ側で Firmware Update アプレットを立ち上げてください。
                // 本サンプルではリトライします。
                NNS_LOG("Firmware Update Failed. retry.\n");
            }
            else if (nn::hidbus::ResultInvalidHandle::Includes(enableResult))
            {
                // コントローラの切断等が発生し、handle が無効になっています。handle 取得からリトライしてください。
                NNS_LOG("Handle is invalid.\n");
                return HidbusState_NotGotHandle;
            }
            else if (nn::hidbus::ResultNotInForcus::Includes(enableResult))
            {
                // アプリケーションが InForcus 状態でないため、拡張デバイスを有効化できません。
                // アプリケーション側で nn::oe のライブラリ群を使用してアプリケーションが InForcus になるのを待ってください。
                // 本サンプルではリトライします。
                NNS_LOG("Not InForcus.\n");
            }
            else if (nn::hidbus::ResultActivateFailureNpadBusy::Includes(enableResult))
            {
                // 指定された Npad が拡張バス機能と排他で使用されるべき機能を使用しているため、有効化できませんでした。
                // アプリケーション側で拡張バスと排他で使用されるべき機能を終了して、再度本関数を実行してください。
                NNS_LOG("Please deactivate the dependency function\n");
            }
            else if (nn::hidbus::ResultExternalDeviceNotAttached::Includes(enableResult))
            {
                // 拡張バスにデバイスが接続されていません。
                // 拡張バスにデバイスが接続されるまでリトライします。
                NNS_LOG("Waiting External Device Connected..\n");
            }
            else if (nn::hidbus::ResultExternalDeviceReadyTimeout::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスとの通信準備中にタイムアウトしました。
                // 本サンプルではリトライします。
                NNS_LOG("Communication Ready Timeout. Please retry.\n");
            }
            else if (nn::hidbus::ResultExternalDeviceTimeout::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスとの通信中にタイムアウトしました。
                // ハンドリング方法はデバイスごとに異なりますが、本サンプルではリトライします。
                NNS_LOG("Communication Timeout. Please retry.\n");
            }
            else if (nn::hidbus::ResultExternalDeviceInvalidId::Includes(enableResult))
            {
                // 拡張バスに接続されたデバイスが EnableExternalDevice() で指定されたデバイスではありません。
                // ユーザーに分かりやすいような表示を行ってください。本サンプルではリトライします。
                NNS_LOG("Attached device is not required device. Please attach required device.\n");
            }
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
        }
    }

    HidbusState CommunicateWithExternalDevice(nn::hidbus::BusHandle handle, char* pBuffer, size_t bufferSize)
    {
        // PollingMode を有効化します。 workBuffer は handle ごとに別の領域を渡してください。
        auto enablePollingModeResult = nn::hidbus::EnableJoyPollingReceiveMode(handle, PollingDataCommand, sizeof(PollingDataCommand),
            pBuffer, bufferSize, nn::hidbus::JoyPollingMode_SixAxisSensorEnable);

        if (enablePollingModeResult.IsFailure())
        {
            if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(enablePollingModeResult))
            {
                // 拡張バスに接続されたデバイスの取り外しや、アプリケーションが InForcus でなくなった等の理由により、
                // 拡張バスに接続されたデバイスとの通信機能が無効化されています。
                // 再度 Enable からやり直してください。
                return HidbusState_GotHandle;
            }
            else if (nn::hidbus::ResultInvalidHandle::Includes(enablePollingModeResult))
            {
                // handle に対応したコントローラーが切断されており、 handle が無効化されています。
                // 再度 handle の取得からやり直してください。
                return HidbusState_NotGotHandle;
            }
        }

        while (NN_STATIC_CONDITION(true))
        {
            nn::hidbus::JoyPollingReceivedData pollingData;

            // ポーリングモードで受信したデータを受け取ります。
            auto pollingDataResult = nn::hidbus::GetJoyPollingReceivedData(&pollingData, handle);
            if (pollingDataResult.IsSuccess())
            {
                // ポーリングモードで取得したデータを表示します。
                NNS_LOG("Polling Data = ");
                for (size_t i = 0; i < pollingData.outSize; i++)
                {
                    NNS_LOG("%2x ", pollingData.data[i]);
                }
                NNS_LOG("  ");
            }
            else if (nn::hidbus::ResultExternalDeviceTimeout::Includes(pollingDataResult))
            {
                // 拡張バスに接続されたデバイスとの通信中にタイムアウトが発生しました。
                // デバイスごとに対応する処理は異なりますが、本サンプルではリトライします。
                NNS_LOG("Timeout Occur. Retry");
            }
            else if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(pollingDataResult))
            {
                // 拡張バスに接続されたデバイスの取り外しや、アプリケーションが InForcus でなくなった等の理由により、
                // 拡張バスに接続されたデバイスとの通信機能が無効化されています。
                // 再度 Enable からやり直してください。
                NNS_LOG("NotEnabled. \n");
                return HidbusState_GotHandle;
            }
            else if (nn::hidbus::ResultInvalidHandle::Includes(pollingDataResult))
            {
                // handle に対応したコントローラーが切断されており、 handle が無効化されています。
                // 再度 handle の取得からやり直してください。
                NNS_LOG("Invalid Handle. \n");
                return HidbusState_NotGotHandle;
            }

            size_t outSize;
            char outBuffer[4];
            const char sendData[4] = {0x01, 0x02, 0x03, 0x04};
            // 拡張バスに接続されたデバイスへデータを送信し、ブロッキングしてデータを受信します。
            auto sendAndReceiveResult = nn::hidbus::SendAndReceive(&outSize, outBuffer, sizeof(outBuffer), handle, sendData, sizeof(sendData));
            if (sendAndReceiveResult.IsSuccess())
            {
                // 取得したデータを表示します。
                NNS_LOG("SendAndReceive Data = ");
                for (size_t i = 0; i < outSize; i++)
                {
                    NNS_LOG("%2x ", outBuffer[i]);
                }
                NNS_LOG("\n");
            }
            else if (nn::hidbus::ResultExternalDeviceTimeout::Includes(sendAndReceiveResult))
            {
                // 拡張バスに接続されたデバイスとの通信中にタイムアウトが発生しました。
                // デバイスごとに対応する処理は異なりますが、本サンプルではリトライします。
                NNS_LOG("Timeout Occur. Retry");
            }
            else if (nn::hidbus::ResultExternalDeviceNotEnabled::Includes(sendAndReceiveResult))
            {
                // 拡張バスに接続されたデバイスの取り外しや、アプリケーションが InForcus でなくなった等の理由により、
                // 拡張バスに接続されたデバイスとの通信機能が無効化されています。
                // 再度 Enable からやり直してください。
                NNS_LOG("NotEnabled. \n");
                return HidbusState_GotHandle;
            }
            else if (nn::hidbus::ResultInvalidHandle::Includes(sendAndReceiveResult))
            {
                // handle に対応したコントローラーが切断されており、 handle が無効化されています。
                // 再度 handle の取得からやり直してください。
                NNS_LOG("Invalid Handle. \n");
                return HidbusState_NotGotHandle;
            }

            if (IsFinished())
            {
                return HidbusState_End;
            }
        }
    }

}//namespace

extern "C" void nnMain()
{
    NN_LOG("Hidbus Simple Sample Start.\n");

    nn::hid::InitializeNpad();

    // Npadのスタイルを設定
    nn::hid::SetSupportedNpadStyleSet(
        ::nn::hid::NpadStyleJoyLeft::Mask |
        ::nn::hid::NpadStyleJoyRight::Mask
    );

    // 使用するNpadId の設定
    nn::hid::SetSupportedNpadIdType(npadIds, sizeof(npadIds) / sizeof(npadIds[0]));

    // 全体のループ
    bool isFinished = false;
    while (!isFinished)
    {
        switch (state)
        {
        case HidbusState_NotGotHandle:
            state = WaitSupportedJoyConConnected(&rightJoyHandle);
            break;

        case HidbusState_GotHandle:
            state = EnableDevice(rightJoyHandle);
            break;

        case HidbusState_Enabled:
            state = CommunicateWithExternalDevice(rightJoyHandle, s_Buffer, sizeof(s_Buffer));
            break;

        case HidbusState_End:
            isFinished = true;
            break;

        default: NN_UNEXPECTED_DEFAULT;
        }
    }
    // ポーリングモードの停止
    nn::hidbus::DisableJoyPollingReceiveMode(rightJoyHandle);

    // 拡張デバイスの無効化
    nn::hidbus::EnableExternalDevice(false, DeviceId, rightJoyHandle);

    // 拡張デバイス機能の終了
    nn::hidbus::Finalize(rightJoyHandle);

    NNS_LOG("Hidbus Simple Sample Done\n");
}
