﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidVibrationRollingBall.cpp,PageSampleHidVibrationRollingBall}

    @brief
    振動を使ったインタラクティブな技術デモです。
 */

/**
    @page PageSampleHidVibrationRollingBall 振動を使ったインタラクティブな技術デモ
    @tableofcontents

    @brief
    振動を使ったインタラクティブな技術デモの解説です。

    @section PageSampleHidVibrationRollingBall_SectionBrief 概要
    振動の効果を体験できるインタラクティブデモです。
    画面内のボールを操作した際に、ボールの重さや位置を振動から感じられます。

    @section PageSampleHidVibrationRollingBall_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidVibrationRollingBall @endlink 以下にあります。

    @section PageSampleHidVibrationRollingBall_SectionNecessaryEnvironment 必要な環境
    Windows 環境では動作しません。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラをペアリングしてください。
    1 台のコントローラを接続する必要があります。

    @section PageSampleHidVibrationRollingBall_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラを持ってください。
    EDEV に Joy-Con を取り付けた状態で、体験されることをお勧めします。

    コントローラを傾けると画面内に描画されたボールが左右に動き出します。
    ボールは緑の枠内を移動し、その挙動にあわせてコントローラが振動します。

    L もしくは R ボタンを押すことでボールの種類を変更できます。
    3 種類のボールがあり、種類を変えると振動の感触が変化します。
    白色の石のボールが一番重く、カラフルな色のバウンスボールが一番軽い振動になります。

    X ボタンで振動の再生の有無を変更することで、
    振動が有るときと無いときの差を感じることが可能です。

    各ボタンを押すと、デモの設定を変更可能です。

    - L ボタン：ボールの種類を変える
    - R ボタン：ボールの種類を変える
    - B ボタン：ボールの位置を初期化する

    - X ボタン：振動再生の有無を変える
    - A ボタン：音声再生の有無を変える
    - Y ボタン：画面表示の有無を変える

    画面右上に、実際に発生している振動値が出力されます。

    サンプルプログラムを終了させるには + ボタンと - ボタンを同時に押してください。

    @section PageSampleHidVibrationRollingBall_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。

    @section PageSampleHidVibrationRollingBall_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidVibrationRollingBall_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - 操作しているコントローラを決定する（最後にボタン操作をしたもしくはセンサに反応があった）
    - そのコントローラの加速度を取得する。
    - 加速度情報から、ボールの位置を更新する。
    - ボールが転がっていれば転がり振動を再生する。
    - ボールが壁に当たっていれば衝突振動を再生する。
    - ボールの種類に応じて振動感を変えています。使用している振動ファイルは同じですが、再生時のパラメータを変化させる（ nn::hid::VibrationModulation の gain と pitch ）ことで実現しています。
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#include <nn/hid.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadJoy.h>
#include <nn/hid/hid_Vibration.h>
#include <nn/hid/hid_SixAxisSensor.h>

#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include "GraphicsSystem.h"
#include "Audio.h"
#include "File.h"
#include "VirtualBall.h"
#include "Sphere.h"
#include "Demo.h"
#include "NpadController.h"
#include "ControllerManager.h"
#include "Color.h"

namespace
{
    // グラフィックス設定
    const int FrameRate = 60;
    const char ProgramName[] = "HidVibration Rolling Ball";

    GraphicsSystem* g_pGraphicsSystem;
    nn::mem::StandardAllocator* g_pAppAllocator;
    nn::Bit8* g_pAppMemory;

    void DrawTitle() NN_NOEXCEPT
    {
        auto pTextWriter = &g_pGraphicsSystem->GetDebugFont();

        pTextWriter->Draw(&g_pGraphicsSystem->GetCommandBuffer());
        pTextWriter->SetScale(2.0f, 2.0f);
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetCursor(10.0f, 10.0f);
        pTextWriter->Print("%s", ProgramName);

        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetTextColor(Color::Red);
        pTextWriter->SetCursor(10.0f, 80.0f);
        pTextWriter->Print("As you tilt the controller, a ball moves left and right on the screen.\nThe controller begins to vibrate as you move the ball inside the green frame.");
    }

    void DrawVibrationValue() NN_NOEXCEPT
    {
        auto pTextWriter = &g_pGraphicsSystem->GetDebugFont();

        VibrationValueDrawer vibDrawer(
            pTextWriter, &g_pGraphicsSystem->GetCommandBuffer(), &g_pGraphicsSystem->GetPrimitiveRenderer());
        vibDrawer.SetScale(1.0f);
        vibDrawer.SetBrightColor(true);

        for (int i = 0; i < ControllerManager::VibrationTargetCountMax; i++)
        {
            float baseX = 700.0f + 280.0f * i;
            float baseY = 50.0f;

            pTextWriter->SetTextColor(Color::White);
            pTextWriter->SetScale(0.8f, 0.8f);
            pTextWriter->SetCursor(baseX, baseY);
            pTextWriter->Print((i == 0) ? "Left" : "Right");

            nn::hid::VibrationValue value
                = ControllerManager::GetInstance().GetVibrationBuffer(i).GetActualVibrationValue();

            vibDrawer.SetPosition(baseX, baseY + 15.f);
            vibDrawer.DrawVibrationValue(value, true);

            baseY += 160.0f;

            bool isLow = true;
            pTextWriter->SetTextColor(Color::White);
            pTextWriter->SetScale(1.0f, 1.0f);
            pTextWriter->SetCursor(baseX, baseY);
            pTextWriter->Print("Time Series amplitude%s value", isLow ? "Low" : "High");
            vibDrawer.SetPosition(baseX, baseY + 40.0f);
            vibDrawer.DrawTimeSeriesGraph(
                ControllerManager::GetInstance().GetVibrationBuffer(i), isLow);

            baseY += 160.0f;

            isLow = false;
            pTextWriter->SetTextColor(Color::White);
            pTextWriter->SetScale(1.0f, 1.0f);
            pTextWriter->SetCursor(baseX, baseY);
            pTextWriter->Print("Time Series amplitude%s value", isLow ? "Low" : "High");
            vibDrawer.SetPosition(baseX, baseY + 40.0f);
            vibDrawer.DrawTimeSeriesGraph(
                ControllerManager::GetInstance().GetVibrationBuffer(i), isLow);
        }
    }

    void InitializeGraphics() NN_NOEXCEPT
    {
        // Memory
        g_pAppAllocator = new nn::mem::StandardAllocator();
        const size_t appMemorySize = 128 * 1024 * 1024;
        g_pAppMemory = new nn::Bit8[appMemorySize];
        g_pAppAllocator->Initialize(g_pAppMemory, appMemorySize);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        const size_t graphicsMemorySize = 256 * 1024 * 1024;
        void* pGraphicsMemory =
            nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);

        nv::SetGraphicsAllocator(
            nns::gfx::GraphicsFramework::DefaultAllocateFunction,
            nns::gfx::GraphicsFramework::DefaultFreeFunction,
            nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);

        nv::SetGraphicsDevtoolsAllocator(
            nns::gfx::GraphicsFramework::DefaultAllocateFunction,
            nns::gfx::GraphicsFramework::DefaultFreeFunction,
            nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);

        nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif
        // Graphics
        g_pGraphicsSystem = new ::GraphicsSystem();
        g_pGraphicsSystem->SetApplicationHeap(g_pAppAllocator);
        g_pGraphicsSystem->Initialize();
    }

    void UpdateGraphics() NN_NOEXCEPT
    {
        g_pGraphicsSystem->BeginDraw();

        // 操作方法の表示
        DrawTitle();
        Demo::GetInstance().Draw(g_pGraphicsSystem);
        DrawVibrationValue();

        // 描画終了
        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000L * 1000L * 1000L / FrameRate));
    }

    void FinalizeGraphics() NN_NOEXCEPT
    {
        g_pGraphicsSystem->Finalize();
        delete g_pGraphicsSystem;

        g_pAppAllocator->Finalize();
        delete g_pAppAllocator;

        delete[] g_pAppMemory;
    }

} // anonymous-namespace

extern "C" void nnMain()
{
    NN_LOG("%s Start.\n", ProgramName);

    // リソースデータの読み込み
    File::GetInstance().Initialize();
    // グラフィックス機能を初期化する
    InitializeGraphics();
    // オーディオ機能を初期化する
    AudioPlayer::GetInstance().Initialize();
    //　使用するオーディオリソースの読み込み
    AudioPlayer::GetInstance().LoadAudioFile("hit");
    // コントローラーを初期化する
    ControllerManager::GetInstance().Initialize();
    // デモに使用するオブジェクトを初期化する
    Demo::GetInstance().Initialize();

    StartVibrationThread();

    bool isQuitRequired = false;

    // Main Loop
    while (!isQuitRequired)
    {
        ControllerManager::GetInstance().Update();
        Demo::GetInstance().Update();
        AudioPlayer::GetInstance().Update();

        UpdateGraphics();

        //終了判定
        if (ControllerManager::GetInstance().GetActiveController()->IsQuitRequired())
        {
            isQuitRequired = true;
        }
    }

    // 終了処理
    StopVibrationThread();
    ControllerManager::GetInstance().Finalize();
    File::GetInstance().Finalize();
    AudioPlayer::GetInstance().Finalize();
    FinalizeGraphics();

    NN_LOG("%s Done\n", ProgramName);
}

