﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/fs.h>
#include <nn/os.h>
#include <nn/nn_Assert.h>

#include "File.h"

namespace
{
    struct Asset
    {
        const FileLabel fileName;
        const char* filePath;
    };

    Asset BnvibAsset[] = {
        { "simple","rom:/simple.bnvib" },
    };

    Asset AudioAsset[] = {
        { "hit","rom:/ballhit.wav" },
    };

    const int AudioCountMax = 10;
    const int BnvibCountMax = 10;

    BnvibFile BnvibData[BnvibCountMax];
    AudioFile AudioData[AudioCountMax];

    // 振動ファイル読み込み
    void LoadBnvibFile(BnvibFile* data, const char* filePath) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(data);
        NN_ASSERT_NOT_NULL(filePath);

        nn::Result result;
        nn::fs::FileHandle file;
        data->size = 0;

        // ファイルを開く
        result = nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ファイルサイズを確認する
        int64_t filesize;
        result = nn::fs::GetFileSize(&filesize, file);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        auto size = static_cast<size_t>(filesize);
        NN_ABORT_UNLESS(size <= BnvibFile::FileSizeMax);

        // データを読み込む
        result = nn::fs::ReadFile(file, 0, data->data, size);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        data->size = size;

        //　ファイルを閉じる
        nn::fs::CloseFile(file);
    }

    // オーディオファイル読み込み
    void LoadAudioFile(AudioFile* data, const char* filePath) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(data);
        NN_ASSERT_NOT_NULL(filePath);

        nn::Result result;
        nn::fs::FileHandle file;
        data->size = 0;

        // ファイルを開く
        result = nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        // ファイルサイズを確認する
        int64_t filesize;
        result = nn::fs::GetFileSize(&filesize, file);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        auto size = static_cast<size_t>(filesize);
        NN_ABORT_UNLESS(size <= AudioFile::FileSizeMax);

        // データを読み込む
        result = nn::fs::ReadFile(file, 0, data->data, size);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        data->size = size;

        //　ファイルを閉じる
        nn::fs::CloseFile(file);
    }

    void LoadFiles(BnvibFile* bnvibFiles, const Asset* bnvibAsset, AudioFile* audioFiles, const Asset* audioAsset) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(bnvibFiles);
        NN_ASSERT_NOT_NULL(bnvibAsset);
        NN_ASSERT_NOT_NULL(audioFiles);
        NN_ASSERT_NOT_NULL(audioAsset);

        size_t cacheSize = 0;
        nn::Result result = nn::fs::QueryMountRomCacheSize(&cacheSize);
        NN_ASSERT(result.IsSuccess());

        char* mountRomCacheBuffer = new(std::nothrow) char[cacheSize];
        NN_ASSERT_NOT_NULL(mountRomCacheBuffer);

        result = nn::fs::MountRom("rom", mountRomCacheBuffer, cacheSize);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);

        int bnvibAssetCount = NN_ARRAY_SIZE(BnvibAsset);
        NN_ASSERT(bnvibAssetCount > 0);
        for (int i = 0; i < bnvibAssetCount; i++)
        {
            LoadBnvibFile(&(bnvibFiles[i]), bnvibAsset[i].filePath);
        }

        int audioDataCount = NN_ARRAY_SIZE(AudioAsset);
        NN_ASSERT(audioDataCount > 0);
        for (int i = 0; i < audioDataCount; i++)
        {
            LoadAudioFile(&(audioFiles[i]), audioAsset[i].filePath);
        }

        nn::fs::Unmount("rom");
        delete[] mountRomCacheBuffer;
    }
}

File& File::GetInstance() NN_NOEXCEPT
{
    static File instance;
    return instance;
}

void File::Initialize() NN_NOEXCEPT
{
    LoadFiles(BnvibData,BnvibAsset,AudioData,AudioAsset);

    for (int i = 0; i < NN_ARRAY_SIZE(BnvibAsset); i++)
    {
        m_BnvibHandles.insert(std::make_pair(BnvibAsset[i].fileName, &(BnvibData[i])));
    }

    for (int i = 0; i < NN_ARRAY_SIZE(AudioAsset); i++)
    {
        m_AudioHandles.insert(std::make_pair(AudioAsset[i].fileName, &(AudioData[i])));
    }

}

void File::Finalize() NN_NOEXCEPT
{
    m_BnvibHandles.clear();
    m_AudioHandles.clear();

}

BnvibFile* File::GetBnvibFile(const FileLabel& name) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(name.empty() == false);
    NN_ABORT_UNLESS(m_BnvibHandles.count(name) == 1);

    return m_BnvibHandles.at(name);
}

AudioFile* File::GetAudioFile(const FileLabel& name) NN_NOEXCEPT
{
    NN_ABORT_UNLESS(name.empty() == false);
    NN_ABORT_UNLESS(m_AudioHandles.count(name) == 1);

    return m_AudioHandles.at(name);
}
