﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <map>
#include <nn/mem.h>
#include <nn/audio.h>
#include <nns/audio/audio_WavFormat.h>

#include "file.h"

// 音声データを取り扱うためのハンドルです
struct AudioHandle
{
    void* pData;
    int   priority;
    bool  isLoop;
    nn::audio::WaveBuffer waveBuffer;
    nn::audio::VoiceType  voiceType;
    nns::audio::WavFormat wavFormat;
};

// 音声データを再生するためのクラスです。
class AudioPlayer
{
    NN_DISALLOW_COPY(AudioPlayer);
    NN_DISALLOW_MOVE(AudioPlayer);

public:
    static AudioPlayer& GetInstance() NN_NOEXCEPT;

    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    void Update() NN_NOEXCEPT;

    // 名前に紐付いたオーディオデータを指定したパラメータで再生する
    void PlaySound(const FileLabel& label, float volume, float pitch, float pos) NN_NOEXCEPT;

    // 名前に紐付いたオーディオデータを読み込む
    void LoadAudioFile(const FileLabel& label) NN_NOEXCEPT;

public:
    // フレーム設定
    static const int DefaultRenderRate = 48000;

    // バッファサイズ
    static const size_t DefaultAudioWorkBufferSize = 8 * 1024 * 1024;
    static const size_t DefaultAudioWaveBufferSize = 16 * 1024 * 1024;
    static const size_t DefaultAudioEffectBufferSize = 12 * 1024 * 1024;

    // MixBuffer
    static const int FinalMixBufferCount = 6;
    static const int SubMixBufferCount = 2;
    static const int DefaultVoiceCount = 24;
    static const int DefaultSinkCount = 1;
    static const int DefaultEffectCount = 4;
    static const int ChannelCount = 2;

protected:
    // 基本情報
    nn::audio::AudioRendererParameter m_Parameter;
    nn::os::SystemEvent m_SystemEvent;

    nn::audio::AudioRendererHandle m_Handle;
    void* m_WorkBuffer = nullptr;
    size_t m_WorkBufferSize;

    nn::audio::AudioRendererConfig m_Config;
    void* m_ConfigBuffer = nullptr;
    size_t m_ConfigBufferSize;

    // アロケータ
    nn::mem::StandardAllocator m_WorkBufferAllocator;
    nn::mem::StandardAllocator m_WaveBufferAllocator;
    nn::mem::StandardAllocator m_EffectBufferAllocator;

    // メモリプール
    void* m_WaveBuffer = nullptr;
    size_t m_WaveBufferSize;
    nn::audio::MemoryPoolType m_WaveBufferMemoryPool;

    void* m_EffectBuffer = nullptr;
    size_t m_EffectBufferSize;
    nn::audio::MemoryPoolType m_EffectBufferMemoryPool;

    // FinalMix / SubMix0-1
    nn::audio::FinalMixType m_FinalMix;
    nn::audio::SubMixType m_SubMix0;
    nn::audio::SubMixType m_SubMix1;

    // Sink
    nn::audio::DeviceSinkType m_DeviceSink;

    // AudioBus
    int8_t m_MainBus[ChannelCount];
    int8_t m_AuxBusA[ChannelCount];
    int8_t m_AuxBusB[ChannelCount];

    // 調整パラメーター
    float m_VoiceVolume = 1.0f;
    float m_VoicePitch = 1.0f;

    // AudioHandle 管理
    std::map<std::string, AudioHandle*> m_AudioHandles;

protected:
    AudioPlayer() NN_NOEXCEPT {}
    virtual ~AudioPlayer() NN_NOEXCEPT {}

    // WavFile を AudioHandle に設定する
    void LoadWavFile(AudioHandle* handle, void* wavData, size_t wavDataSize) NN_NOEXCEPT;
};


