﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/hid/hid_VibrationValueTypes.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

//!< 時系列の振動値を保存しておくバッファを扱うクラスです。
class VibrationValueBuffer
{
    NN_DISALLOW_COPY(VibrationValueBuffer);
    NN_DISALLOW_MOVE(VibrationValueBuffer);

public:
    //!< 保持する振動値の数。
    static const int BufferSize = 60;

public:
    VibrationValueBuffer() NN_NOEXCEPT
        : m_BufferAddr(0)
    {
        //warning C4351対策のためmemsetで配列の初期化を行う
        memset(m_CurrentValues, 0x00, BufferSize * sizeof(nn::hid::VibrationValue));
        memset(m_ActualValues,  0x00, BufferSize * sizeof(nn::hid::VibrationValue));
    }

    ~VibrationValueBuffer() NN_NOEXCEPT {}

    //!< 指令した振動値と実際の振動値を保存します。
    void AddVibrationValues(const nn::hid::VibrationValue& currentVibrationValue,
        const nn::hid::VibrationValue& actualVibrationValue) NN_NOEXCEPT;

    //!< 特定のタイミングの振動値（指令した振動値と実際の振動値）の振幅を取得します。
    void GetAmplitudes(nn::util::Float4* values, int time) const NN_NOEXCEPT;

    //!< 最新の実際の振動値を取得します。
    const nn::hid::VibrationValue& GetActualVibrationValue() const NN_NOEXCEPT
    {
        return m_ActualValues[m_BufferAddr];
    }

    //!< 最新の指令した振動値を取得します。
    const nn::hid::VibrationValue& GetCurrentVibrationValue() const NN_NOEXCEPT
    {
        return m_CurrentValues[m_BufferAddr];
    }

protected:
    int m_BufferAddr;
    nn::hid::VibrationValue m_CurrentValues[BufferSize];
    nn::hid::VibrationValue m_ActualValues[BufferSize];
};

//!< 振動値のグラフィカルな描画を扱うクラスです。
class VibrationValueDrawer
{
    NN_DISALLOW_COPY(VibrationValueDrawer);
    NN_DISALLOW_MOVE(VibrationValueDrawer);

public:
    VibrationValueDrawer(nn::gfx::util::DebugFontTextWriter* pWriter,
                         nn::gfx::CommandBuffer* pCommandBuffer,
                         nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer) NN_NOEXCEPT
        : m_pWriter(pWriter)
        , m_pCommandBuffer(pCommandBuffer)
        , m_pPrimitiveRenderer(pPrimitiveRenderer)
        , m_Scale(1.0f)
        , m_X(0.0f)
        , m_Y(0.0f)
        , m_ColorPatternId(0)
    {
        //何もしない
    }

    ~VibrationValueDrawer() NN_NOEXCEPT {}

    void SetPosition(float x, float y) NN_NOEXCEPT;
    void SetScale(float scale) NN_NOEXCEPT;
    void SetBrightColor(bool isBrightColorEnabled) NN_NOEXCEPT;

    //!< 振動の振幅と周波数のグラフを描画します
    void DrawVibrationValue(const nn::hid::VibrationValue& vibrationValue, bool isTextPrinted) const NN_NOEXCEPT;

    //!< 振動の振幅の時系列のグラフを描画します。
    void DrawTimeSeriesGraph(const VibrationValueBuffer& buffer, bool isDrawLowFreqBand ) const NN_NOEXCEPT;

protected:
    void DrawAxis() const NN_NOEXCEPT;
    void DrawAmplitudeAndFrequency(float amplitude, float frequency,
        bool isDrawLowFreqBand, bool isTextPrinted) const NN_NOEXCEPT;

protected:
    nn::gfx::util::DebugFontTextWriter* const m_pWriter;
    nn::gfx::CommandBuffer* const m_pCommandBuffer;
    nns::gfx::PrimitiveRenderer::Renderer* const m_pPrimitiveRenderer;
    float m_Scale;
    float m_X;
    float m_Y;
    int m_ColorPatternId;
};
