﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/util/util_Vector.h>

#include "Pointer.h"

namespace {

template <typename T>
T Clamp(T value, T min, T max)
{
    if (value < min)
    {
        value = min;
    }
    else if (max < value)
    {
        value = max;
    }

    return value;
}

template <typename T>
T LinearInterpolation(T x1, T x2, T coefficient)
{
    return x1 * (1 - coefficient) + x2 * coefficient;
}

static float CalculateVerticalAngle(const ::nn::util::Vector3f& dir)
{
    // 上が0度、下が180度
    float v = Clamp(::nn::util::VectorGetY(dir), -1.0f, 1.0f);

    float degree = nn::util::RadianToDegree(nn::util::AcosEst(v));

    // 上が-90度、下が90度
    degree -= 90.0f;

    // 上が90度、下が-90度
    return -degree;
}

::nn::util::Vector3f GetBaseDirection(const ::nn::util::Float3& directionY)
{
    return ::nn::util::Vector3f(-directionY.x, directionY.z, directionY.y);
}

const ::nn::util::Float3 InitialFrontVector = { {{ 0.0f, 0.0f, 1.0f }} };

// 画面サイズ
const float Width = 1280.0f;
const float Height = 720.0f;

// カーソルの中心座標
const float CursorCenterX = Width * 0.5f;
const float CursorCenterY = Height * 0.5f;

// バイアス
const float BiasX = 5.0f;
const float BiasY = 5.0f;

} // namespace

SixAxisSensorPointer::SixAxisSensorPointer() NN_NOEXCEPT
    : m_BaseAngle(0.0f)
    , m_Front(InitialFrontVector)
    , m_Cursor()
    , m_Direction()
{
    // 何もしない
}

void SixAxisSensorPointer::Update(const nn::hid::DirectionState& direction) NN_NOEXCEPT
{
    m_Direction = direction;

    // 画面座標に変換

    ::nn::util::Vector3f pointingDirection = GetBaseDirection(direction.y);
    ::nn::util::Vector3f pointingHorizontalDirection = pointingDirection;

    ::nn::util::VectorSetY(&pointingHorizontalDirection, 0.0f);
    pointingHorizontalDirection.Normalize();

    float x = 0.0f;
    float y = 0.0f;
    const float limit = 30.0f;

    // y = 正面ベクトル(dir_z)から
    {
        float verticalAngle = CalculateVerticalAngle(pointingDirection);

        // 基準値からの相対
        verticalAngle -= m_BaseAngle;
        verticalAngle = Clamp(verticalAngle, -1.0f * limit, limit);

        y = nn::util::TanEst(nn::util::DegreeToRadian(verticalAngle));
        y = y * BiasY;
        y = y * 0.5f + 0.5f;
    }

    // x =
    {
        ::nn::util::Vector3f front;
        ::nn::util::VectorLoad(&front, m_Front);

        // 基準となる前方向ベクトル
        ::nn::util::Vector3f rightDirection = front;
        rightDirection = rightDirection.Cross(::nn::util::Vector3f::UnitY());

        // frontとの角度差
        float cos = front.Dot(pointingHorizontalDirection);
        cos = Clamp(cos, -1.0f, 1.0f);

        // 0度～180度
        float horizontalAngle = nn::util::RadianToDegree(nn::util::AcosEst(cos));

        // 符号
        if (rightDirection.Dot(pointingHorizontalDirection) < 0.0f)
        {
            horizontalAngle = -horizontalAngle;
        }

        horizontalAngle = Clamp(horizontalAngle, -1.0f * limit, limit);

        x = nn::util::TanEst(nn::util::DegreeToRadian(horizontalAngle));
        x = x * BiasX;
        x = x * 0.5f + 0.5f;
    }

    const float lpf = 0.75f;
    // x = -640～640, y = -360, 360の範囲に変換
    m_Cursor.x = LinearInterpolation(-Width * 0.5f, Width * 0.5f, x) * lpf + m_Cursor.x * (1.0f - lpf);
    m_Cursor.y = LinearInterpolation(-Height * 0.5f, Height * 0.5f, y) * lpf + m_Cursor.y *  (1.0f - lpf);
}

void SixAxisSensorPointer::Reset() NN_NOEXCEPT
{
    ::nn::util::Vector3f baseDirection = GetBaseDirection(m_Direction.y);

    // 中心へ補正
    m_Front.x = ::nn::util::VectorGetX(baseDirection);
    m_Front.y = 0.0f;
    m_Front.z = ::nn::util::VectorGetZ(baseDirection);

    ::nn::util::Vector3f front;
    ::nn::util::VectorLoad(&front, m_Front);

    if (!front.Normalize())
    {
        front = ::nn::util::Vector3f::UnitZ();
    }
    ::nn::util::VectorStore(&m_Front, front);

    m_BaseAngle = CalculateVerticalAngle(baseDirection);
    m_Cursor.x = 0.0f;
    m_Cursor.y = 0.0f;
}

::nn::util::Vector3f SixAxisSensorPointer::GetCursor() const NN_NOEXCEPT
{
    ::nn::util::Vector3f pointer;

    //ジャイロポインタは常に画面内に収める
    float x = CursorCenterX + m_Cursor.x;
    x = std::max(x, 0.0f);
    x = std::min(x, Width - 10.0f);

    float y = CursorCenterY - m_Cursor.y;
    y = std::max(y, 0.0f);
    y = std::min(y, Height - 10.0f);

    pointer.SetX(x);
    pointer.SetY(y);

    return pointer;
}
