﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidVibrationGenerator.cpp,PageSampleHidVibrationGenerator}

    @brief
    パラメータから簡単に振動波形を生成するプログラム
 */

/**
    @page PageSampleHidVibrationGenerator パラメータによる簡易な振動波形生成
    @tableofcontents

    @brief
    パラメータから簡単に振動波形を生成するプログラムの解説です。

    @section PageSampleHidVibrationGenerator_SectionBrief 概要
    パラメータを設定することで振動波形を生成する方法を説明します。
    ソースコード内に、直接パラメータを記載することで、様々な振動波形をデザインできます。
    この方法では、振動ファイルが不要のため、アセット管理が不要です。
    より手軽に振動機能を使うことが可能です。

    DumpedVibrationGenerator は nn::hid::VibrationNode を継承したクラスです。
    このクラスでは、振幅や周波数、時間、減衰カーブといったパラメータを設定することで、様々な振動波形を生成できます。
    nn::hid::VibrationPlayer と同様に、Play()で振動を再生、Stop()で振動を停止します。

    @section PageSampleHidVibrationGenerator_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidVibrationGenerator @endlink 以下にあります。

    @section PageSampleHidVibrationGenerator_SectionNecessaryEnvironment 必要な環境
    Windows 環境で動作させる場合は、事前に PC に Bluetooth ドングルを接続した上で、PC とコントローラをペアリングしてください。
    SDEV/EDEV 環境で動作させる場合は、事前に SDEV/EDEV とコントローラをペアリングしてください。
    1 セットのコントローラを無線で接続する必要があります。

    @section PageSampleHidVibrationGenerator_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。

    B ボタンをおしてください、ジャイロポインタが画面中央に表示されます。
    ジャイロポインタ（赤色の ＋ ）を操作して、画面右側の青い線群に触れてください。
    ポインタが線に触れるとコントローラが振動します。
    一度に複数の線に触れると、振動が重なり、大きな振動となります。

    振動波形として、複数のプリセットが用意されています。
    L / R ボタンを押すとプリセットを切り替えられます。
    プリセットを変えることで、振動波形の各パラメータが大きく変わり、触れたときの感触が変わります。
    また、 R スティックと方向ボタンで各プリセットのパラメータを細かく変更できます。

    各ボタンを使って以下の操作が可能です。

    - A ボタン：振動を再生します。
    - Y ボタン：実際に発生している振動値のオーバレイ表示の有効にします。
    - + ボタン：振動パラメータ設定の初期化します。
    - B ボタン：ジャイロポインタのリセットします（画面中心にポインタが移動します）。
    - L ボタン：振動波形のパラメータを次のプリセットに変更します。
    - R ボタン：振動波形のパラメータを前のプリセットに変更します。

    - R スティック：上下左右でパラメータのフィールドを変更します。
    - L スティック：上下でパラメータの値を変更します。左右でパラメータのフィールドを変更します。
    - 方向ボタン：上下でパラメータの値を変更します。左右でパラメータのフィールドを変更します。

    サンプルプログラムを終了させるには + ボタンと - ボタンを同時に押してください。

    @section PageSampleHidVibrationGenerator_SectionPrecaution 注意事項
    コントローラは十分に充電した状態でお使いください。

    @section PageSampleHidVibrationGenerator_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。
    SDEV/EDEV 環境の場合、 TV モードもしくはテーブルモードで実行してください。
    1 セットのコントローラを無線で接続してください。

    @section PageSampleHidVibrationGenerator_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - DumpedVibrationGenerator にパラメータを設定する。
    - アクティブなコントローラを判定する。
    - 複数の DumpedVibrationGenerator を nn::hid::VibrationMixer で混ぜた上で、アクティブなコントローラの nn::hid::VibrationTarget に接続する。
    - ジャイロポインタが青色の線群の当たり判定を行う。
    - 線に当たっていれば DumpedVibrationGenerator::Play() を実行する。
    - 別スレッドで nn::hid::VibrationNode を nn::hid::VibrationNode::DefaultVibrationSampleInterval 毎に更新して振動子に振動値を送信する。
    - アクティブなコントローラが振動する。
 */

#include <cstdlib>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os.h>
#include <nn/os/os_Thread.h>
#include <nn/os/os_SystemEvent.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#if defined(NN_BUILD_TARGET_PLATFORM_OS_WIN)
#include <nn/nn_Windows.h>
#endif

#include "GraphicsSystem.h"
#include "NpadController.h"
#include "ControllerManager.h"
#include "Color.h"
#include "Demo.h"
#include "DumpedVibrationGenerator.h"

namespace
{
    // グラフィックス設定
    const int FrameRate = 60;
    const char ProgramName[] = "HidVibrationGenerator & GyroPointer";

    GraphicsSystem* g_pGraphicsSystem;
    nn::mem::StandardAllocator* g_pAppAllocator;
    nn::Bit8* g_pAppMemory;

    void DrawTitle() NN_NOEXCEPT
    {
        auto pTextWriter = &g_pGraphicsSystem->GetDebugFont();

        pTextWriter->Draw(&g_pGraphicsSystem->GetCommandBuffer());
        pTextWriter->SetScale(2.0f, 2.0f);
        pTextWriter->SetTextColor(Color::LimeGreen);
        pTextWriter->SetCursor(10.0f, 10.0f);
        pTextWriter->Print("%s", ProgramName);
    }

    void InitializeGraphics() NN_NOEXCEPT
    {
        // Memory
        g_pAppAllocator = new nn::mem::StandardAllocator();
        const size_t appMemorySize = 128 * 1024 * 1024;
        g_pAppMemory = new nn::Bit8[appMemorySize];
        g_pAppAllocator->Initialize(g_pAppMemory, appMemorySize);

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
        const size_t graphicsMemorySize = 256 * 1024 * 1024;
        void* pGraphicsMemory =
            nns::gfx::GraphicsFramework::DefaultAllocateFunction(graphicsMemorySize, 1, nullptr);

        nv::SetGraphicsAllocator(
            nns::gfx::GraphicsFramework::DefaultAllocateFunction,
            nns::gfx::GraphicsFramework::DefaultFreeFunction,
            nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);

        nv::SetGraphicsDevtoolsAllocator(
            nns::gfx::GraphicsFramework::DefaultAllocateFunction,
            nns::gfx::GraphicsFramework::DefaultFreeFunction,
            nns::gfx::GraphicsFramework::DefaultReallocateFunction, nullptr);

        nv::InitializeGraphics(pGraphicsMemory, graphicsMemorySize);
#endif
        // Graphics
        g_pGraphicsSystem = new ::GraphicsSystem();
        g_pGraphicsSystem->SetApplicationHeap(g_pAppAllocator);
        g_pGraphicsSystem->Initialize();
    }

    void UpdateGraphics() NN_NOEXCEPT
    {
        g_pGraphicsSystem->BeginDraw();

        // 操作方法の表示
        DrawTitle();
        Demo::GetInstance().Draw(g_pGraphicsSystem);

        // 描画終了
        g_pGraphicsSystem->EndDraw();
        g_pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000L * 1000L * 1000L / FrameRate));
    }

    void FinalizeGraphics() NN_NOEXCEPT
    {
        g_pGraphicsSystem->Finalize();
        delete g_pGraphicsSystem;

        g_pAppAllocator->Finalize();
        delete g_pAppAllocator;

        delete[] g_pAppMemory;
    }

} // anonymous-namespace

extern "C" void nnMain()
{
    NN_LOG("%s Start.\n", ProgramName);

    nn::hid::InitializeDebugPad();

    // グラフィックス機能を初期化する
    InitializeGraphics();
    // コントローラーを初期化する
    ControllerManager::GetInstance().Initialize();
    // デモに使用するオブジェクトを初期化する
    Demo::GetInstance().Initialize();

    StartVibrationThread();

    bool isQuitRequired = false;

    // Main Loop
    while (!isQuitRequired)
    {
        ControllerManager::GetInstance().Update();
        Demo::GetInstance().Update();

        UpdateGraphics();

        //終了判定
        if (ControllerManager::GetInstance().GetActiveController()->IsQuitRequired())
        {
            isQuitRequired = true;
        }
    }

    // 終了処理
    StopVibrationThread();
    ControllerManager::GetInstance().Finalize();
    FinalizeGraphics();

    NN_LOG("%s Done\n", ProgramName);
}
