﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   DumpedVibrationGenerator クラスの定義
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_Assert.h>
#include <nn/hid/hid_VibrationNode.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nns/gfx/gfx_GraphicsFramework.h>
#include <nns/gfx/gfx_PrimitiveRenderer.h>

//!< 振動波形のカーブパターンです。
enum DumpedPattern
{
    DumpedPattern_A_Curbe = 0,  //!< 指数
    DumpedPattern_B_Curbe,      //!< リニア
    DumpedPattern_C_Curbe,      //!< 余弦
    DumpedPattern_D_Curbe,      //!< 一定
    DumpedPattern_Max,
};

//!< 振動波形のパラメータです。
struct DumpedVibrationParameter
{
    float amplitude;            //!< 波形の振幅
    float freqFirst;            //!< 波形の開始点での周波数
    float freqEnd;              //!< 波形の終端での周波数
    int offset;                 //!< 波形の立ち上がりまでのオフセット
    int time;                   //!< 波形の長さ
    DumpedPattern pattern;      //!< カーブのパターン
};

//!< 振動波形のパラメータを編集するクラスです。
class DumpedVibrationEditer
{
    NN_DISALLOW_COPY(DumpedVibrationEditer);
    NN_DISALLOW_MOVE(DumpedVibrationEditer);

public:
    DumpedVibrationEditer() NN_NOEXCEPT
        : m_Row(0)
        , m_Column(0)
    {

    }

    ~DumpedVibrationEditer() NN_NOEXCEPT
    {
        //何もしない
    }

    void Set(int column, int row) NN_NOEXCEPT
    {
        if (row < 0)
        {
            m_Row = 0;
        }
        else if (row >= RowMax)
        {
            m_Row = RowMax - 1;
        }
        else
        {
            m_Row = row;
        }

        if (column < 0)
        {
            m_Column = 0;
        }
        else if (column >= ColumnMax[m_Row])
        {
            m_Column = ColumnMax[m_Row] - 1;
        }
        else
        {
            m_Column = column;
        }
    }

    void Up() NN_NOEXCEPT
    {
        Set(m_Column, m_Row - 1);
    }

    void Down() NN_NOEXCEPT
    {
        Set(m_Column, m_Row + 1);
    }

    void Right() NN_NOEXCEPT
    {
        Set(m_Column + 1, m_Row);
    }

    void Left() NN_NOEXCEPT
    {
        Set(m_Column - 1, m_Row);
    }

    int Get() NN_NOEXCEPT
    {
        int x = 0;
        for (int i = 0; i < m_Row; i++)
        {
            x += ColumnMax[i];
        }
        x += m_Column;
        return x;
    }

private:
    static const int RowMax;
    static const int ColumnMax[];
    int m_Row;
    int m_Column;
};

//!< パラメータベースの振動波形のグラフィカルな描画を扱うクラスです。
class DumpedVibrationDrawer
{
    NN_DISALLOW_COPY(DumpedVibrationDrawer);
    NN_DISALLOW_MOVE(DumpedVibrationDrawer);

public:
    DumpedVibrationDrawer(nn::gfx::util::DebugFontTextWriter* pWriter,
        nn::gfx::CommandBuffer* pCommandBuffer,
        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer) NN_NOEXCEPT
        : m_pWriter(pWriter)
        , m_pCommandBuffer(pCommandBuffer)
        , m_pPrimitiveRenderer(pPrimitiveRenderer)
        , m_Scale(1.0f)
        , m_X(0.0f)
        , m_Y(0.0f)
        , m_ColorPatternId(0)
        , m_SelectedParameter(0)
    {
        //何もしない
    }

    ~DumpedVibrationDrawer() NN_NOEXCEPT
    {
        //何もしない
    }

    void SetPosition(float x, float y) NN_NOEXCEPT
    {
        m_X = x;
        m_Y = y;
    }

    void SetScale(float scale) NN_NOEXCEPT
    {
        m_Scale = scale;
    }

    void SetBrightColor(bool isBrightColorEnabled) NN_NOEXCEPT
    {
        m_ColorPatternId = isBrightColorEnabled ? 0 : 1;
    }

    void SetSelectedParameter(int selectedParameter) NN_NOEXCEPT
    {
        m_SelectedParameter = selectedParameter;
    }

    //!< 振動の振幅と周波数のグラフを描画します
    void Draw(const DumpedVibrationParameter& parameter) const NN_NOEXCEPT;

private:
    void DrawText(const DumpedVibrationParameter& parameter) const NN_NOEXCEPT;
    void DrawLine(const DumpedVibrationParameter& parameter) const NN_NOEXCEPT;

private:
    nn::gfx::util::DebugFontTextWriter* const m_pWriter;
    nn::gfx::CommandBuffer* const m_pCommandBuffer;
    nns::gfx::PrimitiveRenderer::Renderer* const m_pPrimitiveRenderer;
    float m_Scale;
    float m_X;
    float m_Y;
    int m_ColorPatternId;
    int m_SelectedParameter;
};

//!< パラメータベースの振動波形を生成するクラスです。VibrationNodeを継承しています。
class DumpedVibrationGenerator : public nn::hid::VibrationNode
{
    NN_DISALLOW_COPY(DumpedVibrationGenerator);
    NN_DISALLOW_MOVE(DumpedVibrationGenerator);

public:
    DumpedVibrationGenerator() NN_NOEXCEPT;

    virtual ~DumpedVibrationGenerator() NN_NOEXCEPT NN_OVERRIDE
    {
        //何もしない
    }

    //!< 指定した振動パラメータを設定します。idx==0の時は低周波帯域、idx==1の時は高周波帯域を設定します。
    void SetParameter(const DumpedVibrationParameter& parameter, int idx) NN_NOEXCEPT
    {
        NN_ASSERT_GREATER_EQUAL(parameter.amplitude, 0.0f);
        NN_ASSERT_GREATER_EQUAL(parameter.time, 0);
        NN_ASSERT_GREATER_EQUAL(parameter.offset, 0);
        NN_ASSERT_GREATER_EQUAL(parameter.freqFirst, 0.0f);
        NN_ASSERT_GREATER_EQUAL(parameter.freqEnd, 0.0f);
        NN_ASSERT_GREATER_EQUAL(idx, 0);
        NN_ASSERT_LESS(idx, 2);

        m_Parameters[idx] = parameter;
    }

    void Play() NN_NOEXCEPT
    {
        if (isPlay)
        {
            //すでに再生中のため何もしない
        }
        else
        {
            isPlay = true;
            m_CurrentTime = 0;
        }
    }

    void Stop() NN_NOEXCEPT
    {
        isPlay = false;
    }

    bool IsPlaying() const NN_NOEXCEPT
    {
        return isPlay;
    }

protected:
    virtual void OnNextSampleRequired(
        nn::hid::VibrationValue* pValue,
        nn::hid::VibrationNodeConnection::List* pInputConnections) NN_NOEXCEPT NN_OVERRIDE;

protected:
    static const int NumOfSuband = 2;   //!< 低周波領域と高周波領域の２つ
    int  m_CurrentTime;                 //!< 再生時間
    bool isPlay;                        //!< 再生しているかどうか

    DumpedVibrationParameter m_Parameters[NumOfSuband];

};
