﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Demo.h"
#include "Color.h"
#include "ControllerManager.h"

namespace
{
    int paraColumn  = 0; //パラメータ
    int paraSubBand = 0; //1:高 0:低

    const int NumOfSubBand = 2;
    const int NumOfSample  = 7;

    DumpedVibrationEditer editer[NumOfSubBand];

    // 振動データを VibrationPlayer で読み込む
    DumpedVibrationParameter parameters[NumOfSubBand*NumOfSample] =
    {
        { 0.3f, 160.0f, 160.0f, 0, 300, DumpedPattern_A_Curbe },
        { 0.0f, 160.0f, 160.0f, 0, 100, DumpedPattern_A_Curbe },

        { 1.0f, 250.0f, 250.0f, 0, 500, DumpedPattern_A_Curbe },
        { 0.0f, 160.0f, 160.0f, 0, 100, DumpedPattern_A_Curbe },

        { 0.5f, 180.0f, 100.0f, 0, 250, DumpedPattern_B_Curbe },
        { 0.0f, 160.0f, 160.0f, 0, 100, DumpedPattern_A_Curbe },

        { 0.3f, 160.0f, 160.0f, 0, 200, DumpedPattern_A_Curbe },
        { 0.3f, 200.0f, 200.0f, 0, 200, DumpedPattern_A_Curbe },

        { 0.8f, 160.0f, 160.0f,   0, 200, DumpedPattern_A_Curbe },
        { 0.2f, 200.0f, 150.0f, 100, 400, DumpedPattern_C_Curbe },

        { 0.2f, 180.0f, 180.0f, 0, 500, DumpedPattern_C_Curbe },
        { 0.8f, 300.0f, 300.0f, 0, 100, DumpedPattern_A_Curbe },

        { 1.0f, 110.0f, 100.0f, 0, 800, DumpedPattern_A_Curbe },
        { 0.0f, 160.0f, 160.0f, 0, 100, DumpedPattern_A_Curbe },
    };

    void ChangeParameter(DumpedVibrationParameter& parameter, int index, bool isUp) NN_NOEXCEPT
    {
        switch (index)
        {
        case 0:
            parameter.amplitude += isUp ? 0.05f : -0.05f;
            parameter.amplitude = std::max(parameter.amplitude, 0.0f);
            break;
        case 1:
            if (parameter.pattern == DumpedPattern_A_Curbe)
            {
                parameter.pattern = isUp ? DumpedPattern_B_Curbe : DumpedPattern_A_Curbe;
            }
            else if (parameter.pattern == DumpedPattern_B_Curbe)
            {
                parameter.pattern = isUp ? DumpedPattern_C_Curbe : DumpedPattern_A_Curbe;
            }
            else if (parameter.pattern == DumpedPattern_C_Curbe)
            {
                parameter.pattern = isUp ? DumpedPattern_D_Curbe : DumpedPattern_B_Curbe;
            }
            else
            {
                parameter.pattern = isUp ? DumpedPattern_D_Curbe : DumpedPattern_C_Curbe;
            }
            break;
        case 2:
            parameter.freqFirst += isUp ? 10.0f : -10.0f;
            parameter.freqFirst = std::max(parameter.freqFirst, 40.0f);
            break;
        case 3:
            parameter.freqEnd += isUp ? 10.0f : -10.0f;
            parameter.freqEnd = std::max(parameter.freqEnd, 40.0f);
            break;
        case 4:
            parameter.time += isUp ? 10 : -10;
            parameter.time = std::max(parameter.time, 0);
            break;
        case 5:
            parameter.offset += isUp ? 10 : -10;
            parameter.offset = std::max(parameter.offset, 0);
            break;
        default:
            break;
        }
    }
}

Demo& Demo::GetInstance() NN_NOEXCEPT
{
    static Demo instance;
    return instance;
}

// デモに使用するオブジェクトの初期化
void Demo::Initialize() NN_NOEXCEPT
{
    for (int i = 0; i < BoxCountMax; i++)
    {
        // 描画場所の設定
        m_Box[i].SetPos(800.0f,  300.0f * i + 100.0f);

        // 振動パラメータの設定
        int addr = NumOfSubBand * m_VibrationPattern;
        m_Box[i].SetVibrationParameter(parameters[addr], parameters[addr + 1]);

        // 各Box単位の振動をMixerに合算する
        auto pOutput = m_Box[i].GetVibrationOutput();
        m_ConnectionFromBoxToMixer[i].Connect(pOutput, &m_Mixer);
    }

    // Mixerは振幅を加算する設定で使用する
    m_Mixer.SetMixMode(nn::hid::VibrationMixMode_AmplitudeSum);

    // Mixerから左右のターゲットに接続する
    for (int i = 0; i < VibrationDeviceCountMax; i++)
    {
        m_ConnectionFromMixerToTarget[i].Connect(&m_Mixer, &m_Target[i]);
    }

    // 振動ターゲットの設定
    ControllerManager::GetInstance().SetActiveVibrationTarget(&m_Target[0], &m_Target[1]);

}

void Demo::Update() NN_NOEXCEPT
{
    UpdateButton();
    UpdateSensor();
}

void Demo::Draw(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
{
    DrawEnv(pGraphicsSystem);

    if (m_IsGraphicsEnabled)
    {
        DrawVibrationValue(pGraphicsSystem);
    }

    DrawObject(pGraphicsSystem);
    DrawVibrationGenerator(pGraphicsSystem);
}

//コントローラによるデモ設定の変更
void Demo::UpdateButton() NN_NOEXCEPT
{
    static nn::hid::NpadButtonSet previousButtons;
    const auto& currentButtons = ControllerManager::GetInstance().GetActiveController()->GetNpadButtonSet();
    nn::hid::NpadButtonSet downButtons = currentButtons & ~previousButtons;
    previousButtons = currentButtons;

    // Bでジャイロポインタの位置リセット
    if (downButtons.Test(nn::hid::NpadButton::B::Index))
    {
        ControllerManager::GetInstance().ResetCursor();
    }
    // Rボタンで次のデモ環境に変更
    else if (downButtons.Test(nn::hid::NpadButton::R::Index))
    {
        if (m_VibrationPattern < NumOfSample - 1)
        {
            m_VibrationPattern++;
        }
        for (int i = 0; i < BoxCountMax; i++)
        {
            //減衰振動設定
            int addr = NumOfSubBand * m_VibrationPattern;
            m_Box[i].SetVibrationParameter(parameters[addr], parameters[addr + 1]);
        }
    }
    // Lボタンで次のデモ環境に変更
    else if (downButtons.Test(nn::hid::NpadButton::L::Index))
    {
        if (m_VibrationPattern > 0)
        {
            m_VibrationPattern--;
        }
        for (int i = 0; i < BoxCountMax; i++)
        {
            //減衰振動設定
            int addr = NumOfSubBand * m_VibrationPattern;
            m_Box[i].SetVibrationParameter(parameters[addr], parameters[addr + 1]);
        }
    }
    // Aボタンで振動再生
    else if (downButtons.Test(nn::hid::NpadButton::A::Index))
    {
        m_Box[0].PlayVibration();
    }
    // Yボタンでデバッグ表示を有効にする
    else if (downButtons.Test(nn::hid::NpadButton::Y::Index))
    {
        m_IsGraphicsEnabled = !m_IsGraphicsEnabled;
        ControllerManager::GetInstance().ResetCursor();
    }
    // +ボタンで振動パラメータを初期化する
    else if (downButtons.Test(nn::hid::NpadButton::Plus::Index))
    {
        int p = m_VibrationPattern * NumOfSubBand + paraSubBand;
        auto& vp = parameters[p];
        vp = { 0.5f, 160.0f, 160.0f, 0, 100, DumpedPattern_A_Curbe };
    }
    else if (downButtons.Test(nn::hid::NpadButton::Up::Index)
        || downButtons.Test(nn::hid::NpadButton::StickLUp::Index))
    {
        int p = m_VibrationPattern * NumOfSubBand + paraSubBand;
        auto& vp = parameters[p];
        ChangeParameter(vp, paraColumn, true);
    }
    else if (downButtons.Test(nn::hid::NpadButton::Down::Index)
        || downButtons.Test(nn::hid::NpadButton::StickLDown::Index))
    {
        int p = m_VibrationPattern * NumOfSubBand + paraSubBand;
        auto& vp = parameters[p];
        ChangeParameter(vp, paraColumn, false);
    }
    else if (downButtons.Test(nn::hid::NpadButton::Left::Index)
        || downButtons.Test(nn::hid::NpadButton::StickLLeft::Index)
        || downButtons.Test(nn::hid::NpadButton::StickRLeft::Index))
    {
        editer[paraSubBand].Left();
        paraColumn = editer[paraSubBand].Get();
    }
    else if (downButtons.Test(nn::hid::NpadButton::Right::Index)
        || downButtons.Test(nn::hid::NpadButton::StickLRight::Index)
        || downButtons.Test(nn::hid::NpadButton::StickRRight::Index))
    {
        editer[paraSubBand].Right();
        paraColumn = editer[paraSubBand].Get();
    }
    else if (downButtons.Test(nn::hid::NpadButton::StickRUp::Index))
    {
        if (paraSubBand == 1 && paraColumn < 2)
        {
            paraSubBand = (paraSubBand + NumOfSubBand - 1) % NumOfSubBand;
            editer[paraSubBand].Set(0, 3);
        }
        else
        {
            editer[paraSubBand].Up();
        }
        paraColumn = editer[paraSubBand].Get();
    }
    else if (downButtons.Test(nn::hid::NpadButton::StickRDown::Index))
    {
        if (paraSubBand == 0 && paraColumn > 3)
        {
            paraSubBand = (paraSubBand + 1) % NumOfSubBand;
            editer[paraSubBand].Set(0, 0);
        }
        else
        {
            editer[paraSubBand].Down();
        }
        paraColumn = editer[paraSubBand].Get();
    }

    for (int i = 0; i < BoxCountMax; i++)
    {
        //減衰振動設定
        int addr = NumOfSubBand * m_VibrationPattern;
        m_Box[i].SetVibrationParameter(parameters[addr], parameters[addr + 1]);
    }
}  // NOLINT(readability/fn_size)

void Demo::UpdateSensor() NN_NOEXCEPT
{
    //ジャイロポインタの座標をBoxに反映させる
    const auto& pController = ControllerManager::GetInstance().GetActiveController();
    auto pos = pController->GetCursor();
    for (int i = 0; i < BoxCountMax; i++)
    {
        m_Box[i].Update(pos.GetX(), pos.GetY());
    }
}

void Demo::DrawVibrationGenerator(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
{
    auto pTextWriter = &pGraphicsSystem->GetDebugFont();

    for (int i = 0; i < NumOfSubBand; i++)
    {
        DumpedVibrationDrawer envDrawer(
            pTextWriter,
            &pGraphicsSystem->GetCommandBuffer(),
            &pGraphicsSystem->GetPrimitiveRenderer());

        envDrawer.SetScale(1.0f);
        envDrawer.SetPosition(20.0f, 200.0f + 250.0f * i);

        int p = m_VibrationPattern * NumOfSubBand + i;
        auto& vp = parameters[p];
        envDrawer.SetSelectedParameter(paraColumn);
        envDrawer.SetBrightColor(i == paraSubBand ? true : false);
        envDrawer.Draw(vp);
    }
}

void Demo::DrawVibrationValue(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
{
    auto pTextWriter = &pGraphicsSystem->GetDebugFont();

    VibrationValueDrawer vibDrawer(
        pTextWriter,
        &pGraphicsSystem->GetCommandBuffer(),
        &pGraphicsSystem->GetPrimitiveRenderer());

    vibDrawer.SetScale(1.0f);
    vibDrawer.SetBrightColor(true);

    for (int i = 0; i < ControllerManager::VibrationTargetCountMax; i++)
    {
        float baseX = 700.0f + 280.0f * i;
        float baseY = 100.0f;

        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(0.8f, 0.8f);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print((i == 0) ? "Left Vibration Device" : "Right Vibration Device");

        nn::hid::VibrationValue value
            = ControllerManager::GetInstance().GetVibrationBuffer(i).GetActualVibrationValue();

        vibDrawer.SetPosition(baseX, baseY + 15.0f);
        vibDrawer.DrawVibrationValue(value, true);

        baseY += 150.0f;

        bool isLow = true;
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("Time Series amplitude%s value", isLow ? "Low" : "High");
        vibDrawer.SetPosition(baseX, baseY + 20.0f);
        vibDrawer.DrawTimeSeriesGraph(
            ControllerManager::GetInstance().GetVibrationBuffer(i), isLow);

        baseY += 150.0f;

        isLow = false;
        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetCursor(baseX, baseY);
        pTextWriter->Print("Time Series amplitude%s value", isLow ? "Low" : "High");
        vibDrawer.SetPosition(baseX, baseY + 20.0f);
        vibDrawer.DrawTimeSeriesGraph(
            ControllerManager::GetInstance().GetVibrationBuffer(i), isLow);
    }

    pTextWriter->SetCursor(700.0f, 540.0f);
    pTextWriter->SetTextColor(Color::Green);
    pTextWriter->Print("GreenLine is actual vibration value");
}

void Demo::DrawEnv(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
{
    auto pTextWriter = &pGraphicsSystem->GetDebugFont();
    pTextWriter->Draw(&pGraphicsSystem->GetCommandBuffer());
    pTextWriter->SetScale(1.0f, 1.0f);

    const float xOffset = 30.0f;
    const float xInnerOffset = 300.0f;
    float yOffset = 60.0f;

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(10.0f, yOffset);
    pTextWriter->Print("KeyConfig:");
    yOffset += 20.0f;

    pTextWriter->SetTextColor(Color::Indigo);
    pTextWriter->SetCursor(xOffset, yOffset);
    pTextWriter->Print("(L/R) Change Pattern (1 - %d) : %d", NumOfSample, m_VibrationPattern + 1);
    yOffset += 20.0f;

    pTextWriter->SetTextColor(Color::Turquoise);
    pTextWriter->SetCursor(xOffset, yOffset);
    pTextWriter->Print("(Right Stick) Change Field.");
    yOffset += 20.0f;

    pTextWriter->SetCursor(xOffset, yOffset);
    pTextWriter->Print("(Left Stick or Directional Buttons) Change Value.");
    yOffset += 20.0f;

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetCursor(xOffset, yOffset);
    pTextWriter->Print("(A) Play Vibration.");
    pTextWriter->SetCursor(xInnerOffset, yOffset);
    pTextWriter->Print("(B) Gyro Position Reset.");
    yOffset += 20.0f;

    pTextWriter->SetCursor(xOffset, yOffset);
    pTextWriter->SetTextColor(m_IsGraphicsEnabled ? Color::Red : Color::White);
    pTextWriter->Print("(Y) Draw Info:");
    pTextWriter->SetCursor(xInnerOffset, yOffset);
    pTextWriter->Print(m_IsGraphicsEnabled ? "[Enable]  Disable" : " Enable  [Disable]");
    pTextWriter->SetTextColor(Color::White);
    yOffset += 20.0f;

    pTextWriter->SetCursor(xOffset, yOffset);
    pTextWriter->Print("(+) Vibration Parameter Reset.");
    yOffset += 20.0f;

    pTextWriter->SetTextColor(Color::White);
}

//!< 6軸センサーの状態を座標系上に描画します。
void WriteSixAxisSensorPointer(nn::gfx::util::DebugFontTextWriter* pTextWriter,
    const ::nn::util::Vector3f& pointer,
    const nn::util::Unorm8x4& Color) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);
    const nn::util::Unorm8x4& textColor = Color;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(2.f, 2.f);
    pTextWriter->SetCursor(pointer.GetX(), pointer.GetY() - 20.0f);
    pTextWriter->Print("+");
}

void Demo::DrawObject(GraphicsSystem* pGraphicsSystem) const NN_NOEXCEPT
{
    nn::util::Matrix4x3fType viewMatrix;
    nn::util::Matrix4x4fType projectionMatrix;
    nn::util::Matrix4x3f modelMatrix;

    nn::util::MatrixIdentity(&viewMatrix);
    nn::util::MatrixIdentity(&projectionMatrix);
    nn::util::MatrixIdentity(&modelMatrix);

    auto pTextWriter = &pGraphicsSystem->GetDebugFont();
    auto pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();
    auto pCommandBuffer = &pGraphicsSystem->GetCommandBuffer();

    pPrimitiveRenderer->SetViewMatrix(&viewMatrix);
    pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);
    pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

    for (int i = 0; i < BoxCountMax; i++)
    {
        m_Box[i].Draw(pPrimitiveRenderer, pCommandBuffer);
    }

    //ポインタの描画
    const auto& pController = ControllerManager::GetInstance().GetActiveController();

    WriteSixAxisSensorPointer(pTextWriter,
        pController->GetCursor(),
        Color::Red);

}

