﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "NpadController.h"

void NpadControllerBase::Initialize() NN_NOEXCEPT
{
    // 振動子のハンドルを取得します
    nn::hid::VibrationDeviceHandle handles[VibrationDeviceCountMax];

    m_VibrationDeviceCount = nn::hid::GetVibrationDeviceHandles(
        handles, VibrationDeviceCountMax, m_NpadId, GetStyleSet());

    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        VibrationState& v = m_VibrationStateArray[i];
        v.deviceHandle = handles[i];
        v.vibrationPatternId = 0;
        v.currentVibrationValue = nn::hid::VibrationValue::Make();

        // 振動子を初期化します
        nn::hid::InitializeVibrationDevice(v.deviceHandle);

        // 振動子の情報を取得します
        nn::hid::GetVibrationDeviceInfo(&v.deviceInfo, v.deviceHandle);
    }
}

void NpadControllerBase::Update() NN_NOEXCEPT
{
    UpdateButtons();
    UpdateVibrationPattern();
    UpdateVibrationValue();
    m_LoopCount++;
}

bool NpadControllerBase::IsQuitRequired() const NN_NOEXCEPT
{
    // プラスボタンとマイナスボタンが同時に押されていたら終了判定をします
    const nn::hid::NpadButtonSet QuitButtonMask =
        nn::hid::NpadButton::Plus::Mask | nn::hid::NpadButton::Minus::Mask;

    return ((m_Buttons & QuitButtonMask) == QuitButtonMask);
}


void NpadControllerBase::UpdateVibrationPattern() NN_NOEXCEPT
{
    const nn::hid::NpadButtonSet LeftHandButtonMask =
        nn::hid::NpadButton::Up::Mask | nn::hid::NpadButton::Right::Mask | nn::hid::NpadButton::Down::Mask |
        nn::hid::NpadButton::Left::Mask | nn::hid::NpadButton::L::Mask | nn::hid::NpadButton::ZL::Mask;
    const nn::hid::NpadButtonSet RightHandButtonMask =
        nn::hid::NpadButton::A::Mask | nn::hid::NpadButton::B::Mask | nn::hid::NpadButton::X::Mask |
        nn::hid::NpadButton::Y::Mask | nn::hid::NpadButton::R::Mask | nn::hid::NpadButton::ZR::Mask;

    // 押されているボタンの個数に応じて振動パターンを更新します
    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        VibrationState& v = m_VibrationStateArray[i];
        switch (v.deviceInfo.position)
        {
        case nn::hid::VibrationDevicePosition_Left:
            v.vibrationPatternId = (m_Buttons & LeftHandButtonMask).CountPopulation();
            break;
        case nn::hid::VibrationDevicePosition_Right:
            v.vibrationPatternId = (m_Buttons & RightHandButtonMask).CountPopulation();
            break;
        default:
            v.vibrationPatternId = 0;
            break;
        }
    }
}

void NpadControllerBase::StopVibration() NN_NOEXCEPT
{
    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        VibrationState& v = m_VibrationStateArray[i];
        nn::hid::VibrationValue vib = nn::hid::VibrationValue::Make();
        nn::hid::SendVibrationValue(v.deviceHandle, vib);
    }
}

void NpadControllerBase::UpdateVibrationValue() NN_NOEXCEPT
{
    for (int i = 0; i < m_VibrationDeviceCount; i++)
    {
        VibrationState& v = m_VibrationStateArray[i];

        // 振動パターンに応じた振動値を生成します
        nn::hid::VibrationValue vib = nn::hid::VibrationValue::Make();
        switch (v.vibrationPatternId)
        {
        case 0:
            // 振動を発生させない
            break;
        case 1:
            // 周波数 160 Hz で振幅が 1.00 と 0.00 に交互に切り替わる振動を発生します
            vib.amplitudeLow = (m_LoopCount % 20 < 1) ? 1.0f : 0.0f;
            vib.frequencyLow = 160.0f;
            vib.amplitudeHigh = 0.0f;
            vib.frequencyHigh = 320.0f;
            break;
        case 2:
            // 周波数 160 Hz で振幅が動的に変化する振動を発生します
            vib.amplitudeLow = 0.5f * (1.0f + static_cast<float>(sin(2 * nn::util::FloatPi * (m_LoopCount % 60 / 60.0))));
            vib.frequencyLow = 160.0f;
            vib.amplitudeHigh = 0.0f;
            vib.frequencyHigh = 320.0f;
            break;
        case 3:
            // 振幅 0.50 で周波数が 160 Hz と 320 Hz に交互に切り替わる振動を発生します
            vib.amplitudeLow = (m_LoopCount % 60 < 30) ? 0.5f : 0.0f;
            vib.frequencyLow = 160.0f;
            vib.amplitudeHigh = (m_LoopCount % 60 < 30) ? 0.0f : 0.5f;
            vib.frequencyHigh = 320.0f;
            break;
        default:
            // 振幅が 0.50 で周波数が動的に変化する振動を発生します
            vib.amplitudeLow = 0.5f;
            vib.frequencyLow = 160.0f * (1.5f + 0.5f * static_cast<float>(sin(2 * nn::util::FloatPi * (m_LoopCount % 120 / 120.0))));
            vib.amplitudeHigh = 0.0f;
            vib.frequencyHigh = 320.0f;
            break;
        }

        if (m_IsConnected)
        {
            // コントローラが接続していれば振動値を送信します
            nn::hid::SendVibrationValue(v.deviceHandle, vib);

            // 振動の指令値をセット
            v.currentVibrationValue = vib;

            // 現在の振動を取得します
            nn::hid::GetActualVibrationValue(&v.actualVibrationValue, v.deviceHandle);
            m_VibrationValueBuffer[i].AddVibrationValues(
                v.currentVibrationValue, v.actualVibrationValue
            );
        }
    }
}

void NpadFullKeyController::UpdateButtons() NN_NOEXCEPT
{
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadId);

    if (style.Test<nn::hid::NpadStyleFullKey>())
    {
        m_IsConnected = true;
        nn::hid::NpadFullKeyState state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_Buttons = state.buttons;
    }
    else
    {
        if (m_IsConnected)
        {
            StopVibration();
            m_Buttons.Reset();
            m_IsConnected = false;
        }
    }
}

void NpadHandheldController::UpdateButtons() NN_NOEXCEPT
{
    nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(m_NpadId);

    if (style.Test<nn::hid::NpadStyleHandheld>())
    {
        m_IsConnected = true;
        nn::hid::NpadHandheldState state;
        nn::hid::GetNpadState(&state, m_NpadId);
        m_Buttons = state.buttons;
    }
    else
    {
        if (m_IsConnected)
        {
            StopVibration();
            m_Buttons.Reset();
            m_IsConnected = false;
        }
    }
}


