﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>

#include <nn/hid/hid_SixAxisSensor.h>
#include <nn/hid/hid_NpadSixAxisSensor.h>
#include <nn/hid/system/hid_SixAxisSensorCalibration.h>
#include <nn/hid/system/hid_UniquePad.h>

#include "HidSixAxisSensorCalibration_StabilizingState.h"
#include "HidSixAxisSensorCalibration_State.h"

namespace
{
    void PrintDevices() NN_NOEXCEPT
    {
        ::nn::hid::system::UniquePadId uniquePadIds[::nn::hid::system::UniquePadIdCountMax];

        int uniquePadIdCount = ::nn::hid::system::ListUniquePads(uniquePadIds,
                                                                 sizeof(uniquePadIds) / sizeof(uniquePadIds[0]));

        for (int i = 0; i < uniquePadIdCount; i++)
        {
            int uniquePadControllerNumber;
            ::nn::hid::system::UniquePadInterface uniquePadInterface;
            ::nn::hid::system::UniquePadType uniquePadType;
            ::nn::hid::system::UniquePadSerialNumber uniquePadSerialNumber;

            auto resultControllerNumber = ::nn::hid::system::GetUniquePadControllerNumber(&uniquePadControllerNumber,
                                                                                          uniquePadIds[i]);
            auto resultInterface = ::nn::hid::system::GetUniquePadInterface(&uniquePadInterface,
                                                                            uniquePadIds[i]);
            uniquePadType = ::nn::hid::system::GetUniquePadType(uniquePadIds[i]);

            auto resultSerialNumber = ::nn::hid::system::GetUniquePadSerialNumber(&uniquePadSerialNumber,
                                                                                  uniquePadIds[i]);

            if (resultControllerNumber.IsSuccess() &&
                resultInterface.IsSuccess() &&
                resultSerialNumber.IsSuccess())
            {
                NN_LOG("Id=0x%x, ControllerNumber=%d, Interface=%x, Type=%x\n",
                    uniquePadIds[i]._storage,
                    resultControllerNumber.IsSuccess() ? uniquePadControllerNumber : -1,
                    resultInterface.IsSuccess() ? uniquePadInterface : -1,
                    uniquePadType);
                NN_LOG("SerialNumber: ");
                for (auto& c : uniquePadSerialNumber._storage)
                {
                    NN_LOG("%c", c);
                }
                NN_LOG("\n");
            }
        }
    }
} // namespace

// StabilizingState クラス
StabilizingState::StabilizingState() NN_NOEXCEPT
{
    // 何もしない
}

StabilizingState::~StabilizingState() NN_NOEXCEPT
{
    // 何もしない
}

void StabilizingState::Update(Context* pContext) NN_NOEXCEPT
{
    const auto Features = pContext->GetFocusedPadFeatures();
    if (::nn::hid::IsSixAxisSensorAtRest(Features.handle))
    {
        pContext->SetState(&State::s_CalibratingState);
        return;
    }
    NN_LOG("[%s] StabilizingState\n", __FUNCTION__);
}

void StabilizingState::Enter(Context* pContext) NN_NOEXCEPT
{
    NN_LOG("[%s] StabilizingState\n", __FUNCTION__);

    // 確認用出力
    PrintDevices();

    const auto features = pContext->GetFocusedPadFeatures();
    ::nn::hid::StartSixAxisSensor(features.handle);
}

void StabilizingState::Exit(Context* pContext) NN_NOEXCEPT
{
    NN_LOG("[%s] StabilizingState\n", __FUNCTION__);

    const auto features = pContext->GetFocusedPadFeatures();
    ::nn::hid::StopSixAxisSensor(features.handle);
}
