﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/hid/system/hid_Result.h>
#include <nn/hid/system/hid_SixAxisSensorCalibration.h>

#include "HidSixAxisSensorCalibration_CalibratingState.h"
#include "HidSixAxisSensorCalibration_State.h"

CalibratingState::CalibratingState() NN_NOEXCEPT
{
    // 何もしない
}

CalibratingState::~CalibratingState() NN_NOEXCEPT
{
    // 何もしない
}

void CalibratingState::Update(Context* pContext) NN_NOEXCEPT
{
    const auto Features = pContext->GetFocusedPadFeatures();
    ::nn::hid::system::SixAxisSensorUserCalibrationStage stage;
    auto result = ::nn::hid::system::GetSixAxisSensorUserCalibrationStage(&stage,
                                                                          Features.uniquePadHandle);

    if (result.IsFailure())
    {
        HandleError(pContext, result);
        return;
    }

    switch (stage)
    {
    case nn::hid::system::SixAxisSensorUserCalibrationStage_Measuring:
        NN_LOG("[%s] CalibratingState: Measuring...\n", __FUNCTION__);
        break;
    case nn::hid::system::SixAxisSensorUserCalibrationStage_Update:
        NN_LOG("[%s] CalibratingState: Update...\n", __FUNCTION__);
        break;
    case nn::hid::system::SixAxisSensorUserCalibrationStage_Completed:
        NN_LOG("[%s] CalibratingState: Completed! Set to InitialState...\n", __FUNCTION__);
        pContext->SetState(&State::s_InitialState);
        return;
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void CalibratingState::Enter(Context* pContext) NN_NOEXCEPT
{
    NN_LOG("[%s] CalibratingState\n", __FUNCTION__);

    // キャリブレーション処理を開始
    const auto features = pContext->GetFocusedPadFeatures();
    auto result = ::nn::hid::system::StartSixAxisSensorUserCalibration(features.uniquePadHandle);

    if (result.IsFailure())
    {
        NN_LOG("[%s] StartSixAxisSensorUserCalibration() Failed. Set to InitialState...\n", __FUNCTION__);
        pContext->SetState(&State::s_InitialState);
        return;
    }
}

void CalibratingState::Exit(Context* pContext) NN_NOEXCEPT
{
    NN_LOG("[%s] CalibratingState\n", __FUNCTION__);

    // キャリブレーション処理をキャンセル
    const auto Features = pContext->GetFocusedPadFeatures();
    ::nn::hid::system::CancelSixAxisSensorUserCalibration(Features.uniquePadHandle);
}

void CalibratingState::HandleError(Context* pContext,
                                   const nn::Result& result) NN_NOEXCEPT
{
    if (::nn::hid::system::ResultSixAxisSensorDisconnected::Includes(result))
    {
        NN_LOG("[%s] CalibratingState: ResultSixAxisSensorDisconnected\n", __FUNCTION__);
    }
    else if (::nn::hid::system::ResultSixAxisSensorNotSupported::Includes(result))
    {
        NN_LOG("[%s] CalibratingState: ResultSixAxisSensorNotSupported\n", __FUNCTION__);
    }
    else if (::nn::hid::system::ResultSixAxisSensorNotStable::Includes(result))
    {
        NN_LOG("[%s] CalibratingState: ResultSixAxisSensorNotStable\n", __FUNCTION__);
    }
    else if (::nn::hid::system::ResultSixAxisSensorNotHorizontal::Includes(result))
    {
        NN_LOG("[%s] CalibratingState: ResultSixAxisSensorNotHorizontal\n", __FUNCTION__);
    }
    else if (::nn::hid::system::ResultSixAxisSensorWriteFailure::Includes(result))
    {
        NN_LOG("[%s] CalibratingState: ResultSixAxisSensorWriteFailure\n", __FUNCTION__);
    }
    else if (::nn::hid::system::ResultSixAxisSensorDisconnected::Includes(result))
    {
        NN_LOG("[%s] CalibratingState: ResultSixAxisSensorDisconnected\n", __FUNCTION__);
    }

    const auto Features = pContext->GetFocusedPadFeatures();
    ::nn::hid::system::CancelSixAxisSensorUserCalibration(Features.uniquePadHandle);

    NN_LOG("[%s] CalibratingState: Calibration Failed. Set to InitialState...\n", __FUNCTION__);

    pContext->SetState(&State::s_InitialState);
    return;
}
