﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
    @examplesource{HidSevenSixAxisSensor_Main.cpp,PageSampleHidSevenSixAxisSensor}

    @brief
    Seven 向け本体 6 軸センサーを使って SixAxisSensor の状態を取得するためのサンプルプログラム
*/

/**
    @page PageSampleHidSevenSixAxisSensor Seven 向け本体 6 軸センサーを使った SixAxisSensor の状態の取得
    @tableofcontents

    @brief
    Seven 向け本体 6 軸センサーを使って SixAxisSensor の状態を取得するためのサンプルプログラムの解説です。

    @section PageSampleHidSevenSixAxisSensor_SectionBrief 概要
    コントローラの最新の SixAxisSensor の状態を取得する方法について説明します。

    @section PageSampleHidSevenSixAxisSensor_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidSevenSixAxisSensor @endlink 以下にあります。

    @section PageSampleHidSevenSixAxisSensor_SectionNecessaryEnvironment 必要な環境
    NX 開発機環境でのみ動作します。

    @section PageSampleHidSevenSixAxisSensor_SectionHowToOperate 操作方法
    サンプルプログラムを実行すると NX 開発機環境ではタッチスクリーン上に SixAxisSensor の状態が表示されます。

    以下の状態を表示します。
    - SixAxisSensor の加速度・角速度・サンプリング番号・パケットロス率の表示
    - acceleration の XY・YZ・ZX成分を 2次元座標系で表示
    - angularVelocity の XY・YZ・ZX成分を 2次元座標系で表示
    - 姿勢回転を 3次元ワイヤーフレームで表示

    サンプルプログラムを終了させるには、Handheld 操作形態で + ボタンと - ボタンを同時に押すか、
    Nintendo Target Manager 経由で終了させてください。

    @section PageSampleHidSevenSixAxisSensor_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidSevenSixAxisSensor_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - Seven 向け本体 6 軸センサーを初期化
    - Handheld 操作形態の現在のボタン入力状態を取得
    - 現在の Seven 向け 6 軸センサー入力状態を取得
    - 現在のタイムスタンプからパケット落ち率を推定
*/

#include <cstdlib>
#include <string>
#include <vector>
#include <nn/nn_Assert.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/nn_TimeSpan.h>
#include <nn/gfx/util/gfx_DebugFontTextWriter.h>
#include <nn/hid.h>
#include <nn/hid/hid_SevenSixAxisSensor.h>
#include <nn/hid/hid_SevenSixAxisSensorFusionApi.h>
#include <nn/hid/hid_Npad.h>
#include <nn/hid/hid_NpadHandheld.h>
#include <nn/TargetConfigs/build_Platform.h>
#include <nn/util/util_MathTypes.h>
#include <nn/util/util_Matrix.h>
#include <nn/util/util_MatrixApi.h>

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
#include <nv/nv_MemoryManagement.h>
#endif

#include "HidSevenSixAxisSensor_ApplicationHeap.h"
#include "HidSevenSixAxisSensor_Color.h"
#include "HidSevenSixAxisSensor_FontSystem.h"
#include "HidSevenSixAxisSensor_GraphicsSystem.h"
#include "HidSevenSixAxisSensor_WindowMessage.h"

namespace {

const size_t ApplicationHeapSize = 128 * 1024 * 1024;

const int FrameBufferWidth = 1280;

const int FrameBufferHeight = 720;

const int FrameRate = 60;

template<typename T, std::size_t U>
std::size_t GetArrayLength(const T (&staticArray)[U])
{
    NN_UNUSED(staticArray);
    return U;
}

const nn::hid::NpadIdType NpadIds[] = { nn::hid::NpadId::Handheld };

const int NpadIdCountMax = static_cast<int>(GetArrayLength(NpadIds));

const nn::util::Unorm8x4 ColorArray[]  = {Color::Red,
                                          Color::Orange,
                                          Color::Yellow,
                                          Color::Green};

/**
 * @brief       描画する座標系の識別子です。
 */
enum CoordinateId
{
    CoordinateId_AccelerometerXy = 0,
    CoordinateId_AccelerometerYz,
    CoordinateId_AccelerometerZx,
    CoordinateId_AngularVelocityXy,
    CoordinateId_AngularVelocityYz,
    CoordinateId_AngularVelocityZx,
};

//!< 座標系の原点座標を取得します。
void GetCoordinateOrigin(nn::util::Float2* pOutOrigin, CoordinateId id)
{
    const float X = 350;
    const float Y = 150;
    const float DeltaX = 250;
    const float DeltaY = 230;

    pOutOrigin->x = X + static_cast<int>(id) / 3 * DeltaX;
    pOutOrigin->y = Y + (static_cast<int>(id) % 3) * DeltaY;
}

//!< 座標系の名前を取得します。
void GetCoordinateName(char* pOutAxis1,
                       char* pOutAxis2,
                       ::std::string* pOutTitle,
                       CoordinateId id)
{
    switch (id)
    {
    case CoordinateId_AccelerometerXy :
        *pOutTitle = "AccelerometerXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_AccelerometerYz :
        *pOutTitle = "AccelerometerYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_AccelerometerZx :
        *pOutTitle = "AccelerometerZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    case CoordinateId_AngularVelocityXy:
        *pOutTitle = "AngularVelocityXy";
        *pOutAxis1 = 'x';
        *pOutAxis2 = 'y';
        break;
    case CoordinateId_AngularVelocityYz:
        *pOutTitle = "AngularVelocityYz";
        *pOutAxis1 = 'y';
        *pOutAxis2 = 'z';
        break;
    case CoordinateId_AngularVelocityZx:
        *pOutTitle = "AngularVelocityZx";
        *pOutAxis1 = 'z';
        *pOutAxis2 = 'x';
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
        break;
    }
}

//!< 6軸センサーの状態を描画します。
void WriteSixAxisSensorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                             const nn::hid::SevenSixAxisSensorState& state,
                             const float packetDropPercentage,
                             const float offsetX,
                             const float offsetY,
                             const nn::util::Unorm8x4& Color,
                             const int playerNumber,
                             const bool isAtRest) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

#if !defined(NN_BUILD_TARGET_PLATFORM_NX)
    NN_UNUSED(packetDropPercentage);
#endif

    const nn::util::Unorm8x4& textColor = Color;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);
    pTextWriter->SetCursor(offsetX, offsetY);
    pTextWriter->Print("Player: %d", playerNumber);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);

    float posY = offsetY + 15.0f;
    const float DeltaY = 15.0f;

    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("acceleration");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.acceleration.x,
                                              state.acceleration.y,
                                              state.acceleration.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("angularVelocity");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", state.angularVelocity.x,
                                              state.angularVelocity.y,
                                              state.angularVelocity.z);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("sampling number");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("%lld", state.samplingNumber);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("coordinate number");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("%lld", state.coordinateNumber);

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("elapsed time[us]");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("%lld", state.timeStamp.GetMicroSeconds());

    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("IsAtRest : %s", isAtRest ? "Y" : "N");

    posY += DeltaY;
    float strength = -1.f;
    auto result = ::nn::hid::GetSevenSixAxisSensorFusionStrength(&strength);
    pTextWriter->SetCursor(offsetX + 10, posY);
    if (result.IsFailure())
    {
        pTextWriter->Print("Strength : INVALID");
    }
    else
    {
        pTextWriter->Print("Strength : %f\n", strength);
    }

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 5, posY);
    pTextWriter->Print("packet loss");
    posY += DeltaY;
    pTextWriter->SetCursor(offsetX + 10, posY);
    pTextWriter->Print("( %.2f % )", packetDropPercentage * 100.0f);
#endif
}

//!< 座標系を描画します。
void WriteCoordinateAxes(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                         CoordinateId id) NN_NOEXCEPT
{
    const nn::util::Unorm8x4& textColor = Color::White;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.7f, 0.7f);

    const int NumberOfCharacters = 10;
    const int Interval = 10;
    nn::util::Float2 origin;
    char axis1;
    char axis2;
    ::std::string title;
    GetCoordinateName(&axis1, &axis2, &title, id);
    GetCoordinateOrigin(&origin, id);

    for(int i = 0; i < NumberOfCharacters; i++)
    {
        pTextWriter->SetCursor(origin.x + Interval * i, origin.y);
        pTextWriter->Print("-");
        pTextWriter->SetCursor(origin.x - Interval * i, origin.y);
        pTextWriter->Print("-");

        pTextWriter->SetCursor(origin.x, origin.y + Interval * i);
        pTextWriter->Print("|");
        pTextWriter->SetCursor(origin.x, origin.y - Interval * i);
        pTextWriter->Print("|");
    }
    pTextWriter->SetCursor(origin.x + NumberOfCharacters * Interval, origin.y);
    pTextWriter->Print(">%c", axis1);
    pTextWriter->SetCursor(origin.x, origin.y - NumberOfCharacters * Interval);
    pTextWriter->Print("^%c", axis2);
    pTextWriter->SetCursor(origin.x - NumberOfCharacters * Interval,
                           origin.y - NumberOfCharacters * Interval - 1.0f);
    pTextWriter->Print("%s", title.c_str());
}

//!< 6軸センサーの状態を座標系上に描画します。
void WriteSixAxisSensorStateFigure(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                   const nn::hid::SevenSixAxisSensorState& state,
                                   const nn::util::Unorm8x4& Color,
                                   const int playerNumber) NN_NOEXCEPT
{
    NN_ASSERT_NOT_NULL(pTextWriter);

    const nn::util::Unorm8x4& textColor = Color;

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(1, 1);

    pTextWriter->SetTextColor(textColor);
    pTextWriter->SetScale(0.8f, 0.8f);

    const float Coefficient  = 50.0f;

    nn::util::Float2 coordinateOrigin;

    // Accelerometer
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.x * Coefficient,
                           coordinateOrigin.y - state.acceleration.y * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.y * Coefficient,
                           coordinateOrigin.y - state.acceleration.z * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.acceleration.z * Coefficient,
                           coordinateOrigin.y - state.acceleration.x * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    // AngularVelocity
    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityXy);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.x * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.y * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityYz);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.y * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.z * Coefficient);
    pTextWriter->Print("%d", playerNumber);

    GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityZx);
    pTextWriter->SetCursor(coordinateOrigin.x + state.angularVelocity.z * Coefficient,
                           coordinateOrigin.y - state.angularVelocity.x * Coefficient);
    pTextWriter->Print("%d", playerNumber);
}

//!< Seven 向け 6 軸センサーの入力状態を出力します。
void PrintSevenSixAxisSensorState(const nn::hid::SevenSixAxisSensorState& state)
{
    NN_LOG("SamplingNumber:%lld, ElapsedTime[us]:%lld\n", state.samplingNumber,
                                                          state.timeStamp.GetMicroSeconds());
    NN_LOG("CoordinateNumber:%lld\n", state.coordinateNumber);
    NN_LOG("Acc: %f, %f, %f\n", state.acceleration.x,
        state.acceleration.y,
        state.acceleration.z);
    NN_LOG("Gyro: %f, %f, %f\n", state.angularVelocity.x,
        state.angularVelocity.y,
        state.angularVelocity.z);
    NN_LOG("Quaternion: %f, %f, %f, %f\n", state.rotation._v[0]
                                         , state.rotation._v[1]
                                         , state.rotation._v[2]
                                         , state.rotation._v[3]);
    NN_LOG("\n");
}

//!< ボタンのトリガー入力を取得します。
nn::hid::NpadButtonSet GetTriggerButtons(const nn::hid::NpadButtonSet current,
                                         const nn::hid::NpadButtonSet previous)
{
    return (current ^ previous) & current;
}

//!< アプリの終了判定をします。
bool Terminate(const nn::hid::NpadButtonSet& buttons)
{
    return (buttons.Test<nn::hid::NpadButton::Plus>() &&
            buttons.Test<nn::hid::NpadButton::Minus>());
}

/**
 * @brief       操作形態ごとの処理を記述したインタフェースです。
 */
class IConsoleSixAxisSensor
{
public:

    virtual ~IConsoleSixAxisSensor() NN_NOEXCEPT { /* 何もしない */ }

    virtual void Initialize(nn::mem::StandardAllocator* pApplicationHeap) NN_NOEXCEPT = 0; //!< ハンドルの取得とサンプリングの開始を行います。

    virtual bool CanPrint() NN_NOEXCEPT = 0;   //!< コンソールにセンサー値をダンプするか否かを返します。

    virtual void Print() NN_NOEXCEPT = 0;      //!< センサー値を出力します。

    virtual bool CanReset() NN_NOEXCEPT = 0;   //!< センサー値の姿勢位置をリセットするか否かを返します。

    virtual void Reset() NN_NOEXCEPT = 0;      //!< センサー値の姿勢位置をリセットします

    virtual void Update() NN_NOEXCEPT = 0;     //!< 入力状態を更新します。

    virtual bool Quit() NN_NOEXCEPT = 0;       //!< アプリを終了する場合に true を返します。

    virtual void Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT = 0;

    virtual void Draw(GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT = 0;

    virtual bool IsConnected() const NN_NOEXCEPT = 0;

    virtual void Finalize() NN_NOEXCEPT = 0; //!< 終了処理を行います。

protected:
    nn::mem::StandardAllocator* m_pApplicationHeap;

};

//!< ConsoleSixAxisSensor 向けの処理を記述したクラスです。
class ConsoleSixAxisSensor : public IConsoleSixAxisSensor
{
    NN_DISALLOW_COPY(ConsoleSixAxisSensor);
    NN_DISALLOW_MOVE(ConsoleSixAxisSensor);

private:
    void*                            m_pWorkBuffer;
    nn::hid::NpadHandheldState       m_ButtonState[2];
    nn::hid::SevenSixAxisSensorState m_SevenState;
    nn::util::Quaternion             m_Quaternion;
    const nn::hid::NpadIdType*       m_pId;

    uint32_t     m_FramerateCounter;
    nn::os::Tick m_FramerateFirstTick;
    int64_t      m_FramerateFirstSample;
    float        m_FramerateComputation;
    float        m_PacketDropPercentage;
public:
    static const int ResetIntervalsInFrame  = 60 * 3;
    static const int UpdateIntervalsInFrame = 20;

    explicit ConsoleSixAxisSensor(const nn::hid::NpadIdType& id) NN_NOEXCEPT
        : m_pId(&id)
    {
        m_Quaternion = nn::util::Quaternion::Identity();
    }

    virtual ~ConsoleSixAxisSensor() NN_NOEXCEPT NN_OVERRIDE { /* 何もしない */ };

    virtual void Initialize(nn::mem::StandardAllocator* pApplicationHeap) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pApplicationHeap = pApplicationHeap;

        m_pWorkBuffer = m_pApplicationHeap->Allocate(::nn::hid::SevenSixAxisSensorWorkBufferSize, 4096);
        nn::hid::InitializeSevenSixAxisSensor(m_pWorkBuffer,
                                              ::nn::hid::SevenSixAxisSensorWorkBufferSize);
    }

    virtual bool CanPrint() NN_NOEXCEPT NN_OVERRIDE
    {
        return (GetTriggerButtons(m_ButtonState[0].buttons, m_ButtonState[1].buttons).IsAnyOn());
    }

    virtual void Print() NN_NOEXCEPT NN_OVERRIDE
    {
        PrintSevenSixAxisSensorState(m_SevenState);
    }

    virtual bool CanReset() NN_NOEXCEPT NN_OVERRIDE
    {
        return (GetTriggerButtons(m_ButtonState[0].buttons, m_ButtonState[1].buttons) & nn::hid::NpadButton::Plus::Mask).IsAnyOn();
    }

    virtual void Reset() NN_NOEXCEPT NN_OVERRIDE
    {
    }

    virtual void Update() NN_NOEXCEPT NN_OVERRIDE
    {
        m_ButtonState[1] = m_ButtonState[0];

        nn::hid::GetNpadState(&m_ButtonState[0], *m_pId);
        auto count = nn::hid::GetSevenSixAxisSensorStates(&m_SevenState, 1);
        if(count <= 0)
        {
            m_SevenState = ::nn::hid::SevenSixAxisSensorState();
        }
        // ResetIntervalsInFrame フレームごとに Packet Loss 向けデータを初期化します
        if ((m_FramerateCounter % ResetIntervalsInFrame ) == 0)
        {
            m_FramerateFirstTick = nn::os::GetSystemTick();
            m_FramerateFirstSample = m_SevenState.samplingNumber;
        }
        nn::os::Tick currentTick = nn::os::GetSystemTick() - m_FramerateFirstTick;
        int64_t currentSample = m_SevenState.samplingNumber - m_FramerateFirstSample;

        // UpdateIntervalsInFrame 周期ごとに、Packet Loss を更新します
        if ( m_FramerateCounter % UpdateIntervalsInFrame == (UpdateIntervalsInFrame - 1) )
        {
            m_FramerateComputation = currentSample / (float(currentTick.GetInt64Value()) / nn::os::GetSystemTickFrequency());

            const float ExpectedFrameRate = 1000.0f;
            m_PacketDropPercentage = 1.0f - std::min( std::max( m_FramerateComputation / ExpectedFrameRate, 0.0f ), 1.0f);

            nn::hid::NpadStyleSet style = nn::hid::GetNpadStyleSet(*m_pId);
            if (style.Test<nn::hid::NpadStyleHandheld>() == false)
            {
                m_ButtonState[0].buttons.Reset();
            }
        }
        m_FramerateCounter++;

        auto triggerButtons = GetTriggerButtons(m_ButtonState[0].buttons, m_ButtonState[1].buttons);
        if (triggerButtons.Test<nn::hid::NpadButton::A>())
        {
            ::nn::hid::StartSevenSixAxisSensor();
        }
        else if (triggerButtons.Test<nn::hid::NpadButton::B>())
        {
            ::nn::hid::StopSevenSixAxisSensor();
        }
        else if (triggerButtons.Test<nn::hid::NpadButton::X>())
        {
            ::nn::hid::SetSevenSixAxisSensorFusionStrength(10.f);
        }
        else if (triggerButtons.Test<nn::hid::NpadButton::Y>())
        {
            ::nn::hid::SetSevenSixAxisSensorFusionStrength(0.f);
        }
    }

    virtual bool Quit() NN_NOEXCEPT NN_OVERRIDE
    {
        return Terminate(m_ButtonState[0].buttons);
    }

    virtual void Draw(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT NN_OVERRIDE
    {
        int i = static_cast<uint32_t>(0);

        WriteSixAxisSensorState(pTextWriter,
                                m_SevenState,
                                m_PacketDropPercentage,
                                25, 15 + 175 * static_cast<float>(i),
                                ColorArray[i],
                                i + 1,
                                ::nn::hid::IsSevenSixAxisSensorAtRest());

        WriteSixAxisSensorStateFigure(pTextWriter,
                                      m_SevenState,
                                      ColorArray[i],
                                      i + 1);
    }

    virtual void Draw(GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT NN_OVERRIDE
    {
        int i = static_cast<uint32_t>(0);

        nn::gfx::ViewportScissorState* pViewportScissorState = &pGraphicsSystem->GetViewportScissor(i + 1);
        pGraphicsSystem->GetCommandBuffer().SetViewportScissorState(pViewportScissorState);

        nn::util::Vector3fType center = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType size = { 1.0f, 2.0f, 0.5f };

        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer = &pGraphicsSystem->GetPrimitiveRenderer();

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;

        nn::util::Vector3fType cameraPosition = { 0.f, -2.5f, 0.0f };
        nn::util::Vector3fType cameraTarget = { 0.f, 0.f, 0.f };
        nn::util::Vector3fType cammeraUp = { 0.f, 0.f, 1.f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, cameraPosition, cameraTarget, cammeraUp);
        pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

        pPrimitiveRenderer->SetColor(ColorArray[i]);
        pPrimitiveRenderer->SetLineWidth(3.f);

        // プロジェクションを初期化
        const float Fovy = nn::util::FloatPi / 3.0f;
        const float Aspect = static_cast< float >(FrameBufferWidth) / static_cast< float >(FrameBufferHeight);
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix, Fovy, Aspect, 0.1f, 100.f);
        pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

        nn::util::Vector3f vecZero;
        nn::util::VectorZero(&vecZero);

        nn::util::Quaternion currentQuaternion(m_SevenState.rotation), releativeQuaternion;
        releativeQuaternion = currentQuaternion / m_Quaternion;
        modelMatrix = nn::util::MatrixRowMajor4x3f::MakeRotation(releativeQuaternion);
        nn::util::MatrixSetAxisW(&modelMatrix, vecZero);
        pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

        pGraphicsSystem->GetPrimitiveRenderer().DrawCube(
            &pGraphicsSystem->GetCommandBuffer(),
            nns::gfx::PrimitiveRenderer::Surface_Wired,
            center, size);

        pViewportScissorState = &pGraphicsSystem->GetViewportScissor(0);
        pGraphicsSystem->GetCommandBuffer().SetViewportScissorState(pViewportScissorState);
    }

    bool IsConnected() const NN_NOEXCEPT NN_OVERRIDE
    {
        return true;
    }

    void Finalize() NN_NOEXCEPT NN_OVERRIDE
    {
        ::nn::hid::FinalizeSevenSixAxisSensor();
        m_pApplicationHeap->Free(m_pWorkBuffer);
    }

};

#if defined(NN_BUILD_TARGET_PLATFORM_NX)
const size_t GraphicsMemorySize = 8 * 1024 * 1024;

void* NvAllocate(size_t size, size_t alignment, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return aligned_alloc(alignment, nn::util::align_up(size, alignment));
}

void NvFree(void* addr, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    free(addr);
}

void* NvReallocate(void* addr, size_t newSize, void* userPtr) NN_NOEXCEPT
{
    NN_UNUSED(userPtr);
    return realloc(addr, newSize);
}
#endif

} // namespace

extern "C" void nnMain()
{
#if defined(NN_BUILD_TARGET_PLATFORM_NX)
    nv::SetGraphicsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(NvAllocate, NvFree, NvReallocate, NULL);
    nv::InitializeGraphics(std::malloc(GraphicsMemorySize), GraphicsMemorySize);
#endif

    nn::mem::StandardAllocator appAllocator;
    nn::Bit8* pAppMemory = new nn::Bit8[ApplicationHeapSize];
    appAllocator.Initialize(pAppMemory, ApplicationHeapSize);

    std::vector<IConsoleSixAxisSensor*> npadStyleSixAxisSensors; //!< 管理対象の操作形態を格納するコンテナーです。
    NN_LOG("ConsoleSixAxisSensor Sample Start.\n");

    nn::hid::InitializeNpad();

    // 使用する Npad を設定
    nn::hid::SetSupportedNpadIdType(NpadIds, NpadIdCountMax);

    for(int i = 0; i < NpadIdCountMax; i++)
    {
        void* area = appAllocator.Allocate(sizeof(ConsoleSixAxisSensor), 16);
        npadStyleSixAxisSensors.push_back(reinterpret_cast<IConsoleSixAxisSensor*>(new (area) ConsoleSixAxisSensor(NpadIds[i])));
    }

    //使用する操作形態を設定
    nn::hid::SetSupportedNpadStyleSet(nn::hid::NpadStyleHandheld::Mask);

    NN_LOG("If you push any button, button state log will appear on the console.\n");
    NN_LOG("Push (+) and (-) Button to shutdown this application.\n");

    // 管理対象の操作形態に関する初期化
    for(std::vector<IConsoleSixAxisSensor*>::iterator it = npadStyleSixAxisSensors.begin();
        it != npadStyleSixAxisSensors.end();
        ++it)
    {
        (*it)->Initialize(&appAllocator);
    }

    GraphicsSystem* pGraphicsSystem = new GraphicsSystem();
    pGraphicsSystem->Initialize(
        &appAllocator, FrameBufferWidth, FrameBufferHeight);

    EnableWindowMessage(pGraphicsSystem->GetNativeWindowHandle());

    FontSystem* pFontSystem = new FontSystem();
    pFontSystem->Initialize(&appAllocator, pGraphicsSystem);

    nn::gfx::util::DebugFontTextWriter& textWriter =
        pFontSystem->GetDebugFontTextWriter();

    while (NN_STATIC_CONDITION(true))
    {
        bool quits = false;

        switch (GetWindowMessage(pGraphicsSystem->GetNativeWindowHandle()))
        {
        case WindowMessage_Close:
            quits = true;
            break;
        default:
            break;
        }

        WriteCoordinateAxes(&textWriter, CoordinateId_AccelerometerXy);
        WriteCoordinateAxes(&textWriter, CoordinateId_AccelerometerYz);
        WriteCoordinateAxes(&textWriter, CoordinateId_AccelerometerZx);
        WriteCoordinateAxes(&textWriter, CoordinateId_AngularVelocityXy);
        WriteCoordinateAxes(&textWriter, CoordinateId_AngularVelocityYz);
        WriteCoordinateAxes(&textWriter, CoordinateId_AngularVelocityZx);

        for(std::vector<IConsoleSixAxisSensor*>::iterator it = npadStyleSixAxisSensors.begin();
            it != npadStyleSixAxisSensors.end();
            ++it)
        {
            if (!(*it)->IsConnected())
            {
                continue;
            }

            (*it)->Update();
            (*it)->Draw(&textWriter);

            if((*it)->CanPrint())
            {
                (*it)->Print();
            }

            if ((*it)->CanReset())
            {
                (*it)->Reset();
            }

            if((*it)->Quit())
            {
                quits = true;
                (*it)->Finalize();
            }
        }

        pGraphicsSystem->BeginDraw();
        for (std::vector<IConsoleSixAxisSensor*>::iterator it = npadStyleSixAxisSensors.begin();
            it != npadStyleSixAxisSensors.end();
            ++it)
        {
            if (!(*it)->IsConnected())
            {
                continue;
            }

            (*it)->Draw(pGraphicsSystem);
        }
        pFontSystem->Draw();
        pGraphicsSystem->EndDraw();

        pGraphicsSystem->Synchronize(
            nn::TimeSpan::FromNanoSeconds(1000 * 1000 * 1000 / FrameRate));

        if (quits)
        {
            break;
        }
    }

    pFontSystem->Finalize();
    delete pFontSystem;

    pGraphicsSystem->Finalize();
    delete pGraphicsSystem;
}
