﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <new>
#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/os.h>
#include "RomFileSystem.h"

namespace {
    char* g_RomCacheBuffer = nullptr;
}

void InitializeRomFileSystem() NN_NOEXCEPT
{
    nn::Result result;
    size_t romCacheSize = 0;
    nn::fs::QueryMountRomCacheSize(&romCacheSize);

    // キャッシュバッファを確保する
    g_RomCacheBuffer = new(std::nothrow) char[romCacheSize];
    NN_ABORT_UNLESS_NOT_NULL(g_RomCacheBuffer);

    // ファイルシステムをマウントする
    result = nn::fs::MountRom("rom", g_RomCacheBuffer, romCacheSize);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
}

void FinalizeRomFileSystem() NN_NOEXCEPT
{
    // ファイルシステムをアンマウントする
    nn::fs::Unmount("rom");

    delete[] g_RomCacheBuffer;
    g_RomCacheBuffer = nullptr;
}

void LoadFile(size_t* outFileSize, char* buffer, size_t size, const char* filePath) NN_NOEXCEPT
{
    NN_ABORT_UNLESS_NOT_NULL(outFileSize);
    NN_ABORT_UNLESS_NOT_NULL(buffer);
    NN_ABORT_UNLESS_NOT_NULL(filePath);

    nn::Result result;
    nn::fs::FileHandle file;
    int64_t fileSize;

    result = nn::fs::OpenFile(&file, filePath, nn::fs::OpenMode_Read);
    if (nn::fs::ResultPathNotFound::Includes(result))
    {
        // ファイルが存在しません
        NN_ASSERT(false, "[RomFileSystem] File not found.\n");
    }

    nn::fs::GetFileSize(&fileSize, file);

    NN_ABORT_UNLESS_LESS_EQUAL(static_cast<size_t>(fileSize), size);

    nn::fs::ReadFile(file, 0, buffer, static_cast<size_t>(fileSize));

    nn::fs::CloseFile(file);

    *outFileSize = static_cast<size_t>(fileSize);
}
