﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


/**
    @examplesource{HidPalmaSimple.cpp,PageSampleHidPalmaSimple}

    @brief
    Palma の基本機能を示すためのサンプルプログラム
*/

/**
    @page PageSampleHidPalmaSimple Palma 利用時の Npad を使ったボタン入力の取得
    @tableofcontents

    @brief
    Npad を使って Palma を含めたボタン入力の取得や、Palma を制御する API の使用方法を示すためのサンプルプログラムの解説です。

    @section PageSampleHidPalmaSimple_SectionBrief 概要
    コントローラの最新のボタン入力を取得する方法について説明します。

    @section PageSampleHidPalmaSimple_SectionFileStructure ファイル構成
    本サンプルプログラムは @link ../../../Samples/Sources/Applications/HidPalmaSimple @endlink 以下にあります。

    @section PageSampleHidPalmaSimple_SectionNecessaryEnvironment 必要な環境
    Windows 環境では、Palma の接続はできません。

    @section PageSampleHidPalmaSimple_SectionHowToOperate 操作方法
    サンプルプログラムを実行して、コントローラのボタンを押してください。
    Bluetooth の接続が確立すると、コントローラの LED が点滅から点灯に変わります。
    Palma の接続が確立すると、Palma が振動します。

    Joy-Con が1本でも無線接続されている場合は、NpadJoyDual の入力としてコンソールにログが出力されます。
    Joy-Con が2本無線接続されている場合(本体にジョイントされていない場合)は、NpadJoyDual 及び NpadFullKey の入力としてログが出力されます。
    Joy-Con がジョイントされている場合は、NpadHandheld の入力としてコンソールにログが出力されます。
    Palma が接続されている場合は、NpadPalma の入力としてコンソールにログが出力されます。

    サンプルプログラムを終了させるには + ボタンと - ボタンを同時に押してください。
    Palma だけ、Joy-Con (L) だけ、もしくは Joy-Con (R) だけが接続された場合はVisualStudioかNintendo Target Manager経由で終了させてください。

    サンプルプログラムを実行すると、Palma のスキャンが開始され、発見次第接続します。
    ペアリング済みの Palma との自動接続が有効になっており、スキャンが OFF 状態でも接続されます。
    ジョイントした Joy-Con で以下の操作が行えます。
    - A ボタンを押すと、Palma のスキャンの ON/OFF 状態を切り替えます。
    - X ボタンを押すと、すべての Palma との接続を切断します。
    - + ボタンを押すと、Palma をペアリングします。
    - Y ボタンを押すと、Palma から情報を取得し、結果をログに出力します。
    - 左ボタン、右ボタンを押すと、Palma に Activity を書き込みます。
    - Palma の Top ボタンを押すと、書き込まれた Activity を再生します。

    @section PageSampleHidPalmaSimple_SectionPrecaution 注意事項
    本サンプルプログラムでは画面描画は行いません。

    @section PageSampleHidPalmaSimple_SectionHowToExecute 実行手順
    サンプルプログラムをビルドし、実行してください。

    @section PageSampleHidPalmaSimple_SectionDetail 解説
    サンプルプログラムの全体像は以下の通りです。
    - BLE を初期化
    - Npad を初期化
    - 現在のボタン入力状態を取得
    - Palma の制御
*/

#include <nn/nn_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Macro.h>
#include <nn/os.h>
#include "HidNpadController.h"
#include "PalmaDeviceController.h"
#include "PalmaConnectionManager.h"
#include "RomFileSystem.h"

namespace
{
    void Init()
    {
        InitializePalmaConnectionManager();

        InitializeNpadController();

        InitializePalmaDeviceController();

        InitializeRomFileSystem();
    }

    void finalize()
    {
        FinalizePalmaDeviceController();

        FinalizePalmaConnectionManager();

        FinalizeRomFileSystem();
    }

}//namespace

extern "C" void nnMain()
{
    NN_LOG("Palma Sample Start.\n");

    Init();

    NN_LOG("If you push any button, button state log will appear on the console.\n");
    NN_LOG("Push (+) and (-) Button to shutdown this application.\n");
    NN_LOG("(A) : Start/Stop scan\n");
    NN_LOG("(X) : Detach all Palma\n");
    NN_LOG("(+) : Pair Palma\n");
    NN_LOG("(Y) : Get Palma properties\n");
    NN_LOG("(<)/(>) : Write Palma activity\n");
    NN_LOG("(Palma Top) Button : Play Palma activity\n");

    while(NN_STATIC_CONDITION(true))
    {
        UpdatePalmaDevices();
        if (UpdateNpadController() < 0) break;
        ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(15));
    }

    // Palma の Scan を停止する
    StopPalmaScanner();

    finalize();

    NN_LOG("Palma Sample Done.\n");
}
