﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HidNpadIntegrate_VibrationValueDrawer.h"

namespace
{

    const float PlotScale = 1.0f;
    const float DrawUnit = 8.0f;
    const int AxisWidth = static_cast<int>(50 * PlotScale);
    const int AxisHeight = static_cast<int>(20 * PlotScale);
    const float FrequencyLow = static_cast<float>(nn::hid::VibrationFrequencyLowDefault);
    const float FrequencyHigh = static_cast<float>(nn::hid::VibrationFrequencyHighDefault);

    float CalcXOffsetFromFrequency(float frequency) NN_NOEXCEPT
    {
        const float Thresholds[] = {
            100.43f, 102.63f, 104.88f, 107.17f, 109.52f, 111.92f, 114.37f, 116.87f,
            119.43f, 122.05f, 124.72f, 127.45f, 130.24f, 133.09f, 136.01f, 138.99f,
            142.03f, 145.14f, 148.32f, 151.57f, 154.88f, 158.28f, 161.74f, 165.28f,
            168.90f, 172.60f, 176.38f, 180.24f, 184.19f, 188.22f, 192.35f, 196.56f,
            200.86f, 205.26f, 209.75f, 214.35f, 219.04f, 223.84f, 228.74f, 233.75f,
            238.87f, 244.10f, 249.44f, 254.90f, 260.48f, 266.19f, 272.02f, 277.97f,
            284.06f, 290.28f, 296.64f, 303.13f, 309.77f, 316.55f, 323.48f, 330.57f,
            337.81f, 345.20f, 352.76f, 360.49f, 368.38f, 376.45f, 384.69f, 393.11f,
            401.72f, 410.52f, 419.51f, 428.69f, 438.08f, 447.67f, 457.48f, 467.49f,
            477.73f, 488.19f, 498.88f, 509.81f };

        const int ThresholdsCount = NN_ARRAY_SIZE(Thresholds);

        for (int id = 0; id < ThresholdsCount; id++)
        {
            if (frequency < Thresholds[id])
            {
                return static_cast<float>(id * AxisWidth) / ThresholdsCount;
            }
        }

        return static_cast<float>(AxisWidth);
    }
} // namespace

void VibrationValueDrawer::SetPosition(float x, float y) NN_NOEXCEPT
{
    m_X = x;
    m_Y = y;
}

void VibrationValueDrawer::SetFontScale(float scale) NN_NOEXCEPT
{
    m_FontScale = scale;
}

void VibrationValueDrawer::DrawVibrationValue(nn::hid::VibrationValue vibrationValue,
                                              bool isTextPrinted) NN_NOEXCEPT
{
    const nn::util::Unorm8x4 AxisColor = Color::Gray;
    const nn::util::Unorm8x4 VibrationValueColor = Color::Lime;

    m_pWriter->SetTextColor(AxisColor);
    DrawAxis();

    m_pWriter->SetTextColor(VibrationValueColor);
    DrawAmplitudeAndFrequency(
        vibrationValue.amplitudeLow,
        vibrationValue.frequencyLow,
        true,
        isTextPrinted);

    m_pWriter->SetTextColor(VibrationValueColor);
    DrawAmplitudeAndFrequency(
        vibrationValue.amplitudeHigh,
        vibrationValue.frequencyHigh,
        false,
        isTextPrinted);
}

void VibrationValueDrawer::DrawAxis() NN_NOEXCEPT
{
    const float Unit = DrawUnit * m_FontScale;
    m_pWriter->SetScale(m_FontScale, m_FontScale);

    for (int y = 1; y <= AxisHeight; y += 2)
    {
        float xOffsetLow = CalcXOffsetFromFrequency(FrequencyLow);
        float xOffsetHigh = CalcXOffsetFromFrequency(FrequencyHigh);
        m_pWriter->SetCursor(m_X + Unit * xOffsetLow, m_Y + Unit * y);
        m_pWriter->Print("|");
        m_pWriter->SetCursor(m_X + Unit * xOffsetHigh, m_Y + Unit * y);
        m_pWriter->Print("|");
    }

    for (int x = 0; x <= AxisWidth; x += 2)
    {
        m_pWriter->SetCursor(m_X + Unit * x, m_Y);
        m_pWriter->Print("-");
        m_pWriter->SetCursor(m_X + Unit * x, m_Y + Unit * AxisHeight / 2);
        m_pWriter->Print("-");
        m_pWriter->SetCursor(m_X + Unit * x, m_Y + Unit * AxisHeight);
        m_pWriter->Print("-");
    }
}

void VibrationValueDrawer::DrawAmplitudeAndFrequency(float amplitude, float frequency,
                                                     bool isLow, bool isTextPrinted) NN_NOEXCEPT
{
    const float Unit = DrawUnit * m_FontScale;
    float xOffset = CalcXOffsetFromFrequency(isLow ? FrequencyLow : FrequencyHigh);
    m_pWriter->SetScale(m_FontScale, m_FontScale);
    m_pWriter->SetCursor(m_X + Unit * xOffset, m_Y + Unit * (AxisHeight + 2));
    if (isTextPrinted)
    {
        m_pWriter->Print("(%4.2f, %5.1fHz)", amplitude, frequency);
    }
    xOffset = CalcXOffsetFromFrequency(frequency);
    int barHeight = static_cast<int>(amplitude * AxisHeight + 0.5f);

    for (int y = 0; y <= barHeight; y++)
    {
        m_pWriter->SetCursor(m_X + Unit * xOffset, m_Y + Unit * (AxisHeight - y));
        m_pWriter->Print("*");
    }
}
