﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "HidNpadIntegrate_PluginManager.h"
#include "HidNpadIntegrate_SixAxisSensor.h"

namespace
{

    SET_PLUGIN(SixAxisSensorDemoScene, "SixAxisSensor Demo");

    const int DeviceCountMax = 2;
    const int SixAxisSensorNpadIdCountMax = 5;
    const int SixAxisSensorNpadStyleMax = 5;

    const nn::util::Unorm8x4 ColorArray[][DeviceCountMax]  = {{Color::Red, Color::Salmon},
                                                              {Color::SteelBlue, Color::Turquoise},
                                                              {Color::Gold, Color::DarkKhaki},
                                                              {Color::LimeGreen, Color::Lime},
                                                              {Color::Orange, Color::Orange},
                                                             };

    // 描画する座標系の識別子です。
    enum CoordinateId
    {
        CoordinateId_AccelerometerXy = 0,
        CoordinateId_AccelerometerYz,
        CoordinateId_AccelerometerZx,
        CoordinateId_AngularVelocityXy,
        CoordinateId_AngularVelocityYz,
        CoordinateId_AngularVelocityZx,
        CoordinateId_AngleX,
        CoordinateId_AngleY,
        CoordinateId_AngleZ,
    };

    // 座標系の原点座標を取得します。
    void GetCoordinateOrigin(nn::util::Float2* pOutOrigin, CoordinateId id) NN_NOEXCEPT
    {
        const float X = 410;
        const float Y = 175;
        const float DeltaX = 245;
        const float DeltaY = 220;

        pOutOrigin->x = X + static_cast<int>(id) / 3 * DeltaX;
        pOutOrigin->y = Y + (static_cast<int>(id) % 3) * DeltaY;
    }

    // 座標系の名前を取得します。
    void GetCoordinateName(char* pOutAxis1,
                           char* pOutAxis2,
                           ::std::string* pOutTitle,
                           CoordinateId id) NN_NOEXCEPT
    {
        switch (id)
        {
        case CoordinateId_AccelerometerXy :
            *pOutTitle = "AccelerometerXy";
            *pOutAxis1 = 'x';
            *pOutAxis2 = 'y';
            break;
        case CoordinateId_AccelerometerYz :
            *pOutTitle = "AccelerometerYz";
            *pOutAxis1 = 'y';
            *pOutAxis2 = 'z';
            break;
        case CoordinateId_AccelerometerZx :
            *pOutTitle = "AccelerometerZx";
            *pOutAxis1 = 'z';
            *pOutAxis2 = 'x';
            break;
        case CoordinateId_AngularVelocityXy:
            *pOutTitle = "AngularVelocityXy";
            *pOutAxis1 = 'x';
            *pOutAxis2 = 'y';
            break;
        case CoordinateId_AngularVelocityYz:
            *pOutTitle = "AngularVelocityYz";
            *pOutAxis1 = 'y';
            *pOutAxis2 = 'z';
            break;
        case CoordinateId_AngularVelocityZx:
            *pOutTitle = "AngularVelocityZx";
            *pOutAxis1 = 'z';
            *pOutAxis2 = 'x';
            break;
        case CoordinateId_AngleX:
            *pOutTitle = "AngleX";
            *pOutAxis1 = ' ';
            *pOutAxis2 = ' ';
            break;
        case CoordinateId_AngleY:
            *pOutTitle = "AngleY";
            *pOutAxis1 = ' ';
            *pOutAxis2 = ' ';
            break;
        case CoordinateId_AngleZ:
            *pOutTitle = "AngleZ";
            *pOutAxis1 = ' ';
            *pOutAxis2 = ' ';
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }

    // 6軸センサーの状態を描画します。
    void WriteSixAxisSensorState(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                 const nn::hid::SixAxisSensorState& State,
                                 const float packetDropPercentage,
                                 const float OffsetX,
                                 const float OffsetY,
                                 const nn::util::Unorm8x4& PlayerColor) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        pTextWriter->SetTextColor(PlayerColor);
        pTextWriter->SetScale(0.7f, 0.7f);

        float posY = OffsetY + 20;
        const float DeltaY = 15;

        pTextWriter->SetCursor(OffsetX, posY);
        pTextWriter->Print("acceleration");
        pTextWriter->SetCursor(OffsetX + 90, posY);
        pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", State.acceleration.x,
                                                  State.acceleration.y,
                                                  State.acceleration.z);

        posY += DeltaY;
        pTextWriter->SetCursor(OffsetX, posY);
        pTextWriter->Print("angularVelocity");
        pTextWriter->SetCursor(OffsetX + 90, posY);
        pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", State.angularVelocity.x,
                                                  State.angularVelocity.y,
                                                  State.angularVelocity.z);

        posY += DeltaY;
        pTextWriter->SetCursor(OffsetX, posY);
        pTextWriter->Print("angle");
        pTextWriter->SetCursor(OffsetX + 90, posY);
        pTextWriter->Print("(%8.5f,%8.5f,%8.5f)", State.angle.x,
                                                  State.angle.y,
                                                  State.angle.z);

        posY += DeltaY;
        pTextWriter->SetCursor(OffsetX, posY);
        pTextWriter->Print("packet loss");
        pTextWriter->SetCursor(OffsetX + 90, posY);
        pTextWriter->Print("( %.2f % )", packetDropPercentage * 100.0f);
    }

    // 座標系を描画します。
    void WriteCoordinateAxes(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                             CoordinateId id) NN_NOEXCEPT
    {
        const nn::util::Unorm8x4& TextColor = Color::White;

        pTextWriter->SetTextColor(TextColor);
        pTextWriter->SetScale(0.7f, 0.7f);

        const int NumberOfCharacters = 10;
        const int Interval = 10;
        nn::util::Float2 origin;
        char axis1;
        char axis2;
        ::std::string title;
        GetCoordinateName(&axis1, &axis2, &title, id);
        GetCoordinateOrigin(&origin, id);

        for(int i = 0; i < NumberOfCharacters; i++)
        {
            pTextWriter->SetCursor(origin.x + Interval * i, origin.y);
            pTextWriter->Print("-");
            pTextWriter->SetCursor(origin.x - Interval * i, origin.y);
            pTextWriter->Print("-");

            pTextWriter->SetCursor(origin.x, origin.y + Interval * i);
            pTextWriter->Print("|");
            pTextWriter->SetCursor(origin.x, origin.y - Interval * i);
            pTextWriter->Print("|");
        }
        pTextWriter->SetCursor(origin.x + NumberOfCharacters * Interval, origin.y);
        pTextWriter->Print(">%c", axis1);
        pTextWriter->SetCursor(origin.x, origin.y - NumberOfCharacters * Interval);
        pTextWriter->Print("^%c", axis2);
        pTextWriter->SetCursor(origin.x - NumberOfCharacters * Interval,
                               origin.y - NumberOfCharacters * Interval - 1.0f);
        pTextWriter->Print("%s", title.c_str());
    }

    // 6軸センサーの状態を座標系上に描画します。
    void WriteSixAxisSensorStateFigure(nn::gfx::util::DebugFontTextWriter* pTextWriter,
                                       const nn::hid::SixAxisSensorState& State,
                                       const nn::util::Unorm8x4& PlayerColor,
                                       const int PlayerNumber) NN_NOEXCEPT
    {
        NN_ASSERT_NOT_NULL(pTextWriter);

        pTextWriter->SetTextColor(PlayerColor);
        pTextWriter->SetScale(0.8f, 0.8f);

        const float Coefficient  = 50.0f;

        nn::util::Float2 coordinateOrigin;

        // 加速度計
        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerXy);
        pTextWriter->SetCursor(coordinateOrigin.x + State.acceleration.x * Coefficient,
                               coordinateOrigin.y - State.acceleration.y * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerYz);
        pTextWriter->SetCursor(coordinateOrigin.x + State.acceleration.y * Coefficient,
                               coordinateOrigin.y - State.acceleration.z * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AccelerometerZx);
        pTextWriter->SetCursor(coordinateOrigin.x + State.acceleration.z * Coefficient,
                               coordinateOrigin.y - State.acceleration.x * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        // 角速度
        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityXy);
        pTextWriter->SetCursor(coordinateOrigin.x + State.angularVelocity.x * Coefficient,
                               coordinateOrigin.y - State.angularVelocity.y * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityYz);
        pTextWriter->SetCursor(coordinateOrigin.x + State.angularVelocity.y * Coefficient,
                               coordinateOrigin.y - State.angularVelocity.z * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngularVelocityZx);
        pTextWriter->SetCursor(coordinateOrigin.x + State.angularVelocity.z * Coefficient,
                               coordinateOrigin.y - State.angularVelocity.x * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        // 角度
        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleX);
        pTextWriter->SetCursor(
            coordinateOrigin.x + cos(2 * nn::util::FloatPi * State.angle.x) * Coefficient,
            coordinateOrigin.y - sin(2 * nn::util::FloatPi * State.angle.x) * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleY);
        pTextWriter->SetCursor(
            coordinateOrigin.x + cos(2 * nn::util::FloatPi * State.angle.y) * Coefficient,
            coordinateOrigin.y - sin(2 * nn::util::FloatPi * State.angle.y) * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);

        GetCoordinateOrigin(&coordinateOrigin, CoordinateId_AngleZ);
        pTextWriter->SetCursor(
            coordinateOrigin.x + cos(2 * nn::util::FloatPi * State.angle.z) * Coefficient,
            coordinateOrigin.y - sin(2 * nn::util::FloatPi * State.angle.z) * Coefficient);
        pTextWriter->Print("%d", PlayerNumber);
    }

    class SixAxisSensorDemo
    {
        NN_DISALLOW_COPY(SixAxisSensorDemo);
        NN_DISALLOW_MOVE(SixAxisSensorDemo);

    public:
        static const int ResetIntervalsInFrame  = 60 * 3;
        static const int UpdateIntervalsInFrame = 20;

    public:
        SixAxisSensorDemo() NN_NOEXCEPT
        {
            for(int i = 0; i < DeviceCountMax; i++)
            {
                m_Quaternion[i] = nn::util::Quaternion::Identity();
            }
        }

        void Initialize(const nn::hid::NpadIdType& Id) NN_NOEXCEPT
        {
            m_pId = &Id;
            for(int i = 0; i < GetPluginManager().GetNpadStylePluginCount(); i++)
            {
                nn::hid::NpadStyleSet NpadStyle =
                                    GetPluginManager().GetNpadStyle(i)->GetNpadStyleSet();
                m_DeviceCount[i] = nn::hid::GetSixAxisSensorHandles(m_Handle[i],
                    DeviceCountMax, *m_pId, NpadStyle);

                for(int j = 0; j < m_DeviceCount[i]; j++)
                {
                    nn::hid::StartSixAxisSensor(m_Handle[i][j]);
                }
            }
        }

        void Reset() NN_NOEXCEPT
        {
            for(int i = 0; i < m_DeviceCount[m_StyleIndex]; i++)
            {
                m_State[i].GetQuaternion(&m_Quaternion[i]);
            }
        }

        void Update() NN_NOEXCEPT
        {
            m_Style = nn::hid::GetNpadStyleSet(*m_pId);
            NpadStylePluginBase* pNpad = GetPluginManager().GetEnableNpad(m_Style);
            if(pNpad != NULL)
            {
                m_StyleIndex = pNpad->GetStyleIndex();
            }
            else
            {
                return;
            }

            NpadCommonState state = GetNpadButtonSet(*m_pId);
            if(state.buttons.Test<nn::hid::NpadButton::Plus>() ||
               state.buttons.Test<nn::hid::NpadButton::Minus>())
            {
                Reset();
            }

            for(int i = 0; i < m_DeviceCount[m_StyleIndex]; i++)
            {
                nn::hid::GetSixAxisSensorState(&m_State[i], m_Handle[m_StyleIndex][i]);

                // ResetIntervalsInFrame フレームごとに Packet Loss 向けデータを初期化します
                if ((m_FramerateCounter[i] % ResetIntervalsInFrame ) == 0)
                {
                    m_FramerateFirstTick[i] = nn::os::GetSystemTick();
                    m_FramerateFirstSample[i] = m_State[i].samplingNumber;
                }
                nn::os::Tick currentTick = nn::os::GetSystemTick() - m_FramerateFirstTick[i];
                int64_t currentSample = m_State[i].samplingNumber - m_FramerateFirstSample[i];

                // UpdateIntervalsInFrame 周期ごとに、Packet Loss を更新します
                if (m_FramerateCounter[i] % UpdateIntervalsInFrame == (UpdateIntervalsInFrame - 1))
                {
                    m_FramerateComputation[i] = currentSample /
                        (float(currentTick.GetInt64Value()) / nn::os::GetSystemTickFrequency());

                    const float ExpectedFrameRate =
                        1000.0f / m_State[i].deltaTime.GetMilliSeconds();
                    m_PacketDropPercentage[i] = 1.0f -
                        std::min(std::max( m_FramerateComputation[i] / ExpectedFrameRate, 0.0f),
                            1.0f);
                }
                m_FramerateCounter[i]++;
            }
        }

        nn::hid::SixAxisSensorState GetSixAxisSensorState(int i) NN_NOEXCEPT
        {
            return m_State[i];
        }

        int GetDeviceCount() NN_NOEXCEPT
        {
            return m_DeviceCount[m_StyleIndex];
        }

        nn::hid::NpadStyleSet GetNpadStyle() NN_NOEXCEPT
        {
            return m_Style;
        }

        nn::util::Quaternion GetQuaternion(int i) NN_NOEXCEPT
        {
            return m_Quaternion[i];
        }

        nn::hid::NpadIdType GetNpadId() NN_NOEXCEPT
        {
            return *m_pId;
        }

        float GetPacketDropPercentage(int i) NN_NOEXCEPT
        {
            return m_PacketDropPercentage[i];
        }

        void Finalize() NN_NOEXCEPT
        {
            for(int i = GetPluginManager().GetNpadStylePluginCount() - 1; i >= 0; i--)
            {
                for(int j = m_DeviceCount[i] - 1; j >= 0; j--)
                {
                    nn::hid::StopSixAxisSensor(m_Handle[i][j]);
                }
            }
        }

    private:
        nn::hid::SixAxisSensorHandle m_Handle[SixAxisSensorNpadStyleMax][DeviceCountMax];
        nn::hid::SixAxisSensorState  m_State[DeviceCountMax];
        nn::util::Quaternion         m_Quaternion[DeviceCountMax];
        int                          m_DeviceCount[SixAxisSensorNpadStyleMax];
        nn::hid::NpadStyleSet        m_Style;
        int                          m_StyleIndex;
        const nn::hid::NpadIdType*   m_pId;
        uint32_t                     m_FramerateCounter[DeviceCountMax];
        nn::os::Tick                 m_FramerateFirstTick[DeviceCountMax];
        int64_t                      m_FramerateFirstSample[DeviceCountMax];
        float                        m_FramerateComputation[DeviceCountMax];
        float                        m_PacketDropPercentage[DeviceCountMax];
    };

    SixAxisSensorDemo& GetSixAxisSensor(int i) NN_NOEXCEPT
    {
        static SixAxisSensorDemo s_SixAxisSensorStates[SixAxisSensorNpadIdCountMax];
        return s_SixAxisSensorStates[i % SixAxisSensorNpadIdCountMax];
    }

    void WriteSixAxisSensorGraphics(GraphicsSystem* pGraphicsSystem,
        const nn::hid::SixAxisSensorState& State,
        const nn::util::Unorm8x4& PlayerColor,
        const int PlayerNumber,
        const int DeviceNumber
    ) NN_NOEXCEPT
    {
        nn::gfx::ViewportScissorState* pViewportScissorState =
            &pGraphicsSystem->GetViewportScissor((PlayerNumber * 2) + DeviceNumber + 1);
        pGraphicsSystem->GetCommandBuffer().SetViewportScissorState(pViewportScissorState);


        nns::gfx::PrimitiveRenderer::Renderer* pPrimitiveRenderer =
            &pGraphicsSystem->GetPrimitiveRenderer();

        nn::util::Matrix4x3fType viewMatrix;
        nn::util::Matrix4x4fType projectionMatrix;
        nn::util::Matrix4x3f modelMatrix;

        const nn::util::Vector3fType Center = { 0.0f, 0.0f, 0.0f };
        const nn::util::Vector3fType Size = { 0.8f, 1.6f, 0.4f };
        const nn::util::Vector3fType CameraPosition = { 0.0f, -2.5f, 0.0f };
        const nn::util::Vector3fType CameraTarget = { 0.0f, 0.0f, 0.0f };
        const nn::util::Vector3fType CameraUp = { 0.0f, 0.0f, 1.0f };
        nn::util::MatrixLookAtRightHanded(&viewMatrix, CameraPosition, CameraTarget, CameraUp);
        pPrimitiveRenderer->SetViewMatrix(&viewMatrix);

        pPrimitiveRenderer->SetColor(PlayerColor);
        pPrimitiveRenderer->SetLineWidth(1.0f);

        // プロジェクションを初期化
        const float Fovy = nn::util::FloatPi / 3.0f;
        const float Aspect =
            static_cast< float >(FrameBufferWidth) / static_cast< float >(FrameBufferHeight);
        nn::util::MatrixPerspectiveFieldOfViewRightHanded(&projectionMatrix,
            Fovy, Aspect, 0.1f, 100.0f);
        pPrimitiveRenderer->SetProjectionMatrix(&projectionMatrix);

        nn::util::Vector3f vecZero;
        nn::util::VectorZero(&vecZero);

        nn::util::Quaternion currentQuaternion;
        nn::util::Quaternion releativeQuaternion;
        State.GetQuaternion(&currentQuaternion);
        releativeQuaternion =
            currentQuaternion / GetSixAxisSensor(PlayerNumber).GetQuaternion(DeviceNumber);
        modelMatrix = nn::util::MatrixRowMajor4x3f::MakeRotation(releativeQuaternion);
        nn::util::MatrixSetAxisW(&modelMatrix, vecZero);
        pPrimitiveRenderer->SetModelMatrix(&modelMatrix);

        pGraphicsSystem->GetPrimitiveRenderer().DrawCube(
            &pGraphicsSystem->GetCommandBuffer(),
            nns::gfx::PrimitiveRenderer::Surface_Wired,
            Center, Size);

        pViewportScissorState = &pGraphicsSystem->GetViewportScissor(0);
        pGraphicsSystem->GetCommandBuffer().SetViewportScissorState(pViewportScissorState);
    }

    void UpdateGraphics(nn::gfx::util::DebugFontTextWriter* pTextWriter,
        GraphicsSystem* pGraphicsSystem) NN_NOEXCEPT
    {
        WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerXy);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerYz);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AccelerometerZx);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityXy);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityYz);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AngularVelocityZx);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AngleX);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AngleY);
        WriteCoordinateAxes(pTextWriter, CoordinateId_AngleZ);
        WriteCommonGuide(pTextWriter);

        for(int i = 0; i < SixAxisSensorNpadIdCountMax; i++)
        {
            nn::hid::NpadStyleSet style = GetSixAxisSensor(i).GetNpadStyle();
            int deviceCount = 0;
            NpadStylePluginBase* pNpad = GetPluginManager().GetEnableNpad(style);
            if(pNpad != NULL)
            {
                deviceCount = GetSixAxisSensor(i).GetDeviceCount();
            }

            pTextWriter->SetTextColor(deviceCount != 0 ? ColorArray[i][0] : Color::Gray);
            pTextWriter->SetScale(0.9f, 0.9f);
            pTextWriter->SetCursor(15, 50 + 145 * static_cast<float>(i));

            if(GetSixAxisSensor(i).GetNpadId() == nn::hid::NpadId::Handheld)
            {
                pTextWriter->Print("NpadHandheld(No.4)");
            }
            else
            {
                if(pNpad != NULL)
                {
                    pTextWriter->Print("NpadID:No.%d, NpadStyle:%s", i, pNpad->GetName());
                }
                else
                {
                    pTextWriter->Print("NpadID:No.%d, NpadStyle:---", i);
                    continue;
                }
            }

            for(int j = 0; j < deviceCount; j++)
            {
                if(!GetSixAxisSensor(i).GetSixAxisSensorState(j).
                        attributes.Test<nn::hid::SixAxisSensorAttribute::IsConnected>())
                {
                    continue;
                }

                WriteSixAxisSensorState(pTextWriter,
                    GetSixAxisSensor(i).GetSixAxisSensorState(j),
                    GetSixAxisSensor(i).GetPacketDropPercentage(j),
                    25,
                    50 + 145 * static_cast<float>(i) + 60 * static_cast<float>(j),
                    ColorArray[i][j]
                );

                WriteSixAxisSensorStateFigure(pTextWriter,
                    GetSixAxisSensor(i).GetSixAxisSensorState(j),
                    ColorArray[i][j],
                    i
                );

                WriteSixAxisSensorGraphics(pGraphicsSystem,
                    GetSixAxisSensor(i).GetSixAxisSensorState(j),
                    ColorArray[i][j],
                    i,
                    j
                );
            }
        }
    }

} // namespace

SixAxisSensorDemoScene::SixAxisSensorDemoScene() NN_NOEXCEPT
{
}

SixAxisSensorDemoScene::~SixAxisSensorDemoScene() NN_NOEXCEPT
{
}

void SixAxisSensorDemoScene::InitializeScene(ApplicationHeap* pAppAllocator,
                    GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    NN_UNUSED(pAppAllocator);
    NN_UNUSED(pGraphicsSystem);

    m_SceneNum = GetPluginManager().GetCurrentSceneNum();
    m_IsRunnableAlways = true;
    m_StatusName       = "SAS";
    m_ToggleOperation  = "Right + L Stick ( A + R Stick )";

    if(!m_IsRunning)
    {
        for (int i = 0; i < SixAxisSensorNpadIdCountMax; i++)
        {
            GetSixAxisSensor(i).Initialize(NpadIds[i]);
        }
        m_IsRunning = true;
    }
}

void SixAxisSensorDemoScene::FinalizeScene(ApplicationHeap* pAppAllocator) NN_NOEXCEPT
{
    NN_UNUSED(pAppAllocator);

    if(m_IsRunning)
    {
        for (int i = SixAxisSensorNpadIdCountMax - 1; i >= 0; i--)
        {
            GetSixAxisSensor(i).Finalize();
        }
        m_IsRunning = false;
    }
}

void SixAxisSensorDemoScene::StopScene() NN_NOEXCEPT
{
}

void SixAxisSensorDemoScene::RestartScene() NN_NOEXCEPT
{
}

void SixAxisSensorDemoScene::SwitchScene() NN_NOEXCEPT
{
}

void SixAxisSensorDemoScene::ToggleProcess(const nn::hid::NpadButtonSet& Buttons,
                        ApplicationHeap* pAppAllocator,
                        GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    const int ChengeWaitTime = 500;

    // 6軸センサによる計算処理のON/OFFを切り替える
    if ((Buttons.Test<nn::hid::NpadButton::A>() &&
         Buttons.Test<nn::hid::NpadButton::StickR>()) ||
        (Buttons.Test<nn::hid::NpadButton::Right>() &&
         Buttons.Test<nn::hid::NpadButton::StickL>()))
    {
        if( m_SceneNum != GetPluginManager().GetCurrentSceneNum())
        {
            if(m_IsRunning)
            {
                FinalizeScene(pAppAllocator);
            }
            else
            {
                InitializeScene(pAppAllocator, pGraphicsSystem);
            }
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(ChengeWaitTime));
        }
    }
}

void SixAxisSensorDemoScene::RunScene(nn::gfx::util::DebugFontTextWriter& textWriter,
    GraphicsSystem*  pGraphicsSystem) NN_NOEXCEPT
{
    for (int i = 0; i < SixAxisSensorNpadIdCountMax; i++)
    {
        GetSixAxisSensor(i).Update();
    }

    UpdateGraphics(&textWriter, pGraphicsSystem);
}
