﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>

#include "HidNpadIntegrate_Color.h"
#include "HidNpadIntegrate_GraphicsSystem.h"
#include "HidNpadIntegrate_Main.h"
#include "HidNpadIntegrate_MenuBase.h"
#include "HidNpadIntegrate_PluginManager.h"
#include "HidNpadIntegrate_SettingScreen.h"

namespace
{
    SettingScreen* g_pSettingScreen;

    void WriteNpadStates(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
    {
        const float OffsetX = 780;
        const float OffsetY = 120;
        const nn::hid::NpadStyleSet NpadStyle = nn::hid::GetSupportedNpadStyleSet();

        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.3f, 1.3f);
        pTextWriter->SetCursor(OffsetX - 40, OffsetY - 40);
        pTextWriter->Print("Current Settings");

        pTextWriter->SetFixedWidthEnabled(true);
        pTextWriter->SetFixedWidth(12);

        pTextWriter->SetScale(1.0f, 1.0f);
        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->Print("SupportedNpadStyleSet");

        for(int i = 0; i < GetPluginManager().GetNpadStylePluginCount(); i++)
        {
            pTextWriter->SetCursor(OffsetX, OffsetY + 30 + 30 * i);

            NpadStylePluginBase* pNpad = GetPluginManager().GetNpadStyle(i);
            pTextWriter->Print("  %s: %s", pNpad->GetName(),
                pNpad->IsEnableNpadStyle(NpadStyle) ? "Valid" : "Invalid");
        }

        for(int i = 0; i < GetPluginManager().GetSettingApiPluginCount(); i++)
        {
            bool isDrawn = GetPluginManager().GetSettingApi(i)->
                                DrawHoldType(pTextWriter, OffsetX, OffsetY + 180);
            if(isDrawn)
            {
                break;
            }
        }

        pTextWriter->SetFixedWidthEnabled(false);

        for(int i = 0; i < NpadIdMax; i++)
        {
            pTextWriter->SetTextColor(Color::White);
            pTextWriter->SetCursor(OffsetX + 20, OffsetY + 270 + 90 * i);

            const nn::hid::NpadStyleSet CurrentStyle = nn::hid::GetNpadStyleSet(NpadIds[i]);
            NpadStylePluginBase* pNpad = GetPluginManager().GetEnableNpad(CurrentStyle);
            if(pNpad != NULL)
            {
                pTextWriter->Print("NpadStyle: %s", pNpad->GetName());
            }
            else
            {
                pTextWriter->SetTextColor(Color::Gray);
                pTextWriter->Print("NpadStyle: ---");
            }

            pTextWriter->SetCursor(OffsetX, OffsetY + 240 + 90 * i);
            pTextWriter->Print("NpadID:No.%d", i);

            for(int j = 0; j < GetPluginManager().GetSettingApiPluginCount(); j++)
            {
                bool isDrawn = GetPluginManager().GetSettingApi(j)->
                    DrawAssignment(i, pTextWriter, OffsetX + 20, OffsetY + 300 + 90 * i);
                if(isDrawn)
                {
                    break;
                }
            }
        }
    }

    void WriteSettingGuide(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
    {
        const float OffsetX = 405;
        const float OffsetY = 0;

        pTextWriter->SetTextColor(Color::White);
        pTextWriter->SetScale(1.5f, 1.5f);
        pTextWriter->SetCursor(10, 10);
        pTextWriter->Print("Change Mode and Assignment");

        pTextWriter->SetTextColor(Color::Aqua);
        pTextWriter->SetScale(0.9f, 0.9f);
        pTextWriter->SetCursor(OffsetX, OffsetY);
        pTextWriter->Print("Change Component");
        pTextWriter->SetCursor(OffsetX, OffsetY + 15);
        pTextWriter->Print("Change Value");
        pTextWriter->SetCursor(OffsetX, OffsetY + 30);
        pTextWriter->Print("Apply Setting");
        pTextWriter->SetCursor(OffsetX + 470, OffsetY);
        pTextWriter->Print("Exit Settings");
        pTextWriter->SetCursor(OffsetX + 470, OffsetY + 15);
        pTextWriter->Print("Finish Demo");

        pTextWriter->SetCursor(OffsetX + 150, OffsetY);
        pTextWriter->Print(": Up or Down ( X or B )");
        pTextWriter->SetCursor(OffsetX + 150, OffsetY + 15);
        pTextWriter->Print(": Left or Right ( Y or A )");
        pTextWriter->SetCursor(OffsetX + 150, OffsetY + 30);
        pTextWriter->Print(": L ( R )");
        pTextWriter->SetCursor(OffsetX + 580, OffsetY);
        pTextWriter->Print(": ZL + L Stick ( ZR + R Stick )");
        pTextWriter->SetCursor(OffsetX + 580, OffsetY + 15);
        pTextWriter->Print(": Minus + L Stick ( Plus + R Stick )");
    }

} // namespace

SettingScreen::SettingScreen() NN_NOEXCEPT
    : m_CurrentProcessor()
    , m_NextProcessor()
    , m_NpadId()
    , m_MenuSelection()
{
    for (int i = 0; i < NpadIdMax; i++)
    {
        m_PreviousButtonState[i] = GetNpadButtonSet(NpadIds[i]).buttons;
    }
}

void SettingScreen::Initialize() NN_NOEXCEPT
{
    for(int i = 0; i < GetPluginManager().GetSettingApiPluginCount(); i++)
    {
        GetPluginManager().GetSettingApi(i)->Initialize(&m_NextProcessor, &m_MenuSelection);
    }
}

void SettingScreen::Finalize() NN_NOEXCEPT
{
    for(int i = 0; i < GetPluginManager().GetSettingApiPluginCount(); i++)
    {
        GetPluginManager().GetSettingApi(i)->Finalize();
    }
}

void SettingScreen::Run(nn::gfx::util::DebugFontTextWriter* pTextWriter) NN_NOEXCEPT
{
    const float OffsetX = 10;
    const float OffsetY = 80;

    CheckNpadButtons();

    pTextWriter->SetTextColor(Color::White);
    pTextWriter->SetScale(1.3f, 1.3f);
    pTextWriter->SetCursor(OffsetX, OffsetY);
    pTextWriter->Print("Mode Change Function");

    pTextWriter->SetFixedWidthEnabled(true);
    pTextWriter->SetFixedWidth(12);

    DrawMenu(GetPluginManager().GetSettingApi(m_NextProcessor)->GetReadWriteMenu(),
        OffsetX + 40, OffsetY + 40, GetMenuIndex(), pTextWriter);

    pTextWriter->SetFixedWidthEnabled(false);

    GetPluginManager().GetSettingApi(m_NextProcessor)->DrawCaution(pTextWriter, 50, 310);
    GetPluginManager().GetSettingApi(m_NextProcessor)->DrawResult(pTextWriter, 50, 340);

    GetPluginManager().GetSettingApi(m_NextProcessor)->Explain(pTextWriter, 50, 400);
    DrawRect(pTextWriter, 40, 380, 80, 40);
}

void SettingScreen::CheckNpadButtons() NN_NOEXCEPT
{
    // 押されたボタンを検出
    for(int i = 0; i < NpadIdCountMax; i++)
    {
        nn::hid::NpadButtonSet currentButtonState;
        nn::hid::NpadButtonSet buttonDownSet;

        currentButtonState = GetNpadButtonSet(NpadIds[i]).buttons;
        buttonDownSet = currentButtonState & ~m_PreviousButtonState[i];
        m_PreviousButtonState[i] = currentButtonState;

        // Menu Item を選ぶ
        if ((buttonDownSet & nn::hid::NpadButton::B::Mask).IsAnyOn() ||
            (buttonDownSet & nn::hid::NpadButton::Down::Mask).IsAnyOn())
        {
            GetPluginManager().GetSettingApi(m_NextProcessor)->ResetResult();
            m_MenuSelection += 1;
        };

        if ((buttonDownSet & nn::hid::NpadButton::X::Mask).IsAnyOn() ||
            (buttonDownSet & nn::hid::NpadButton::Up::Mask).IsAnyOn())
        {
            GetPluginManager().GetSettingApi(m_NextProcessor)->ResetResult();
            m_MenuSelection -= 1;
        };

        // Menu の設定値を変更する
        if ((buttonDownSet & nn::hid::NpadButton::Y::Mask).IsAnyOn()    ||
            (buttonDownSet & nn::hid::NpadButton::A::Mask).IsAnyOn()    ||
            (buttonDownSet & nn::hid::NpadButton::Left::Mask).IsAnyOn() ||
            (buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn())
        {
            GetPluginManager().GetSettingApi(m_NextProcessor)->ResetResult();
            int8_t delta = ((buttonDownSet & nn::hid::NpadButton::A::Mask).IsAnyOn() ||
                            (buttonDownSet & nn::hid::NpadButton::Right::Mask).IsAnyOn()) ?
                                1 : -1;

            std::vector<MenuItem> menu = GetPluginManager().GetSettingApi(m_NextProcessor)->GetReadWriteMenu();
            const int MenuIndex = GetMenuIndex();
            if (MenuIndex >= 0)
            {
                menu[MenuIndex].Increment(delta);
            }
        }

        // イベントトリガー
        if ((buttonDownSet & nn::hid::NpadButton::R::Mask).IsAnyOn() ||
            (buttonDownSet & nn::hid::NpadButton::L::Mask).IsAnyOn())
        {
            GetPluginManager().GetSettingApi(m_NextProcessor)->ResetResult();
            GetPluginManager().GetSettingApi(m_NextProcessor)->Start();
            m_CurrentProcessor = m_NextProcessor;
            const int WaitTime = 500;
            ::nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(WaitTime));
        }
    }
}

int SettingScreen::GetMenuIndex() const NN_NOEXCEPT
{
    const std::vector<MenuItem> Menu = GetPluginManager().GetSettingApi(m_NextProcessor)->GetReadWriteMenu();
    int size = static_cast<int>(Menu.size());
    int menuIndex = ((m_MenuSelection % size) + size) % size;

    NN_ASSERT(menuIndex >= 0);
    NN_ASSERT(menuIndex < size);

    return menuIndex;
}

void InitializeSettingScreen() NN_NOEXCEPT
{
    g_pSettingScreen = new SettingScreen();

    g_pSettingScreen->Initialize();
}

void FinalizeSettingScreen() NN_NOEXCEPT
{
    g_pSettingScreen->Finalize();

    delete g_pSettingScreen;
}

void SettingScreenMain(nn::gfx::util::DebugFontTextWriter& textWriter) NN_NOEXCEPT
{
    g_pSettingScreen->Run(&textWriter);

    WriteNpadStates(&textWriter);
    WriteSettingGuide(&textWriter);
}
